// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "fitsimage.h"
#include "util.h"

#include "NaN.h"

void FitsImage::setBinCursor(const Vector& v)
{
  histCursor = v * userToPhysical;
}

void FitsImage::setBinFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] *= bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] *= bb[1] <= 0 ? 1 : bb[1];
}

void FitsImage::setBinToFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] = bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] = bb[1] <= 0 ? 1 : bb[1];
}

Matrix FitsImage::updateHistCenter()
{
  if (hist_)
    return nextHist(getHistCenter());
  else
    return Matrix();
}

Matrix FitsImage::updateHistCursor()
{
  if (hist_)
    return nextHist(histCursor);
  else
    return Matrix();
}

Matrix FitsImage::updateHist(const Vector& v)
{
  if (hist_)
    return nextHist(v * userToPhysical);
  else
    return Matrix();
}

Matrix FitsImage::nextHist(const Vector& c)
{
  // remember where we are pointing
  histCursor = c;

  // cursor is in bin (physical) coords
  //  Vector s = getHistDim()/binFactor_;
  Vector d = getHistDim();
  Vector s;
  s[0] = d[0]/binFactor_[0];
  s[1] = d[1]/binFactor_[1];

  int width = (int)(s[0]<binBufferSize_? s[0] : binBufferSize_);
  int height= (int)(s[1]<binBufferSize_? s[1] : binBufferSize_);
  int depth = binDepth_;

  Vector center = Vector(width, height)/2;

  if (DebugBin) {
    cerr << "width height: " << width << ' ' << height << endl;
    cerr << "center: " << center << endl;
    cerr << "center.ceil(): " << center.ceil() << endl;
  }

  if (binFactor_[0]<1 || binFactor_[1]<1) {
    actualHistCursor = histCursor;

    if (DebugBin)
      cerr << "histCursor: " << histCursor << endl;
  }
  else {
    // force to a bin boundary, then translate to center of bin cell
    //    actualHistCursor = ((histCursor/binFactor_).floor() * binFactor_) +
    //      Vector(.5,.5);
    actualHistCursor[0] = (floor(histCursor[0]/binFactor_[0]) * binFactor_[0])
      +  .5;
    actualHistCursor[1] = (floor(histCursor[1]/binFactor_[1]) * binFactor_[1])
      +  .5;

    // now, figure out any offset due to mod(lowerleft,binFactor_)
    FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
    Vector xd = hdu->dimension(fits_->pBinX());
    Vector yd = hdu->dimension(fits_->pBinY());
    Vector ll(xd[0],yd[0]);

    //    Vector a = ((ll/binFactor_).floor() * binFactor_) + Vector(.5,.5);
    Vector a;
    a[0] = (floor(ll[0]/binFactor_[0]) * binFactor_[0]) + .5;
    a[1] = (floor(ll[1]/binFactor_[1]) * binFactor_[1]) + .5;
    Vector offset = a-ll;
    actualHistCursor -= offset;

    if (DebugBin) {
      cerr << "histCursor: " << histCursor << endl;
      cerr << "actualHistCursor: " << actualHistCursor << endl;
      cerr << "ll: " << ll << endl;
      cerr << "offset: " << offset << endl;
    }
  }

  // new physicalToData
  Matrix mm = 
    Translate(-actualHistCursor) * 
    Scale(1./binFactor_[0],1./binFactor_[1]) *
    Translate(center.ceil());

  if (DebugBin)
    cerr << "mm: " << mm << endl << endl;

  if (hist_)
    delete hist_;

  hist_ = new FitsHist(fits_, width, height, depth, mm,
		       binFunction_, binFactor_);

  // reset LTMV keywords
  keyLTMV = 0;

  if (!hist_->isValid()) {
    reset();
    return Matrix();
  }

  load();

  return refToPhysical * mm * dataToRef;
}

Vector FitsImage::getHistColMinMax(const char* col)
{
  if (fits_->isTable())
    return fits_->getColMinMax(col);
  else
    return Vector();
}

Vector FitsImage::getHistDim()
{
  // assumes we aready have our columns

  FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
  Vector xd = hdu->dimension(fits_->pBinX());
  Vector yd = hdu->dimension(fits_->pBinY());

  // if DBL_MAX, we will get nan
  Vector r(xd[1]-xd[0],yd[1]-yd[0]);

  if (isnand(r[0]) || isnand(r[1]))
    return Vector(DBL_MAX,DBL_MAX);
  else
    return r;
}

Vector FitsImage::getHistCenter()
{
  // assumes we aready have our columns

  FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
  Vector xd = hdu->dimension(fits_->pBinX());
  Vector yd = hdu->dimension(fits_->pBinY());

  // if DBL_MAX, we will get nan
  Vector r = Vector(xd[1]-xd[0],yd[1]-yd[0])/2 + Vector(xd[0],yd[0]);

  if (isnand(r[0]) || isnand(r[1]))
    return Vector();
  else
    return r;
}

const char* FitsImage::getHistFilter()
{
  if (fits_)
    return fits_->pFilter();
  else
    return NULL;
}

const char* FitsImage::getHistX()
{
  return fits_ ? fits_->pBinX() : NULL;
}

const char* FitsImage::getHistY()
{
  return fits_ ? fits_->pBinY() : NULL;
}

const char* FitsImage::getHistZ()
{
  return fits_ ? fits_->pBinZ() : NULL;
}

char* FitsImage::getHistList()
{
  if (fits_ && fits_->isTable()) {
    FitsHead* head = fits_->head();
    return ((FitsTableHDU*)head->hdu())->list();
  }
  else
    return NULL;
}

Vector FitsImage::getHistZlim()
{
  return fits_ ? fits_->getBinZlim() : Vector();
}

void FitsImage::setBinX(const char* str)
{
  if (fits_)
    fits_->setpBinX(str);
}

void FitsImage::setBinY(const char* str)
{
  if (fits_)
    fits_->setpBinY(str);
}

void FitsImage::setBinZ(const char* str)
{
  if (fits_)
    fits_->setpBinZ(str);
}

void FitsImage::setBinFilter(const char* filter)
{
  if (fits_)
    fits_->setpFilter(filter);
}

void FitsImage::setBinSliceFilter(const char* filter)
{
  if (fits_)
    fits_->setpSliceFilter(filter);
}

void FitsImage::setBinZlim(const Vector& lim)
{
  if (fits_)
    fits_->setBinZlim(lim);
}

