// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "line.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Line::Line(const Line& a) : BaseLine(a)
{
  p1Arrow = a.p1Arrow;
  p2Arrow = a.p2Arrow;
}

Line::Line(FrameBase* p, const Vector& ptr1, const Vector& ptr2,
	   int a1, int a2,
	   const char* clr, int wth, const char* fnt, const char* txt,
	   unsigned short prop, const char* cmt, 
	   const List<Tag>& tg, const List<CallBack>& cb) 
  : BaseLine(p, ptr1, ptr2, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  p1Arrow = a1;
  p2Arrow = a2;

  strcpy(type,"line");
  handle = new Vector[2];
  numHandle = 2;

  updateBBox();
}

void Line::updateBBox()
{
  center = (p2-p1)/2 + p1;
  angle = (p2-p1).angle();
  if (parent->isIIS())
    angle = -angle;

  // bound marker
  bbox = BBox(p1 * parent->refToCanvas, p2 * parent->refToCanvas);

  // generate handles in canvas coords
  handle[0] = p1 * parent->refToCanvas;
  handle[1] = p2 * parent->refToCanvas;

  // make room for handles/arrows
  bbox.expand(5);

  // calculate overall bbox
  calcAllBBox();
}

void Line::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;

  ostringstream str;
  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  if (p1Arrow)
    psArrow(a, (b-a).normalize());

  if (p2Arrow)
    psArrow(b, (a-b).normalize());
}

int Line::isIn(const Vector& v)
{
  // do this in canvas coords, not ref coords

  Vector l1 = p1*parent->refToCanvas;
  Vector l2 = p2*parent->refToCanvas;
  double a = (l2-l1).angle();

  Matrix m = Translate(-l1) * Rotate(a);
  Vector vv = v*m;
  Vector end = l2*m;
  return (vv[0]>0 && vv[0]<end[0] && vv[1]>-6 && vv[1]<6);
}

void Line::setArrows(int w1, int w2)
{
  p1Arrow = w1 ? 1 : 0;
  p2Arrow = w2 ? 1 : 0;
  updateBBox();
}

// Private

void Line::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector a = (p1 * mx).round();
  Vector b = (p2 * mx).round();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)b[0], (int)b[1]);

  if (p1Arrow)
    renderArrow(drawable, a, (b-a).normalize());

  if (p2Arrow)
    renderArrow(drawable, b, (a-b).normalize());
}

// list

void Line::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v1 = ptr->mapFromRef(p1,sys);
      Vector v2 = ptr->mapFromRef(p2,sys);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v1 = ptr->mapFromRef(p1,sys,sky);
	    Vector v2 = ptr->mapFromRef(p2,sys,sky);
	    str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		<< ',' << v2[0] << ',' << v2[1] << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    char ra1[16], ra2[16];
	    char dec1[16], dec2[16];
	    {
	      ptr->mapFromRef(p1,sys,sky,format,buf,64);
	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra1 >> dec1;
	    }
	    {
	      ptr->mapFromRef(p2,sys,sky,format,buf,64);
	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra2 >> dec2;
	    }
	    str << type << '(' << ra1 << ',' << dec1 << ',' 
		<< ra2 << ',' << dec2 << ')';
	  }
	  break;
	}
      }
      else {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	    << ',' << v2[0] << ',' << v2[1] << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Line::listPost(ostream& str, int conj, int strip)
{
  // no props for semicolons
  if (!strip) {
    if (conj)
      str << " ||";

    str << " # line=" << p1Arrow << ' ' << p2Arrow;
    listProperties(str, 0);
  }
  else {
    if (conj)
      str << "||";
    else
      str << ';';
  }
}

void Line::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOtngPre(str, strip);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v1 = ptr->mapFromRef(p1,IMAGE);
      Vector v2 = ptr->mapFromRef(p2,IMAGE);
      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	  << v2[0] << ',' << v2[1] << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v1 = ptr->mapFromRef(p1,sys,sky);
	  Vector v2 = ptr->mapFromRef(p2,sys,sky);
	  str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	      << v2[0] << ',' << v2[1] << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  char ra1[16], ra2[16];
	  char dec1[16], dec2[16];
	  {
	    ptr->mapFromRef(p1,sys,sky,format,buf,64);
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra1 >> dec1;
	  }
	  {
	    ptr->mapFromRef(p2,sys,sky,format,buf,64);
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra2 >> dec2;
	  }
	  str << type << '(' << ra1 << ',' << dec1 << ',' 
	      << ra2 << ',' << dec2 << ')';
	}
	break;
      }

    }
  }

  listSAOtngPost(str, strip);
}
