// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_BitStringLiteral.cc,v 1.2 1999/03/09 20:53:09 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "IIRScram_BitStringLiteral.hh"
#include "set.hh"
#include "dl_list.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include <strstream.h>

IIRScram_BitStringLiteral::~IIRScram_BitStringLiteral() {}

void
IIRScram_BitStringLiteral::_publish_cc() {
  IIR_TypeDefinition *type = _get_subtype();
  //Work Around till this node is resolved properly
  if (type != NULL) {
    ASSERT(type->_is_resolved());
    
    if (type->_is_anonymous() == TRUE) {
      if (type->_get_base_type() != NULL) {
	type = type->_get_base_type();
      }
      else {
	if (type->_get_type_mark() != NULL) {
	  ASSERT ( type->_get_type_mark()->_is_anonymous() == FALSE );
	  type = type->_get_type_mark();
	}
      }
    }

    type->_publish_cc_type_name();
  }
  else {
    cerr << "Work Around for IIR_BitStringLiteral: Assumes the type to be bit_vector " << endl;
    _cc_out << "Savantbit_vectorType";
  }
  
  _cc_out << "(ObjectBase::VARIABLE, ";
  //Work Around again
  if (type != NULL) {
    if ((type->_is_anonymous() == TRUE) || (type->_is_unconstrained_array_type() == TRUE)) {
      _publish_cc_range();
      _cc_out << ", ";
    }
  }
  
  _publish_cc_initialization_value();
  if ((type->_is_unconstrained_array_type() == TRUE) &&
      (type->_get_element_subtype()->_is_scalar_type() == TRUE)) {
    type->_get_element_subtype()->_publish_cc_object_type_info();
  }
  _cc_out << ")";
}

void
IIRScram_BitStringLiteral::_publish_cc_initialization_value() {
  IIR_Int32 length = get_text_length();
  IIR_Char base_specifier = (*this)[0];
  IIR_Char *string;

  switch(base_specifier) {
  case 'b':
  case 'B':
    // We have to new the string here since this will be deleted at the end.
    string = new char[length+1];
    strncpy(string, get_text(),length);
    string[length] = '\0';
    break;
  case 'o':
  case 'O':
    string = _convert_octal_to_binary();
    length = strlen(string);
    break;
  case 'x':
  case 'X':
    string = _convert_hex_to_binary();
    length = strlen(string);
    break;
  default:
    cerr << "IIRScram_BitStringLiteral::_publish_cc_initialization_value():"
	 << " unrecognized base specifier `" << base_specifier << "'" << endl;
    abort();
  }

  _cc_out << "\"";
  for (register IIR_Int32 i = 0; i < length; i++) { 
    if(string[i] == '1' || string[i] == '0') {
      _cc_out << string[i];
    }
  }
  _cc_out << "\"";

  delete [] string;		// The caller of get_text() has to deallocate.
}

IIR_Char*
IIRScram_BitStringLiteral::_convert_octal_to_binary() {
  ostrstream bitStream;
  IIR_Char* bitString;
  register IIR_Int32 i;
  IIR_Int32 length = get_text_length();

  // Omit the first character since it is B, X, or O
  for(i = 1; i < length; i++) {
    switch((*this)[i]) {
    case '0':
      bitStream << "000"; break;
    case '1':
      bitStream << "001"; break;
    case '2':
      bitStream << "010"; break;
    case '3':
      bitStream << "011"; break;
    case '4':
      bitStream << "100"; break;
    case '5':
      bitStream << "101"; break;
    case '6':
      bitStream << "110"; break;
    case '7':
      bitStream << "111"; break;
    default:
      // The code-generator assumes that all the error-checking has been
      // done by the front-end, and hence no error is reported here.
      break;
    } // switch
  } // for
  bitStream << ends;
  // The return memory is newed here.  It is the caller's responsibility
  // to deallocate this memory.
  bitString = bitStream.str();
  return bitString;
}
      
IIR_Char*
IIRScram_BitStringLiteral::_convert_hex_to_binary() {
  ostrstream bitStream;
  IIR_Char* bitString;
  register IIR_Int32 i;
  IIR_Int32 length = get_text_length();

  // Omit the first character since it is B, X, or O
  for(i = 1; i < length; i++) {
    switch((*this)[i]) {
    case '0':
      bitStream << "0000"; break;
    case '1':
      bitStream << "0001"; break;
    case '2':
      bitStream << "0010"; break;
    case '3':
      bitStream << "0011"; break;
    case '4':
      bitStream << "0100"; break;
    case '5':
      bitStream << "0101"; break;
    case '6':
      bitStream << "0110"; break;
    case '7':
      bitStream << "0111"; break;
    case '8':
      bitStream << "1000"; break;
    case '9':
      bitStream << "1001"; break;
    case 'a':
    case 'A':
      bitStream << "1010"; break;
    case 'b':
    case 'B':
      bitStream << "1011"; break;
    case 'c':
    case 'C':
      bitStream << "1100"; break;
    case 'd':
    case 'D':
      bitStream << "1101"; break;
    case 'e':
    case 'E':
      bitStream << "1110"; break;
    case 'f':
    case 'F':
      bitStream << "1111"; break;
    default:
      // The code-generator assumes that all the error-checking has been
      // done by the front-end, and hence no error is reported here.
      break;
    } // switch
  } // for
  bitStream << ends;
  // The return memory is newed here.  It is the caller's responsibility
  // to deallocate this memory.
  bitString = bitStream.str();
  return bitString;
}
IIR_Int32
IIRScram_BitStringLiteral::_get_effective_length() {
  IIR_Int32 length = _get_num_digits();
  IIR_Char base_specifier = (*this)[0];
  
  // Get the length of the string as a bit string.
  switch(base_specifier) {
  case 'b':
  case 'B':
    break;
  case 'o':
  case 'O':
    length *= 3;		// One octal digit = 3 bits.
    break;
  case 'x':
  case 'X':
    length *= 4;		// One hex digit = 4 bits.
    break;
  default:
    cerr << "IIRScram_BitStringLiteral::_publish_cc_range():"
	 << " unrecognized base specifier `" << base_specifier << "'" << endl;
    abort();
  }
  return length;
}
      
void
IIRScram_BitStringLiteral::_publish_cc_range() {
  IIR_ScalarTypeDefinition *index_type = NULL;
  IIR_ArrayTypeDefinition *array_type = NULL;

  if ( _get_subtype()->_is_array_type() == TRUE && _get_subtype()->_is_single_dimensional_array_type() == TRUE ){
    array_type = (IIR_ArrayTypeDefinition *)_get_subtype();
    index_type = array_type->get_index_subtype();
    if ( index_type != NULL && index_type->get_left() != NULL && index_type->get_right() != NULL ){
      index_type->get_left()->_publish_cc_value();
      if ( index_type->_is_ascending_range() == TRUE){
	_cc_out << ", to, ";
      } else {
	_cc_out << ", downto, ";
      }
      index_type->get_right()->_publish_cc_value();
    } else {
      IIR_Int32 length = _get_effective_length();
      _cc_out << "0, to, " << length-1;	
    }
  } else {
    IIR_Int32 length = _get_effective_length();
    _cc_out << "0, to, " << length-1;	
  }
  //  IIR_Int32 length = _get_effective_length();
  //  _cc_out << "0, to, " << length-1;	
}

void
IIRScram_BitStringLiteral::_publish_cc_bounds() {
  IIR_ScalarTypeDefinition *index_type = NULL;
  IIR_ArrayTypeDefinition  *array_type = NULL;
  
  if ( _get_subtype()->_is_array_type() == TRUE && _get_subtype()->_is_single_dimensional_array_type() == TRUE ){
    array_type = (IIR_ArrayTypeDefinition *)_get_subtype();
    index_type = array_type->get_index_subtype();
    if (index_type != NULL && index_type->get_left() != NULL && index_type->get_right() != NULL) {
      _cc_out << "ArrayInfo(";
      index_type->get_left()->_publish_cc_value();
      if ( index_type->_is_ascending_range() == TRUE ){
	_cc_out << ", to, ";
      } else {
	_cc_out << ", downto, ";
      }
      index_type->get_right()->_publish_cc_value();
      _cc_out << ")";
    } else {
      _cc_out << "nullInfo";
    }
  } else {
    _cc_out << "nullInfo";
  }
  
  // should publish &ArrayInfo((int)0, to, (int)lenght-1)
  //IIR_Int32 rbound = _get_effective_length() - 1;
  //_cc_out << "&ArrayInfo((int)0, to, (int)" << rbound << ")";
}

void
IIRScram_BitStringLiteral::_publish_cc_state_object_init() {
  _cc_out << " ";
  if(_get_subtype()->_is_iir_access_type_definition() != TRUE) {
    _cc_out << "*";
  }
  _cc_out << "( new ";
  _publish_cc();
  _cc_out << ")";
}

void
IIRScram_BitStringLiteral::_publish_cc_headers() {
  if(_get_subtype() != NULL) {
    _get_subtype()->_publish_cc_headers();
  } else {
    _cc_out << "#include \"Savantbit_vectorType.hh\"" << endl;
  }
}

// This method counts the number of digits in the string.  It assumes that
// the string is made of a base specifier, a quote, a sequence of digits
// and ending with a quote.  Further, the string within the quotes is made
// of *valid* digits and the '_' character only.
IIR_Int32
IIRScram_BitStringLiteral::_get_num_digits() {
  IIR_Int32 length = get_text_length();
  IIR_Int32 num_digits = 0;

  // We omit the bit specifier and the two quotes.
  for(register IIR_Int32 i = 2; i < length-1; i++) {
    if((*this)[i] != '_') {
      num_digits++;
    }
  }
  return num_digits;
}

set<IIR_TypeDefinition> *
IIRScram_BitStringLiteral::_get_rval_set( IIR_Boolean (IIR::*constraint_function)() ){
  return _get_rval_set_for_string();
}

void
IIRScram_BitStringLiteral::_type_check( set<IIR_TypeDefinition> * ){
  // do nothing...
}

