// Copyright (c) 1995-1999 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey    phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer	tmcbraye@ececs.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: main.cc,v 1.7 1999/09/22 00:32:01 tmcbraye Exp $
// 
//---------------------------------------------------------------------------
#include <signal.h>
#include <ctype.h>
#include <fstream.h>
#include "version.hh"
#include "savant.hh"
#include "scram.hh"
#include "IIR_DesignFile.hh"
#include "IIR_DesignFileList.hh"
#include "IIR_LibraryUnit.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "arg_parser.hh"
#include "symbol_table.hh"
#include "dl_list.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_Identifier.hh"

// temporary elaboration info stuff
#include "elaborate_info.hh"
elaborate_info elab_info;

// class description of file descriptors
#include "switch_file.hh"

// to invoke the library manager
#include "library_manager.hh"

// These are global flags.
bool debug_symbol_table;
bool publish_vhdl;
bool publish_cc;
bool no_file_output;
bool print_warranty;
bool keep_design_units;
bool ignore_standard_library;
bool no_mangling;
bool verbose_output;

// for library manager
char *design_library_name;

dl_list<char> lib_list;

#ifdef PROCESS_COMBINATION
bool static_elaborate;
char *entity_name;
char *arch_name;
char *config_name;
char *partition_algorithm;
int num_partitions;

// This method lowercases a null-terminated string by walking through the
// argument and calling tolower on each character.
static void
lowercase(char* string) {
  while (*string != '\0') {
    *string = tolower(*string);
    string++;
  }
}
#endif

#ifdef PROCESS_GRAPH
bool signal_graph;
#endif

static arg_parser::arg_record arg_list[] = {
  //  {"-ignore-standard-library","do not parse the VHDL standard library",   &ignore_standard_library, arg_parser::BOOLEAN}, 
  {"-debug-symbol-table","print out debugging info relating to symbol table",
   &debug_symbol_table, arg_parser::BOOLEAN}, 
  {"-design-library-name", "design library name", &design_library_name, 
   arg_parser::STRING},
  {"-publish-vhdl","publish VHDL", &publish_vhdl, arg_parser::BOOLEAN},
  {"-publish-cc","publish c++", &publish_cc, arg_parser::BOOLEAN},
  {"-no-file-output", "send publish_cc output to stdout instead of files",
   &no_file_output, arg_parser::BOOLEAN},
  {"-warranty-info", "print information about (lack of) warranty",
   &print_warranty, arg_parser::BOOLEAN},
#ifdef PROCESS_COMBINATION
  {"-static-elaborate", "statically elaborate the specified design",  &static_elaborate, arg_parser::BOOLEAN},
  {"-partitions", "specify # of partitions for elaborated design",  &num_partitions, arg_parser::INTEGER},
  {"-entity", "specify entity to elaborate", &entity_name, arg_parser::STRING},
  {"-arch", "specify architecture to elaborate", &arch_name, arg_parser::STRING},
  {"-config", "specify configuration to use (if needed)", &config_name, arg_parser::STRING},
  {"-alg", "specify static partitioning algorithm to use", &partition_algorithm, arg_parser::STRING},
#endif
#ifdef PROCESS_GRAPH
  {"-publish-process-graph", "publish process graph information", &signal_graph, arg_parser::BOOLEAN},
#endif
  {"-v", "verbose output", &verbose_output, arg_parser::BOOLEAN },
    // {"-lib", "specify libraries to use.  Multiple libraries allowed--first specified is WORK", &lib_list, arg_parser::STRING_LIST},
    //  {"-keep-design-units","all design units remain visible when parsing multiple files", &keep_design_units, arg_parser::BOOLEAN},
    // {"-no-mangling", "supress mangling while publishing c++", &no_mangling, arg_parser::BOOLEAN},
  {NULL, NULL}
};

// global flag for parse errors.  Parse errors include semantic errors
// like undefined symbols and such
bool parse_error = false;

// Right now, the error handling in scram is more or less
// non-existant.  That is to say, if a symbol is undefined then there
// is a very good chance that after reporting the error, it will core
// dump.  The current plan is to implement error handling using C++
// exceptions.  However, until then, we'll just catch seg faults, and
// check to see if the global parse_error flag is set.  If so, we'll
// just exit - otherwise, we'll let the process core dump.
void 
catch_core_dumps( int ){
  if( parse_error == true ){
    exit(-1);
  }
  else{
    signal( SIGSEGV, SIG_DFL );  
  }
}


void 
help_func(){
    cerr << SAVANT_VERSION << "\n\n";
    cerr <<"THE UNIVERSITY OF CINCINNATI (UC) MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT\nTHE SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT\nNOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A\nPARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY\nDAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING RESULT OF USING, MODIFYING\nOR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.\n";
    cerr << "\nReport problems, comments, etc. to \"savant@ececs.uc.edu\".\n\n";
}

#if defined(__GNUG__)
#define IOS_BIN ios::bin
#else
#define IOS_BIN ios::binary
#endif

#ifdef DEVELOPER_ASSERTIONS
#define _CC_OUT _cc_out
#else
#define _CC_OUT IIRScram::_cc_out
#endif	  

int 
main (int argc, char *argv[]) {
  signal( SIGSEGV, catch_core_dumps );

  // here are defaults for global argument variables
  debug_symbol_table = false;
  publish_vhdl = false;
  publish_cc = false;
  no_file_output = false;
  print_warranty = false;
  keep_design_units = false;
  ignore_standard_library = false;
  no_mangling = false;
  verbose_output = false;

#ifdef PROCESS_COMBINATION
  design_library_name = entity_name = arch_name = config_name = NULL;
  partition_algorithm = NULL;
  num_partitions = 1;
  static_elaborate = false;
#endif

  arg_parser ap( arg_list, help_func );
  ap.check_args(argc, argv);

  if( argc <= 1 ){
    ap.print_usage( argv[0] );
    exit( -1 );
  }

#ifdef PROCESS_COMBINATION
  if (entity_name != NULL && arch_name != NULL) {
    lowercase(entity_name);
    lowercase(arch_name);
     cerr << "Design unit to elaborate is ";
     if (config_name != NULL) {
       lowercase(config_name);
       cerr << "configuration " << config_name << " of ";
     }
     cerr << entity_name << "(" << arch_name << ")\n";
  }
  else if (entity_name == NULL && arch_name == NULL) {
    // do nothing
  }
  else {
    cerr << "You must specify both an entity and an architecture\n";
    exit(1);
  }

  if ( 1 < num_partitions ) {
    cerr << num_partitions << " partitions specified.\n";
  }

  if ( NULL != partition_algorithm ) {
    cerr << "'" << partition_algorithm << "' specified for a "
	 << "partitioning algorithm.\n";
  }

  if (static_elaborate == true && (entity_name == NULL || arch_name == NULL)) {
    cerr << "Error: you must specify a top-level entity to elaborate" << endl;
    exit(1);
  }
#endif

  library_manager *libMgr;
  if(design_library_name != NULL){
    libMgr = new library_manager( design_library_name ); 
  }  
  else{
    libMgr = new library_manager( "work" ); 
  }
    
  if (lib_list.num_elements() != 0) {
     cerr << "Libraries specified: ";
     char* lib = lib_list.first();
     while (lib != NULL) {
        cerr << lib << " ";
        lib = lib_list.successor(lib);
     }
     cerr << "\n";
  }
    
  if( print_warranty == true ){
    help_func();
    exit( -1 );
  }

  IIR_DesignFileList *iir_design_files_processed = NULL;    
  scram parser( libMgr );

  if(argc > 1) {

    iir_design_files_processed = parser.parse_files( argc, argv );  

    if( parse_error == FALSE ){
      cerr << "Parse complete - no errors." << endl;
    }
    else{
      exit( -1 );
    }
  }
  
  if( iir_design_files_processed != NULL ){
    // Now we'll walk the list, and publish if we're supposed to.
    // stream for the publishing vhdl
    switch_file _vhdl_out(1);

    IIR_DesignFile *to_publish = iir_design_files_processed->first();
    while( to_publish != NULL ){
#ifdef PROCESS_COMBINATION
      IIR_ArchitectureDeclaration *elaborated_design = NULL;
      if( static_elaborate == true ) {
	cerr << "Starting elaboration of specified design...\n";
	elaborated_design = (IIR_ArchitectureDeclaration*)to_publish->_static_elaborate_design();
	if (elaborated_design == NULL) {
	  cerr << "Specified design not found in " 
	       << (to_publish->get_file_name()) << "\n";
	  to_publish = iir_design_files_processed->successor( to_publish );
	  continue;
	}
	ASSERT(elaborated_design->get_kind() == IIR_ARCHITECTURE_DECLARATION);
	cerr << "Elaboration complete.\n";

	elaborated_design->_combine();
	cerr << "Process combination complete.\n";
      }
#endif
      if (publish_vhdl == true) {
	to_publish->_publish_vhdl(_vhdl_out);
	_vhdl_out.flush();
      }
	
      if (publish_cc == true) {
	_CC_OUT.set_file_output( !no_file_output );
	cerr << "Starting C++ code generation..." << endl;
	to_publish->_publish_cc();
	_CC_OUT.flush();
	cerr << "Code generation finished successfully." << endl;
      }

      to_publish = iir_design_files_processed->successor( to_publish );
    }
  }
  _CC_OUT.reset();
  return 0;
}

