module Schleuder
  # Represents a single list: its name, its config, its members, its logger.
  class List
    # Name of this list. Must match the name of the subdirectory in
    # +SchleuderConfig::lists_dir+
    attr_reader :listname
    # Provides the SchleuderLogger instance to log through
    attr_reader :log
    
    # Prepare some variables, set up the SchleuderLogger and set GNUPGHOME
    def initialize(listname,newlist=false)
      @listname = listname
      @listdir = File.expand_path(File.join(Schleuder.config.lists_dir, listname))
      file = File.join(@listdir, Schleuder.config.lists_logfile)
      @log = SchleuderLogger.new("Schleuder #{listname}", file)
      
      # setting GNUPGHOME to list's home, to make use of the keys there
      @log.debug "setting ENV[GNUPGHOME] to #{@listdir}"
      ENV["GNUPGHOME"] = @listdir
      @_members = nil
      _load_config(false) if newlist
     
      # This works but is ugly: the log.debug()-statements in config() won't be
      # logged regardless of the loglevel set in the config (as the
      # SchleuderLogger uses WARN as default).
      @log.sev_threshold = eval("Logger::" + config.loglevel.upcase)

      # delete out-of-age logfiles. Do it here and not in
      # schleuder_logger because that one also cares for the global log
      removed = File.unlink(*Dir["#{file}.20[0-9]*"].sort.slice(0..-self.config.log_rotate_keep))
      @log.info "Removed #{removed} old logfiles" if removed > 0
    end
    
    # Provides an array of Schleuder::Member's, read from +members.conf+
    def members
      unless @_members
        file = File.join(@listdir, Schleuder.config.lists_memberfile)
        @log.debug("reading #{file}")
        @_members = []
        YAML::load_file(file).each do |h|
          unless h.kind_of? Schleuder::Member
            h = Schleuder::Member.new(h)
          end
          @_members << h
        end
      end
      @_members
    end
    
    # Saves an array of Schleuder::Member's into +members.conf++
    def members=(arr)
      @log.debug 'writing members'
      file = File.join(@listdir, Schleuder.config.lists_memberfile)
      @log.info("writing #{file}")
      a = []
      arr.each do |m|
        m = Member.new(m,false) if m.kind_of?(Hash)
        a << m.to_hash
      end
      _write(YAML.dump(a), File.join(@listdir, Schleuder.config.lists_memberfile))
      @_members = nil
    end
    
    # Tests if +address+ is admin of this list. +address+ must be a string.
    def is_admin? address
      raise 'Error: need string as input' unless address.kind_of? String
      return self.config.adminaddr.include?(address)
    end

    # Tests if +address+ is a member of this list. +address+ must be a string.
    def is_member? address
      raise 'Error: need string as input' unless address.kind_of? String
      self.members.each do |m|
        return m if m.email == address
      end
      false
    end

    # Finds a member by +key+. Useful to determine email addresses from gpg
    # signatures.
    def find_member_by_key key
      crypt = Crypt.new(Schleuder.list.config.gpg_password)
      self.members.each do |m|
        mk = crypt.get_key(m.key || m.email)
        return m if mk.to_s == key.to_s
      end
      return false
    end

    # Finds a member by its email address.
    def find_member_by_email address
      self.members.each do |m|
        return m if m.email == address
      end
      return false
    end

    # Provides the list config as Schleuder::ListConfig-object 
    def config
      _load_config unless @_config
      @_config
    end

    # Loads the configuration
    # fromfile = Wether to load the config from file.
    def _load_config(fromfile=true)
        @log.debug("reading list-config for: #{@listname}") unless @log.nil?
        @_config = ListConfig.new(File.join(@listdir, Schleuder.config.lists_configfile),fromfile)
    end
    
    # Saves +data+ into the list-config-file (default: list.conf). +data+ must
    # be a Schleuder::ListConfig or valid input to +ListConfig.new+
    def config=(data)
      @log.info("writing list-config for: #{@listname}")
      if data.is_a?(ListConfig)
        @_config = data
      else
        @_config = ListConfig.new(data)
      end
      _write(YAML::dump(@_config.to_hash), File.join(@listdir, Schleuder.config.lists_configfile))
    end
    
    # Builds the bounce-address for the list
    def bounce_addr
      self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-bounce@\2')
    end

    # builds the send-key-command-address for the list
    def sendkey_addr
      self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-sendkey@\2')
    end

    private

    def _write(data,filename)
      if File.open(filename, 'w') { |f| f << data }  
        true
      else
        false
      end
    end
    
  end
end
