require 'yaml'

module Schleuder
  # Abstract class to provide loading of files and overloading of values.
  # Note: don't use Schleuder.log in here, is might be not yet available, and
  # you will produce loops.
  class Storage

    class << self
      def schleuder_attr(attr_name, default_value=nil,&block)
        attr_name = attr_name.to_s unless attr_name.is_a?(String)
        default_schleuder_attributes[attr_name] = block_given? ? block : Proc.new{ default_value }

        class_eval <<-EOE
          def #{attr_name}
            if schleuder_attributes['#{attr_name}'].nil?
              schleuder_attributes['#{attr_name}'] = self.instance_eval(&self.class.default_schleuder_attributes['#{attr_name}'])
            end
            schleuder_attributes['#{attr_name}']
          end
          def #{attr_name}=(value)
            schleuder_attributes['#{attr_name}'] = value
          end
        EOE
      end

      def default_schleuder_attributes
        @default_schleuder_attributes ||= {}
      end
    end

    def schleuder_attributes
      @schleuder_attributes ||= {}
    end

    # If +input+ is String or Hash it will be used to fill instance variables
    # fromfile = whether to load the information from file
    def initialize(input=nil, fromfile=true)
      if input.kind_of?(Hash)
        overload_from_hash!(input)
      elsif input.kind_of?(String) && fromfile
        overload_from_file!(input)
      else
        raise "Unknown input: #{input.class}"
      end
    end

    private

    # Load content from +filename+ and overwrite existing instance variables of
    # self
    def overload_from_file!(filename)
      h = YAML.load_file(filename) || Hash.new # yaml returns nil if the Hash is empty
      overload_from_hash!(h)
    end

    # Load content from +h+ into self (if allowed so according to
    # attr_reader/writer/accessor)
    def overload_from_hash!(h)
      h.each_pair do |k,v|
        k = k.to_s unless k.is_a?(String)
        if self.class.default_schleuder_attributes.keys.include?(k)
          schleuder_attributes[k] = v
        else
          Schleuder.log.warn "Attempt to set illegal attribute: #{k} => #{v}"
        end
      end
    end

    def to_hash
      self.class.default_schleuder_attributes.keys.inject({}) do |res, key|
        val = send(key)
        res[key] = if val.is_a?(Array)
          val.collect { |e| e.respond_to?(:to_hash) ? e.to_hash : e }
        elsif val.respond_to?(:to_hash)
          val.to_hash
        else
          val
        end
        res
      end
    end
  end
end
