//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [cjdtai]=CL_dat_utc2tai(cjdutc)
// UTC to TAI conversion
//
// Calling Sequence
// [cjdtai]=CL_dat_utc2tai(cjdutc)
//
// Description
// <itemizedlist><listitem>
// Converts the time (from 1950.0) of an event given in the UTC time scale (<emphasis role="bold">cjdutc</emphasis>) into the time (from 1950.0) of the same event in the TAI time scale (<emphasis role="bold">cjdtai</emphasis>). 
// </listitem>
// <listitem>
// Notes: 
// <para> - Conversion from UTC to TAI uses the global variable %CL_TAI_UTC (vector) that defines the leap seconds in the UTC time scale (it defines values for cjdtai minus cjdutc). </para>
// <para> - TAI was not defined until 1 January 1972. Before this time, UTC and TAI are supposed identical. </para>
// </listitem>
// <listitem>
// See <link linkend="DatesAndTime">Dates and Time</link> for more details.
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// cjdutc: Modified (1950.0) julian day in UTC time scale (1xN)
// cjdtai: Modified (1950.0) julian day in TAI time scale (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_dat_tai2utc
//
// Examples
// cjdutc=[21000:0.1:21001]';
// [cjdtai]=CL_dat_utc2tai(cjdutc);
//

// Declarations:
if(~exists('%CL_TAI_UTC')) then global %CL_TAI_UTC; end;

// Code:

sense = 1;
if find(cjdutc<8035)~=[] then warning('TAI not defined for dates before 1972'); end

jj = floor(cjdutc);
nsec = (cjdutc-jj)*86400.0;

TAI_UTC = %CL_TAI_UTC;

mjj=min(jj);
Mjj=max(jj);

im=min(find(TAI_UTC(:,1)>=mjj)) - 1;
if im<0 then im=size(TAI_UTC,1); end,
iM=max(find(TAI_UTC(:,2)<=Mjj)) + 1;

for I=im:iM
  nn=find(jj>=TAI_UTC(I,1) & jj<TAI_UTC(I,2));
  nsec(nn)=nsec(nn)+sense*TAI_UTC(I,3);
  nn=find(nsec>86400); nsec(nn)=nsec(nn)-86400; jj(nn)=jj(nn)+1;
  nn=find(nsec<0);     nsec(nn)=nsec(nn)+86400; jj(nn)=jj(nn)-1;
end

cjdtai = jj + nsec / 86400.0;


endfunction
