# Copyright (C) 2005 Tiago Cogumbreiro <cogumbreiro@users.sf.net>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Authors: Tiago Cogumbreiro <cogumbreiro@users.sf.net>

import gtk
import gobject
import sys
import weakref
import logging

logger = logging.getLogger('serpentine')

from gtk import glade
from os import path
from types import IntType, TupleType
from gettext import gettext as _
from gettext import ngettext as N_

from serpentine.errors import UnsupportedLocationError

# Local modules
from serpentine import operations
from serpentine import audio
from serpentine import xspf
from serpentine import gtkutil
from serpentine import urlutil

from serpentine.gtkutil import DictStore
from serpentine.operations import OperationsQueue
from serpentine.gdkpiechart import SerpentineUsage
from serpentine.common import parse_key_event

# Python 2.3 compability
try:
    set
except NameError:
    from sets import Set as set

################################################################################
# Operations used on AudioMastering
#
# 

class ErrorTrapper (operations.Operation, operations.OperationListener):
    def __init__ (self, parent = None):
        operations.Operation.__init__ (self)
        self.__errors = []
        self._parent = parent
    
    errors = property (lambda self: self.__errors)
    parent = property (lambda self: self._parent)
    
    def on_finished (self, event):
        if event.is_error:
            self.errors.append (event.source)
    
    def start (self):
        if len (self.errors) == 0:
            self._send_finished_event(operations.SUCCESSFUL)
            return

        filenames = []
        for e in self.errors:
            filenames.append (urlutil.basename(e.hints['location']))
        del self.__errors
        
        title = N_(
            "Unsupported file type",
            "Unsupported file types",
            len(filenames)
        )

        msg = N_(
            "The following file was not added:",
            "The following files were not added:",
            len(filenames)
        )

        gtkutil.list_dialog(
            title,
            _("If you're having problems opening certain files make sure you "
              "have the GStreamer plugins needed to decode them."),
            list_title=msg,
            parent=self.parent,
            items=filenames,
            stock = gtk.STOCK_DIALOG_ERROR,
            buttons =(gtk.STOCK_CLOSE, gtk.RESPONSE_OK),
        )
        
        self._send_finished_event(operations.SUCCESSFUL)

class AddFile(operations.Operation, audio.AudioMetadataListener):
    # TODO: Implement full Operation here
    
    running = property (lambda self: False)

    def __init__ (self, music_list, hints, insert=None, app=None):
        operations.Operation.__init__ (self)
        self.hints = hints
        self.music_list = music_list
        self.insert = insert
        # We normalize the paths by default
        hints["location"] = urlutil.normalize(hints["location"])
        self.app = app
    
    def start (self):
        if self.app.preferences.useGnomeVfs:
            oper = audio.get_metadata(audio.GVFS_SRC, self.hints["location"])
            
        else:
            url = urlutil.UrlParse(self.hints["location"])
            if not url.is_local:
                self._send_finished_event(
                    operations.ERROR,
                    error=StandardError(self.hints["location"])
                )
                return
                
            filename = url.path
            oper = audio.get_metadata("filesrc", filename)

        oper.add_listener(self)
        try:
            oper.start()
        except audio.GstPlayingFailedError:
            self._send_finished_event(
                operations.ERROR,
                error=StandardError(self.hints["location"])
            )
    
    def on_metadata (self, event, metadata):
        title = urlutil.basename(self.hints['location'])
        title = path.splitext(title)[0]
        
        row = {
            "location": self.hints['location'],
            "title": title or _("Unknown"),
            "artist": _("Unknown Artist"),
            "duration": int(metadata['duration']),
        }
        
        if metadata.has_key ('title'):
            row['title'] = metadata['title']
        if metadata.has_key ('artist'):
            row['artist'] = metadata['artist']
            
        if self.hints.has_key ('title'):
            row['title'] = self.hints['title']
        if self.hints.has_key ('artist'):
            row['artist'] = self.hints['artist']

        if self.insert is not None:
            self.music_list.insert (self.insert, row)
        else:
            self.music_list.append (row)
        
    
    def on_finished(self, evt):
        self._propagate(evt)
            
 
class UpdateDiscUsage (operations.Operation):
    def __init__ (self, masterer, update):
        operations.Operation.__init__ (self)
        self.__update = update
        self.__masterer = masterer
    
    running = property (lambda self: False)
    
    can_run = property (lambda self: True)
        
    def start (self):
        self.__masterer.update = self.__update
        if self.__update:
            self.__masterer.update_disc_usage()
        self._send_finished_event(operations.SUCCESSFUL)

################################################################################

class MusicListListener(object):
    """The interface from a music list listener."""
    def on_musics_added (self, event):
        """Musics were added to the source of the event."""
    
    def on_musics_removed (self, event):
        """Musics were removed from the source of the event."""

class MusicList (operations.Listenable):
    def __getitem__ (self):
        pass
        
    def append_many (self, rows):
        pass
    
    def append (self, row):
        pass
    
    def insert (self, index, row):
        pass
    
    def insert_many (self, index, rows):
        pass
    
    def __len__ (self):
        pass
    
    def __delitem__ (self, index):
        pass
    
    def delete_many (self, indexes):
        pass
    
    def clear (self):
        pass
    
    def has_key (self, key):
        pass
    
    def from_playlist (self, playlist):
        rows = []
        for t in playlist.tracks:
            rows.append ({'location': t.location,
                          'duration': t.duration,
                          'title': t.title,
                          'artist': t.creator})
        self.append_many (rows)

    def to_playlist (self, playlist):
        for r in self:
            t = xspf.Track()
            t.location = r['location']
            t.duration = r['duration']
            t.title = r['title']
            t.creator = r['artist']
            playlist.tracks.append (t)


class GtkMusicList(MusicList):
    """The GtkMusicList uses a ListStore as a backend, it is not visual and
    depends only on glib.
    
    Takes care of the data source. Supports events and listeners.
    """
    SPEC = (
        # URI is used in converter
        {"name": "location", "type": gobject.TYPE_STRING},
        # Remaining items are for the list
        {"name": "duration", "type": gobject.TYPE_INT},
        {"name": "title", "type": gobject.TYPE_STRING},
        {"name": "artist", "type": gobject.TYPE_STRING},
        {"name": "time", "type": gobject.TYPE_STRING}
    )
            
    def __init__ (self):
        operations.Listenable.__init__(self)
        self.__model = DictStore(*self.SPEC)
        self.__model.connect("row-deleted", self._on_row_deleted)
        self.__model.connect("row-inserted", self._on_row_inserted)
        self.__total_duration = 0
        self.__frozen = False
        self.use_gap = False
    
    model = property (fget=lambda self: self.__model,
                      doc="Associated ListStore.")

    def _recalculate(self):
        self.__total_duration = 0
        for row in self:
            self.__total_duration += row['duration']

    def _notify_removed(self):
        if not self.__frozen:
            e = operations.Event (self)
            gobject.idle_add(lambda: self._notify('on_musics_removed', e))

    def _notify_added(self):
        if not self.__frozen:
            e = operations.Event (self)
            gobject.idle_add(lambda: self._notify('on_musics_added', e))
    
    def _on_row_deleted(self, source, rows):
        self._recalculate()
        self._notify_removed()

    def _on_row_inserted(self, source, path, tree_iter):
        self._notify_added()
    
    def get_total_duration(self):
        if len(self) == 0:
            assert self.__total_duration == 0
            return 0

        total = self.__total_duration
        if self.use_gap:
            total += (len(self) - 1) * 2
            
        return total

    total_duration = property(get_total_duration,
                              doc="Total disc duration, in seconds.")
    
    def __getitem__ (self, index):
        return self.model.get (index)
    

    def append_many (self, rows):
        
        self.__frozen = True
        for row in rows:
            self.append (row)
        self.__frozen = False

        self._notify_added()
    

    def __correct_row (self, row):
        if not row.has_key ('time'):
            row['time'] = "%.2d:%.2d" % (row['duration'] / 60, row['duration'] % 60)
        return row
        

    def append (self, row):
        row = self.__correct_row(row)
        self.model.append(row)
        self.__total_duration += row['duration']
    

    def insert (self, index, row):
        row = self.__correct_row(row)
        self.model.insert_before(self.model.get_iter (index), row)
        self.__total_duration += row['duration']
        

    def __len__ (self):
        return len(self.model)
    

    def __delitem__ (self, index):
        # Copy native row
        row = dict(self[index])
        del self.model[index]
        self.__total_duration -= row['duration']
        rows = (row,)
    

    def delete_many (self, indexes):
        assert isinstance(indexes, list)

        indexes.sort()
        low = indexes[0] - 1
        # Remove duplicate entries
        for i in indexes:
            if low == i:
                indexes.remove (i)
            low = i
        # Now decrement the offsets
        for i in range (len (indexes)):
            indexes[i] -= i

        # Remove the elements directly
        self.__frozen = True
        for i in indexes:
            # Copy native row
            r = dict(self.model.get(i))
            self.__total_duration -= r['duration']
            del self.model[i]
        self.__frozen = False
        
        # Warn the listeners
        self._notify_removed()
        
    def clear (self):
        self.__frozen = True
        self.model.clear()
        self.__frozen = False
        self.__total_duration = 0

        # Warn the listeners
        self._notify_removed()

    def __repr__(self):
        return repr(self.model)



################################################################################
# Audio Mastering widget
#    

def copy_store(store):
    """
    Convert a store to a dictionary of filenames and their usage.
    """
    result = {}
    for row in store:
        location = row['location']
        try:
            result[location] += 1
        except:
            result[location] = 1
            
    return result
    

def diff_stores(old_store, new_store):
    """
    Creates a new dictionary that represents which locations
    were removed and which are added.
    """
    diff = {}
    for location in new_store:
        old_value = old_store.get(location, 0)
        new_value = new_store.get(location, 0)
        diff[location] = new_value - old_value

    for location in old_store:
        if location not in new_store:
            diff[location] = -old_store[location]
            
    return diff


class AudioMasteringMusicListener(MusicListListener):
    def __init__ (self, audio_mastering):
        self.master = audio_mastering
        self.old_store = {}

    def update_pool(self):
        """
        Updates the music pool.
        """
        store = copy_store(self.master.source)

        diff = diff_stores(old_store=self.old_store,
                           new_store=store)

        for (location, value) in diff.items():
            if value > 0:
                for x in range(value):
                    self.master.pool.request_music(location)
            else:
                for x in range(-value):
                    self.master.pool.release_music(location)

        self.old_store = store

        
    def on_musics_added(self, evt):
        self.master.update_disc_usage()
        self.update_pool()
        

    def on_musics_removed(self, evt):
        self.master.update_disc_usage()
        self.update_pool()

            

class HintsFilter(object):

    priority = 0
    
    def filter_location (self, location):
        """Returns a list of dictionaries of hints of a given location.
        The 'location' field is obligatory.
        
        For example if your filter parses a directory it should return a list
        of hints of each encountered file.
        """
        raise NotImplementedError()
    
    def __cmp__ (self, value):
        if not hasattr(value, 'priority'):
            raise TypeError('%r has no \'priority\' attribute' % value)

        return self.priority - value.priority


class MetadataRetriever(object):
    """Manages objects that retrieve metadata from a location."""
    def __init__(self):
        self.filters = []

    def filter_location(self, location):
        """
        Checks if a given location is a composite of 'hint' instances.
        
        throws UnsupportedLocationError"""
        buggy_filters = []
        
        for loc_filter in self.filters:
            try:
                metadata = loc_filter.filter_location(location)
                if metadata is None:
                    logger.info('Metadata retriever %r is not returning an appropriate value, unregistring it.' % loc_filter)
                    buggy_filters.append(loc_filter)
                    continue
                self._remove_filters(buggy_filters)
                return metadata
            except UnsupportedLocationError:
                # Try next location filter
                pass
        self._remove_filters(buggy_filters)
        raise UnsupportedLocationError()

    def _remove_filters(self, filters):
        for a_filter in filters:
            self.filters.remove(a_filter)
            
    def add(self, location_filter):
        self.filters.append(location_filter)
        # Sort filters priority
        self.filters.sort()
    
    def remove(self, location_filter):
        self.filters.remove(location_filter)


class MusicListGateway:
    """This class wraps the MusicList interface in a friendlier one
    with a method `add_files` easier to use then the `insert` method
    which expects a hints `dict`. It also serves as a hints filter
    which is a list of client objects which must provide the
    `filter_location` method.
    """

    class Handler:
        """A handler is created each time a method is created, it must
        return objects with this class signature."""
        
        def prepare_queue (self, gateway, queue):
            """Method called before the AddFile operations are added to the queue"""
        
        def finish_queue (self, gateway, queue):
            """Method called after the AddFile operations are added to the queue"""
        
        def prepare_add_file (self, gateway, add_file):
            """Method called before add_file object is added to queue"""
    
    def __init__ (self, app):
        # Filters
        self._metadata = MetadataRetriever()
        self._app = weakref.ref(app)
    
    music_list = None
    
    def add_hints_filter(self, location_filter):
        self._metadata.add(location_filter)
    
    def remove_hints_filter(self, location_filter):
        self._metadata.remove(location_filter)


    def add_hints (self, hints_list, insert=None, known=()):
        assert insert is None or isinstance (insert, IntType)
        logger.info('adding %r in %r' % (hints_list, insert))
        queue = OperationsQueue()
        queue.abort_on_failure = False
        
        handler = self.Handler ()
        
        handler.prepare_queue (self, queue)
        
        for i, hint in enumerate(hints_list):
            # prepare the operation 'AddFile'
            try:
                # try to check if the location is a composite of hints
                pls = self._metadata.filter_location(hint["location"])

                # Filter known files
                pls = [p for p in pls if p not in known]
                known += tuple(pls)
                
                for child_hint in pls:
                    # normalize the location of the returning elements
                    child_hint["location"] = urlutil.normalize(child_hint["location"])
                                
                queue.append(self.add_hints(pls, insert, known))

                # composite hints have no children, thus do not need
                # to be probed
                continue
                
            except UnsupportedLocationError:
                pass
            
            ins = insert
            if ins is not None:
                ins += i
            
            oper = AddFile(self.music_list, hint, ins, self._app())
            handler.prepare_add_file(self, oper)

            # Enqueue 'AddFile'
            queue.append(oper)
            

        
        handler.finish_queue (self, queue)
        return queue

    def add_files(self, filenames):
        return self.add_hints({"location": urlutil.normalize(filename)}
                              for filename in filenames)


class AudioMastering (gtk.VBox, operations.Listenable):
    SIZE_21 = 0
    SIZE_74 = 1
    SIZE_80 = 2
    SIZE_90 = 3
    
    class MusicListGateway (MusicListGateway):
        def __init__ (self, parent):
            MusicListGateway.__init__ (self, parent._application())
            self.parent = parent
        
        def music_list (self):
            return self.parent.music_list
            
        music_list = property (music_list)
        
        def window (self):
            return gtkutil.get_root_parent (self.parent)
        
        window = property (window)
        
        class Handler:
            def prepare_queue (self, gateway, queue):
                queue.append (UpdateDiscUsage (gateway.parent, False))
                self.trapper = ErrorTrapper (gateway.window)
            
            def finish_queue (self, gateway, queue):
                queue.append (UpdateDiscUsage (gateway.parent, True))
                queue.append (self.trapper)
                del self.trapper
            
            def prepare_add_file (self, gateway, add_file):
                add_file.add_listener(self.trapper)
        
    
    disc_sizes = [21 * 60, 74 * 60, 80 * 60, 90 * 60]
    
    DND_TARGETS = [
        ('SERPENTINE_ROW', gtk.TARGET_SAME_WIDGET, 0),
        ('text/uri-list', 0, 1),
        ('text/plain', 0, 2),
        ('STRING', 0, 3),
    ]
    
    def __init__ (self, application):
        gtk.VBox.__init__ (self)
        self._application = weakref.ref(application)
        self._use_gap = application.preferences.connect("use_gap",
                                                        self.on_use_gap)
        
        operations.Listenable.__init__ (self)
        self.__disc_size = 74 * 60
        self.update = True
        self.source = GtkMusicList ()
        self.source.add_listener(AudioMasteringMusicListener(self))
        
        self.__gateway = AudioMastering.MusicListGateway (self)
        
        gtk.VBox.__init__ (self)
        filename = application.locations.get_data_file("serpentine.glade")
        g = glade.XML (filename, "audio_container")
        
        self.add (g.get_widget ("audio_container"))
        
        self.__setup_track_list(g)
        self.__setup_container_misc(g)
        self.connect("key-press-event", self.on_key_pressed)
        self.keys = {
            "Delete": self.remove_selected,
        }

    def on_use_gap(self, signal, use_gap):
        self.update_disc_usage()
    
    def on_key_pressed(self, widget, evt):
        try:
            self.keys[parse_key_event(evt)]()
        except KeyError:
            pass
    
    def __set_disc_size (self, size):
        assert size in AudioMastering.disc_sizes
        self.__disc_size = size
        self.__size_list.set_active(AudioMastering.disc_sizes.index(size))
        self.update_disc_usage()
        self._notify('on_disc_size_changed', operations.Event(self))
        
    music_list_gateway = property (lambda self: self.__gateway)
    music_list = property (lambda self: self.source)
    disc_size = property (
            lambda self: self.__disc_size,
            __set_disc_size,
            doc = "Represents the disc size, in seconds.")
    
    disc_size_widget = property (lambda self: self.__size_list)
    
    pool = property(lambda self: self._application().cache_pool)
    
    def __setup_container_misc (self, g):
        self.__size_list = g.get_widget ("size_list")
        self.__usage_label = g.get_widget ("usage_label")
        
        self.__usage_gauge = SerpentineUsage (self)
        self.__usage_gauge.widget.show ()
        self.__usage_gauge.widget.set_size_request (92, 92)
        hbox = g.get_widget ("disc_details")
        hbox.pack_start (self.__usage_gauge.widget, expand = False, fill = False)
        
        self.__capacity_exceeded = g.get_widget ("capacity_exceeded")
        
        self.__size_list.connect ("changed", self.__on_size_changed)
        self.__size_list.set_active (AudioMastering.SIZE_74)
    
    def __setup_track_list (self, g):
        lst = g.get_widget ("track_list")
        lst.set_model (self.source.model)
        # Track value is dynamicly calculated
        r = gtk.CellRendererText()
        col = gtk.TreeViewColumn (_("Track"), r)
        col.set_cell_data_func (r, self.__generate_track)
        
        r = gtk.CellRendererText()
        r.set_property ('editable', True)
        r.connect ('edited', self.__on_title_edited)
        lst.append_column (col)
        col = gtk.TreeViewColumn (_("Title"), r, text = self.source.model.index_of("title"))
        lst.append_column (col)
        
        r = gtk.CellRendererText()
        r.set_property ('editable', True)
        r.connect ('edited', self.__on_artist_edited)
        col = gtk.TreeViewColumn (_("Artist"), r, text = self.source.model.index_of("artist"))
        lst.append_column (col)
        r = gtk.CellRendererText()
        col = gtk.TreeViewColumn (_("Duration"), r, text = self.source.model.index_of("time"))
        lst.append_column (col)
        
        # TreeView Selection
        self.__selection = lst.get_selection()
        self.__selection.connect ("changed", self.__selection_changed)
        self.__selection.set_mode (gtk.SELECTION_MULTIPLE)
        
        # Listen for drag-n-drop events
        lst.set_reorderable (True)

        lst.enable_model_drag_source (gtk.gdk.BUTTON1_MASK,
                                      AudioMastering.DND_TARGETS,
                                      gtk.gdk.ACTION_DEFAULT |
                                      gtk.gdk.ACTION_MOVE)

        lst.enable_model_drag_dest (AudioMastering.DND_TARGETS,
                                    gtk.gdk.ACTION_DEFAULT |
                                    gtk.gdk.ACTION_MOVE)
        lst.connect ("drag_data_received", self.__on_dnd_drop)
        lst.connect ("drag_data_get", self.__on_dnd_send)
        
    
    def __generate_track(self, col, renderer, tree_model, treeiter,
                         user_data=None):
        index = tree_model.get_path(treeiter)[0]
        renderer.set_property ('text', index + 1)
    
    def __on_size_changed (self, *args):
        self.disc_size = AudioMastering.disc_sizes[self.__size_list.get_active()]
    
    def __on_title_edited (self, cell, path, new_text, user_data = None):
        self.source[path]["title"] = new_text
    
    def __on_artist_edited (self, cell, path, new_text, user_data = None):
        self.source[path]["artist"] = new_text
    
    def __on_dnd_drop (self, treeview, context, x, y, selection, info, timestamp, user_data = None):
        data = selection.data
        hints_list = []
        
        # Insert details
        insert = None
        drop_info = treeview.get_dest_row_at_pos(x, y)
        if drop_info:
            insert, insert_before = drop_info
            assert isinstance(insert, TupleType), len(insert) == 1
            insert, = insert
            if (insert_before != gtk.TREE_VIEW_DROP_BEFORE and
                insert_before != gtk.TREE_VIEW_DROP_INTO_OR_BEFORE):
                insert += 1
                if insert == len (self.source):
                    insert = None
            del insert_before
                
        del drop_info
        
        if selection.type == 'application/x-rhythmbox-source':
            #TODO: handle rhythmbox playlists
            return
        elif selection.type == 'SERPENTINE_ROW':
            # Private row
            store, path_list = self.__selection.get_selected_rows ()
            if not path_list or len (path_list) != 1:
                return
            path, = path_list
            # Copy the row
            row = dict(self.source[path])
            # Remove old row
            del self.source[path]
            
            # When we insert in the last position it's the same thing as appending
            if len (self.source) == insert:
                insert = None

            # Append this row
            if insert is not None:
                self.source.insert (insert, row)
            else:
                self.source.append (row)
            return
            
        for line in data.split("\n"):
            line = line.strip()
            if len (line) < 1 or line == "\x00":
                continue
                
            assert "\x00" not in line, "Malformed DnD string: %s" % line
            hint = {'location': line}
            hints_list.append (hint)
        self.music_list_gateway.add_hints (hints_list, insert).start ()
    
    def __on_dnd_send (self, widget, context, selection, target_type, timestamp):
        store, path_list = self.__selection.get_selected_rows ()
        assert path_list and len(path_list) == 1
        path, = path_list # unpack the only element
        selection.set (selection.target, 8, self.source[path]['location'])
    
    def __hig_duration (self, duration):
        hig_duration = ""
        minutes = duration / 60
        if minutes:
            # To translators: I know this is ugly for you
            hig_duration = ("%s %s") %(minutes, minutes == 1 and _("minute") or _("minutes"))
        seconds = duration % 60
        if seconds:
            # To translators: I know this is ugly for you
            hig_secs = ("%s %s") %(seconds, seconds == 1 and _("second") or _("seconds"))
            if len (hig_duration):
                hig_duration += _(" and ")
                
            hig_duration += hig_secs
        return hig_duration

    def get_preferences(self):
        return self._application().preferences

    preferences = property(get_preferences)

    def update_disc_usage (self):
        
        if not self.update:
            return

        self.source.use_gap = self.preferences.useGap
        
        if self.source.total_duration > self.disc_size:
            self.__capacity_exceeded.show()
            
        else:
            self.__capacity_exceeded.hide()

        # Flush events so progressbar redrawing gets done
        while gtk.events_pending():
            gtk.main_iteration(True)
        
        if self.source.total_duration > 0:
            duration = self.__disc_size - self.source.total_duration
            if duration > 0:
                dur = _("%s remaining")  % self.__hig_duration (duration)
            else:
                dur = _("%s overlaping") % self.__hig_duration (abs (duration))
        else:
            dur = _("Empty")
        
        self.__usage_label.set_text (dur)
            
        e = operations.Event(self)
        self._notify('on_contents_changed', e)
    
    def __selection_changed(self, treeselection):
        e = operations.Event(self)
        self._notify('on_selection_changed', e)

    
    def get_selected (self):
        """Returns the selected indexes"""
        store, path_list = self.__selection.get_selected_rows()
        
        if path_list is None:
            return []
            
        indexes = []
        for p in path_list:
            assert len(p) == 1
            indexes.append(*p)
        return indexes
    
    def remove_selected (self):
        self.source.delete_many (self.get_selected ())
            
    def count_selected (self):
        return self.__selection.count_selected_rows()

    
if __name__ == '__main__':
    win = gtk.Window()
    win.connect ("delete-event", gtk.main_quit)
    w = AudioMastering ()
    w.show()
    win.add (w)
    win.show()
    

    w.add_file (sys.argv[1])
    w.source.clear()
    gtk.main()
