//  libsigcperl -- a helper library for writing XSUB wrappers of libsigc++
//  Copyright (C) 2002 Ron Steinke
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; if not, write to the 
//  Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
//  Boston, MA  02111-1307  USA.

#ifndef SIGC_PERL_CONVERT_H
#define SIGC_PERL_CONVERT_H

#include <assert.h>
#include <exception>
#include <string>

extern "C" {
#include "EXTERN.h"
#include "perl.h"
}

#include <sigcperl/data.h>

namespace SigCPerl {

class BadConvertVal : public std::exception {};

template <class R>
struct ArgBox
{
  ArgBox(SV*) throw();
  R val() {return m_val;}
  R ret_val() {return m_val;}

  R m_val;
};

// General types
template<> inline ArgBox<int>::ArgBox(SV *val) throw()
  : m_val(SvIV(val)) {}
template<> inline ArgBox<char>::ArgBox(SV *val) throw()
  : m_val(SvIV(val)) {}
template<> inline ArgBox<short>::ArgBox(SV *val) throw()
  : m_val(SvIV(val)) {}
template<> inline ArgBox<long>::ArgBox(SV *val) throw()
  : m_val(SvIV(val)) {}
template<> inline ArgBox<bool>::ArgBox(SV *val) throw()
  : m_val(SvIV(val) != 0) {}
template<> inline  ArgBox<unsigned char>::ArgBox(SV *val) throw()
  : m_val(SvUV(val)) {}
template<> inline  ArgBox<unsigned short>::ArgBox(SV *val) throw()
  : m_val(SvUV(val)) {}
template<> inline  ArgBox<unsigned int>::ArgBox(SV *val) throw()
  : m_val(SvUV(val)) {}
template<> inline ArgBox<unsigned long>::ArgBox(SV *val) throw()
  : m_val(SvUV(val)) {}
template<> inline ArgBox<float>::ArgBox(SV *val) throw()
  : m_val(SvNV(val)) {}
template<> inline ArgBox<double>::ArgBox(SV *val) throw()
  : m_val(SvNV(val)) {}

inline std::string SigCPerl::get_string(SV *val) throw()
{
  STRLEN len;
  const char *str = SvPV(val, len);
  return std::string(str, len);
}

template<> inline ArgBox<std::string>::ArgBox(SV *val) throw()
  : m_val(get_string(val)) {}

template<>
struct ArgBox<const std::string&>
{
  ArgBox(SV *val) throw() : m_val(get_string(val)) {}
  const std::string& val() const {return m_val;}

  std::string m_val;
};

template<>
struct ArgBox<const char*>
{
  ArgBox(SV *val) throw() : m_val(get_string(val)) {}
  const char* val() const {return m_val.c_str();}

  std::string m_val;
};

// For passing pointers around, the
// constructer is implemented using typemap
// code from the Perl module which is using
// the library

template<class R>
struct ArgBox<R*>
{
  ArgBox(SV*) throw(BadConvertVal);
  R* val() {return m_val;}
  R* ret_val() {return m_val;}

  R* m_val;
};

inline SV* GetSV(int val) throw() {return newSViv(val);}
inline SV* GetSV(char val) throw() {return newSViv(val);}
inline SV* GetSV(short val) throw() {return newSViv(val);}
inline SV* GetSV(long val) throw() {return newSViv(val);}
inline SV* GetSV(bool val) throw() {return newSViv(val);}
inline SV* GetSV(unsigned char val) throw() {return newSVuv(val);}
inline SV* GetSV(unsigned int val) throw() {return newSVuv(val);}
inline SV* GetSV(unsigned short val) throw() {return newSVuv(val);}
inline SV* GetSV(unsigned long val) throw() {return newSVuv(val);}
inline SV* GetSV(double val) throw() {return newSVnv(val);}
inline SV* GetSV(float val) throw() {return newSVnv(val);}
inline SV* GetSV(const char *val) throw() {return newSVpv(val, 0);}
inline SV* GetSV(const std::string &val) throw()
	{return newSVpv(val.c_str(), val.size());}

template<class R>
struct Returns
{
  static I32 getFlags() throw() {return G_SCALAR;}
  static R ret_val(const Data &data) throw(BadConvertVal)
  {
    if(data.size() < 1)
      throw BadConvertVal();

    return ArgBox<R>(data[0]).ret_val();
  }

  static Data getRetData(R r) throw()
  {
    Data data;
    data.push_back(r);
    return data;
  }
};

// pass two args back from perl using std::pair<>,
// this can be combined with SigC::convert()
// to connect to signals which expect a function
// which returns a second argument to a passed pointer
template<class R1, class R2>
struct Returns<std::pair<R1,R2> >
{
  typedef std::pair<R1,R2> PairType;
  static I32 getFlags() throw() {return G_ARRAY;}
  static PairType ret_val(const Data &data) throw(BadConvertVal)
  {
    if(data.size() < 2)
      throw BadConvertVal();

    return PairType(ArgBox<R1>(data[0]).ret_val(),
      ArgBox<R2>(data[1]).ret_val());
  }

  static Data getRetData(const PairType &r) throw()
  {
    Data data;
    data.push_back(r.first);
    data.push_back(r.second);
    return data;
  }
};

} // namespace SigCPerl

#endif // SIGC_PERL_CONVERT_H
