/***************************************************************************
    smb4kcore  -  The main core class of Smb4K.
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qdir.h>

// KDE includes
#include <kdeversion.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <krun.h>
#include <kapplication.h>
#include <kpassivepopup.h>

// system specific includes
#include <stdlib.h>

// application specific includes
#include "smb4kscanner.h"
#include "smb4kmounter.h"
#include "smb4kpasswordhandler.h"
#include "smb4kfileio.h"
#include "smb4kbookmarkhandler.h"
#include "smb4kprint.h"
#include "smb4kcore.h"
#include "smb4kdefs.h"
#include "smb4kglobal.h"

using namespace Smb4KGlobal;

Smb4KCore *Smb4KCore::m_this_class = 0L;


Smb4KCore::Smb4KCore( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_this_class = this;

  searchPrograms();

  m_fileIO = new Smb4KFileIO( this, "Core_FileIO" );
  m_scanner = new Smb4KScanner( this, "Core_Scanner" );
  m_mounter = new Smb4KMounter( this, "Core_Mounter" );
  m_bookmarkHandler = new Smb4KBookmarkHandler( this, "Core_BookmarkHandler" );
  m_print = new Smb4KPrint( this, "Core_PrinterHandler" );
  m_password_handler = new Smb4KPasswordHandler( this, "Core_PasswordHandler" );
  m_synchronizer = new Smb4KSynchronizer( this, "Core_Synchronizer" );

  m_scanner_state = SCANNER_STOP;
  m_mounter_state = MOUNTER_STOP;
  m_print_state = PRINT_STOP;
  m_syncer_state = SYNCER_STOP;

  // Connections:
  connect( m_scanner,         SIGNAL( state( int ) ),
           this,              SLOT( slotSetScannerState( int ) ) );
  connect( m_mounter,         SIGNAL( state( int ) ),
           this,              SLOT( slotSetMounterState( int ) ) );
  connect( m_print,           SIGNAL( state( int ) ),
           this,              SLOT( slotSetPrinterHandlerState( int ) ) );
  connect( m_synchronizer,    SIGNAL( state( int ) ),
           this,              SLOT( slotSetSynchronizerState( int ) ) );

  connect( kapp,              SIGNAL( shutDown() ),
           this,              SLOT( slotShutdown() ) );
}


Smb4KCore::~Smb4KCore()
{
  // Do not call abort() here. This will lead
  // to crashes.
}


/****************************************************************************
   Returns a bool that tells the program whether a core process is running.
****************************************************************************/

bool Smb4KCore::isRunning()
{
  if ( m_scanner->isRunning() || m_mounter->isRunning() || m_print->isRunning() || m_synchronizer->isRunning() )
  {
    return true;
  }
  else
  {
    return false;
  }
}


/****************************************************************************
   Sets  the current state of the core.
****************************************************************************/

void Smb4KCore::setCurrentState( int state )
{
  if ( state != SCANNER_STOP && state != MOUNTER_STOP && state != PRINT_STOP && state != SYNCER_STOP )
  {
    m_current_state = state;
  }
  else
  {
    if ( !m_scanner->isRunning() && !m_mounter->isRunning() && !m_print->isRunning() && !m_synchronizer->isRunning() )
    {
      m_current_state = CORE_STOP;
    }
    else
    {
      if ( m_scanner->isRunning() )
      {
        m_current_state = m_scanner_state;
      }
      else if ( m_print->isRunning() )
      {
        m_current_state = m_print_state;
      }
      else if ( m_mounter->isRunning() )
      {
        m_current_state = m_mounter_state;
      }
      else if ( m_synchronizer->isRunning() )
      {
        m_current_state = m_syncer_state;
      }
    }
  }
}


/****************************************************************************
   Aborts any process of the core.
****************************************************************************/

void Smb4KCore::abort()
{
  m_scanner->abort();
  m_mounter->abort();
  m_print->abort();
  m_synchronizer->abort();
}


/****************************************************************************
   Opens the given URL.
****************************************************************************/

void Smb4KCore::open( Smb4KShare *share )
{
  if ( !share || share->isBroken() )
  {
    return;
  }

#if KDE_VERSION_MAJOR == 3 && KDE_VERSION_MINOR <= 3 && KDE_VERSION_RELEASE <= 92

  if ( QString::compare( share->filesystem(), "cifs" ) == 0 )
  {
    if( KMessageBox::warningContinueCancel( (QWidget *)this, i18n( "Up to KDE 3.3.x, KIO and Konqueror cannot handle CIFS shares. Konqueror will hang if you try to access it.\nDo you want to continue?" ) ) == KMessageBox::Cancel )
    {
      return;
    }
  }

#endif

  KURL url;
  url.setPath( share->canonicalPath() );

  (void) new KRun( url, 0, true, true );
}


/****************************************************************************
   Searches for the programs needed by Smb4K
****************************************************************************/

void Smb4KCore::searchPrograms()
{
  config()->deleteGroup( "Programs" );
  config()->setGroup( "Programs" );

  QStringList pathList = QStringList::split( ":", QString( "%1" ).arg( getenv( "PATH" ) ), false );
  pathList << "/sbin";
  pathList << "/usr/sbin";
  pathList << "/usr/local/sbin";

  // all mandatory programs that are needed by Smb4K
  QStringList fileList;
  fileList << "grep";
  fileList << "awk";
  fileList << "sed";
  fileList << "xargs";
  fileList << "rmdir";
  fileList << "ls";
  fileList << "sh";
  fileList << "nmblookup";
  fileList << "smbclient";
  fileList << "smbspool";
  fileList << "net";
#ifndef __FreeBSD__
  fileList << "mount.cifs";
  fileList << "umount.cifs";
  fileList << "smbmount";
  fileList << "smbumount";
  fileList << "mount";
#else
  fileList << "mount_smbfs";
#endif
  fileList << "umount";
  fileList << "smb4k_mount";
  fileList << "smb4k_umount";
  fileList << "smb4k_kill";
  fileList << "smb4k_cat";
  fileList << "smb4k_mv";

  // optional programs
  QStringList optional;
  optional << "super";
  optional << "sudo";
  optional << "dvips";
  optional << "enscript";
  optional << "rsync";

  fileList += optional;

  config()->setGroup( "Programs" );

  for ( QStringList::ConstIterator it = pathList.begin(); it != pathList.end(); ++it )
  {
    QDir::setCurrent( *it );

    for ( QStringList::Iterator bin = fileList.begin(); bin != fileList.end(); ++bin )
    {
      if ( QFile::exists( *bin ) )
      {
        config()->writeEntry( *bin, QString( "%1/%2" ).arg( QDir::currentDirPath(), *bin ) );
        *bin = QString::null;
        continue;
      }
      else
      {
        continue;
      }
    }
  }

  fileList.remove( QString::null );

  if ( fileList.count() != 0 )
  {
    for ( QStringList::Iterator it = optional.begin(); it != optional.end(); ++it )
    {
      if ( fileList.find( *it ) == fileList.end() )
      {
        *it = QString::null;
        continue;
      }
      else
      {
        fileList.remove( fileList.find( *it ) );
        continue;
      }
    }

    optional.remove( QString::null );

    if ( optional.count() != 0 )
    {
      config()->setGroup( "Mount Options" );

      bool disable_suid = false;
      QString suid_program = config()->readEntry( "SUID Program", QString::null );

      if ( !suid_program.isEmpty() )
      {
        if ( QString::compare( suid_program, "super" ) == 0 && optional.find( "super" ) != optional.end() )
        {
          disable_suid = true;
          showCoreError( ERROR_SUID_PROGRAM_MISSING, suid_program );
        }
        else if ( QString::compare( suid_program, "sudo" ) == 0 && optional.find( "sudo" ) != optional.end() )
        {
          disable_suid = true;
          showCoreError( ERROR_SUID_PROGRAM_MISSING, suid_program );
        }
      }

      if ( disable_suid )
      {
        config()->setGroup( "Super User Privileges" );
        config()->writeEntry( "Run SUID", false );
        config()->writeEntry( "Force Unmount", false );
      }
    }

    if ( fileList.count() != 0 )
    {
#ifndef __FreeBSD__

      // If either smbmount/smbumount or mount.cifs/umount.cifs
      // are present, we want to allow Smb4K to start up.
      // We also need to change the filesystem, if necessary. But
      // only modify, if the user does *not* use super or sudo.

      config()->setGroup( "Samba" );
      QString filesystem = config()->readEntry( "Mount Filesystem", "smbfs" );

      config()->setGroup( "Super User Privileges" );
      bool run_suid = config()->readBoolEntry( "Run SUID", false );

      if ( fileList.find( "smbmount" ) != fileList.end() &&
          fileList.find( "mount.cifs" ) == fileList.end() )
      {
        *(fileList.find( "smbmount" )) = QString::null;

        if ( QString::compare( filesystem, "cifs" ) != 0 && !run_suid )
        {
          filesystem = "cifs";

          config()->setGroup( "Samba" );
          config()->writeEntry( "Mount Filesystem", filesystem );
        }
      }

      if ( fileList.find( "mount.cifs" ) != fileList.end() &&
          fileList.find( "smbmount" ) == fileList.end() )
      {
        *(fileList.find( "mount.cifs" )) = QString::null;

        if ( QString::compare( filesystem, "smbfs" ) != 0 && !run_suid )
        {
          filesystem = "smbfs";

          config()->setGroup( "Samba" );
          config()->writeEntry( "Mount Filesystem", filesystem );
        }
      }

      if ( fileList.find( "smbumount" ) != fileList.end() &&
          fileList.find( "umount.cifs" ) == fileList.end() )
      {
        *(fileList.find( "smbumount" )) = QString::null;

        if ( QString::compare( filesystem, "cifs" ) != 0 && !run_suid )
        {
          filesystem = "cifs";

          config()->setGroup( "Samba" );
          config()->writeEntry( "Mount Filesystem", filesystem );
        }
      }

      if ( fileList.find( "umount.cifs" ) != fileList.end() &&
          fileList.find( "smbumount" ) == fileList.end() )
      {
        *(fileList.find( "umount.cifs" )) = QString::null;

        if ( QString::compare( filesystem, "smbfs" ) != 0 && !run_suid )
        {
          filesystem = "smbfs";

          config()->setGroup( "Samba" );
          config()->writeEntry( "Mount Filesystem", filesystem );
        }
      }

      fileList.remove( QString::null );

      if ( fileList.count() != 0 )
      {
        if ( fileList.find( "mount.cifs" ) != fileList.end() &&
             fileList.find( "smbmount" ) != fileList.end() )
        {
          *(fileList.find( "smbmount" )) = QString::null;
          *(fileList.find( "mount.cifs" )) += "/smbmount";
        }

        if ( fileList.find( "umount.cifs" ) != fileList.end() &&
             fileList.find( "smbumount" ) != fileList.end() )
        {
          *(fileList.find( "smbumount" )) = QString::null;
          *(fileList.find( "umount.cifs" )) += "/smbumount";
        }

        fileList.remove( QString::null );
        fileList.sort();

        showCoreError( ERROR_MISSING_PROGRAMS, fileList.join( ", " ) );
        exit( EXIT_SUCCESS );
      }

#else

      fileList.sort();
      showCoreError( ERROR_MISSING_PROGRAMS, fileList.join( ", " ) );
      exit( EXIT_SUCCESS );

#endif
    }
  }

  config()->sync();
}


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KCore::slotSetScannerState( int state )
{
  m_scanner_state = state;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotSetMounterState( int state )
{
  m_mounter_state = state;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotSetPrinterHandlerState( int state )
{
  m_print_state = state;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotSetSynchronizerState( int state )
{
  m_syncer_state = state;
  setCurrentState( state );
  emit runStateChanged();
}


void Smb4KCore::slotShutdown()
{
  config()->sync();
}

#include "smb4kcore.moc"
