/*
 * smil2raw.cc -- Extracts a Raw DV Stream from a kino smil project
 * Copyright (C) 2002 Raphael Amaury Jacquot <sxpert@esitcom.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <iostream>
#include <Diagnostics.h>
#include <extensions.h>
#include <PlayListDVProvider.h>
#include <Threader.h>
#include <unistd.h>

class SMIL2Raw : public Threader
{
	private:
		ExtendedPlayList playlist;
		int offset;
		int frames;
		double speed;
		int pump_size;
		int scene;
		bool progress;

	public: 
		SMIL2Raw( ) : offset( 0 ), frames( -1 ), speed( 1 ), pump_size( 50 ), scene( -1 ), progress( false )
		{
		}

		virtual ~SMIL2Raw( )
		{
		}

		virtual string LogId( )
		{
			return "SMIL2Raw";
		}

		void SetOffset( int _offset )
		{
			offset = _offset;
		}

		void SetFrames( int _frames )
		{
			frames = _frames - 1;
		}

		bool Append( string file_name )
		{
			return playlist.Append( file_name );
		}

		void SetSpeed( double _speed )
		{
			speed = _speed;
		}

		void SetPumpSize( int _pump_size )
		{
			pump_size = _pump_size;
		}

		bool IsRunnable( )
		{
			return playlist.GetNumFrames( ) != 0;
		}

		void SetScene( int _scene )
		{
			scene = _scene;
		}

		void SetProgress( bool _progress )
		{
			progress = _progress;
		}

	protected:
		void Thread( )
		{
			bool running = true;
			int total = playlist.GetNumFrames( );
			int current = 0;

			PlayListDVProvider input( playlist );
			input.SetPumpSize( pump_size );
			input.SetSpeed( speed );

			if ( scene != -1 )
			{
				PlayList copy;
				int index = 0;
				int begin = 0;
				int end = input.FindEndOfScene( begin );
				while ( index != scene )
				{
					index ++;
					begin = end + 1;
					end = input.FindEndOfScene( begin );
				}
				input.GetPlayList( begin, end, copy );
				total = copy.GetNumFrames( );
				input.SetPlayList( copy );
			}

			if ( frames != -1 || offset != 0 )
			{
				PlayList copy;
				int end = frames == -1 ? input.GetNumFrames( ) - offset - 1 : offset + frames;
				input.GetPlayList( offset, end, copy );
				total = copy.GetNumFrames( );
				input.SetPlayList( copy );
			}

			input.ThreadStart( );

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );
					fwrite( frame.data, frame.GetFrameSize( ), 1, stdout );
					fflush( stdout );
					input.QueueInputFrame( );
					current++;
					if ( progress && ( current % 100 == 0 ) )
					{
					  std::cerr << "Progress: " << current << "/" << total << " (" << (100 * current / total) << "%)\r";
					}
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}
			input.ThreadStop( );
			if ( progress )
			{
				std::cerr << "Progress: " << current << "/" << total << " (" << (100 * current / total) << "%)" << std::endl;
			}
		}
};

void Usage( )
{
	std::cerr << "Usage: smil2raw [ options ] file [ file ... ]" << std::endl;
	std::cerr << "Where: file is smil, dv avi or raw dv" << std::endl;
	std::cerr << "       -o offset - frame offset (default: 0)" << std::endl;
	std::cerr << "       -f count  - frame count (default: all)" << std::endl;
	std::cerr << "       -s speed  - floating point speed (default: 1)" << std::endl;
	std::cerr << "       -c scene  - scene to output (default: all)" << std::endl;
	std::cerr << "       -p        - output progress to stderr (default: quiet)" << std::endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	bool result = true;
	int index = 1;
	SMIL2Raw output;

	Diagnostics::SetApp( "smil2raw" );

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	try
	{
		for ( index = 1; result && index < argc; index ++ )
		{
			if ( !strcmp( argv[ index ], "--help" ) )
				Usage( );
			else if ( !strcmp( argv[ index ], "-o" ) )
				output.SetOffset( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-f" ) )
				output.SetFrames( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-b" ) )
				output.SetPumpSize( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-p" ) )
				output.SetProgress( true );
			else if ( !strcmp( argv[ index ], "-s" ) )
				output.SetSpeed( atof( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-c" ) )
				output.SetScene( atoi( argv[ ++ index ] ) );
			else
				result = output.Append( argv[ index ] );
		}

		if ( result && output.IsRunnable( ) )
		{
			output.ThreadExecute( );
		}
		else if ( argc > 1 )
		{
			std::cerr << "Failed to load " << argv[ index - 1 ] << std::endl;
			Usage( );
		}
		else
		{
			Usage( );
		}
	}
	catch ( string exc )
	{
		std::cerr << "Exception caught: " << exc << std::endl;
	}

	return !result;
}
