/*==================================================================
 * sfload.c - sound font loading functions
 * Based on the awesfx utility Copyright (C) 1996-1999 Takashi Iwai
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <glib.h>
#include "sffile.h"
#include "sfont.h"
#include "sample.h"
#include "util.h"
#include "i18n.h"

/*
   functions for loading data from sfont files, with appropriate byte swapping
   on big endian machines. Sfont IDs are not swapped because the ID read is
   equivalent to the matching ID list in memory regardless of LE/BE machine
*/

#define READCHUNK(var,fd)	G_STMT_START {		\
    if (!safe_fread(var, 8, fd))			\
	return(FAIL);					\
    ((SFChunk *)(var))->size = GUINT32_FROM_LE(((SFChunk *)(var))->size);  \
} G_STMT_END
#define READID(var,fd)		G_STMT_START {		\
    if (!safe_fread(var, 4, fd))			\
	return(FAIL);					\
} G_STMT_END
#define READSTR(var,fd)		G_STMT_START {		\
    if (!safe_fread(var, 20, fd))			\
	return(FAIL);					\
    (*var)[20] = '\0';					\
} G_STMT_END
#define READD(var,fd)		G_STMT_START {		\
    guint32 _temp;					\
    if (!safe_fread(&_temp, 4, fd))			\
	return(FAIL);					\
    var = GINT32_FROM_LE(_temp);			\
} G_STMT_END
#define READW(var,fd)		G_STMT_START {		\
    guint16 _temp;					\
    if (!safe_fread(&_temp, 2, fd))			\
	return(FAIL);					\
    var = GINT16_FROM_LE(_temp);			\
} G_STMT_END
#define READB(var,fd)		G_STMT_START {		\
    if (!safe_fread(&var, 1, fd))			\
	return(FAIL);					\
} G_STMT_END
#define FSKIP(size,fd)		G_STMT_START {		\
    if (!safe_fseek(fd, size, SEEK_CUR))		\
	return(FAIL);					\
} G_STMT_END
#define FSKIPW(fd)		G_STMT_START {		\
    if (!safe_fseek(fd, 2, SEEK_CUR))			\
	return(FAIL);					\
} G_STMT_END

/* removes and advances a GSList pointer */
#define SLADVREM(list, item)	G_STMT_START {		\
    GSList *_temp = item;				\
    item = g_slist_next(item);				\
    list = g_slist_remove_link(list, _temp);		\
    g_slist_free_1(_temp);				\
} G_STMT_END

static gint chunkid (guint32 id);
static gint load_body (gint32 size, SFData * sf, FILE * fd);
static gint read_listchunk (SFChunk * chunk, FILE * fd);
static gint process_info (gint32 size, SFData * sf, FILE * fd);
static gint process_sdta (gint32 size, SFData * sf, FILE * fd);
static gint pdtahelper (guint32 expid, guint32 reclen, SFChunk * chunk,
  gint32 * size, FILE * fd);
static gint process_pdta (gint32 size, SFData * sf, FILE * fd);
static gint load_phdr (gint32 size, SFData * sf, FILE * fd);
static gint load_pbag (gint32 size, SFData * sf, FILE * fd);
static gint load_pmod (gint32 size, SFData * sf, FILE * fd);
static gint load_pgen (gint32 size, SFData * sf, FILE * fd);
static gint load_ihdr (gint32 size, SFData * sf, FILE * fd);
static gint load_ibag (gint32 size, SFData * sf, FILE * fd);
static gint load_imod (gint32 size, SFData * sf, FILE * fd);
static gint load_igen (gint32 size, SFData * sf, FILE * fd);
static gint load_shdr (gint32 size, SFData * sf, FILE * fd);
static gint fixup_pgen (SFData * sf);
static gint fixup_igen (SFData * sf);
static gint fixup_sample (SFData * sf);

gchar idlist[] = {
  "RIFFLISTsfbkINFOsdtapdtaifilisngINAMiromiverICRDIENGIPRD"
    "ICOPICMTISFTsnamsmplphdrpbagpmodpgeninstibagimodigenshdr"
};

static gint sdtachunk_size;

/* sound font file load functions */
static gint
chunkid (guint32 id)
{
  gint i;
  guint32 *p;

  p = (guint32 *) & idlist;

  for (i = 0; i < sizeof (idlist) / sizeof (gint); i++, p += 1)
    if (*p == id)
      return (i + 1);

  return (UNKN_ID);
}

SFData *
sfload_file (gchar * fname)
{
  SFData *sf = NULL;
  FILE *fd;
  guint32 fsize;
  gint err = FALSE;

  if (!(fd = fopen (fname, "r")))
    {
      logit (LogFubar | LogErrno, _("Unable to open file \"%s\""), fname);
      return (NULL);
    }

  if (!(sf = safe_malloc (sizeof (SFData))))
    err = TRUE;
  if (!err)
    {
      memset (sf, 0, sizeof (SFData));	/* zero sfdata */
      sf->fname = g_strdup (fname);	/* copy file name */
      sf->sffd = fd;
    }

  /* get size of file */
  if (!err && fseek (fd, 0L, SEEK_END) == -1)
    {				/* seek to end of file */
      err = TRUE;
      logit (LogFubar | LogErrno, _("Seek to end of file failed"));
    }
  if (!err && (fsize = ftell (fd)) == -1)
    {				/* position = size */
      err = TRUE;
      logit (LogFubar | LogErrno, _("Get end of file position failed"));
    }
  if (!err)
    rewind (fd);

  if (!load_body (fsize, sf, fd))
    err = TRUE;			/* load the sfont */

  if (err)
    {
      if (sf)
	sfont_close (sf);
      return (NULL);
    }

  /* do this last, as most sfont_* routines set it to FALSE */
  sf->up2date = TRUE;		/* sfont in memory is in sync with file */
  sf->beensaved = FALSE;	/* has not been saved yet */

  return (sf);
}

static gint
load_body (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  READCHUNK (&chunk, fd);	/* load RIFF chunk */
  if (chunkid (chunk.id) != RIFF_ID)	/* error if not RIFF */
    return (logit (LogFubar, _("Not a RIFF file")));

  READID (&chunk.id, fd);	/* load file ID */
  if (chunkid (chunk.id) != SFBK_ID)	/* error if not SFBK_ID */
    return (logit (LogFubar, _("Not a sound font file")));

  if (chunk.size != size - 8)
    return (logit (LogFubar, _("Sound font file size mismatch")));

  /* Process INFO block */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != INFO_ID)
    return (logit (LogFubar, _("Invalid ID found when expecting INFO chunk")));
  if (!process_info (chunk.size, sf, fd))
    return (FAIL);

  /* Process sample chunk */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != SDTA_ID)
    return (logit (LogFubar,
	_("Invalid ID found when expecting SAMPLE chunk")));
  if (!process_sdta (chunk.size, sf, fd))
    return (FAIL);

  /* process HYDRA chunk */
  if (!read_listchunk (&chunk, fd))
    return (FAIL);
  if (chunkid (chunk.id) != PDTA_ID)
    return (logit (LogFubar, _("Invalid ID found when expecting HYDRA chunk")));
  if (!process_pdta (chunk.size, sf, fd))
    return (FAIL);

  if (!fixup_pgen (sf))
    return (FAIL);
  if (!fixup_igen (sf))
    return (FAIL);
  if (!fixup_sample (sf))
    return (FAIL);

  /* sort preset list by bank, preset # */
  sf->preset = g_slist_sort (sf->preset,
    (GCompareFunc) sfont_preset_compare_func);

  return (OK);
}

static gint
read_listchunk (SFChunk * chunk, FILE * fd)
{
  READCHUNK (chunk, fd);	/* read list chunk */
  if (chunkid (chunk->id) != LIST_ID)	/* error if ! list chunk */
    return (logit (LogFubar, _("Invalid chunk id in level 0 parse")));
  READID (&chunk->id, fd);	/* read id string */
  chunk->size -= 4;
  return (OK);
}

static gint
process_info (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;
  guint8 id;
  gchar *item;
  guint16 ver;

  while (size > 0)
    {
      READCHUNK (&chunk, fd);
      size -= 8;

      id = chunkid (chunk.id);

      if (id == IFIL_ID)
	{			/* sound font version chunk? */
	  if (chunk.size != 4)
	    return (logit (LogFubar,
		_("Sound font version info chunk has invalid size")));

	  READW (ver, fd);
	  sf->version.major = ver;
	  READW (ver, fd);
	  sf->version.minor = ver;

	  if (sf->version.major < 2)
	    return (logit (LogFubar,
		_("Sound font version is %d.%d which is not"
		  " supported, convert to version 2.0x"), sf->version.major,
		sf->version.minor));
	  if (sf->version.major > 2)
	    logit (LogWarn, _("Sound font version is %d.%d which is newer than"
		" what this version of Smurf was designed for (v2.0x)"));
	}
      else if (id == IVER_ID)
	{			/* ROM version chunk? */
	  if (chunk.size != 4)
	    return (logit (LogFubar,
		_("ROM version info chunk has invalid size")));

	  READW (ver, fd);
	  sf->romver.major = ver;
	  READW (ver, fd);
	  sf->romver.minor = ver;
	}
      else if (id != UNKN_ID)
	{
	  if ((id != ICMT_ID && chunk.size > 256) || (chunk.size > 65536)
	    || (chunk.size % 2))
	    return (logit (LogFubar,
		_("INFO sub chunk %.4s has invalid chunk size"
		  " of %d bytes"), &chunk.id, chunk.size));

	  /* alloc for chunk id and da chunk */
	  if (!(item = safe_malloc (chunk.size + 1)))
	    return (FAIL);

	  /* attach to INFO list, sfont_close will cleanup if FAIL occurs */
	  sf->info = g_slist_append (sf->info, item);

	  *(guint8 *) item = id;
	  if (!safe_fread (&item[1], chunk.size, fd))
	    return (FAIL);

	  /* force terminate info item (don't forget uint8 info ID) */
	  *(item + chunk.size) = '\0';
	}
      else
	return (logit (LogFubar, _("Invalid chunk id in INFO chunk")));
      size -= chunk.size;
    }

  if (size < 0)
    return (logit (LogFubar, _("INFO chunk size mismatch")));

  return (OK);
}

static gint
process_sdta (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  if (size == 0)
    return (OK);		/* no sample data? */

  /* read sub chunk */
  READCHUNK (&chunk, fd);
  size -= 8;

  if (chunkid (chunk.id) != SMPL_ID)
    return (logit (LogFubar,
	_("Expected SMPL chunk found invalid id instead")));

  if ((size - chunk.size) != 0)
    return (logit (LogFubar, _("SDTA chunk size mismatch")));

  /* sample data follows */
  sf->samplepos = ftell (fd);

  /* used in fixup_sample() to check validity of sample headers */
  sdtachunk_size = chunk.size;

  FSKIP (chunk.size, fd);

  return (OK);
}

static gint
pdtahelper (guint32 expid, guint32 reclen, SFChunk * chunk,
  gint32 * size, FILE * fd)
{
  guint32 id;
  gchar *expstr;

  expstr = CHNKIDSTR (expid);	/* in case we need it */

  READCHUNK (chunk, fd);
  *size -= 8;

  if ((id = chunkid (chunk->id)) != expid)
    return (logit (LogFubar, _("Expected"
	  " PDTA sub-chunk \"%.4s\" found invalid id instead"), expstr));

  if (chunk->size % reclen)	/* valid chunk size? */
    return (logit (LogFubar,
	_("\"%.4s\" chunk size is not a multiple of %d bytes"), expstr,
	reclen));
  if ((*size -= chunk->size) < 0)
    return (logit (LogFubar,
	_("\"%.4s\" chunk size exceeds remaining PDTA chunk size"), expstr));
  return (OK);
}

static gint
process_pdta (gint32 size, SFData * sf, FILE * fd)
{
  SFChunk chunk;

  if (!pdtahelper (PHDR_ID, SFPHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_phdr (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PBAG_ID, SFBAGSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pbag (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PMOD_ID, SFMODSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pmod (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (PGEN_ID, SFGENSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_pgen (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IHDR_ID, SFIHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_ihdr (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IBAG_ID, SFBAGSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_ibag (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IMOD_ID, SFMODSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_imod (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (IGEN_ID, SFGENSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_igen (chunk.size, sf, fd))
    return (FAIL);

  if (!pdtahelper (SHDR_ID, SFSHDRSIZE, &chunk, &size, fd))
    return (FAIL);
  if (!load_shdr (chunk.size, sf, fd))
    return (FAIL);

  return (OK);
}

/* preset header loader */
static gint
load_phdr (gint32 size, SFData * sf, FILE * fd)
{
  gint i, i2;
  SFPreset *p, *pr = NULL;	/* ptr to current & previous preset */
  guint16 zndx, pzndx;

  if (size % SFPHDRSIZE || size == 0)
    return (logit (LogFubar, _("Preset header chunk size is invalid")));

  i = size / SFPHDRSIZE - 1;
  if (i == 0)
    {				/* at least one preset + term record */
      logit (LogWarn, _("File contains no presets"));
      FSKIP (SFPHDRSIZE, fd);
      return (OK);
    }

  for (; i > 0; i--)
    {				/* load all preset headers */
      p = sfont_preset_alloc ();
      sf->preset = g_slist_append (sf->preset, p);
      READSTR (&p->name, fd);
      READW (p->prenum, fd);
      READW (p->bank, fd);
      READW (zndx, fd);
      READD (p->libr, fd);
      READD (p->genre, fd);
      READD (p->morph, fd);

      if (pr)
	{			/* not first preset? */
	  if (zndx < pzndx)
	    return (logit (LogFubar, _("Preset header indices not monotonic")));
	  i2 = zndx - pzndx;
	  while (i2--)
	    {
	      pr->zone = g_slist_prepend (pr->zone, NULL);
	    }
	}
      else if (zndx > 0)	/* 1st preset, warn if ofs >0 */
	logit (LogWarn, _("%d preset zones not referenced, discarding"), zndx);
      pr = p;			/* update preset ptr */
      pzndx = zndx;
    }

  FSKIP (24, fd);
  READW (zndx, fd);		/* Read terminal generator index */
  FSKIP (12, fd);

  if (zndx < pzndx)
    return (logit (LogFubar, _("Preset header indices not monotonic")));
  i2 = zndx - pzndx;
  while (i2--)
    {
      pr->zone = g_slist_prepend (pr->zone, NULL);
    }

  return (OK);
}

/* preset bag loader */
static gint
load_pbag (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2;
  SFZone *z, *pz = NULL;
  guint16 genndx, modndx;
  guint16 pgenndx, pmodndx;
  guint16 i;

  if (size % SFBAGSIZE || size == 0)	/* size is multiple of SFBAGSIZE? */
    return (logit (LogFubar, _("Preset bag chunk size is invalid")));

  p = sf->preset;
  while (p)
    {				/* traverse through presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse preset's zones */
	  if ((size -= SFBAGSIZE) < 0)
	    return (logit (LogFubar, _("Preset bag chunk size mismatch")));
	  z = sfont_zone_alloc ();
	  p2->data = z;
	  z->gen = NULL;	/* Init gen and mod before possible failure, */
	  z->mod = NULL;	/* to ensure proper cleanup (sfont_close) */
	  READW (genndx, fd);	/* possible read failure ^ */
	  READW (modndx, fd);
	  z->instsamp = NULL;

	  if (pz)
	    {			/* if not first zone */
	      if (genndx < pgenndx)
		return (logit (LogFubar,
		    _("Preset bag generator indices not monotonic")));
	      if (modndx < pmodndx)
		return (logit (LogFubar,
		    _("Preset bag modulator indices not monotonic")));
	      i = genndx - pgenndx;
	      while (i--)
		pz->gen = g_slist_prepend (pz->gen, NULL);
	      i = modndx - pmodndx;
	      while (i--)
		pz->mod = g_slist_prepend (pz->mod, NULL);
	    }
	  pz = z;		/* update previous zone ptr */
	  pgenndx = genndx;	/* update previous zone gen index */
	  pmodndx = modndx;	/* update previous zone mod index */
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  size -= SFBAGSIZE;
  if (size != 0)
    return (logit (LogFubar, _("Preset bag chunk size mismatch")));

  READW (genndx, fd);
  READW (modndx, fd);

  if (!pz)
    {
      if (genndx > 0)
	logit (LogWarn, _("No preset generators and terminal index not 0"));
      if (modndx > 0)
	logit (LogWarn, _("No preset modulators and terminal index not 0"));
      return (OK);
    }

  if (genndx < pgenndx)
    return (logit (LogFubar, _("Preset bag generator indices not monotonic")));
  if (modndx < pmodndx)
    return (logit (LogFubar, _("Preset bag modulator indices not monotonic")));
  i = genndx - pgenndx;
  while (i--)
    pz->gen = g_slist_prepend (pz->gen, NULL);
  i = modndx - pmodndx;
  while (i--)
    pz->mod = g_slist_prepend (pz->mod, NULL);

  return (OK);
}

/* preset modulator loader */
static gint
load_pmod (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3;
  SFMod *m;

  p = sf->preset;
  while (p)
    {				/* traverse through all presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse this preset's zones */
	  p3 = ((SFZone *) (p2->data))->mod;
	  while (p3)
	    {			/* load zone's modulators */
	      if ((size -= SFMODSIZE) < 0)
		return (logit (LogFubar,
		    _("Preset modulator chunk size mismatch")));
	      m = sfont_mod_alloc ();
	      p3->data = m;
	      READW (m->src, fd);
	      READW (m->dest, fd);
	      READW (m->amount, fd);
	      READW (m->amtsrc, fd);
	      READW (m->trans, fd);
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  /*
     If there isn't even a terminal record
     Hmmm, the specs say there should be one, but..
   */
  if (size == 0)
    return (OK);

  size -= SFMODSIZE;
  if (size != 0)
    return (logit (LogFubar, _("Preset modulator chunk size mismatch")));
  FSKIP (SFMODSIZE, fd);	/* terminal mod */

  return (OK);
}

/* -------------------------------------------------------------------
 * preset generator loader
 * generator (per preset) loading rules:
 * Zones with no generators or modulators shall be annihilated
 * Global zone must be 1st zone, discard additional ones (instrumentless zones)
 *
 * generator (per zone) loading rules (in order of decreasing precedence):
 * KeyRange is 1st in list (if exists), else discard
 * if a VelRange exists only preceded by a KeyRange, else discard
 * if a generator follows an instrument discard it
 * if a duplicate generator exists replace previous one
 * ------------------------------------------------------------------- */
static gint
load_pgen (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3, *dup, **hz;
  SFZone *z;
  SFGen *g;
  SFGenAmount genval;
  guint16 genid;
  gint level, skip, drop, gzone, discarded;

  p = sf->preset;
  while (p)
    {				/* traverse through all presets */
      gzone = FALSE;
      discarded = FALSE;
      p2 = ((SFPreset *) (p->data))->zone;
      if (p2)
	hz = &p2;
      while (p2)
	{			/* traverse preset's zones */
	  level = 0;
	  z = (SFZone *) (p2->data);
	  p3 = z->gen;
	  while (p3)
	    {			/* load zone's generators */
	      dup = NULL;
	      skip = FALSE;
	      drop = FALSE;
	      if ((size -= SFGENSIZE) < 0)
		return (logit (LogFubar,
		    _("Preset generator chunk size mismatch")));

	      READW (genid, fd);

	      if (genid == Gen_KeyRange)
		{		/* nothing precedes */
		  if (level == 0)
		    {
		      level = 1;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_VelRange)
		{		/* only KeyRange precedes */
		  if (level <= 1)
		    {
		      level = 2;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_Instrument)
		{		/* inst is last gen */
		  level = 3;
		  READW (genval.uword, fd);
		  GPOINTER_TO_INT (((SFZone *) (p2->data))->instsamp) =
		    genval.uword + 1;
		  break;	/* break out of generator loop */
		}
	      else
		{
		  level = 2;
		  if (gen_validp (genid))
		    {		/* generator valid? */
		      READW (genval.sword, fd);
		      dup = gen_inlist (genid, z->gen);
		    }
		  else
		    skip = TRUE;
		}

	      if (!skip)
		{
		  if (!dup)
		    {		/* if gen ! dup alloc new */
		      g = sfont_gen_alloc ();
		      p3->data = g;
		      g->id = genid;
		    }
		  else
		    {
		      g = (SFGen *) (dup->data);	/* ptr to orig gen */
		      drop = TRUE;
		    }
		  g->amount = genval;
		}
	      else
		{		/* Skip this generator */
		  discarded = TRUE;
		  drop = TRUE;
		  FSKIPW (fd);
		}

	      if (!drop)
		p3 = g_slist_next (p3);	/* next gen */
	      else
		SLADVREM (z->gen, p3);	/* drop place holder */

	    }			/* generator loop */

	  if (level == 3)
	    SLADVREM (z->gen, p3);	/* zone has inst? */
	  else
	    {			/* congratulations its a global zone */
	      if (!gzone)
		{		/* Prior global zones? */
		  gzone = TRUE;

		  /* if global zone is not 1st zone, relocate */
		  if (*hz != p2)
		    {
		      gpointer save = p2->data;
		      logit (LogWarn,
			_("Preset \"%s\": Global zone is not first zone"),
			((SFPreset *) (p->data))->name);
		      SLADVREM (*hz, p2);
		      *hz = g_slist_prepend (*hz, save);
		      continue;
		    }
		}
	      else
		{		/* previous global zone exists, discard */
		  logit (LogWarn,
		    _("Preset \"%s\": Discarding invalid global zone"),
		    ((SFPreset *) (p->data))->name);
		  sfont_remove_zone (sf, hz, (SFZone *) (p2->data));
		}
	    }

	  while (p3)
	    {			/* Kill any zones following an instrument */
	      discarded = TRUE;
	      if ((size -= SFGENSIZE) < 0)
		return (logit (LogFubar,
		    _("Preset generator chunk size mismatch")));
	      FSKIP (SFGENSIZE, fd);
	      SLADVREM (z->gen, p3);
	    }

	  p2 = g_slist_next (p2);	/* next zone */
	}
      if (discarded)
	logit (LogWarn,
	  _("Preset \"%s\": Some invalid generators were discarded"),
	  ((SFPreset *) (p->data))->name);
      p = g_slist_next (p);
    }

  /* in case there isn't a terminal record */
  if (size == 0)
    return (OK);

  size -= SFGENSIZE;
  if (size != 0)
    return (logit (LogFubar, _("Preset generator chunk size mismatch")));
  FSKIP (SFGENSIZE, fd);	/* terminal gen */

  return (OK);
}

/* instrument header loader */
static gint
load_ihdr (gint32 size, SFData * sf, FILE * fd)
{
  gint i, i2;
  SFInst *p, *pr = NULL;	/* ptr to current & previous instrument */
  guint16 zndx, pzndx;

  if (size % SFIHDRSIZE || size == 0)	/* chunk size is valid? */
    return (logit (LogFubar, _("Instrument header has invalid size")));

  size = size / SFIHDRSIZE - 1;
  if (size == 0)
    {				/* at least one preset + term record */
      logit (LogWarn, _("File contains no instruments"));
      FSKIP (SFIHDRSIZE, fd);
      return (OK);
    }

  for (i = 0; i < size; i++)
    {				/* load all instrument headers */
      p = sfont_inst_alloc ();
      sf->inst = g_slist_append (sf->inst, p);
      READSTR (&p->name, fd);
      READW (zndx, fd);

      if (pr)
	{			/* not first instrument? */
	  if (zndx < pzndx)
	    return (logit (LogFubar,
		_("Instrument header indices not monotonic")));
	  i2 = zndx - pzndx;
	  while (i2--)
	    pr->zone = g_slist_prepend (pr->zone, NULL);
	}
      else if (zndx > 0)	/* 1st inst, warn if ofs >0 */
	logit (LogWarn, _("%d instrument zones not referenced, discarding"),
	  zndx);
      pzndx = zndx;
      pr = p;			/* update instrument ptr */
    }

  FSKIP (20, fd);
  READW (zndx, fd);

  if (zndx < pzndx)
    return (logit (LogFubar, _("Instrument header indices not monotonic")));
  i2 = zndx - pzndx;
  while (i2--)
    pr->zone = g_slist_prepend (pr->zone, NULL);

  return (OK);
}

/* instrument bag loader */
static gint
load_ibag (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2;
  SFZone *z, *pz = NULL;
  guint16 genndx, modndx, pgenndx, pmodndx;
  gint i;

  if (size % SFBAGSIZE || size == 0)	/* size is multiple of SFBAGSIZE? */
    return (logit (LogFubar, _("Instrument bag chunk size is invalid")));

  p = sf->inst;
  while (p)
    {				/* traverse through inst */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* load this inst's zones */
	  if ((size -= SFBAGSIZE) < 0)
	    return (logit (LogFubar, _("Instrument bag chunk size mismatch")));
	  z = sfont_zone_alloc ();
	  p2->data = z;
	  z->gen = NULL;	/* In case of failure, */
	  z->mod = NULL;	/* sfont_close can clean up */
	  READW (genndx, fd);	/* READW = possible read failure */
	  READW (modndx, fd);
	  z->instsamp = NULL;

	  if (pz)
	    {			/* if not first zone */
	      if (genndx < pgenndx)
		return (logit (LogFubar,
		    _("Instrument generator indices not monotonic")));
	      if (modndx < pmodndx)
		return (logit (LogFubar,
		    _("Instrument modulator indices not monotonic")));
	      i = genndx - pgenndx;
	      while (i--)
		pz->gen = g_slist_prepend (pz->gen, NULL);
	      i = modndx - pmodndx;
	      while (i--)
		pz->mod = g_slist_prepend (pz->mod, NULL);
	    }
	  pz = z;		/* update previous zone ptr */
	  pgenndx = genndx;
	  pmodndx = modndx;
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  size -= SFBAGSIZE;
  if (size != 0)
    return (logit (LogFubar, _("Instrument chunk size mismatch")));

  READW (genndx, fd);
  READW (modndx, fd);

  if (!pz)
    {				/* in case that all are no zoners */
      if (genndx > 0)
	logit (LogWarn,
	  _("No instrument generators and terminal index not 0"));
      if (modndx > 0)
	logit (LogWarn,
	  _("No instrument modulators and terminal index not 0"));
      return (OK);
    }

  if (genndx < pgenndx)
    return (logit (LogFubar, _("Instrument generator indices not monotonic")));
  if (modndx < pmodndx)
    return (logit (LogFubar, _("Instrument modulator indices not monotonic")));
  i = genndx - pgenndx;
  while (i--)
    pz->gen = g_slist_prepend (pz->gen, NULL);
  i = modndx - pmodndx;
  while (i--)
    pz->mod = g_slist_prepend (pz->mod, NULL);

  return (OK);
}

/* instrument modulator loader */
static gint
load_imod (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3;
  SFMod *m;

  p = sf->inst;
  while (p)
    {				/* traverse through all inst */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* traverse this inst's zones */
	  p3 = ((SFZone *) (p2->data))->mod;
	  while (p3)
	    {			/* load zone's modulators */
	      if ((size -= SFMODSIZE) < 0)
		return (logit (LogFubar,
		    _("Instrument modulator chunk size mismatch")));
	      m = sfont_mod_alloc ();
	      p3->data = m;
	      READW (m->src, fd);
	      READW (m->dest, fd);
	      READW (m->amount, fd);
	      READW (m->amtsrc, fd);
	      READW (m->trans, fd);
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  /*
     If there isn't even a terminal record
     Hmmm, the specs say there should be one, but..
   */
  if (size == 0)
    return (OK);

  size -= SFMODSIZE;
  if (size != 0)
    return (logit (LogFubar, _("Instrument modulator chunk size mismatch")));
  FSKIP (SFMODSIZE, fd);	/* terminal mod */

  return (OK);
}

/* load instrument generators (see load_pgen for loading rules) */
static gint
load_igen (gint32 size, SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3, *dup, **hz;
  SFZone *z;
  SFGen *g;
  SFGenAmount genval;
  guint16 genid;
  gint level, skip, drop, gzone, discarded;

  p = sf->inst;
  while (p)
    {				/* traverse through all instruments */
      gzone = FALSE;
      discarded = FALSE;
      p2 = ((SFInst *) (p->data))->zone;
      if (p2)
	hz = &p2;
      while (p2)
	{			/* traverse this instrument's zones */
	  level = 0;
	  z = (SFZone *) (p2->data);
	  p3 = z->gen;
	  while (p3)
	    {			/* load zone's generators */
	      dup = NULL;
	      skip = FALSE;
	      drop = FALSE;
	      if ((size -= SFGENSIZE) < 0)
		return (logit (LogFubar, _("IGEN chunk size mismatch")));

	      READW (genid, fd);

	      if (genid == Gen_KeyRange)
		{		/* nothing precedes */
		  if (level == 0)
		    {
		      level = 1;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_VelRange)
		{		/* only KeyRange precedes */
		  if (level <= 1)
		    {
		      level = 2;
		      READB (genval.range.lo, fd);
		      READB (genval.range.hi, fd);
		    }
		  else
		    skip = TRUE;
		}
	      else if (genid == Gen_SampleId)
		{		/* sample is last gen */
		  level = 3;
		  READW (genval.uword, fd);
		  GPOINTER_TO_INT (((SFZone *) (p2->data))->instsamp) =
		    genval.uword + 1;
		  break;	/* break out of generator loop */
		}
	      else
		{
		  level = 2;
		  if (gen_valid (genid))
		    {		/* gen valid? */
		      READW (genval.sword, fd);
		      dup = gen_inlist (genid, z->gen);
		    }
		  else
		    skip = TRUE;
		}

	      if (!skip)
		{
		  if (!dup)
		    {		/* if gen ! dup alloc new */
		      g = sfont_gen_alloc ();
		      p3->data = g;
		      g->id = genid;
		    }
		  else
		    {
		      g = (SFGen *) (dup->data);
		      drop = TRUE;
		    }
		  g->amount = genval;
		}
	      else
		{		/* skip this generator */
		  discarded = TRUE;
		  drop = TRUE;
		  FSKIPW (fd);
		}

	      if (!drop)
		p3 = g_slist_next (p3);	/* next gen */
	      else
		SLADVREM (z->gen, p3);

	    }			/* generator loop */

	  if (level == 3)
	    SLADVREM (z->gen, p3);	/* zone has sample? */
	  else
	    {			/* its a global zone */
	      if (!gzone)
		{
		  gzone = TRUE;

		  /* if global zone is not 1st zone, relocate */
		  if (*hz != p2)
		    {
		      gpointer save = p2->data;
		      logit (LogWarn,
			_("Instrument \"%s\": Global zone is not first zone"),
			((SFPreset *) (p->data))->name);
		      SLADVREM (*hz, p2);
		      *hz = g_slist_prepend (*hz, save);
		      continue;
		    }
		}
	      else
		{		/* previous global zone exists, discard */
		  logit (LogWarn,
		    _("Instrument \"%s\": Discarding invalid global zone"),
		    ((SFInst *) (p->data))->name);
		  sfont_remove_zone (sf, hz, (SFZone *) (p2->data));
		}
	    }

	  while (p3)
	    {			/* Kill any zones following a sample */
	      discarded = TRUE;
	      if ((size -= SFGENSIZE) < 0)
		return (logit (LogFubar,
		    _("Instrument generator chunk size mismatch")));
	      FSKIP (SFGENSIZE, fd);
	      SLADVREM (z->gen, p3);
	    }

	  p2 = g_slist_next (p2);	/* next zone */
	}
      if (discarded)
	logit (LogWarn,
	  _("Instrument \"%s\": Some invalid generators were discarded"),
	  ((SFInst *) (p->data))->name);
      p = g_slist_next (p);
    }

  /* for those non-terminal record cases, grr! */
  if (size == 0)
    return (OK);

  size -= SFGENSIZE;
  if (size != 0)
    return (logit (LogFubar, _("IGEN chunk size mismatch")));
  FSKIP (SFGENSIZE, fd);	/* terminal gen */

  return (OK);
}

/* sample header loader */
static gint
load_shdr (gint32 size, SFData * sf, FILE * fd)
{
  guint32 i;
  SFSample *p;

  if (size % SFSHDRSIZE || size == 0)	/* size is multiple of SHDR size? */
    return (logit (LogFubar, _("Sample header has invalid size")));

  size = size / SFSHDRSIZE - 1;
  if (size == 0)
    {				/* at least one sample + term record? */
      logit (LogWarn, _("File contains no samples"));
      FSKIP (SFSHDRSIZE, fd);
      return (OK);
    }

  /* load all sample headers */
  for (i = 0; i < size; i++)
    {
      p = sfont_sample_alloc ();
      sf->sample = g_slist_append (sf->sample, p);

      p->datainfo = sam_datainfo_new ();  /* create new sam data info struct */
      p->datainfo->samfile = FALSE;  /* sample is in sound font */
      p->datainfo->sf = sf;
      p->datainfo->refcount++;  /* update reference count */

      READSTR (&p->name, fd);
      READD (p->datainfo->start, fd);
      READD (p->end, fd);	/* - end, loopstart and loopend */
      READD (p->loopstart, fd);	/* - will be checked and turned into */
      READD (p->loopend, fd);	/* - offsets in fixup_sample() */
      READD (p->samplerate, fd);
      READB (p->origpitch, fd);
      READB (p->pitchadj, fd);
      FSKIPW (fd);		/* skip sample link */
      READW (p->sampletype, fd);
    }

  FSKIP (SFSHDRSIZE, fd);	/* skip terminal shdr */

  return (OK);
}

/* "fixup" (inst # -> inst ptr) instrument references in preset list */
static gint
fixup_pgen (SFData * sf)
{
  GSList *p, *p2, *p3;
  SFZone *z;
  gint i;

  p = sf->preset;
  while (p)
    {
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* traverse this preset's zones */
	  z = (SFZone *) (p2->data);
	  if ((i = GPOINTER_TO_INT (z->instsamp)))
	    {			/* load instrument # */
	      p3 = g_slist_nth (sf->inst, i - 1);
	      if (!p3)
		return (logit (LogFubar,
		    _("Preset %03d %03d: Invalid instrument reference"),
		    ((SFPreset *) (p->data))->bank,
		    ((SFPreset *) (p->data))->prenum));
	      z->instsamp = p3;
	    }
	  else
	    z->instsamp = NULL;
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  return (OK);
}

/* "fixup" (sample # -> sample ptr) sample references in instrument list */
static gint
fixup_igen (SFData * sf)
{
  GSList *p, *p2, *p3;
  SFZone *z;
  gint i;

  p = sf->inst;
  while (p)
    {
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* traverse instrument's zones */
	  z = (SFZone *) (p2->data);
	  if ((i = GPOINTER_TO_INT (z->instsamp)))
	    {			/* load sample # */
	      p3 = g_slist_nth (sf->sample, i - 1);
	      if (!p3)
		return (logit (LogFubar,
		    _("Instrument \"%s\": Invalid sample reference"),
		    ((SFInst *) (p->data))->name));
	      z->instsamp = p3;
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  return (OK);
}

/* convert sample end, loopstart and loopend to offsets and check if valid */
static gint
fixup_sample (SFData * sf)
{
  GSList *p;
  SFSample *sam;

  p = sf->sample;
  while (p)
    {
      sam = (SFSample *) (p->data);

      /* if sample is not a ROM sample and end is over the sample data chunk
         or sam start is greater than 4 less than the end (at least 4 samples) */
      if ((!(sam->sampletype & SF_SAMPLETYPE_ROM)
	  && sam->end > sdtachunk_size)
	  || sam->datainfo->start > (sam->end - 4))
	{
	  logit (LogWarn, _("Sample '%s' start/end file positions are invalid,"
	      " disabling and will not be saved"), sam->name);

	  /* disable sample by setting all sample markers to 0 */
	  sam->datainfo->start = sam->end = sam->loopstart = sam->loopend = 0;

	  return (OK);
	}
      else if (sam->loopend > sam->end || sam->loopstart >= sam->loopend
	|| sam->loopstart <= sam->datainfo->start)
	{			/* loop is fowled?? (cluck cluck :) */
	  /* can pad loop by 8 samples and ensure at least 4 for loop (2*8+4) */
	  if ((sam->end - sam->datainfo->start) >= 20)
	    {
	      sam->loopstart = sam->datainfo->start + 8;
	      sam->loopend = sam->end - 8;
	    }
	  else
	    {			/* loop is fowled, sample is tiny (can't pad 8 samples) */
	      sam->loopstart = sam->datainfo->start + 1;
	      sam->loopend = sam->end - 1;
	    }
	}

      /* convert sample end, loopstart, loopend to offsets from sam->start */
      sam->end -= sam->datainfo->start + 1;	/* marks last sample, contrary to SF spec. (wish I hadn't done this, will change again later) */
      sam->loopstart -= sam->datainfo->start;
      sam->loopend -= sam->datainfo->start;

      sam->datainfo->size = sam->end + 1;  /* update sam data info size */

      p = g_slist_next (p);
    }

  return (OK);
}
