/*==================================================================
 * sfont.h - Header file for Sound font routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#ifndef SFONT_H_DEF
#define SFONT_H_DEF

#include <stdio.h>
#include <glib.h>

/* NOTE! : ISFT version used to be saved as "Smurf v" prior to v0.49.8,
   now this string is set here with more complete name, this string is used
   for the name portion of the ISFT info field */
#define SFONT_ISFT_NAME "Smurf Sound Font Editor v"

/*
    A safe method of changing a sound font variable and setting the up2date flag.
    Compare a sound font variable to a value: if they differ, assign the value
    to the variable and set the up2date flag to FALSE
*/
#define SFONT_SETVAR(var, val, sf)	G_STMT_START {	\
    if (var != val) {					\
	var = val;					\
	sf->up2date = FALSE;				\
    }							\
} G_STMT_END

#define SF_SAMPLETYPE_MONO	1
#define SF_SAMPLETYPE_RIGHT	2
#define SF_SAMPLETYPE_LEFT	4
#define SF_SAMPLETYPE_LINKED	8
#define SF_SAMPLETYPE_ROM	0x8000

#define SF_SAMPMODES_LOOP	1
#define SF_SAMPMODES_UNROLL	2

#define SF_MIN_SAMPLERATE	400
#define SF_MAX_SAMPLERATE	50000

#define SF_MIN_SAMPLE_LENGTH	32

/* SFItemIDs in sound font items (SFData, SFPreset, SFInst, SFSample
   and SFZone): This field is filled in when an item gets added to the
   SFTree. The ID is a reference to a balanced binary tree of
   ctree nodes. If the ID field for an item is 0 when it is
   inserted into the tree then a new unique id is generated and
   placed in the ID field, otherwise the ID is retained.
   NOTE: The ID field is expected to be first of the SFPreset and
   SFInst structures */

typedef gint SFItemID;

/* Sound Font structure defines */

typedef struct _SFVersion
{				/* version structure */
  guint16 major;
  guint16 minor;
}
SFVersion;

typedef struct _SFMod
{				/* Modulator structure */
  guint16 src;			/* source modulator */
  guint16 dest;			/* destination generator */
  gint16 amount;		/* signed, degree of modulation */
  guint16 amtsrc;		/* second source controls amnt of first */
  guint16 trans;		/* transform applied to source */
}
SFMod;

typedef union _SFGenAmount
{				/* Generator amount structure */
  gint16 sword;			/* signed 16 bit value */
  guint16 uword;		/* unsigned 16 bit value */
  struct
  {
    guint8 lo;			/* low value for ranges */
    guint8 hi;			/* high value for ranges */
  }
  range;
}
SFGenAmount;

typedef struct _SFGen
{				/* Generator structure */
  guint16 id;			/* generator ID */
  SFGenAmount amount;		/* generator value */
}
SFGen;

typedef struct _SFZone
{				/* Sample/instrument zone structure */
  SFItemID itemid;		/* unique item id (see SFItemID note above) */
  GSList *instsamp;		/* instrument/sample pointer for zone */
  GSList *gen;			/* list of generators */
  GSList *mod;			/* list of modulators */
}
SFZone;

typedef struct _SFSamDataInfo
{
  gpointer patchid;		/* pointer to wavetable driver specific patch
				   ID info (or NULL if not set) */
  gboolean samfile;		/* Loaded sfont/sample buffer = FALSE/TRUE */
  struct _SFData *sf;		/* if samfile == FALSE, sfont of sam data */
  guint start;			/* Offset to start of sample (in samples) */
  guint size;			/* size of sample data (in samples) */
  gint refcount;		/* active item reference count */
  gint dorefcount;		/* undo/redo reference count */
}
SFSamDataInfo;

typedef struct _SFSample
{				/* Sample structure */
  SFItemID itemid;		/* unique item id (see SFItemID note above) */

  SFSamDataInfo *datainfo;	/* the sample data information */

  gchar name[21];		/* Name of sample */
  guint32 end;			/* Offset from start to end of sample, this
				   is the last point of the sample, the SF
				   spec has this as the 1st point after,
				   corrected on load/save */
  guint32 loopstart;		/* Offset from start to start of loop */
  guint32 loopend;		/* Offset from start to end of loop, marks the
				   first point after loop, whose sample value
				   is ideally equivalent to loopstart */
  guint32 samplerate;		/* Sample rate recorded at */
  guint8 origpitch;		/* root midi key number */
  gint8 pitchadj;		/* pitch correction in cents */
  guint16 sampletype;		/* 1 mono,2 right,4 left,linked 8,0x8000=ROM */
}
SFSample;

typedef struct _SFInst
{				/* Instrument structure */
  SFItemID itemid;		/* unique item id (see SFItemID note above) */
  gchar name[21];		/* Name of instrument */
  GSList *zone;			/* list of instrument zones */
}
SFInst;

typedef struct _SFPreset
{				/* Preset structure */
  SFItemID itemid;		/* unique item id (see SFItemID note above) */
  gchar name[21];		/* preset name */
  guint16 prenum;		/* preset number */
  guint16 bank;			/* bank number */
  guint32 libr;			/* Not used (preserved) */
  guint32 genre;		/* Not used (preserved) */
  guint32 morph;		/* Not used (preserved) */
  GSList *zone;			/* list of preset zones */
}
SFPreset;

/* NOTE: sffd is also used to determine if sound font is new (NULL) */
typedef struct _SFData
{				/* Sound font data structure */
  SFItemID itemid;		/* unique item id (see SFItemID note above) */
  SFVersion version;		/* sound font version */
  SFVersion romver;		/* ROM version */
  guint32 samplepos;		/* position within sffd of the sample chunk */
  gchar *fname;			/* file name */
  FILE *sffd;			/* loaded sfont file descriptor */
  GSList *info;		     /* linked list of info strings (1st byte is ID) */
  GSList *preset;		/* linked list of preset info */
  GSList *inst;			/* linked list of instrument info */
  GSList *sample;		/* linked list of sample info */
  gboolean up2date;		/* saved sound font is up to date? */
  gboolean beensaved;		/* been saved since open/creation? */
}
SFData;

/* sf file chunk IDs */
enum
{ UNKN_ID, RIFF_ID, LIST_ID, SFBK_ID,
  INFO_ID, SDTA_ID, PDTA_ID,	/* info/sample/preset */

  IFIL_ID, ISNG_ID, INAM_ID, IROM_ID, /* info ids (1st byte of info strings) */
  IVER_ID, ICRD_ID, IENG_ID, IPRD_ID,	/* more info ids */
  ICOP_ID, ICMT_ID, ISFT_ID,	/* and yet more info ids */

  SNAM_ID, SMPL_ID,		/* sample ids */
  PHDR_ID, PBAG_ID, PMOD_ID, PGEN_ID,	/* preset ids */
  IHDR_ID, IBAG_ID, IMOD_ID, IGEN_ID,	/* instrument ids */
  SHDR_ID			/* sample info */
};
/* define the range of known info tags */
#define TAG_INFOSTART IFIL_ID
#define TAG_INFOEND ISFT_ID

/* generator types */
enum
{ Gen_StartAddrOfs, Gen_EndAddrOfs, Gen_StartLoopAddrOfs,
  Gen_EndLoopAddrOfs, Gen_StartAddrCoarseOfs, Gen_ModLFO2Pitch,
  Gen_VibLFO2Pitch, Gen_ModEnv2Pitch, Gen_FilterFc, Gen_FilterQ,
  Gen_ModLFO2FilterFc, Gen_ModEnv2FilterFc, Gen_EndAddrCoarseOfs,
  Gen_ModLFO2Vol, Gen_Unused1, Gen_ChorusSend, Gen_ReverbSend, Gen_Pan,
  Gen_Unused2, Gen_Unused3, Gen_Unused4,
  Gen_ModLFODelay, Gen_ModLFOFreq, Gen_VibLFODelay, Gen_VibLFOFreq,
  Gen_ModEnvDelay, Gen_ModEnvAttack, Gen_ModEnvHold, Gen_ModEnvDecay,
  Gen_ModEnvSustain, Gen_ModEnvRelease, Gen_Key2ModEnvHold,
  Gen_Key2ModEnvDecay, Gen_VolEnvDelay, Gen_VolEnvAttack,
  Gen_VolEnvHold, Gen_VolEnvDecay, Gen_VolEnvSustain, Gen_VolEnvRelease,
  Gen_Key2VolEnvHold, Gen_Key2VolEnvDecay, Gen_Instrument,
  Gen_Reserved1, Gen_KeyRange, Gen_VelRange,
  Gen_StartLoopAddrCoarseOfs, Gen_Keynum, Gen_Velocity,
  Gen_Attenuation, Gen_Reserved2, Gen_EndLoopAddrCoarseOfs,
  Gen_CoarseTune, Gen_FineTune, Gen_SampleId, Gen_SampleModes,
  Gen_Reserved3, Gen_ScaleTune, Gen_ExclusiveClass, Gen_OverrideRootKey,
  Gen_Dummy
}
Gen_Type;

#define SFGen_MaxValid 	Gen_Dummy - 1	/* maximum valid generator */
#define SFGen_Count	Gen_Dummy	/* count of generators */
#define SFGen_ArraySize sizeof(SFGenAmount)*SFGen_Count	/* gen array size */

/* generator unit type */
enum
{
  None,				/* No unit type */
  Unit_Smpls,			/* in samples */
  Unit_32kSmpls,		/* in 32k samples */
  Unit_Cent,			/* in cents (1/100th of a semitone) */
  Unit_HzCent,			/* in Hz Cents */
  Unit_TCent,			/* in Time Cents */
  Unit_cB,			/* in centibels (1/100th of a decibel) */
  Unit_Percent,			/* in percentage */
  Unit_Semitone,		/* in semitones */
  Unit_Range			/* a range of values */
}
Gen_Unit;

/* unit conversion info structure */
typedef struct _GenConv
{
  gint (*user2sf) (float val);	/* user units -> sfont units */
  float (*sf2user) (SFGenAmount val);	/* sfont units -> user units */
    gint (*ofs2sf) (float val);	/* user ofs units -> sfont units */
  float (*sf2ofs) (SFGenAmount val);	/* sfont units -> user ofs units */
  guint8 digits;		/* digits to display after '.' */
  gchar *unittxt;		/* user unit text */
  gchar *ofstxt;		/* user unit offset text */
}
GenConv;

/* generator constraints structure */
typedef struct _GenParms
{
  gint16 min;			/* Min value allowed */
  gint16 max;			/* Max value allowed */
  gint16 def;			/* Default value */
  gint unit;			/* Unit type */
  gchar *label;			/* Descriptive label */
}
GenParms;

/* duplicate mode for sfont_(preset|inst|sample|zone)_dup functions */
typedef enum { SFDUP_NORMAL, SFDUP_ARCHIVE, SFDUP_RESTORE } SFDupEnum;

/* global data */

GTree *sfont_itemids;

extern guint16 badgen[];	/* list of bad generators */
extern guint16 badpgen[];	/* list of bad preset generators */

extern GenConv genconv[];	/* generator unit conversion function table */
extern GenParms genparms[];	/* generator constraint table */

/* functions */
void sfont_init_chunks (void);
SFItemID sfont_next_itemid (void);
void sfont_register_itemid (SFItemID itemid, gpointer val);
gpointer sfont_lookup_itemid (SFItemID itemid);
void sfont_remove_itemid (SFItemID itemid);

SFData *sfont_new (gint titlenum);
void sfont_close (SFData * sf);
void sfont_destroy (SFData * sf);
SFData *sfont_sfdata_alloc (void);
void sfont_sfdata_free (SFData *sf);

SFPreset *sfont_new_preset (SFData * sf, gchar * name, guint16 bank,
			    guint16 prenum);
void sfont_add_preset (SFData * sf, SFPreset * pset);
void sfont_remove_preset (SFData * sf, SFPreset * pset);
gint sfont_preset_compare_func (gconstpointer a, gconstpointer b);

SFInst *sfont_new_inst (SFData * sf, gchar * name);
void sfont_add_inst (SFData * sf, SFInst * inst);
void sfont_insert_inst (SFData * sf, SFInst * inst, gint pos);
void sfont_remove_inst (SFData * sf, SFInst * inst);
void sfont_add_sample (SFData * sf, SFSample * sam);
void sfont_insert_sample (SFData * sf, SFSample * sam, gint pos);
void sfont_remove_sample (SFData * sf, SFSample * sam);
SFZone *sfont_new_preset_zone (SFData * sf, SFPreset * pset, GSList * inst);
gint sfont_add_preset_zone (SFData * sf, SFPreset * pset, SFZone * zone);
SFZone *sfont_new_inst_zone (SFData * sf, SFInst * inst, GSList * sam);
gint sfont_add_inst_zone (SFData * sf, SFInst * inst, SFZone * zone);
void sfont_remove_zone (SFData * sf, GSList ** zlist, SFZone * zone);
void sfont_find_free_preset (SFData *sf, gint *bank, gint *prenum,
			     gboolean melodic);
GSList *sfont_find_preset (SFData * sf, gchar * name, guint bank, guint prenum,
			   SFPreset * excl);
GSList *sfont_find_inst (SFData * sf, gchar * name, SFInst * excl);
GSList *sfont_find_sample (SFData * sf, gchar * name, SFSample * excl);
void sfont_set_fname (SFData * sf, gchar * fname);
gchar *sfont_get_info (SFData * sf, guint8 id);
void sfont_set_info (SFData * sf, guint8 id, gchar * str);
void sfont_set_namestr (SFData * sf, gchar * namestr, gchar * name);
void sfont_set_namestr_nosf (gchar * namestr, gchar * name);

SFPreset *sfont_preset_dup (SFPreset * pset, SFDupEnum mode);
void sfont_preset_destroy (SFPreset * pset);
SFPreset *sfont_preset_alloc (void);
void sfont_preset_free (SFPreset *preset);

SFInst *sfont_inst_dup (SFInst * inst, SFDupEnum mode);
void sfont_inst_destroy (SFInst * inst);
SFInst *sfont_inst_alloc (void);
void sfont_inst_free (SFInst *inst);

SFSample *sfont_sample_dup (SFSample *sam, SFDupEnum mode, gboolean refcount);
void sfont_sample_destroy (SFSample *sam, gboolean refcount);
SFSample *sfont_sample_alloc (void);
void sfont_sample_free (SFSample *sam);
SFSamDataInfo *sfont_samdatainfo_alloc (void);
void sfont_samdatainfo_free (SFSamDataInfo *datainfo);

void sfont_zone_destroy (SFZone * zone);
SFZone * sfont_zone_dup (SFZone * zone, SFDupEnum mode);
SFZone * sfont_zone_alloc (void);
void sfont_zone_free (SFZone *zone);

SFMod *sfont_mod_alloc (void);
void sfont_mod_free (SFMod * mod);

SFGen *sfont_gen_alloc (void);
void sfont_gen_free (SFGen * gen);
float gen_sf2userval (guint16 gen, SFGenAmount amt, gboolean absval);
float gen_sf2userstr (guint16 gen, SFGenAmount amt, gchar * buf,
		      gboolean absval);
gint16 gen_userstr2sf (guint16 gen, gchar * val, gboolean absval,
		       gboolean * err);
void sfont_gen_set (SFData * sf, SFZone * zone, guint16 gen, SFGenAmount amt,
		    gboolean preset);
void sfont_gen_unset (SFData * sf, SFZone * zone, guint16 gen);
SFGenAmount *sfont_gen_get (SFZone * zone, guint16 gen);
void gen_initdef (SFGenAmount * garr);
void gen_initofs (SFGenAmount * garr);
void gen_process_zone (SFZone * zone, SFGenAmount * garr);
void gen_offset_zone (SFGenAmount * abs, SFGenAmount * ofs);
SFGenAmount range_intersect (SFGenAmount a, SFGenAmount b);
GSList *gen_inlist (gint gen, GSList * genlist);
gint gen_valid (gint gen);
gint gen_validp (gint gen);

gint float2int (float f);
float int2float (SFGenAmount val);
gint hz2cents (float hz);
float cents2hz (SFGenAmount cents);
gint db2cb (float db);
float cb2db (SFGenAmount cb);
gint sec2tcent (float sec);
float tcent2sec (SFGenAmount tcent);
gint perc2tperc (float perc);
float tperc2perc (SFGenAmount tperc);

#endif
