/*==================================================================
 * uif_pref.c - User interface routines for preferences
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "uif_pref.h"
#include "uif_sftree.h"
#include "uif_piano.h"
#include "glade_interface.h"
#include "uiface.h"
#include "smurfcfg.h"
#include "widgets/popdog.h"
#include "sfont.h"		/* to set sfont_minsamloop/pad */
#include "wavetable.h"		/* list of wavetable drivers */
#include "sequencer.h"		/* list of sequencer drivers */
#include "midi.h"		/* list of midi drivers */
#include "util.h"
#include "i18n.h"

/* Prototypes */

static void pref_apply_changes (GtkWidget * prefwin);
static void pref_save_piano_keytable (void);
static void pref_cb_path_browse_ok (GtkWidget * btn, GtkWidget * filesel);
static void pref_cb_vkeyb_selection_done (GtkWidget * menu,
					  GtkWidget * prefwin);
static void pref_toggle_sensitive_widgets (GtkWidget * btn,
					   GtkWidget * prefwin, gchar **names);
static void pref_cb_mthru_selection_done (GtkWidget * menu,
					  GtkWidget * prefwin);
static void pref_load_piano_key_clist (GtkWidget * clist, gboolean lowoct);
static void pref_update_keycapture_msg (GtkWidget *prefwin);


static gboolean pref_keycapture;  /* if piano key capture is in effect */
static gboolean pref_change_all_keys;  /* change all keys active? */

static gchar *keynames[] =
{
  "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"
};

guint pref_temp_keytable [UIPIANO_TOTAL_NUMKEYS];

void
pref_create (void)
{
  GtkWidget *prefwin;
  GtkWidget *widg;
  GtkWidget *menu, *item;
  GTokenValue *val;
  gchar *s;
  gint i, i2;
  gint sfspec;

  if (util_activate_unique_dialog ("pref", 0)) return;

  prefwin = create_prefwin ();

  util_register_unique_dialog (prefwin, "pref", 0);

  /* Initialize the General notebook page */

  val = smurfcfg_get_val (SMURFCFG_DEF_SFONT_PATH);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENsfpath");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  val = smurfcfg_get_val (SMURFCFG_DEF_SAMPLE_PATH);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENsampath");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  val = smurfcfg_get_val (SMURFCFG_VBNK_SEARCH_PATH);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENvbnkpaths");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  val = smurfcfg_get_val (SMURFCFG_SAVE_WIN_GEOM);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKwingeom");
  if (val->v_int)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  val = smurfcfg_get_val (SMURFCFG_SET_WIN_POS);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKwinpos");
  if (val->v_int)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  val = smurfcfg_get_val (SMURFCFG_SET_PANE_SIZE);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKpanesize");
  if (val->v_int)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKtips");
  if (smurfcfg_get_val (SMURFCFG_TIPS_ENABLED)->v_int)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKsplash");
  if (smurfcfg_get_val (SMURFCFG_DISABLE_SPLASH)->v_int)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  val = smurfcfg_get_val (SMURFCFG_QUIT_CONFIRM);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPquit");
  if (val->v_int <= SMURFCFG_QUIT_CONFIRM_NEVER)
    gtk_option_menu_set_history (GTK_OPTION_MENU (widg), val->v_int);

  /* Initialize the Sound Font notebook page */

  val = smurfcfg_get_val (SMURFCFG_TEMP_AUDIBLE_BANK);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBbank");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)val->v_int);

  val = smurfcfg_get_val (SMURFCFG_TEMP_AUDIBLE_PRESET);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBpset");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)val->v_int);

  val = smurfcfg_get_val (SMURFCFG_WAVETBL_SAM_CACHING);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPsamcache");
  i = val->v_int;
  if (i < 0 || i > 2) i = 2;
  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i);

  val = smurfcfg_get_val (SMURFCFG_SAM_SFSPEC);
  sfspec = val->v_int;

  if (sfspec)
    widg = gtk_object_get_data (GTK_OBJECT (prefwin), "RADsamspec");
  else
    widg = gtk_object_get_data (GTK_OBJECT (prefwin), "RADsamcustom");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);

  val = smurfcfg_get_val (SMURFCFG_SAM_MINLOOP);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminloop");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (gfloat) val->v_int);
  if (sfspec)
    gtk_widget_set_sensitive (widg, FALSE);

  val = smurfcfg_get_val (SMURFCFG_SAM_MINLOOPPAD);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminpad");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (gfloat) val->v_int);
  if (sfspec)
    gtk_widget_set_sensitive (widg, FALSE);

  val = smurfcfg_get_val (SMURFCFG_SAM_BUF_WASTE);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmaxwaste");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (gfloat) val->v_int);

  val = smurfcfg_get_val (SMURFCFG_SAM_CH1_POSTFIX);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENleftch");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  val = smurfcfg_get_val (SMURFCFG_SAM_CH2_POSTFIX);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENrightch");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  /* Initialize the Driver config page */

  /* wavetable driver frame */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPwavtbl");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));
  s = smurfcfg_get_val (SMURFCFG_WAVETBL_DRIVER)->v_string;

  /* add list of compiled in drivers to menu (AUTO already added) */
  for (i = 0, i2 = 0; i < WTBL_COUNT; i++)
    {
      /* if this driver is the current setting, save index for later */
      if (strcmp (s, wtbl_drivers[i].name) == 0)
	i2 = i + 1;
      item = gtk_menu_item_new_with_label (_(wtbl_drivers[i].name));
      gtk_widget_show (item);
      gtk_menu_append (GTK_MENU (menu), item);
    }

  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i2);

  val = smurfcfg_get_val (SMURFCFG_WAVETBL_OSSDEV);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENwtblossdev");
  gtk_entry_set_text (GTK_ENTRY (widg), val->v_string);

  /* Virtual Keyboard driver frame */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPvkeyb");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));

  gtk_signal_connect (GTK_OBJECT (menu), "selection-done",
		      (GtkSignalFunc) pref_cb_vkeyb_selection_done, prefwin);

  s = smurfcfg_get_val (SMURFCFG_SEQ_DRIVER)->v_string;

  /* add list of compiled in drivers to menu (AUTO already added) */
  for (i = 0, i2 = 0; i < SEQ_COUNT; i++)
    {
      /* if this driver is the current setting, save index for later */
      if (strcmp (s, seq_drivers[i].name) == 0)
	i2 = i + 1;
      item = gtk_menu_item_new_with_label (_(seq_drivers[i].name));
      gtk_widget_show (item);
      gtk_menu_append (GTK_MENU (menu), item);
    }

  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i2);

  /* "selection-done" signal is not activated by gtk_option_menu_set_history */
  pref_cb_vkeyb_selection_done (menu, prefwin);

  i = smurfcfg_get_val (SMURFCFG_SEQ_ALSACLIENT)->v_int;
  if (i == 0)			/* client #0 is for AUTO */
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKvkauto");
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);
      i = 1;
      i2 = 0;
    }
  else
    i2 = smurfcfg_get_val (SMURFCFG_SEQ_ALSAPORT)->v_int;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBvkclient");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBvkport");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i2);

  /* Midi thru driver frame */

  /* MIDI driver selection menu */
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPmidi");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));

  gtk_signal_connect (GTK_OBJECT (menu), "selection-done",
		      (GtkSignalFunc) pref_cb_mthru_selection_done, prefwin);

  s = smurfcfg_get_val (SMURFCFG_MIDI_DRIVER)->v_string;

  for (i = 0, i2 = 0; i < MIDI_COUNT; i++)
    {
      /* if this driver is the current setting, save index for later */
      if (strcmp (s, midi_drivers[i].name) == 0)
	i2 = i + 1;
      item = gtk_menu_item_new_with_label (_(midi_drivers[i].name));
      gtk_widget_show (item);
      gtk_menu_append (GTK_MENU (menu), item);
    }

  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i2);

  /* "selection-done" signal is not activated by gtk_option_menu_set_history */
  pref_cb_mthru_selection_done (menu, prefwin);

  i = smurfcfg_get_val (SMURFCFG_MIDI_ALSA_INCLIENT)->v_int;
  if (i == 0)			/* client #0 is for AUTO */
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKmtinauto");
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);
      i = 1;
      i2 = 0;
    }
  else
    i2 = smurfcfg_get_val (SMURFCFG_MIDI_ALSA_INPORT)->v_int;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtinclient");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtinport");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i2);

  i = smurfcfg_get_val (SMURFCFG_MIDI_ALSA_OUTCLIENT)->v_int;
  if (i == 0)			/* client #0 is for AUTO */
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKmtoutauto");
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);
      i = 1;
      i2 = 0;
    }
  else
    i2 = smurfcfg_get_val (SMURFCFG_MIDI_ALSA_OUTPORT)->v_int;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtoutclient");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtoutport");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)i2);

  val = smurfcfg_get_val (SMURFCFG_MIDI_ALSACARD);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtcard");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (gfloat) val->v_int);

  val = smurfcfg_get_val (SMURFCFG_MIDI_ALSADEVICE);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtdev");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (gfloat) val->v_int);


  /* Piano key mapping frame */

  /* copy key codes to temporary array */
  for (i = 0; i < UIPIANO_TOTAL_NUMKEYS; i++)
    pref_temp_keytable[i] = uipiano_keytable[i];

  /* load piano key CLIST */
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");
  pref_load_piano_key_clist (widg, TRUE);

  pref_keycapture = FALSE;
  pref_change_all_keys = FALSE;

  gtk_widget_show (prefwin);
}

/* user said Okay to preferences, so apply all pref changes */
void
pref_cb_set_pref_okay (GtkWidget * btn, GtkWidget * prefwin)
{
  pref_apply_changes (prefwin);
}

/* user clicked "Save Preferences" button, apply pref changes, save smurfcfg */
void
pref_cb_save_pref_clicked (GtkWidget * btn, GtkWidget * prefwin)
{
  pref_apply_changes (prefwin);
  smurfcfg_save ();
}

/* applies changes user made to preferences to smurfcfg vars */
static void
pref_apply_changes (GtkWidget * prefwin)
{
  GTokenValue val;
  GtkWidget *widg;
  gint i, i2;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENsfpath");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_DEF_SFONT_PATH, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENsampath");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_DEF_SAMPLE_PATH, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENvbnkpaths");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_VBNK_SEARCH_PATH, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKwingeom");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_SAVE_WIN_GEOM, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKwinpos");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_SET_WIN_POS, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKpanesize");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_SET_PANE_SIZE, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKtips");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_TIPS_ENABLED, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKsplash");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_DISABLE_SPLASH, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPquit");
  UTIL_OPMENU_INDEX (val.v_int, widg);
  smurfcfg_set_val (SMURFCFG_QUIT_CONFIRM, &val);

  /* sound font tab */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBbank");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_TEMP_AUDIBLE_BANK, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBpset");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_TEMP_AUDIBLE_PRESET, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPsamcache");
  UTIL_OPMENU_INDEX (val.v_int, widg);
  smurfcfg_set_val (SMURFCFG_WAVETBL_SAM_CACHING, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "RADsamspec");
  val.v_int = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_SAM_SFSPEC, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminloop");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  val.v_int = CLAMP (val.v_int, 1, 99);	/* clamp to slightly sane vals */
  smurfcfg_set_val (SMURFCFG_SAM_MINLOOP, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminpad");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  val.v_int = CLAMP (val.v_int, 1, 99);	/* clamp to slightly sane vals */
  smurfcfg_set_val (SMURFCFG_SAM_MINLOOPPAD, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENleftch");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_SAM_CH1_POSTFIX, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENrightch");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_SAM_CH2_POSTFIX, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmaxwaste");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  val.v_int = CLAMP (val.v_int, 0, 100); /* slightly sane? */
  smurfcfg_set_val (SMURFCFG_SAM_BUF_WASTE, &val);

  /* drivers tab */

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPwavtbl");
  UTIL_OPMENU_INDEX (i, widg);
  if (i == 0)
    val.v_string = "AUTO";
  else
    val.v_string = wtbl_drivers[i - 1].name;
  smurfcfg_set_val (SMURFCFG_WAVETBL_DRIVER, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "ENwtblossdev");
  val.v_string = gtk_entry_get_text (GTK_ENTRY (widg));
  smurfcfg_set_val (SMURFCFG_WAVETBL_OSSDEV, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPvkeyb");
  UTIL_OPMENU_INDEX (i, widg);
  if (i == 0)
    val.v_string = "AUTO";
  else
    val.v_string = seq_drivers[i - 1].name;
  smurfcfg_set_val (SMURFCFG_SEQ_DRIVER, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKvkauto");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg)))
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBvkclient");
      i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBvkport");
      i2 = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
    }
  else
    {
      i = 0;
      i2 = 0;
    }
  val.v_int = i;
  smurfcfg_set_val (SMURFCFG_SEQ_ALSACLIENT, &val);
  val.v_int = i2;
  smurfcfg_set_val (SMURFCFG_SEQ_ALSAPORT, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "OPmidi");
  UTIL_OPMENU_INDEX (i, widg);
  if (i == 0)
    val.v_string = "AUTO";
  else
    val.v_string = midi_drivers[i - 1].name;
  smurfcfg_set_val (SMURFCFG_MIDI_DRIVER, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKmtinauto");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg)))
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtinclient");
      i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));

      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtinport");
      i2 = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
    }
  else
    {
      i = 0;
      i2 = 0;
    }

  val.v_int = i;
  smurfcfg_set_val (SMURFCFG_MIDI_ALSA_INCLIENT, &val);
  val.v_int = i2;
  smurfcfg_set_val (SMURFCFG_MIDI_ALSA_INPORT, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "CHKmtoutauto");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widg)))
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtoutclient");
      i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));

      widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtoutport");
      i2 = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
    }
  else
    {
      i = 0;
      i2 = 0;
    }

  val.v_int = i;
  smurfcfg_set_val (SMURFCFG_MIDI_ALSA_OUTCLIENT, &val);
  val.v_int = i2;
  smurfcfg_set_val (SMURFCFG_MIDI_ALSA_OUTPORT, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtcard");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_MIDI_ALSACARD, &val);

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBmtdev");
  val.v_int = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));
  smurfcfg_set_val (SMURFCFG_MIDI_ALSADEVICE, &val);

  /* copy temp keytable to real one */
  for (i = 0; i < UIPIANO_TOTAL_NUMKEYS; i++)
    uipiano_keytable[i] = pref_temp_keytable[i];

  pref_save_piano_keytable ();

  gtk_widget_destroy (prefwin);
}

static void
pref_save_piano_keytable (void)
{
  gint i;
  GTokenValue val;

  /* check if piano key map is the default */
  for (i = 0; i < UIPIANO_TOTAL_NUMKEYS; i++)
    if (uipiano_keytable[i] != uipiano_default_keytable[i])
      break;

  if (i == UIPIANO_TOTAL_NUMKEYS)  /* same as default? */
    {
      val.v_string = "";  /* write blank strings */
      smurfcfg_set_val (SMURFCFG_PIANO_LOWOCTKEYS, &val);
      smurfcfg_set_val (SMURFCFG_PIANO_HIOCTKEYS, &val);
      return;
    }

  /* keytable is not the same as the default one */

  val.v_string = uipiano_encode_smurfcfg_octkeys (uipiano_keytable,
						  UIPIANO_LOW_NUMKEYS);
  if (val.v_string)
    {
      smurfcfg_set_val (SMURFCFG_PIANO_LOWOCTKEYS, &val);
      g_free (val.v_string);
    }


  val.v_string = uipiano_encode_smurfcfg_octkeys (&uipiano_keytable
						  [UIPIANO_LOW_NUMKEYS],
						  UIPIANO_HI_NUMKEYS);
  if (val.v_string)
    {
      smurfcfg_set_val (SMURFCFG_PIANO_HIOCTKEYS, &val);
      g_free (val.v_string);
    }
}

/* (glade) pop browse dialog for default sfont and sample paths */
void
pref_cb_path_browse_clicked (GtkWidget * btn, GtkWidget * entry)
{
  GtkWidget *filesel;
  gchar *s;

#if 0
  dirsel = dirbrowse_create (_("Choose directory"));

  gtk_object_set_data (GTK_OBJECT (dirsel), "entry", entry);
#endif

  filesel = gtk_file_selection_new (_("Choose directory"));

  gtk_object_set_data (GTK_OBJECT (filesel), "entry", entry);

  s = gtk_entry_get_text (GTK_ENTRY (entry));
  if (strlen (s))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), s);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
    "clicked", (GtkSignalFunc) pref_cb_path_browse_ok, filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (filesel));

  gtk_widget_show (filesel);
}

/* user says Okay to selected path in file dialog */
static void
pref_cb_path_browse_ok (GtkWidget * btn, GtkWidget * filesel)
{
  GtkWidget *entry;
  gchar *s;

  s = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));

  entry = gtk_object_get_data (GTK_OBJECT (filesel), "entry");
  gtk_entry_set_text (GTK_ENTRY (entry), s);

  gtk_widget_destroy (filesel);
}

/* (glade) callback for sample restraint option */
void
pref_cb_sam_sfspec_toggled (GtkWidget * btn, GtkWidget * prefwin)
{
  GtkWidget *widg;
  gboolean actv;

  actv = !GTK_TOGGLE_BUTTON (btn)->active;

  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminloop");
  gtk_widget_set_sensitive (widg, actv);
  widg = gtk_object_get_data (GTK_OBJECT (prefwin), "SPBminpad");
  gtk_widget_set_sensitive (widg, actv);
}

/* callback for virtual keyboard option menu, to hide/show the correct
   parameter widgets for the menu option selected */
static void
pref_cb_vkeyb_selection_done (GtkWidget * menu, GtkWidget * prefwin)
{
  GtkWidget *widg;
  gint i, index;
  gchar *s;

  widg = gtk_menu_get_active (GTK_MENU (menu));
  index = g_list_index (GTK_MENU_SHELL (menu)->children, widg);
  index--;

  for (i = 0; i < SEQ_COUNT; i++)
    {
      s = g_strconcat ("vk", seq_drivers[i].name, NULL);
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), s);
      g_free (s);

      if (widg)
	{
	  if (i == index)
	    gtk_widget_show (widg);
	  else
	    gtk_widget_hide (widg);
	}
    }
}

/* for "auto" toggle buttons in driver preferences, sets sensitivity of a
   list of widgets */
static void
pref_toggle_sensitive_widgets (GtkWidget * btn, GtkWidget * prefwin,
			       gchar **names)
{
  GtkWidget *widg;
  gboolean actv;

  actv = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  while (*names)
    {
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), *names);
      gtk_widget_set_sensitive (GTK_WIDGET (widg), !actv);
      names++;
    }
}

void
pref_cb_vkeyb_auto_toggled (GtkWidget * btn, GtkWidget * prefwin)
{
  gchar *names[] = { "LBLvkclient", "SPBvkclient", "LBLvkport", "SPBvkport",
		     "BTNvkselect", NULL };
  pref_toggle_sensitive_widgets (btn, prefwin, names);
}

/* callback for midi thru option menu, to hide/show the correct parameter
   widgets for the menu option selected */
static void
pref_cb_mthru_selection_done (GtkWidget * menu, GtkWidget * prefwin)
{
  GtkWidget *widg;
  gint i, index;
  gchar *s;

  widg = gtk_menu_get_active (GTK_MENU (menu));
  index = g_list_index (GTK_MENU_SHELL (menu)->children, widg);
  index--;

  for (i = 0; i < MIDI_COUNT; i++)
    {
      s = g_strconcat ("mt", midi_drivers[i].name, NULL);
      widg = gtk_object_get_data (GTK_OBJECT (prefwin), s);
      g_free (s);

      if (widg)
	{
	  if (i == index)
	    gtk_widget_show (widg);
	  else
	    gtk_widget_hide (widg);
	}
    }
}

void
pref_cb_mthru_inauto_toggled (GtkWidget * btn, GtkWidget * prefwin)
{
  gchar *names[] = { "SPBmtinclient", "SPBmtinport", "BTNmtinselect", NULL };
  pref_toggle_sensitive_widgets (btn, prefwin, names);
}

void
pref_cb_mthru_outauto_toggled (GtkWidget * btn, GtkWidget * prefwin)
{
  gchar *names[] ={ "SPBmtoutclient", "SPBmtoutport", "BTNmtoutselect", NULL };
  pref_toggle_sensitive_widgets (btn, prefwin, names);
}

void
pref_cb_piano_key_change_octave (GtkWidget * radbtn, GtkWidget * prefwin)
{
  GtkWidget *btn, *clist;

  btn = gtk_object_get_data (GTK_OBJECT (prefwin), "RADlowoct");
  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");

  pref_load_piano_key_clist (clist,
			     gtk_toggle_button_get_active
			     (GTK_TOGGLE_BUTTON (btn)));
}

static void
pref_load_piano_key_clist (GtkWidget * clist, gboolean lowoct)
{
  gint i, c;
  guint *keyp;
  gchar *coltxt[2];
  gchar note[5];

  if (lowoct)
    {
      c = UIPIANO_LOW_NUMKEYS;
      keyp = pref_temp_keytable;
    }
  else
    {
      c = UIPIANO_HI_NUMKEYS;
      keyp = &pref_temp_keytable[UIPIANO_LOW_NUMKEYS];
    }

  gtk_clist_clear (GTK_CLIST (clist));

  coltxt[0] = note;

  for (i = 0; i < c; i++)
    {
      sprintf (note, "%s-%d", keynames[i % 12], i / 12);

      if (!(coltxt[1] = gdk_keyval_name (*(keyp++))))
	coltxt[1] = _("INVALID");

      gtk_clist_append (GTK_CLIST (clist), coltxt);
    }

  gtk_object_set_data (GTK_OBJECT (clist), "lowoct", GINT_TO_POINTER (lowoct));
}

static void
pref_update_keycapture_msg (GtkWidget *prefwin)
{
  GtkWidget *lbl;
  gchar *s;

  lbl = gtk_object_get_data (GTK_OBJECT (prefwin), "LBLkeymsg");

  if (pref_keycapture)
    s = _("Press a key, click list to cancel");
  else
    s = "";

  gtk_label_set_text (GTK_LABEL (lbl), s);
}

gboolean
pref_cb_key_pressed (GtkWidget *prefwin, GdkEventKey *event)
{
  GtkWidget *clist;
  gint keyndx;
  gint lowoct;
  gchar *keystr;

  if (!pref_keycapture)
    return (FALSE);

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");

  if (!GTK_CLIST (clist)->selection)
    return (FALSE);

  keyndx = GPOINTER_TO_UINT (GTK_CLIST (clist)->selection->data);

  lowoct = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
						 "lowoct"));
  if ((lowoct && keyndx >= UIPIANO_LOW_NUMKEYS)
      || (!lowoct && keyndx >= UIPIANO_HI_NUMKEYS))
    return (FALSE);

  keystr = gdk_keyval_name (event->keyval);

  if (!keystr) return (FALSE);

  pref_temp_keytable[keyndx + (lowoct ? 0 : UIPIANO_LOW_NUMKEYS)]
    = event->keyval;

  gtk_clist_set_text (GTK_CLIST (clist), keyndx, 1, keystr);

  if (pref_change_all_keys && (++keyndx < (lowoct ? UIPIANO_LOW_NUMKEYS :
					   UIPIANO_HI_NUMKEYS)))
    {
      gtk_clist_moveto (GTK_CLIST (clist), keyndx, 0, 0.5, 0.0);
      gtk_clist_select_row (GTK_CLIST (clist), keyndx, 0);
      return (FALSE);
    }
  else if (pref_change_all_keys)
    pref_change_all_keys = FALSE;

  pref_keycapture = FALSE;
  pref_update_keycapture_msg (prefwin);

  return (FALSE);
}

void
pref_cb_keymap_change_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  GtkWidget *clist;
  guint keyndx;
  gint lowoct;

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");

  if (!GTK_CLIST (clist)->selection)   /* an item must be selected */
    return;

  keyndx = GPOINTER_TO_UINT (GTK_CLIST (clist)->selection->data);
  lowoct = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (clist),
						 "lowoct"));
  if ((lowoct && keyndx >= UIPIANO_LOW_NUMKEYS)
      || (!lowoct && keyndx >= UIPIANO_HI_NUMKEYS))
    return;

  pref_keycapture = TRUE;
  pref_update_keycapture_msg (prefwin);
}

void
pref_cb_keymap_change_all_clicked (GtkWidget *btn, GtkWidget *prefwin)
{
  GtkWidget *clist;

  clist = gtk_object_get_data (GTK_OBJECT (prefwin), "CLSTkeys");
  gtk_clist_select_row (GTK_CLIST (clist), 0, 0);
  pref_change_all_keys = TRUE;

  pref_cb_keymap_change_clicked (btn, prefwin);
}

gboolean
pref_cb_pianokey_clist_event (GtkWidget *widget, GdkEvent *event,
			      GtkWidget *prefwin)
{
  if (!pref_keycapture)
    return (FALSE);

  switch (event->type)
    {
    case GDK_MOTION_NOTIFY:
    case GDK_EXPOSE:
      /* do nothing */
      break;
    default:
      pref_keycapture = FALSE;
      pref_update_keycapture_msg (prefwin);
      break;
    }

  return (FALSE);
}
