/*==================================================================
 * uif_sfont.c - Sound font user interface routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include "uif_sfont.h"
#include "glade_interface.h"
#include "uif_menutbar.h"
#include "uif_sftree.h"
#include "uif_piano.h"
#include "uif_pianospan.h"
#include "uif_sfgen.h"
#include "uif_samview.h"
#include "uiface.h"
#include "pixmap.h"
#include "sfont.h"
#include "wavetable.h"
#include "sequencer.h"
#include "sample.h"
#include "sffile.h"
#include "sfdump.h"
#include "sfundo.h"
#include "sfdofunc.h"
#include "smurfcfg.h"
#include "util.h"
#include "vbank.h"
#include "i18n.h"
#include "widgets/popdog.h"

GList *uisf_sfonts = NULL;	/* list of loaded sound fonts (UISFont *) */
GList *uisf_vbanks = NULL;	/* list of loaded virtual banks (VBnkData *) */

/* the following variables reflect the currently selected sftree item */
UISFont *uisf_selected_uisfont = NULL; /* currently selected sound font */
UIVBank *uisf_selected_uivbank = NULL; /* currently selected virtual bank */
gpointer uisf_selected_elem = NULL; /* selected element (preset/inst/sample) */
gint uisf_selected_elem_type = NODE_NONE;	/* type of element selected */
SFZone *uisf_selected_zone = NULL;	/* zone selected */

static gchar *uisf_sfont_save_path = NULL; /* path to last saved sound font */
static gchar *uisf_sfont_load_path = NULL; /* path to last loaded sound font */

gboolean uisf_piano_follows = TRUE; /* piano follows sftree selection */
gboolean uisf_auto_temp_audible = TRUE;	/* auto load temporary audible */

static gint untitlecount = 0;	/* untitled counter (new sound fonts) */

static void uisf_cb_open_sfont_okay (GtkWidget * filewin);
static void uisf_close_sfont_save_yes (gpointer data, GtkWidget *popdog);
static void uisf_cb_close_sfont_savewin_destroy (GtkWidget * savewin,
  GtkWidget * popdog);
static void uisf_close_sfont_save_no (gpointer data, GtkWidget *popdog);
static void uisf_cb_saveas_dialog_okay (GtkWidget * btn, SFItemID itemid);
static void uisf_cb_save_ovrwrite_okay (gpointer data, GtkWidget *popdog);
static gint uisf_cb_save_continue (gchar * fname, SFItemID itemid);

static void uisf_cb_newmod_preset_cbfunc (GtkWidget * psetwin);
static void uisf_cb_newmod_inst_cbfunc (GtkWidget * instwin);
static void uisf_cb_modify_sample_cbfunc (GtkWidget * samwin);
static void uisf_cb_sfitem_delete_okay (gpointer data, GtkWidget *popdog);
static void uisf_cb_find_optype_selection_done (GtkWidget * menu,
  GtkWidget * findwin);
static void uisf_cb_find_search (GtkWidget * btn, GtkWidget * findwin);

static gint uisf_find_pzone_by_inst_compare (SFTreeRef * nref,
  GSList * pinst);
static gint uisf_find_izone_by_sample_compare (SFTreeRef * nref,
  GSList * psam);
static void uisf_preset_delete (SFItemID itemid);
static void uisf_pzone_delete (SFItemID itemid);
static void uisf_inst_delete (SFItemID itemid);
static void uisf_izone_delete (SFItemID itemid);
static void uisf_sample_delete (SFItemID itemid);
static void uisf_vbank_map_delete (SFItemID itemid);
static void uisf_piano_follow_update (gboolean temp_audible,
				      gint bank, gint prenum);
static void uisf_set_selected_preset (SFPreset * preset, UISFont * uisf);
static void uisf_set_selected_inst (SFInst * inst, UISFont * uisf);
static void uisf_set_selected_sample (SFSample * sam, UISFont * uisf);
static void uisf_set_selected_pzone (SFZone * zone, SFPreset * preset,
				     UISFont * uisf);
static void uisf_set_selected_izone (SFZone *zone, SFInst *inst, UISFont *uisf);
static void uisf_set_selected_vbnk_map (VBnkItem *item, UIVBank *uivb);


/***---        Sound Font FILE level routines        ---***/


void
uisf_init (void)
{
  uisf_new_sfont ();
}

/* add sfont to user interface */
UISFont *
uisf_add_sfont (SFData * sf)
{
  UISFont *uisf;

  uisf = g_malloc (sizeof (UISFont));
  uisf->sf = sf;
  uisf_sfonts = g_list_append (uisf_sfonts, uisf);
  sftree_add_sfont (uisf);	/* add sfont to tree */

  return (uisf);
}

/* add virtual sfont bank to user interface */
UIVBank *
uisf_add_vbank (VBnkData *vbnk)
{
  UIVBank *uivb;

  uivb = g_malloc (sizeof (UIVBank));
  uivb->vbnk = vbnk;
  uisf_vbanks = g_list_append (uisf_vbanks, uivb);
  sftree_add_vbank (uivb);

  return (uivb);
}

/* remove sound font from user interface */
void
uisf_remove_sfont (UISFont * uisf)
{
  /* if removing the selected sound font then deactivate selection */
  if (uisf_selected_uisfont == uisf)
    uisf_deactivate_selection ();

  sftree_remove_sfont (uisf);	/* remove from tree */
  sfont_close (uisf->sf);	/* close the sfont */
  uisf_sfonts = g_list_remove (uisf_sfonts, uisf);	/* remove from list */
  g_free (uisf);
}

/* remove virtual bank from user interface */
void
uisf_remove_vbank (UIVBank *uivb)
{
  sftree_remove_vbank (uivb);	/* remove from sftree */
  vbank_close (uivb->vbnk);	/* free the virtual bank */
  uisf_vbanks = g_list_remove (uisf_vbanks, uivb); /* remove from vbank list */
  g_free (uivb);
}

/* create a new sound font */
void
uisf_new_sfont (void)
{
  untitlecount++;
  uisf_add_sfont (sfont_new (untitlecount));
}

/* create a new virtual bank */
void
uisf_new_vbank (void)
{
  untitlecount++;
  uisf_add_vbank (vbank_new (untitlecount));
}

/* finds a loaded sound font by its file name, excluding excl */
UISFont *
uisf_find_sfont_by_fname (gchar * fname, SFData * excl)
{
  GList *p;
  UISFont *uisf;

  p = uisf_sfonts;
  while (p)
    {
      uisf = (UISFont *) (p->data);
      if (uisf->sf != excl && strcmp (fname, uisf->sf->fname) == 0)
	return (uisf);
      p = g_list_next (p);
    }
  return (NULL);
}

/* finds a loaded virtual bank by its file name, excluding excl */
UIVBank *
uisf_find_vbank_by_fname (gchar * fname, VBnkData * excl)
{
  GList *p;
  UIVBank *uivb;

  p = uisf_vbanks;
  while (p)
    {
      uivb = (UIVBank *) (p->data);
      if (uivb->vbnk != excl && strcmp (fname, uivb->vbnk->fname) == 0)
	return (uivb);
      p = g_list_next (p);
    }
  return (NULL);
}

SFItemID
uisf_load_sfont (gchar * fname)
{
  SFData *sf;
  VBnkData *vbnk;
  SFItemID retid;

  /* if its a virtual bank file (*.bnk) then load as such */
  if (vbank_is_virtual_bank (fname))
    {
      if (!(vbnk = vbank_load (fname)))
	return (SFITEMID_NONE);

      uisf_add_vbank (vbnk);
      retid = vbnk->itemid;
    }
  else
    {
      if (!(sf = sfload_file (fname)))
	return (SFITEMID_NONE);

      uisf_add_sfont (sf);
      retid = sf->itemid;
    }

  return (retid);
}

void
uisf_open_sfont (void)
{
  GTokenValue *val;
  GtkWidget *filewin;

  if (util_activate_unique_dialog ("opensf", 0))
    return;

  filewin = gtk_file_selection_new (_("Open Sound Font"));
  util_register_unique_dialog (filewin, "opensf", 0);

  /* if load path isn't set, use default sfont path from smurf.cfg */
  if (!uisf_sfont_load_path)
    {
      val = smurfcfg_get_val (SMURFCFG_DEF_SFONT_PATH);
      uisf_sfont_load_path = g_strdup (val->v_string);
    }

  if (strlen (uisf_sfont_load_path))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filewin),
      uisf_sfont_load_path);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filewin)->
      ok_button), "clicked", (GtkSignalFunc) uisf_cb_open_sfont_okay,
    GTK_OBJECT (filewin));

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filewin)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (filewin));

  gtk_widget_show (filewin);
}

static void
uisf_cb_open_sfont_okay (GtkWidget * filewin)
{
  gchar *fname;
  gchar *s;

  fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filewin));

  if (uisf_find_sfont_by_fname (fname, NULL))
    {
      util_quick_popup (_("That sound font is already loaded and being used"),
	NULL);
      return;
    }

  if (!uisf_load_sfont (fname))	/* load sfont */
    return;

  g_free (uisf_sfont_load_path); /* free old load path */
  s = g_dirname (fname);	/* new load path */
  uisf_sfont_load_path = g_strconcat (s, "/", NULL);
  g_free (s);
  gtk_widget_destroy (filewin); /* destroy file sel dialog on success */
}

void
uisf_dump_sfont (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  FILE *dumpfd;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  uisf = SFTREE_UPFIND_UISF (node);

  if (!(dumpfd = fopen ("/tmp/smurf_dump.txt", "w")))
    {
      logit (LogFubar | LogErrno,
	     _("Failed to open /tmp/smurf_dump.txt for writing"));
      return;
    }

  sfont_dump (uisf->sf, dumpfd);
  fclose (dumpfd);
}

void
uisf_close_sfont (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;
  GtkWidget *popdog;
  gchar *s;

  /* get active single item (NULL if multiple active items) */
  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);
  else uivb = NULL;

  if (uisf && uisf->sf->up2date)
    uisf_remove_sfont (uisf);
  else if (uivb && uivb->vbnk->up2date)
    uisf_remove_vbank (uivb);
  else
    {
      s =
	g_strdup_printf (_("Save changes before closing \"%s\"?"),
	(uisf != NULL) ? uisf->sf->fname : uivb->vbnk->fname);
      popdog =
	util_quick_popup (s, _("Yes"), uisf_close_sfont_save_yes, NULL,
			  _("No"), uisf_close_sfont_save_no, NULL,
			  _("Cancel"), NULL, NULL,
			  NULL);
      g_free (s);
      gtk_object_set_data (GTK_OBJECT (popdog), "itemid",
			   GINT_TO_POINTER (itemid));
    }
}

static void
uisf_close_sfont_save_yes (gpointer data, GtkWidget *popdog)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;
  GtkWidget *savewin;

  itemid =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (popdog), "itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (popdog);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);
  else uivb = NULL;

  sftree_set_selection_single (itemid);
  savewin = uisf_save_sfont (FALSE);
  sftree_unset_selection ();

  if (uisf && uisf->sf->up2date)
    {				/* sound font is now up to date? */
      uisf_remove_sfont (uisf);
      gtk_widget_destroy (popdog);
    }
  else if (uivb && uivb->vbnk->up2date)
    {				/* vbank now up to date? */
      uisf_remove_vbank (uivb);
      gtk_widget_destroy (popdog);
    }
  else
    {				/* not saved, hook savewin destroy signal */
      if (savewin)
	gtk_signal_connect (GTK_OBJECT (savewin), "destroy",
	  (GtkSignalFunc) uisf_cb_close_sfont_savewin_destroy, popdog);
    }
}

/* triggered by death of save file select widget from within close sfont */
static void
uisf_cb_close_sfont_savewin_destroy (GtkWidget * savewin, GtkWidget * popdog)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;

  itemid =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (popdog), "itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (popdog);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);
  else uivb = NULL;

  if (uisf && uisf->sf->up2date)
    {				/* did sound font get saved? */
      uisf_remove_sfont (uisf);
      gtk_widget_destroy (popdog);
    }		/* ?: no, then leave "Save before close" dialog un-molested */
  else if (uivb && uivb->vbnk->up2date)
    {
      uisf_remove_vbank (uivb);
      gtk_widget_destroy (popdog);
    }
}

static void
uisf_close_sfont_save_no (gpointer data, GtkWidget *popdog)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;

  itemid =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (popdog), "itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (popdog);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf)			/* ?: is it a virtual bank? */
    {				/* ?: Yes */
      uivb = SFTREE_UPFIND_UIVB (node);
      uisf_remove_vbank (uivb);
    }
  else uisf_remove_sfont (uisf); /* ?: No */

  gtk_widget_destroy (popdog);
}

/*----  uisf_save_sfont -----------
    saves a sound font with 2 different modes
    save_as = TRUE or sound font never been saved: pops a file selection dialog
    save_as = FALSE: just saves it
    returns: NULL if just saved it or ignored, or file selection widget pointer
 *--------------------------------- */
GtkWidget *
uisf_save_sfont (gint save_as)
{
  SFItemID itemid, pitemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;
  GTokenValue *val;
  GtkWidget *savewin;

  if (!(itemid = sftree_get_selection_single ())) return (NULL);
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return (NULL);
  uisf = SFTREE_UPFIND_UISF (node);

  /* if not a sound font, then its a virtual bank */
  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);

  /* Save As not requested and not first save, then just save it */

  if (uisf)			/* ?: sound font? */
    {
      if (!save_as && uisf->sf->sffd && uisf->sf->beensaved)
	{
	  uisf_cb_save_continue (uisf->sf->fname, itemid);
	  return (NULL);
	}
      pitemid = uisf->sf->itemid; /* parent (sfont) item ID */
    }
  else				/* ?: no, virtual bank */
    {
      if (!save_as && uivb->vbnk->beensaved)
	{
	  uisf_cb_save_continue (uivb->vbnk->fname, itemid);
	  return (NULL);
	}
      pitemid = uivb->vbnk->itemid; /* parent (vbank) item ID */
    }

  /* Save As, i.e. pops a save file selection window */

  /* check if save file selection dialog already active */
  if (util_activate_unique_dialog ("savesf", pitemid))
    return (NULL);

  /* pop save dialog window */
  savewin = gtk_file_selection_new ((uisf != NULL) ? _("Save Sound Font")
				    : _("Save Virtual Bank"));

  util_register_unique_dialog (savewin, "savesf", pitemid);

  /* if save path isn't set, use default sfont path from smurf.cfg */
  if (!uisf_sfont_save_path)
    {
      val = smurfcfg_get_val (SMURFCFG_DEF_SFONT_PATH);
      uisf_sfont_save_path = g_strdup (val->v_string);
    }

  if (strlen (uisf_sfont_save_path))
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (savewin),
      uisf_sfont_save_path);

  gtk_object_set_data (GTK_OBJECT (GTK_FILE_SELECTION (savewin)->ok_button),
    "savewin", savewin);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (savewin)->ok_button),
    "clicked", (GtkSignalFunc) uisf_cb_saveas_dialog_okay,
    GINT_TO_POINTER (itemid));
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (savewin)->
      cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy,
    GTK_OBJECT (savewin));

  gtk_widget_show (savewin);

  return (savewin);
}

static void
uisf_cb_saveas_dialog_okay (GtkWidget * btn, SFItemID itemid)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;
  GtkWidget *popup;		/* popup used for overwrite confirmation */
  GtkWidget *savewin;		/* save file selection widget */
  gchar *fname;
  struct stat st;
  gchar *s;

  /* get save file selection widget */
  savewin = gtk_object_get_data (GTK_OBJECT (btn), "savewin");

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (savewin);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);
  else uivb = NULL;

  fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (savewin));
  if (strlen (fname) == 0)
    return;

  if ((uisf && uisf_find_sfont_by_fname (fname, uisf->sf))
      || (uivb && uisf_find_vbank_by_fname (fname, uivb->vbnk)))
    {
      util_quick_popup (
	_("That sound font is currently loaded and should not be saved over"),
	NULL);
      return;
    }

  if (!strlen (g_basename (fname)))
    {
      logit (LogFubar, _("No file specified in path '%s'"), fname);
      return;
    }

  if (stat (fname, &st) != -1)
    {				/* file exists? */
      s = g_strdup_printf (_("Overwrite existing file '%s'?"), fname);
      popup =
	util_quick_popup (s, _("OK"), uisf_cb_save_ovrwrite_okay, itemid,
	_("Cancel"), NULL, NULL, NULL);
      g_free (s);
      gtk_object_set_data (GTK_OBJECT (popup), "savewin", savewin);
      return;
    }

  if (uisf_cb_save_continue (fname, itemid))
    {
      g_free (uisf_sfont_save_path);	/* free old save path */
      s = g_dirname (fname);	/* new save path */
      uisf_sfont_save_path = g_strconcat (s, G_DIR_SEPARATOR_S, NULL);
      g_free (s);
      gtk_widget_destroy (savewin);	/* destroy save dialog on success */
    }
}

/* user confirmed file overwrite for sound font save */
static void
uisf_cb_save_ovrwrite_okay (gpointer data, GtkWidget *popdog)
{
  GtkCTreeNode *node;
  GtkWidget *savewin;		/* save file selection widget */
  gchar *fname;
  SFItemID itemid = GPOINTER_TO_INT (data);

  savewin = gtk_object_get_data (GTK_OBJECT (popdog), "savewin");
  gtk_widget_destroy (popdog);

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (savewin));

  if (uisf_cb_save_continue (fname, itemid))
    gtk_widget_destroy (savewin);	/* destroy save dialog on success */
}

static gint
uisf_cb_save_continue (gchar * fname, SFItemID itemid)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  UIVBank *uivb;
  gchar *dirname;
  FILE *tmpfd;
  gchar *tmpfname;
  gint mksfd = -1;
  struct stat st;
  gint retval;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return (FAIL);
  uisf = SFTREE_UPFIND_UISF (node);

  if (!uisf) uivb = SFTREE_UPFIND_UIVB (node);

  dirname = g_dirname (fname);

  /* stat the target directory */
  if (stat (dirname, &st) == -1)
    {
      g_free (dirname);
      return (logit (LogFubar | LogErrno,
	  _("Target directory doesn't exist or error occured")));
    }

  /* is it really a directory? (is this necessary?) */
  if (!S_ISDIR (st.st_mode))
    {
      g_free (dirname);
      return (logit (LogFubar, _("Target path is not a directory")));
    }

  tmpfname = g_strconcat (dirname, G_DIR_SEPARATOR_S "smurf_tmp.XXXXXX", NULL);

  g_free (dirname);		/* string from g_dirname() needs to be freed */

  if ((mksfd = mkstemp (tmpfname)) == -1 || !(tmpfd = fdopen (mksfd, "w+")))
    {
      if (mksfd != -1)
	{
	  unlink (tmpfname);
	  close (mksfd);
	}

      logit (LogFubar | LogErrno,
	     _("Unable to open temp file '%s' for writing"), tmpfname);
      g_free (tmpfname);
      return (FAIL);
    }

  if (uisf) retval = sfsave_file (uisf->sf, tmpfd);
  else retval = vbank_save (uivb->vbnk, tmpfd);

  if (!retval)
    {
      fclose (tmpfd);
      if (unlink (tmpfname) == -1)
	logit (LogWarn, _("Could not delete temporary file '%s'"), tmpfname);
      g_free (tmpfname);
      return (FAIL);
    }

  if (rename (tmpfname, fname) == -1)
    {
      /* sample file offsets need restoration (sfload.c)? */
      if (uisf)			/* only valid if a sound font */
	sfsave_undo_sample_posofs ();	/* restore sample file offsets */

      logit (LogFubar | LogErrno,
	     _("Failed to rename temp file to destination file name"));
      fclose (tmpfd);
      if (unlink (tmpfname) == -1)
	logit (LogWarn, _("Could not delete temporary file \"%s\""), tmpfname);
      g_free (tmpfname);
      return (FAIL);
    }

  g_free (tmpfname);

  if (uisf)			/* ?: sound font? */
    {
      uisf->sf->up2date = TRUE;	/* sfont is now up to date */
      uisf->sf->beensaved = TRUE;	/* the sound font has now been saved */

      if (uisf->sf->sffd)
	fclose (uisf->sf->sffd);	/* close old sound font fd */
      uisf->sf->sffd = tmpfd;	/* assign new fd */

      fflush (tmpfd);		/* flush all writes to disk, fd stays open */

      /* update sound font file name */
      sfont_set_fname (uisf->sf, fname);
      sftree_update_sfont_node_label (uisf);

      /* ohh! Didn't need it after all (see sfload.c) */
      sfsave_free_sample_posofs ();
    }
  else				/* ?: No, virtual bank */
    {
      uivb->vbnk->up2date = TRUE; /* vbank now up to date */
      uivb->vbnk->beensaved = TRUE; /* been saved now */

      vbank_set_fname (uivb->vbnk, fname);
      sftree_update_vbank_node_label (uivb);

      fclose (tmpfd);		/* close file descriptor */
    }

  return (OK);
}

/* pops a dialog to edit/view sound font info (itemid should be sf node) */
void
uisf_sfont_info (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  SFData *sf;
  GtkWidget *infowin;
  GtkWidget *lbl;
  GtkWidget *entry;
  GtkWidget *text;
  gchar *s, *s2, *s3;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  uisf = SFTREE_UPFIND_UISF (node);
  sf = uisf->sf;

  /* check if info dialog already active for this sound font */
  if (util_activate_unique_dialog ("sfinfo", sf->itemid))
    return;

  infowin = create_infowin ();

  util_register_unique_dialog (infowin, "sfinfo", sf->itemid);

  gtk_object_set_data (GTK_OBJECT (infowin), "itemid",GINT_TO_POINTER (itemid));

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENinam");
  s = sfont_get_info (sf, INAM_ID);
  if (s) gtk_entry_set_text (GTK_ENTRY (entry), s);

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENieng");
  s = sfont_get_info (sf, IENG_ID);
  if (s) gtk_entry_set_text (GTK_ENTRY (entry), s);

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENicop");
  s = sfont_get_info (sf, ICOP_ID);
  if (s) gtk_entry_set_text (GTK_ENTRY (entry), s);

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENiprd");
  s = sfont_get_info (sf, IPRD_ID);
  if (s) gtk_entry_set_text (GTK_ENTRY (entry), s);

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENicrd");
  s = sfont_get_info (sf, ICRD_ID);
  if (s) gtk_entry_set_text (GTK_ENTRY (entry), s);

  lbl = gtk_object_get_data (GTK_OBJECT (infowin), "LBLiver");
  s = g_strdup_printf ("%d.%d", sf->version.major, sf->version.minor);
  gtk_label_set_text (GTK_LABEL (lbl), s);
  g_free (s);

  s = sfont_get_info (sf, ISFT_ID);
  s2 = strchr (s, ':');
  if (s2)
    {
      s3 = g_strndup (s, (gint) (s2 - s));
      s2 = g_strdup (s2 + 1);
    }
  else
    {
      s3 = g_strdup (s);
      s2 = g_strdup ("");
    }

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENcreate");
  gtk_entry_set_text (GTK_ENTRY (entry), s3);
  g_free (s3);

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENedit");
  gtk_entry_set_text (GTK_ENTRY (entry), s2);
  g_free (s2);

  if (!(s = sfont_get_info (sf, ISNG_ID))) s = "";
  lbl = gtk_object_get_data (GTK_OBJECT (infowin), "LBLisng");
  gtk_label_set_text (GTK_LABEL (lbl), s);

  if ((s = sfont_get_info (sf, IROM_ID)))
    s = g_strdup_printf ("%s v%d.%d", s, sf->romver.major, sf->romver.minor);
  else s = g_strdup ("");
  lbl = gtk_object_get_data (GTK_OBJECT (infowin), "LBLirom");
  gtk_label_set_text (GTK_LABEL (lbl), s);
  g_free (s);

  text = gtk_object_get_data (GTK_OBJECT (infowin), "TXTicmt");
  s = sfont_get_info (sf, ICMT_ID);
  if (s)
    {
      s2 = str_crlf2lf (s);
      gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, s2, -1);
      g_free (s2);
    }

  gtk_widget_show (infowin);
}

void
uisf_cb_sfont_info_date_today (GtkWidget * btn, GtkWidget * entry)
{
  time_t t;
  struct tm *tm;
  gchar s[41];

  time (&t);
  tm = localtime (&t);
  strftime (s, 40, "%B %e, %Y", tm);
  gtk_entry_set_text (GTK_ENTRY (entry), s);
}

void
uisf_cb_sfont_info_okay (GtkWidget * btn, GtkWidget *infowin)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  SFData *sf;
  SFItemID sfitemid;
  GtkWidget *entry;
  GtkWidget *text;
  gchar *s, *s2;

  itemid = GPOINTER_TO_INT(gtk_object_get_data (GTK_OBJECT (infowin),"itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {				/* node still exists? */
      gtk_widget_destroy (infowin);
      return;
    }
  uisf = SFTREE_UPFIND_UISF (node);
  sf = uisf->sf;
  sfitemid = sf->itemid;

  sfdo_group(_("Modified sound font info"));     /* start undo group */

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENinam");
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  s2 = sfont_get_info (sf, INAM_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, INAM_ID);
      sfont_set_info (sf, INAM_ID, s);
      sftree_update_sfont_node_label (uisf);	/* update sfont tree label */
    }

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENieng");
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  s2 = sfont_get_info (sf, IENG_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, IENG_ID);
      sfont_set_info (sf, IENG_ID, s);
    }

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENicop");
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  s2 = sfont_get_info (sf, ICOP_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, ICOP_ID);
      sfont_set_info (sf, ICOP_ID, s);
    }

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENiprd");
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  s2 = sfont_get_info (sf, IPRD_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, IPRD_ID);
      sfont_set_info (sf, IPRD_ID, s);
    }

  entry = gtk_object_get_data (GTK_OBJECT (infowin), "ENicrd");
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  s2 = sfont_get_info (sf, ICRD_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, ICRD_ID);
      sfont_set_info (sf, ICRD_ID, s);
    }

  text = gtk_object_get_data (GTK_OBJECT (infowin), "TXTicmt");
  s2 = gtk_editable_get_chars (GTK_EDITABLE (text), 0, -1);
  s = str_lf2crlf (s2);		/* convert end of line characters */
  g_free (s2);
  /* yeah, that is awefully large, but who knows what someone might want to
     store here, perhaps complementary ascii pictures of cows mating */
  if (strlen (s) > 65535)
    s2[65535] = '\0';
  s2 = sfont_get_info (sf, ICMT_ID);
  if (strcmp (s, s2) != 0)
    {
      dofunc_sfinfo_save (sfitemid, ICMT_ID);
      sfont_set_info (sf, ICMT_ID, s);
    }
  g_free (s);

  sfdo_done();      /* done with undo group */

  gtk_widget_destroy (infowin);
}


/* load virtual bank sound font dependencies */
void
uisf_vbank_load_depends (VBnkData *vbnk)
{
  GList *found, *notfound, *p, *p2;
  gchar *path;

  vbank_find_unloaded_sfonts (vbnk, &found, NULL);

  p = found;
  while (p)			/* loop over located sound fonts */
    {
      path = (gchar *)(p->data);

      /* see if sound font is NOT already loaded */
      if (!(uisf_find_sfont_by_fname (path, NULL)))
	{
	  if (!uisf_load_sfont (path)) /* load sound font */
	    logit (LogBad, _("Failed to load sound font '%s' for virtual bank"),
		  path);
	}

      p2 = p;
      p = g_list_next (p);

      g_free (path);		/* free path string */
      found = g_list_remove_link (found, p2); /* remove and free GList item */
      g_list_free_1 (p2);
    }

  g_list_foreach (notfound, (GFunc)g_free, NULL);
  g_list_free (notfound);
}


/***---        Sound Font NODE level interactive routines        ---***/


void
uisf_item_properties (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  ref = SFTREE_NODE_REF (node);

  switch (ref->type)
    {
    case NODE_SFONT:
      uisf_sfont_info ();
      break;
    case NODE_PRESET:
      uisf_newmod_preset (FALSE);
      break;
    case NODE_INST:
      uisf_newmod_inst (FALSE);
      break;
    case NODE_SAMPLE:
      uisf_modify_sample ();
      break;
    default:
      break;
    }
}

void
uisf_newmod_preset (gboolean create)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;
  gboolean melodic = TRUE;
  UISFont *uisf;
  SFPreset *pset;
  GtkWidget *psetwin;
  gint bank, prenum;
  gchar *name, *title;
  SFItemID uniitemid;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  ref = SFTREE_NODE_REF (node);

  if (create)
    {
      if (sftree_find_parent_by_type (node, NODE_PERCUSS))
	melodic = FALSE;	/* is node hint under percussion branch? */

      uisf = SFTREE_UPFIND_UISF (node);

      uniitemid = uisf->sf->itemid; /* use SF itemid for unique dialog key */

      sfont_find_free_preset (uisf->sf, &bank, &prenum, melodic);
      title = _("New Preset");
      name = "";
    }
  else
    {
      if (ref->type != NODE_PRESET) return;

      pset = (SFPreset *)(((GSList *)(ref->dptr))->data);
      uniitemid = pset->itemid;
      title = _("Modify Preset");
      name = pset->name;
      bank = pset->bank;
      prenum = pset->prenum;
    }

  if (util_activate_unique_dialog ("newmod_pset", uniitemid))
    return;

  psetwin = uisf_create_preset_config_dialog (title, name, bank, prenum,
					      uisf_cb_newmod_preset_cbfunc);

  util_register_unique_dialog (psetwin, "newmod_pset", uniitemid);

  gtk_object_set_data (GTK_OBJECT (psetwin), "itemid", GINT_TO_POINTER(itemid));
  gtk_object_set_data (GTK_OBJECT (psetwin), "create",
		       GINT_TO_POINTER (create));

  gtk_widget_show (psetwin);
}

GtkWidget *
uisf_create_preset_config_dialog (gchar *title, gchar *name, guint8 bank,
				  guint8 prenum, GtkSignalFunc cbfunc)
{
  GtkWidget *psetwin;
  GtkWidget *entry;
  GtkWidget *btn;

  psetwin = create_presetwin ();	/* create preset config window */

  gtk_window_set_title (GTK_WINDOW (psetwin), title);

  /* call back function for preset window */
  gtk_object_set_data (GTK_OBJECT (psetwin), "cbfunc", cbfunc);

  entry = gtk_object_get_data (GTK_OBJECT (psetwin), "ENname");
  gtk_entry_set_text (GTK_ENTRY (entry), name);

  btn = gtk_object_get_data (GTK_OBJECT (psetwin), "SPBbank");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (btn), bank);

  btn = gtk_object_get_data (GTK_OBJECT (psetwin), "SPBprenum");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (btn), prenum);

  /* set state of melodic/percussion radio buttons and bank # */
  btn = gtk_object_get_data (GTK_OBJECT (psetwin), "RADmelod");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), (bank != 128));

  return (psetwin);
}

/* glade_interface.c references this function */
void
uisf_cb_newmod_preset_type_toggled (GtkToggleButton * togglebutton,
  GtkWidget * presetwin)
{
  GtkWidget *rad;
  GtkWidget *spb;

  rad = gtk_object_get_data (GTK_OBJECT (presetwin), "RADmelod");
  spb = gtk_object_get_data (GTK_OBJECT (presetwin), "SPBbank");

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rad)))
    {
      gtk_widget_set_sensitive (GTK_WIDGET (spb), TRUE);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spb), 0.0);
    }
  else
    {
      gtk_widget_set_sensitive (GTK_WIDGET (spb), FALSE);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spb), 128.0);
    }
}

/* glade_interface.c references this function */
void
uisf_cb_newmod_preset_okay (GtkButton * button, GtkWidget * psetwin)
{
  gchar *name;
  guint8 bank;
  guint8 prenum;
  void (*cbfunc) (GtkWidget * psetwin);

  /* Preset name is valid? */
  name =
    gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT (psetwin),
	"ENname")));
  if (!strlen (name))
    {
      util_quick_popup (_("Preset name is required"), NULL);
      return;
    }

  /* bank # is valid? */
  bank =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (gtk_object_get_data
      (GTK_OBJECT (psetwin), "SPBbank")));

  if (bank > 128)
    {
      util_quick_popup (_("Invalid bank #, must be 0-128"), NULL);
      return;
    }

  /* preset # is valid? */
  prenum =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (gtk_object_get_data
      (GTK_OBJECT (psetwin), "SPBprenum")));

  if (prenum > 127)
    {
      util_quick_popup (_("Invalid preset #, must be 0-127"), NULL);
      return;
    }

  cbfunc = gtk_object_get_data (GTK_OBJECT (psetwin), "cbfunc");

  if (cbfunc)
    (*cbfunc) (psetwin);
  else
    gtk_widget_destroy (psetwin);
}

/* Processes user input for new or modified preset */
static void
uisf_cb_newmod_preset_cbfunc (GtkWidget * psetwin)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;
  GSList *p;
  gboolean create;
  SFPreset *pset;
  UISFont *uisf;
  gchar *name;
  guint8 bank, prenum;
  gboolean expanded;

  /* node only set if modifying preset, NULL if new */
  itemid = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT(psetwin),"itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (psetwin);
      return;
    }

  uisf = SFTREE_UPFIND_UISF (node);

  create =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (psetwin), "create"));

  if (!create)
    {
      ref = SFTREE_NODE_REF (node);
      p = ref->dptr;
      pset = (SFPreset *) (p->data);
    }
  else
    pset = NULL;

  /* load user input values */
  name =
    gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT (psetwin),
	"ENname")));
  bank =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (gtk_object_get_data
      (GTK_OBJECT (psetwin), "SPBbank")));
  prenum =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (gtk_object_get_data
      (GTK_OBJECT (psetwin), "SPBprenum")));

  /* look for conflicting preset (exclude itself from the search) */
  if ((p = sfont_find_preset (uisf->sf, name, bank, prenum, pset)))
    {
      SFPreset *pr = (SFPreset *) (p->data);
      if (pr->prenum == prenum && pr->bank == bank)
	util_quick_popup (_("A preset with that bank # and preset # exists"),
	  NULL);
      else
	util_quick_popup (_("A preset with that name exists"), NULL);
      return;
    }

  if (pset)
    {				/* Modified preset? */
      expanded = GTK_CTREE_ROW (node)->expanded; /* save expanded state */
      SFTREE_FREEZE ();
      SFTREE_REMOVE_NODE (node);

      sfont_set_namestr (uisf->sf, pset->name, name);
      pset->bank = bank;
      pset->prenum = prenum;
      uisf->sf->preset = g_slist_remove (uisf->sf->preset, pset);
      sfont_add_preset (uisf->sf, pset);
      node = sftree_add_preset_sorted (g_slist_find (uisf->sf->preset, pset),
	uisf->nodes);

      /* restore expanded state */
      if (expanded) gtk_ctree_expand (GTK_CTREE (sftree_widg), node);
      SFTREE_THAW ();
    }
  else
    {				/* ?: No, its new */
      pset = sfont_new_preset (uisf->sf, name, bank, prenum);
      node = sftree_add_preset_sorted (g_slist_find (uisf->sf->preset, pset),
	uisf->nodes);

      /* save undo state */
      sfdo_group (_("New preset"));
      dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);
      sfdo_done ();
    }

  gtk_widget_destroy (psetwin);
}

/*  name or rename an instrument 
    if create == TRUE then new instrument, rename otherwise */
void
uisf_newmod_inst (gboolean create)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFTreeRef *ref;
  GtkWidget *instwin;
  SFInst *inst;
  gchar *title, *name;
  SFItemID uniitemid;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  ref = SFTREE_NODE_REF (node);

  if (create)
    {
      UISFont *uisf = SFTREE_UPFIND_UISF (node);
      uniitemid = uisf->sf->itemid;
      title = _("New Instrument");
      name = "";
    }
  else
    {
      if (ref->type != NODE_INST) return;

      title = _("Rename Instrument");
      inst = (SFInst *) (((GSList *)(ref->dptr))->data);
      uniitemid = inst->itemid;
      name = inst->name;
    }

  if (util_activate_unique_dialog ("newmod_inst", uniitemid)) return;

  instwin = uisf_create_inst_config_dialog (title, name,
					    uisf_cb_newmod_inst_cbfunc);

  util_register_unique_dialog (instwin, "newmod_inst", uniitemid);

  gtk_object_set_data (GTK_OBJECT (instwin), "itemid", GINT_TO_POINTER(itemid));
  gtk_object_set_data (GTK_OBJECT (instwin), "create",
		       GINT_TO_POINTER (create));

  gtk_widget_show (instwin);
}

GtkWidget *
uisf_create_inst_config_dialog (gchar * title, gchar * name,
				GtkSignalFunc cbfunc)
{
  GtkWidget *instwin;
  GtkWidget *entry;

  instwin = create_instwin ();

  gtk_window_set_title (GTK_WINDOW (instwin), title);
  gtk_object_set_data (GTK_OBJECT (instwin), "cbfunc", cbfunc);
  entry = gtk_object_get_data (GTK_OBJECT (instwin), "ENname");
  gtk_entry_set_text (GTK_ENTRY (entry), name);

  return (instwin);
}

void
uisf_cb_newmod_inst_okay (GtkButton * button, GtkWidget * instwin)
{
  gchar *name;
  void (*cbfunc) (GtkWidget * instwin);

  /* Instrument name is valid? */
  name =
    gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT (instwin),
	"ENname")));
  if (!strlen (name))
    {
      util_quick_popup (_("Instrument name is required"), NULL);
      return;
    }

  cbfunc = gtk_object_get_data (GTK_OBJECT (instwin), "cbfunc");

  if (cbfunc)
    (*cbfunc) (instwin);
  else
    gtk_widget_destroy (instwin);
}

/* user confirmed name or rename of instrument */
static void
uisf_cb_newmod_inst_cbfunc (GtkWidget * instwin)
{
  SFItemID itemid;
  GtkCTreeNode *node, *nptr;
  gboolean create;
  SFTreeRef *ref;
  UISFont *uisf;
  GSList *pinst;
  GList *match;
  SFInst *inst;
  gchar *name;

  itemid = GPOINTER_TO_INT(gtk_object_get_data (GTK_OBJECT (instwin),"itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (instwin);
      return;
    }

  create =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (instwin), "create"));

  if (!create)
    {
      ref = SFTREE_NODE_REF (node);
      pinst = ref->dptr;
      inst = (SFInst *) (pinst->data);
    }
  else
    inst = NULL;

  name =
    gtk_entry_get_text (GTK_ENTRY (gtk_object_get_data (GTK_OBJECT (instwin),
	"ENname")));

  uisf = SFTREE_UPFIND_UISF (node);

  /* check for duplicates (exclude inst from search) */
  if (sfont_find_inst (uisf->sf, name, inst))
    {
      util_quick_popup (_("An instrument with that name exists"), NULL);
      return;
    }

  if (inst)
    {				/* rename instrument? */
      sfont_set_namestr (uisf->sf, inst->name, name);
      sftree_set_tree_node_label (node, name);

      nptr = uisf->nodes->preset;	/* preset branch of tree */

      /* find all preset zone references to this instrument */
      match = gtk_ctree_find_all_by_row_data_custom (sftree_widg, nptr, pinst,
	(GCompareFunc) uisf_find_pzone_by_inst_compare);

      g_list_foreach (match, (GFunc) sftree_set_tree_node_label, name);
      g_list_free (match);
    }
  else
    {				/* ? : No, new instrument */
      inst = sfont_new_inst (uisf->sf, name);
      node = sftree_add_inst (g_slist_find (uisf->sf->inst, inst), uisf->nodes,
		       SFTREE_NODE_APPEND);

      /* save undo state */
      sfdo_group (_("New instrument"));
      dofunc_noitem_save (SFTREE_NODE_REF (node)->itemid);
      sfdo_done ();
    }

  gtk_widget_destroy (instwin);
}

/* modify sample parameters (name, rate, fine tune) */
void
uisf_modify_sample (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  SFSample *sam;
  GtkWidget *samwin;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  if (SFTREE_NODE_REF (node)->type != NODE_SAMPLE) return;

  sam = (SFSample *) (((GSList *) (SFTREE_NODE_REF (node)->dptr))->data);

  if (util_activate_unique_dialog ("newmod_sam", sam->itemid)) return;

  samwin = uisf_create_sample_config_dialog (_("Rename Sample"), sam->name,
					     sam->samplerate, sam->pitchadj,
					     uisf_cb_modify_sample_cbfunc);
  util_register_unique_dialog (samwin, "newmod_sam", sam->itemid);

  gtk_object_set_data (GTK_OBJECT (samwin), "itemid", GINT_TO_POINTER (itemid));
  gtk_widget_show (samwin);
}

GtkWidget *
uisf_create_sample_config_dialog (gchar *title, gchar *name, guint32 samplerate,
				  gint8 pitchadj, GtkSignalFunc cbfunc)
{
  GtkWidget *samwin;
  GtkWidget *entry;
  GtkWidget *widg;
  gint ndx;
  gchar *s;

  samwin = create_samwin ();

  gtk_window_set_title (GTK_WINDOW (samwin), title);
  gtk_object_set_data (GTK_OBJECT (samwin), "cbfunc", cbfunc);

  entry = gtk_object_get_data (GTK_OBJECT (samwin), "ENname");
  gtk_entry_set_text (GTK_ENTRY (entry), name);

  /* if samplerate is specified (not 0), display rate and fine tuning options */
  if (samplerate != 0)
    {
      /* display sample rate and fine tune settings */
      widg = gtk_object_get_data (GTK_OBJECT (samwin), "settings");
      gtk_widget_show (widg);

      switch (samplerate)
	{
	case 44100: ndx = 0; break;
	case 22050: ndx = 1; break;
	case 11025: ndx = 2; break;
	default:
	  ndx = 3;
	  widg = gtk_object_get_data (GTK_OBJECT (samwin), "ENcustom");
	  s = g_strdup_printf ("%d", samplerate);
	  gtk_entry_set_text (GTK_ENTRY (widg), s);
	  g_free (s);
	}

      widg = gtk_object_get_data (GTK_OBJECT (samwin), "OPrate");
      gtk_option_menu_set_history (GTK_OPTION_MENU (widg), ndx);

      widg = gtk_object_get_data (GTK_OBJECT (samwin), "SPBftune");
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (widg), (float)pitchadj);
    }

  return (samwin);
}

void
uisf_cb_modify_sample_okay (GtkButton * button, GtkWidget * samwin)
{
  gchar *name;
  void (*cbfunc) (GtkWidget * samwin);
  GtkWidget *widg;
  gint ndx, rate;

  /* Sample name is valid? */
  widg = gtk_object_get_data (GTK_OBJECT (samwin), "ENname");
  name = gtk_entry_get_text (GTK_ENTRY (widg));
  if (!strlen (name))
    {
      util_quick_popup (_("Sample name is required"), NULL);
      return;
    }

  gtk_object_get_data (GTK_OBJECT (samwin), "settings");
  if (GTK_WIDGET_VISIBLE (widg))
    {
      widg = gtk_object_get_data (GTK_OBJECT (samwin), "OPrate");

      /* macro stores selected item index into ndx */
      UTIL_OPMENU_INDEX (ndx, widg);

      if (ndx == 3)
	{
	  widg = gtk_object_get_data (GTK_OBJECT (samwin), "ENcustom");
	  rate = atoi (gtk_entry_get_text (GTK_ENTRY (widg)));

	  if (rate < SF_MIN_SAMPLERATE || rate > SF_MAX_SAMPLERATE)
	    {
	      util_quick_popup (_("Sample rate is out of bounds"), NULL);
	      return;
	    }
	}
    }

  cbfunc = gtk_object_get_data (GTK_OBJECT (samwin), "cbfunc");

  if (cbfunc)
    (*cbfunc) (samwin);
  else
    gtk_widget_destroy (samwin);
}

static void
uisf_cb_modify_sample_cbfunc (GtkWidget * samwin)
{
  SFItemID itemid;
  GtkWidget *widg;
  GtkCTreeNode *node, *nptr;
  SFTreeRef *ref;
  UISFont *uisf;
  GSList *psam;
  GList *match;
  SFSample *sam;
  gchar *name;
  gint ndx;

  itemid = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (samwin),"itemid"));
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    {
      gtk_widget_destroy (samwin);
      return;
    }

  ref = SFTREE_NODE_REF (node);
  psam = ref->dptr;
  sam = (SFSample *) (psam->data);

  uisf = SFTREE_UPFIND_UISF (node);

  widg = gtk_object_get_data (GTK_OBJECT (samwin), "ENname");
  name = gtk_entry_get_text (GTK_ENTRY (widg));

  if (sfont_find_sample (uisf->sf, name, sam))
    {
      util_quick_popup (_("A sample already exists with that name"), NULL);
      return;
    }

  sfont_set_namestr (uisf->sf, sam->name, name);
  sftree_set_tree_node_label (node, name);

  nptr = uisf->nodes->inst;	/* instrument branch of tree */

  /* find all instrument zone references to this instrument */
  match = gtk_ctree_find_all_by_row_data_custom (sftree_widg, nptr, psam,
    (GCompareFunc) uisf_find_izone_by_sample_compare);

  g_list_foreach (match, (GFunc) sftree_set_tree_node_label, name);
  g_list_free (match);

  /* get sample rate parameter */
  widg = gtk_object_get_data (GTK_OBJECT (samwin), "OPrate");
  UTIL_OPMENU_INDEX (ndx, widg); /* macro stores selected item index into ndx */

  switch (ndx)
    {
    case 0: sam->samplerate = 44100; break;
    case 1: sam->samplerate = 22050; break;
    case 2: sam->samplerate = 11025; break;
    case 3:
      widg = gtk_object_get_data (GTK_OBJECT (samwin), "ENcustom");
      sam->samplerate = atoi (gtk_entry_get_text (GTK_ENTRY (widg)));
      break;
    }

  /* get the fine tunning parameter */
  widg = gtk_object_get_data (GTK_OBJECT (samwin), "SPBftune");
  sam->pitchadj = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widg));

  gtk_widget_destroy (samwin);
}

/* pop a dialog to confirm delete of sf item */
void
uisf_dialog_sfitem_delete (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  GSList *p, *p2, *p3;
  UISFont *uisf;
  SFPreset *pset;
  SFInst *inst;
  SFSample *sam;
  SFZone *z;
  VBnkItem *vbitem;
  gchar *s;
  gchar keystr[8];
  gint refcount = 0;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  p = SFTREE_NODE_REF (node)->dptr;

  switch (SFTREE_NODE_REF (node)->type)
    {
    case NODE_PRESET:
      pset = (SFPreset *) (p->data);
      s = g_strdup_printf (_("Delete preset \"%03d-%03d %s\"?"), pset->bank,
	pset->prenum, pset->name);
      break;
    case NODE_PZONE:
      z = (SFZone *) (p->data);
      s = g_strdup_printf (_("Delete preset zone \"%s\"?"),
	z->instsamp ? ((SFInst *) (z->instsamp->data))->
	name : _("Global Zone"));
      break;
    case NODE_INST:
      inst = (SFInst *) (p->data);
      uisf = SFTREE_UPFIND_UISF (node);

      /* get preset list for this instrument's sfont */
      p2 = uisf->sf->preset;

      /* count the number of preset references to this instrument */
      while (p2)
	{			/* traverse presets */
	  p3 = ((SFPreset *) (p2->data))->zone;
	  while (p3)
	    {			/* traverse preset zones */
	      if (((SFZone *) (p3->data))->instsamp == p)
		refcount++;
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}

      s =
	g_strdup_printf (_("Delete instrument \"%s\" (%d preset references)?"),
			 inst->name,
	refcount);
      break;
    case NODE_IZONE:
      z = (SFZone *) (p->data);
      s = g_strdup_printf (_("Delete instrument zone \"%s\"?"),
	z->instsamp ? ((SFSample *) (z->instsamp->data))->
	name : _("Global Zone"));
      break;
    case NODE_SAMPLE:
      sam = (SFSample *) (p->data);
      uisf = SFTREE_UPFIND_UISF (node);

      /* get instrument list for this samples's sfont */
      p2 = uisf->sf->inst;

      /* count the number of instrument references to this sample */
      while (p2)
	{			/* traverse instruments */
	  p3 = ((SFInst *) (p2->data))->zone;
	  while (p3)
	    {			/* traverse instrument zones */
	      if (((SFZone *) (p3->data))->instsamp == p)
		refcount++;
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}

      s =
	g_strdup_printf (_("Delete sample \"%s\" (%d instrument references)?"),
			 sam->name,
	refcount);
      break;
    case NODE_VBNK_MAP:
      vbitem = (VBnkItem *) (p->data);

      if (vbitem->map.keynote >= 0)
	sprintf (keystr, " [%03d]", vbitem->map.keynote);
      else keystr[0] = '\0';

      s = g_strdup_printf (_("Delete virtual bank map \"%03d-%03d%s\"?"),
			   vbitem->map.bank, vbitem->map.psetnum,
			   keystr);
      break;
    default:
      return;
      break;
    }

  util_quick_popup (s, _("OK"), uisf_cb_sfitem_delete_okay,
		    GINT_TO_POINTER (itemid),
		    _("Cancel"), NULL, NULL, NULL);
  g_free (s);
}

/* user confirmed sound font item delete */
static void
uisf_cb_sfitem_delete_okay (gpointer data, GtkWidget *popdog)
{
  SFItemID itemid = GPOINTER_TO_INT (data);

  if (itemid)
    uisf_sfitem_delete (itemid);

  gtk_widget_destroy (popdog);
}

/* create global zone for preset/inst */
void
uisf_item_global_zone (void)
{
  SFItemID itemid;
  GtkCTreeNode *node, *n;
  UISFont *uisf;
  SFTreeRef *ref;
  SFPreset *pset;
  SFInst *inst;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  ref = SFTREE_NODE_REF (node);

  if (ref->type == NODE_PRESET)
    {
      pset = ((GSList *) (ref->dptr))->data;

      if (pset->zone && !(((SFZone *) (pset->zone->data))->instsamp))
	{
	  util_quick_popup (_("Preset already has a Global Zone"), NULL);
	  return;
	}
    }
  else if (ref->type == NODE_INST)
    {
      inst = ((GSList *) (ref->dptr))->data;

      if (inst->zone && !(((SFZone *) (inst->zone->data))->instsamp))
	{
	  util_quick_popup (_("Instrument already has a Global Zone"), NULL);
	  return;
	}
    }
  else return;			/* not INST or PRESET */

  /* find sound font node that owns target preset */
  uisf = SFTREE_UPFIND_UISF (node);

  if (ref->type == NODE_PRESET)
    {
      sfont_new_preset_zone (uisf->sf, pset, NULL);
      n = sftree_add_pzone (pset->zone, node, SFTREE_NODE_APPEND);

      /* save undo state */
      sfdo_group (_("Preset global zone"));
      dofunc_noitem_save (SFTREE_NODE_REF (n)->itemid);
      sfdo_done ();
    }
  else
    {
      sfont_new_inst_zone (uisf->sf, inst, NULL);
      n = sftree_add_izone (inst->zone, node, SFTREE_NODE_APPEND);

      /* save undo state */
      sfdo_group (_("Instrument global zone"));
      dofunc_noitem_save (SFTREE_NODE_REF (n)->itemid);
      sfdo_done ();
    }
}

void
uisf_find (void)
{
  SFItemID itemid;
  GtkCTreeNode *node = NULL;
  GtkWidget *findwin;
  GtkWidget *opmenu;
  GtkWidget *menu;
  GtkWidget *mitem;
  GtkWidget *btn;
  SFTreeRef *ref;
  UISFont *uisf, *startuisf;
  GList *p;
  gint type, i, index;

  /* don't care if it doesn't exist.. Will just do a default find window */
  if ((itemid = sftree_get_selection_single ()))
    node = SFTREE_LOOKUP_ITEMID (itemid);

  if (util_activate_unique_dialog ("findwin", 0)) return;

  findwin = create_findwin ();
  util_register_unique_dialog (findwin, "findwin", 0);

  opmenu = gtk_object_get_data (GTK_OBJECT (findwin), "OPtype");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opmenu));
  gtk_signal_connect (GTK_OBJECT (menu), "selection-done",
    (GtkSignalFunc) uisf_cb_find_optype_selection_done, findwin);

  btn = gtk_object_get_data (GTK_OBJECT (findwin), "BTNsearch");
  gtk_signal_connect (GTK_OBJECT (btn), "clicked",
    (GtkSignalFunc) uisf_cb_find_search, findwin);

  btn = gtk_object_get_data (GTK_OBJECT (findwin), "BTNclose");
  gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
    (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (findwin));

  btn = gtk_object_get_data (GTK_OBJECT (findwin), "BTNcancel");
  gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
    (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (findwin));

  startuisf = NULL;
  if (node)
    {			/* if an initial node was specified for search hints */
      startuisf = SFTREE_UPFIND_UISF (node);	/* sfont to start in */
      ref = SFTREE_NODE_REF (node);
      i = ref->type;

      if (i == NODE_INSTROOT || i == NODE_INST)
	type = 1;
      else if (i == NODE_SAMPLEROOT || i == NODE_USER || i == NODE_ROM
	|| i == NODE_SAMPLE) type = 2;
      else
	type = 0;

      /* set type of node to search for from hint node */
      opmenu = gtk_object_get_data (GTK_OBJECT (findwin), "OPtype");
      gtk_option_menu_set_history (GTK_OPTION_MENU (opmenu), type);
    }

  if (uisf_sfonts)
    {				/* load menu of sound font file names */
      opmenu = gtk_object_get_data (GTK_OBJECT (findwin), "OPfile");
      menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opmenu));

      index = 0;
      i = 1;
      p = uisf_sfonts;
      while (p)
	{
	  uisf = (UISFont *) (p->data);

	  if (uisf == startuisf)
	    index = i;

	  mitem = gtk_menu_item_new_with_label (g_basename (uisf->sf->fname));
	  ref = SFTREE_NODE_REF (uisf->nodes->sfont);
	  gtk_object_set_data (GTK_OBJECT (mitem), "sfitemid",
	    GINT_TO_POINTER (ref->itemid));
	  gtk_widget_show (mitem);
	  gtk_menu_append (GTK_MENU (menu), mitem);

	  i++;
	  p = g_list_next (p);
	}

      gtk_option_menu_set_history (GTK_OPTION_MENU (opmenu), index);
    }

  gtk_widget_show (findwin);
}

static void
uisf_cb_find_optype_selection_done (GtkWidget * menu, GtkWidget * findwin)
{
  GtkWidget *widg;
  GtkWidget *mnuitem;
  gint index;
  gboolean sensitive;

  mnuitem = gtk_menu_get_active (GTK_MENU (menu));
  index = g_list_index (GTK_MENU_SHELL (menu)->children, mnuitem);

  if (index == -1)
    index = 0;

  if (index != 0)
    {				/* preset type NOT selected */
      sensitive = FALSE;
      widg = gtk_object_get_data (GTK_OBJECT (findwin), "RADname");
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);
    }
  else
    sensitive = TRUE;

  widg = gtk_object_get_data (GTK_OBJECT (findwin), "RADname");
  gtk_widget_set_sensitive (widg, sensitive);

  widg = gtk_object_get_data (GTK_OBJECT (findwin), "RADbank");
  gtk_widget_set_sensitive (widg, sensitive);

  widg = gtk_object_get_data (GTK_OBJECT (findwin), "SPBbank");
  gtk_widget_set_sensitive (widg, sensitive);

  widg = gtk_object_get_data (GTK_OBJECT (findwin), "SPBpreset");
  gtk_widget_set_sensitive (widg, sensitive);

  widg = gtk_object_get_data (GTK_OBJECT (findwin), "LBLpreset");
  gtk_widget_set_sensitive (widg, sensitive);
}

/* agghh, this routine is a mess!!!! */
static void
uisf_cb_find_search (GtkWidget * widg, GtkWidget * findwin)
{
  SFItemID itemid;
  GtkCTreeNode *startnode;
  GtkCTreeNode *node, *found = NULL;
  SFTreeRef *ref;
  gint nodetype;
  gint alt;
  gboolean substr, nametog, allfiles, beginning, nofirst;
  gchar *name;
  gint bank, prenum;
  UISFont *uisf, *uisf2;
  GSList *slptr;
  SFPreset *preset;
  SFInst *inst;
  SFSample *sample;
  GtkWidget *opmenu;
  GtkWidget *menu;
  GtkWidget *mnuitem;
  GtkWidget *btn;
  GtkWidget *entry;

  /* selected method of name string comparison (sub string or starts with) */
  btn = gtk_object_get_data (GTK_OBJECT (findwin), "RADsubstr");
  substr = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  /* selected search criteria, name or bank/preset (preset search only) */
  btn = gtk_object_get_data (GTK_OBJECT (findwin), "RADname");
  nametog = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  /* get name entry to find */
  entry = gtk_object_get_data (GTK_OBJECT (findwin), "ENname");
  name = gtk_entry_get_text (GTK_ENTRY (entry));

  /* get bank spin button value to find */
  btn = gtk_object_get_data (GTK_OBJECT (findwin), "SPBbank");
  bank = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (btn));

  /* get preset number spin button value to find */
  btn = gtk_object_get_data (GTK_OBJECT (findwin), "SPBpreset");
  prenum = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (btn));

  /* type of node to search for (preset, instrument or sample) */
  opmenu = gtk_object_get_data (GTK_OBJECT (findwin), "OPtype");
  UTIL_OPMENU_INDEX (nodetype, opmenu);

  btn = gtk_object_get_data (GTK_OBJECT (findwin), "CKBbegin");
  beginning = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  /* get selected sound font file uisf ptr (or all files option) */
  opmenu = gtk_object_get_data (GTK_OBJECT (findwin), "OPfile");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opmenu));
  mnuitem = gtk_menu_get_active (GTK_MENU (menu));
  itemid =
    GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (mnuitem), "sfitemid"));
  node = SFTREE_LOOKUP_ITEMID (itemid);
  allfiles = (node == NULL);
  if (node)
    uisf = SFTREE_UPFIND_UISF (node);	/* use selected sound font file */
  else
    uisf = uisf_sfonts->data;	/* start with first loaded sound font */

  if (!beginning)
    {				/* start from beginning? */
      /* retrieve node to start search from (or NULL) */
      itemid =
	GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (findwin),
   "startitemid"));
      startnode = SFTREE_LOOKUP_ITEMID (itemid);
    }
  else
    startnode = NULL;

  alt = 0;	/* toggles 0/1 for preset and sample alternate branches */
  if (startnode)
    {			/* make sure start node matches search criteria */
      ref = SFTREE_NODE_REF (startnode);
      uisf2 = SFTREE_UPFIND_UISF (startnode);	/* uisf of start node */

      if (allfiles || uisf == uisf2)
	{
	  uisf = uisf2;

	  switch (ref->type)
	    {
	    case NODE_PRESET:
	      if (nodetype == 0)
		{
		  node = GTK_CTREE_ROW (startnode)->parent;
		  ref = SFTREE_NODE_REF (node);
		  if (ref->type == NODE_MELODIC)
		    alt = 1;
		}
	      else
		startnode = NULL;
	      break;
	    case NODE_INST:
	      if (nodetype != 1)
		startnode = NULL;
	      break;
	    case NODE_SAMPLE:
	      if (nodetype == 2)
		{
		  node = GTK_CTREE_ROW (startnode)->parent;
		  ref = SFTREE_NODE_REF (node);
		  if (ref->type == NODE_USER)
		    alt = 1;
		}
	      else
		startnode = NULL;
	      break;
	    default:
	      break;
	    }
	}
      else
	startnode = NULL;
    }

  if (startnode)
    {
      /* startnode is last found node, so advance to next node */
      startnode = GTK_CTREE_ROW (startnode)->sibling;

      /* NOTE: if startnode == NULL (last node of branch) the master while
         loop will still handle this correctly! */
      node = startnode;
      nofirst = TRUE;
    }
  else
    nofirst = FALSE;

  while (uisf && !found)
    {				/* while sound fonts */
      if (!nofirst)
	{
	  switch (nodetype)
	    {
	    case 1:		/* instrument search */
	      node = uisf->nodes->inst;
	      break;
	    case 2:		/* sample search */
	      if (!alt)
		node = uisf->nodes->loaded;
	      else
		node = uisf->nodes->rom;
	      alt ^= 1;
	      break;
	    default:		/* preset search (0) */
	      if (!alt)
		node = uisf->nodes->melodic;
	      else
		node = uisf->nodes->percuss;
	      alt ^= 1;
	      break;
	    }

	  if (node)
	    node = GTK_CTREE_ROW (node)->children;
	}

      nofirst = FALSE;

      while (node && !found)
	{			/* loop over items of same type */
	  slptr = (GSList *) (SFTREE_NODE_REF (node)->dptr);
	  switch (nodetype)
	    {
	    case 1:		/* instrument search */
	      inst = (SFInst *) (slptr->data);
	      /* match instrument by name (substr or starts with) */
	      if ((substr && substrcmp (name, inst->name)) || (!substr
		  && g_strncasecmp (name, inst->name, strlen (name)) == 0))
		found = node;
	      break;
	    case 2:		/* sample search */
	      sample = (SFSample *) (slptr->data);
	      if ((substr && substrcmp (name, sample->name)) || (!substr
		  && g_strncasecmp (name, sample->name, strlen (name)) == 0))
		found = node;
	      break;
	    default:		/* preset search (0) */
	      preset = (SFPreset *) (slptr->data);
	      if (!nametog && (bank == -1 || bank == preset->bank)
		&& (prenum == -1 || prenum == preset->prenum))
		found = node;	/* bank/preset # match */
	      else if (nametog && (!strlen (name) ||
		  (substr && substrcmp (name, preset->name)) || (!substr
		    && g_strncasecmp (name, preset->name,
		      strlen (name)) == 0)))
		found = node;	/* name string match */
	      break;
	    }

	  node = GTK_CTREE_ROW (node)->sibling;	/* next node */
	}

      if (!alt)
	{			/* next sound font */
	  if (!allfiles)
	    break;

	  node = uisf->nodes->sfont;
	  node = GTK_CTREE_ROW (node)->sibling;
	  if (node)
	    uisf = SFTREE_UPFIND_UISF (node);
	  else
	    uisf = NULL;
	}
    }

  if (found)
    {				/* found a matching node? */
      if (uisf_selected_elem)
	uisf_deactivate_selection ();
      sftree_clear_selection ();

      ref = SFTREE_NODE_REF (found);
      gtk_object_set_data (GTK_OBJECT (findwin), "startitemid",
	GINT_TO_POINTER (ref->itemid));

      sftree_spotlight_node (found);

      if (beginning)
	{			/* turn off "start from beginning" if on */
	  btn = gtk_object_get_data (GTK_OBJECT (findwin), "CKBbegin");
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), FALSE);
	}
    }
  else
    util_quick_popup (_("No more matching items"), NULL);
}

/* expects a zone, advances to the item referenced by the zone */
void
uisf_goto_zone_ref (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;
  UISFont *uisf;
  SFTreeRef *ref;
  GtkCTreeNode *n, *found;
  GSList *finditem;
  SFZone *z;
  gint alt;		 /* for sample search, to alternate between branches */

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  uisf = SFTREE_UPFIND_UISF (node);
  ref = SFTREE_NODE_REF (node);

  if (ref->type != NODE_PZONE && ref->type != NODE_IZONE)
    return;

  finditem = (GSList *) (ref->dptr);
  z = (SFZone *) (finditem->data);
  finditem = z->instsamp;

  if (!finditem)
    return;			/* global zone? (no reference) */

  if (ref->type == NODE_PZONE)
    {
      n = uisf->nodes->inst;
      n = GTK_CTREE_ROW (n)->children;
      while (n)
	{
	  ref = SFTREE_NODE_REF (n);
	  if (ref->dptr == finditem)
	    break;
	  n = GTK_CTREE_ROW (n)->sibling;
	}
    }
  else
    {
      n = uisf->nodes->loaded;	/* user loaded samples branch */
      n = GTK_CTREE_ROW (n)->children;
      alt = 0;
      while (n)
	{
	  ref = SFTREE_NODE_REF (n);
	  if (ref->dptr == finditem)
	    break;
	  n = GTK_CTREE_ROW (n)->sibling;
	  if (!n && !alt)
	    {			/* done with loaded sample branch? */
	      alt = 1;
	      n = uisf->nodes->rom;	/* switch to rom branch */
	      n = GTK_CTREE_ROW (n)->children;
	    }
	}
    }

  if (!(found = n))
    return;			/* !found??!! */

  if (uisf_selected_elem)
    uisf_deactivate_selection ();
  sftree_clear_selection ();

  sftree_spotlight_node (found);
}


/***---        Sound Font NODE level NON-interactive routines        ---***/


static gint
uisf_find_pzone_by_inst_compare (SFTreeRef * nref, GSList * pinst)
{
  if (nref->type == NODE_PZONE &&
    ((SFZone *) (((GSList *) (nref->dptr))->data))->instsamp == pinst)
    return (0);			/* matched */
  return (1);			/* didn't match */
}

static gint
uisf_find_izone_by_sample_compare (SFTreeRef * nref, GSList * psam)
{
  if (nref->type == NODE_IZONE &&
    ((SFZone *) (((GSList *) (nref->dptr))->data))->instsamp == psam)
    return (0);			/* matched */
  return (1);			/* didn't match */
}

/*
  add a sound font item
   "parent" is the parent Preset or Inst if adding a zone, otherwise sf itemid
   "data" is a pointer to the SF(Preset | Inst | Sample | Zone) to add
   "pos" list index to insert item at SF(Inst | Sample | Zone) only,< 0 = append
   "type" is NODE_PRESET | NODE_INST | NODE_SAMPLE | NODE_PZONE | NODE_IZONE
*/
void
uisf_sfitem_add (SFItemID parent, gpointer data, gint pos, SFNodeType type)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  SFPreset *pset;
  SFInst *inst;
  GSList *p;

  g_return_if_fail (parent != SFITEMID_NONE);
  g_return_if_fail (data != NULL);

  if (!(node = SFTREE_LOOKUP_ITEMID (parent))) return;

  uisf = SFTREE_UPFIND_UISF (node);

  switch (type)
    {
    case NODE_PRESET:
      sfont_add_preset (uisf->sf, (SFPreset *)data);
      p = g_slist_find (uisf->sf->preset, (SFPreset *)data);
      sftree_add_preset_sorted (p, uisf->nodes);
      break;
    case NODE_INST:
      sfont_insert_inst (uisf->sf, (SFInst *)data, pos);
      p = g_slist_find (uisf->sf->inst, (SFInst *)data);
      sftree_add_inst (p, uisf->nodes, pos);
      break;
    case NODE_SAMPLE:
      sfont_insert_sample (uisf->sf, (SFSample *)data, pos);
      p = g_slist_find (uisf->sf->sample, (SFSample *)data);
      sftree_add_sample (p, uisf->nodes, pos);
      break;
    case NODE_PZONE:
      /* fetch parent preset structure */
      pset = (SFPreset *)(((GSList *)(SFTREE_NODE_REF (node)->dptr))->data);

      sfont_add_preset_zone (uisf->sf, pset, (SFZone *)data);
      p = g_slist_find (pset->zone, (SFZone *)data);
      sftree_add_pzone (p, node, pos);

      if (uisf_selected_elem == pset) /* zone added to selected item? */
	pianospan_update ();	/* update key spans */

      /* notify wavetable of changed preset */
      wtbl_sfitem_changed (parent, WTBL_ITEM_CHANGE);
      break;
    case NODE_IZONE:
      /* fetch parent instrument structure */
      inst = (SFInst *)(((GSList *)(SFTREE_NODE_REF (node)->dptr))->data);

      sfont_add_inst_zone (uisf->sf, inst, (SFZone *)data);
      p = g_slist_find (inst->zone, (SFZone *)data);
      sftree_add_izone (p, node, pos);

      if (uisf_selected_elem == inst) /* zone added to selected item? */
	pianospan_update ();	/* update key spans */

      /* notify wavetable of changed instrument */
      wtbl_sfitem_changed (parent, WTBL_ITEM_CHANGE);
      break;
    default:
      break;
    }
}

void
uisf_sfitem_delete (SFItemID itemid)
{
  GtkCTreeNode *node;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  switch (SFTREE_NODE_REF (node)->type)
    {
    case NODE_PRESET:
      uisf_preset_delete (itemid);
      break;
    case NODE_PZONE:
      uisf_pzone_delete (itemid);
      break;
    case NODE_INST:
      uisf_inst_delete (itemid);
      break;
    case NODE_IZONE:
      uisf_izone_delete (itemid);
      break;
    case NODE_SAMPLE:
      uisf_sample_delete (itemid);
      break;
    case NODE_VBNK_MAP:
      uisf_vbank_map_delete (itemid);
      break;
    default:
      break;
    }
}

static void
uisf_preset_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  SFTreeRef *ref;		/* Node reference */
  GSList *p;
  SFPreset *pset;
  UISFont *uisf;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  /* save undo state */
  sfdo_group (_("Delete preset"));
  dofunc_sfitem_save (itemid);
  sfdo_done ();

  ref = SFTREE_NODE_REF (node);
  p = ref->dptr;
  pset = (SFPreset *) (p->data);

  uisf = SFTREE_UPFIND_UISF (node);

  if (pset == uisf_selected_elem)
    uisf_deactivate_selection ();

  wtbl_sfitem_changed (itemid, WTBL_ITEM_DELETE);

  SFTREE_REMOVE_NODE (node);
  sfont_remove_preset (uisf->sf, pset);
}

static void
uisf_pzone_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  SFTreeRef *ref;
  GSList *z, *p;
  UISFont *uisf;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  /* save undo state */
  sfdo_group (_("Delete preset zone"));
  dofunc_sfitem_save (itemid);
  sfdo_done ();

  ref = SFTREE_NODE_REF (node);
  z = ref->dptr;

  /* get owning preset pointer */
  ref = SFTREE_NODE_REF (GTK_CTREE_ROW (node)->parent);
  p = ref->dptr;

  uisf = SFTREE_UPFIND_UISF (node);

  /* if its the active zone then set to nozone */
  if (z->data == uisf_selected_zone)
    uisf_set_selected_nozone ();

  SFTREE_REMOVE_NODE (node);	/* delete zone node from tree */
  sfont_remove_zone (uisf->sf, &(((SFPreset *) (p->data))->zone),
    (SFZone *) (z->data));

  if (p->data == uisf_selected_elem)	/* selected elem's zone deleted? */
    pianospan_update ();	/* update key spans */

  /* notify wavetable of changed preset */
  wtbl_sfitem_changed (((SFPreset *)(p->data))->itemid, WTBL_ITEM_CHANGE);
}

static void
uisf_inst_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  SFInst *inst;
  SFTreeRef *ref;
  GtkCTreeNode *nptr;
  GSList *pinst;
  GList *match, *p;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  /* start undo group (state saved at end of routine to group del pzones) */
  sfdo_group (_("Delete instrument"));

  /* fetch pointer to instrument list item */
  ref = SFTREE_NODE_REF (node);
  pinst = ref->dptr;

  uisf = SFTREE_UPFIND_UISF (node);	/* fetch the uisf structure */

  nptr = uisf->nodes->preset;	/* preset branch of tree */

  /* find all preset zone references to this instrument */
  match = gtk_ctree_find_all_by_row_data_custom (sftree_widg, nptr, pinst,
    (GCompareFunc) uisf_find_pzone_by_inst_compare);

  p = match;
  while (p)
    {				/* traverse preset zones */
      uisf_pzone_delete (SFTREE_NODE_REF (GTK_CTREE_NODE (p->data))->itemid);
      p = g_list_next (p);
    }
  g_list_free (match);		/* free matching zone list */

  /* save the instrument undo state */
  dofunc_sfitem_save (itemid);
  sfdo_done ();   /* close undo group, groups deleted pzones too */

  inst = (SFInst *) (pinst->data);

  if (inst == uisf_selected_elem)	/* selected item deleted? */
    uisf_deactivate_selection ();

  /* notify wavetable of the instrument delete */
  wtbl_sfitem_changed (itemid, WTBL_ITEM_DELETE);

  SFTREE_REMOVE_NODE (node);	/* remove the inst from tree */

  /* remove from sound font */
  sfont_remove_inst (uisf->sf, inst);
}

static void
uisf_izone_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  SFTreeRef *ref;
  GSList *z, *p;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  /* save undo state */
  sfdo_group (_("Delete instrument zone"));
  dofunc_sfitem_save (itemid);
  sfdo_done ();

  ref = SFTREE_NODE_REF (node);
  z = ref->dptr;		/* GSList * to zone */

  ref = SFTREE_NODE_REF (GTK_CTREE_ROW (node)->parent);
  p = ref->dptr;		/* GSList * to instrument that owns zone */

  uisf = SFTREE_UPFIND_UISF (node);

  /* if its the active zone then set nozone */
  if (z->data == uisf_selected_zone)
    uisf_set_selected_nozone ();

  SFTREE_REMOVE_NODE (node);	/* remove zone node from tree */
  sfont_remove_zone (uisf->sf, &(((SFInst *) (p->data))->zone),
    (SFZone *) (z->data));

  if (uisf_selected_elem == p->data)	/* selected element zone deleted? */
    pianospan_update ();	/* update key spans */

  /* notify wavetable of the instrument change */
  wtbl_sfitem_changed (itemid, WTBL_ITEM_CHANGE);
}

/* delete a sample in a sound font */
static void
uisf_sample_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  UISFont *uisf;
  SFSample *sam;
  SFTreeRef *ref;
  GtkCTreeNode *nptr;
  GSList *psam;
  GList *match, *p;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  /* start undo group (state saved at end of routine to group del izones) */
  sfdo_group (_("Delete sample"));

  /* fetch pointer to sample list item */
  ref = SFTREE_NODE_REF (node);
  psam = ref->dptr;

  uisf = SFTREE_UPFIND_UISF (node);

  nptr = uisf->nodes->inst;	/* instrument branch of tree */

  /* find all instrument zone references to this sample */
  match = gtk_ctree_find_all_by_row_data_custom (sftree_widg, nptr, psam,
    (GCompareFunc) uisf_find_izone_by_sample_compare);

  p = match;
  while (p)
    {				/* traverse instrument zones */
      uisf_izone_delete (SFTREE_NODE_REF (GTK_CTREE_NODE (p->data))->itemid);
      p = g_list_next (p);
    }
  g_list_free (match);		/* free matching zone list */

  /* save sample undo state */
  dofunc_sfitem_save (itemid);
  sfdo_done ();	    /* finished undo group, groups deleted izones too */

  sam = (SFSample *) (psam->data);

  if (sam == uisf_selected_elem)	/* selected item deleted? */
    uisf_deactivate_selection ();

  /* notify wavetable of the sample delete */
  wtbl_sfitem_changed (itemid, WTBL_ITEM_DELETE);

  SFTREE_REMOVE_NODE (node);	/* remove the sample from tree */

  /* remove from sound font */
  sfont_remove_sample (uisf->sf, sam);
}

static void
uisf_vbank_map_delete (SFItemID itemid)
{
  GtkCTreeNode *node;
  UIVBank *uivb;
  VBnkItem *item;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  uivb = SFTREE_UPFIND_UIVB (node);
  item = (VBnkItem *)(((GSList *)(SFTREE_NODE_REF (node)->dptr))->data);

  SFTREE_REMOVE_NODE (node);	/* remove map node from tree */

  vbank_delete_map (item, uivb->vbnk); /* remove map from vbank */
}

/* sets the sample to view (or makes samview inactive if NULL), if forceupd
   is TRUE then the sample is forced to reload */
void
uisf_set_sam_in_view (SFSample * sam, SFData * sf, gboolean forceupd)
{
  /* requested sample already in view (or sam==NULL and already NULL)? */
  if (!forceupd && sam == sam_in_view)
    return;

  /* free old sample in view if there is one */
  if (sam_in_view && sam_data_in_view)
    g_free (sam_data_in_view);

  /* in case we crap out because of an error */
  sam_in_view = NULL;

  /* disable sample viewer on ROM samples */
  if (!sam || sam->sampletype & SF_SAMPLETYPE_ROM)
    {
      samview_set_mode (SAMVIEW_INACTIVE);
      return;
    }

  /* everything's cool, so set up the view */

  if (!(sam_data_in_view = sam_load_sample (sam, sam->end + 1, 0, NULL)))
    {
      samview_set_mode (SAMVIEW_INACTIVE);
      return;
    }

  sam_in_view = sam;
}

/* deactivates the tree selection interface (i.e. all widgets tied to selection) */
void
uisf_deactivate_selection (void)
{
  uisf_set_selected_nozone ();
  pianospan_clear ();
  pianospan_clear_rootkey_ptrstrip ();
  uisf_set_sam_in_view (NULL, NULL, FALSE);	/* de-activate sample viewer */

  uisf_selected_uisfont = NULL;
  uisf_selected_uivbank = NULL;
  uisf_selected_elem = NULL;
  uisf_selected_elem_type = NODE_NONE;

  /* disable Save and Save As on main file menu */
  sfmenu_set_filemenu_save_sensitive (FALSE);
}

/* update temporary audible interface (piano follows selection, etc) */
static void
uisf_piano_follow_update (gboolean temp_audible, gint bank, gint prenum)
{
  if (!uisf_piano_follows) return;

  if (temp_audible)
    {
      bank = smurfcfg_get_int (SMURFCFG_TEMP_AUDIBLE_BANK);
      prenum = smurfcfg_get_int (SMURFCFG_TEMP_AUDIBLE_PRESET);
    }

  uipiano_set_bank (bank);
  uipiano_set_preset (prenum);
}

/* called by sftree_cb_select_row when any SINGLE item in the sfont tree is
   selected */
void
uisf_sfitem_selected (GtkCTreeNode *node)
{
  GtkCTreeNode *parent;
  SFTreeRef *ref;
  GSList *p, *p2;
  UISFont *uisf;
  UIVBank *uivb;

  ref = SFTREE_NODE_REF (node);
  p = (GSList *)(ref->dptr);

  if (!(uisf = SFTREE_UPFIND_UISF (node)))
    uivb = SFTREE_UPFIND_UIVB (node);

  switch (ref->type)
    {				/* switch node type */
    case NODE_PRESET:		/* A preset was clicked */
      uisf_set_selected_preset ((SFPreset *) (p->data), uisf);
      break;
    case NODE_INST:		/* An instrument was clicked */
      uisf_set_selected_inst ((SFInst *) (p->data), uisf);
      break;
    case NODE_SAMPLE:		/* A sample was clicked */
      uisf_set_selected_sample ((SFSample *) (p->data), uisf);
      break;
    case NODE_PZONE:		/* A preset zone was clicked */
      /* get preset node for this zone */
      parent = GTK_CTREE_ROW (node)->parent;
      ref = SFTREE_NODE_REF (parent);
      p2 = ref->dptr;

      uisf_set_selected_pzone ((SFZone *) (p->data), (SFPreset *) (p2->data),
	uisf);
      break;
    case NODE_IZONE:		/* An instrument zone was clicked */
      /* get instrument node for this zone */
      parent = GTK_CTREE_ROW (node)->parent;
      ref = SFTREE_NODE_REF (parent);
      p2 = ref->dptr;

      uisf_set_selected_izone ((SFZone *) (p->data), (SFInst *) (p2->data),
	uisf);
      break;
    case NODE_VBNK_MAP:
      uisf_set_selected_vbnk_map ((VBnkItem *)(p->data), uivb);
      break;
    default:
      if (uisf)
	{
	  uisf_selected_uisfont = uisf;
	  uisf_selected_uivbank = NULL;
	}
      else
	{
	  uisf_selected_uisfont = NULL;
	  uisf_selected_uivbank = uivb;
	}
      break;
    }

  /* enable Save and Save As on main file menu */
  sfmenu_set_filemenu_save_sensitive (TRUE);
}

/* reset all zone related user interface elements */
void
uisf_set_selected_nozone (void)
{
  sfgen_set_mode (GENMODE_INACTIVE);
  pianospan_clear_selection ();
  uisf_set_sam_in_view (NULL, NULL, FALSE);	/* de-activate sample viewer */
  uisf_selected_zone = NULL;
}

/* set selected preset and update user interface widgets */
static void
uisf_set_selected_preset (SFPreset *preset, UISFont *uisf)
{
  GtkCTreeNode *node;

  /* if there is a selected zone, de-activate it */
  if (uisf_selected_zone)
    uisf_set_selected_nozone ();
  else
    uisf_set_sam_in_view (NULL, NULL, FALSE);	/* de-activate sample viewer */

  /* if requested preset is already active, then return */
  if (uisf_selected_elem == preset) return;

  /* update selected item variables */
  uisf_selected_uisfont = uisf;
  uisf_selected_uivbank = NULL;
  uisf_selected_elem = preset;
  uisf_selected_elem_type = NODE_PRESET;

  pianospan_update ();		/* update key spans */

  pianospan_clear_rootkey_ptrstrip (); /* clear the rootkey pointer */

  /* check if preset is part of a loaded sound font */
  node = SFTREE_LOOKUP_ITEMID (wtbl_loaded_sfbank);
  if (!node || uisf != SFTREE_UPFIND_UISF (node))
    {
      /* preset is not part of loaded sound font, load as a temporary audible */
      uisf_wavetable_load_sfitem (preset->itemid, FALSE);

      /* point piano to new temporary audible */
      uisf_piano_follow_update (TRUE, -1, -1);
    }
  else uisf_piano_follow_update (FALSE, preset->bank, preset->prenum);
}

/* set selected element to instrument */
static void
uisf_set_selected_inst (SFInst *inst, UISFont *uisf)
{
  /* if there is a selected zone, de-activate it */
  if (uisf_selected_zone)
    uisf_set_selected_nozone ();
  else
    uisf_set_sam_in_view (NULL, NULL, FALSE);	/* just de-activate sample viewer */

  /* if requested instrument is already active, then return */
  if (uisf_selected_elem == inst)
    return;

  uisf_selected_uisfont = uisf;	/* set selected sound font */
  uisf_selected_uivbank = NULL;
  uisf_selected_elem = inst;
  uisf_selected_elem_type = NODE_INST;

  pianospan_update ();

  pianospan_clear_rootkey_ptrstrip (); /* clear the rootkey pointer */

  /* load instrument as temporary audible */
  uisf_wavetable_load_sfitem (inst->itemid, FALSE);

  /* point piano to new temporary audible */
  uisf_piano_follow_update (TRUE, -1, -1);
}

/* set selected element to sample */
static void
uisf_set_selected_sample (SFSample *sam, UISFont *uisf)
{
  /* if there is a selected zone, de-activate it */
  if (uisf_selected_zone)
    uisf_set_selected_nozone ();

  /* if requested sample is already active, then return */
  if (uisf_selected_elem == sam)
    return;

  uisf_selected_uisfont = uisf;	/* set selected sound font */
  uisf_selected_uivbank = NULL;
  uisf_selected_elem = sam;
  uisf_selected_elem_type = NODE_SAMPLE;

  pianospan_clear ();		/* clear key spans */
  pianospan_update_rootkey_ptrstrip ();	/* update root key ptr */

  /* load sample as temporary audible */
  uisf_wavetable_load_sfitem (sam->itemid, FALSE);

  /* point piano to new temporary audible */
  uisf_piano_follow_update (TRUE, -1, -1);

  samview_set_mode (SAMVIEW_SAMPLE);	/* set the sample view mode */
  uisf_set_sam_in_view (sam, uisf->sf, FALSE);	/* show sample in view */
  samview_update ();
}

/* set selected element to preset zone */
static void
uisf_set_selected_pzone (SFZone * zone, SFPreset * preset, UISFont * uisf)
{
  GtkCTreeNode *node;
  gint i;

  if (uisf_selected_elem != preset)
    {				/* is this preset already selected? */
      uisf_selected_uisfont = uisf;	/* set selected sound font */
      uisf_selected_uivbank = NULL;
      uisf_selected_elem = preset;
      uisf_selected_elem_type = NODE_PRESET;

      pianospan_update ();	/* update key spans */
      pianospan_clear_rootkey_ptrstrip (); /* clear the rootkey pointer */

      uisf_set_sam_in_view (NULL, NULL, FALSE);	/* de-activate sample viewer */

      /* check if preset is part of a loaded sound font */
      node = SFTREE_LOOKUP_ITEMID (wtbl_loaded_sfbank);
      if (!node || uisf != SFTREE_UPFIND_UISF (node))
	{
	  /* preset not part of loaded sfont, load as a temporary audible */
	  uisf_wavetable_load_sfitem (preset->itemid, FALSE);

	  /* point piano to new temporary audible */
	  uisf_piano_follow_update (TRUE, -1, -1);
	}
      else uisf_piano_follow_update (FALSE, preset->bank, preset->prenum);
    }

  if (uisf_selected_zone == zone)
    return;			/* return if zone already selected */

  uisf_selected_zone = zone;

  /* get zone index (minus global zone if any) for selecting key span */
  pianospan_clear_selection ();
  i = g_slist_index (preset->zone, zone);
  if (!((SFZone *) (preset->zone->data))->instsamp)
    i--;
  if (i >= 0)
    pianospan_select_span (i);

  sfgen_set_mode (GENMODE_OFS);	/* set to offset generator mode */
  sfgen_update (zone);		/* update generator widgets */
}

/* set selected element to instrument zone */
static void
uisf_set_selected_izone (SFZone * zone, SFInst * inst, UISFont * uisf)
{
  gint i;

  if (uisf_selected_elem != inst)
    {				/* is this instrument already selected? */
      uisf_selected_uisfont = uisf;	/* set selected sound font */
      uisf_selected_uivbank = NULL;
      uisf_selected_elem = inst;
      uisf_selected_elem_type = NODE_INST;

      pianospan_update ();	/* update key spans */

      /* load instrument into seq device */
      uisf_wavetable_load_sfitem (inst->itemid, FALSE);

      /* point piano to new temporary audible */
      uisf_piano_follow_update (TRUE, -1, -1);
    }

  if (uisf_selected_zone == zone)
    return;

  uisf_selected_zone = zone;

  /* get zone index (minus global zone if any) for selecting key span */
  pianospan_clear_selection ();
  i = g_slist_index (inst->zone, zone);
  if (!((SFZone *) (inst->zone->data))->instsamp)
    i--;
  if (i >= 0)
    pianospan_select_span (i);

  pianospan_update_rootkey_ptrstrip ();	/* update root key override ptr */

  sfgen_set_mode (GENMODE_ABS);	/* set to absolute generator mode */
  sfgen_update (zone);

  /* check if this is a global zone, and display zone sample if not */
  if (zone->instsamp)
    {
      samview_set_mode (SAMVIEW_IZONE);	/* set to izone sample view mode */
      uisf_set_sam_in_view ((SFSample *) (zone->instsamp->data), uisf->sf,
	FALSE);
      samview_update ();
    }
  else
    uisf_set_sam_in_view (NULL, NULL, FALSE);
}

static void
uisf_set_selected_vbnk_map (VBnkItem *item, UIVBank *uivb)
{
  GtkCTreeNode *node;

  uisf_set_selected_nozone ();
  pianospan_clear ();
  pianospan_clear_rootkey_ptrstrip ();
  uisf_set_sam_in_view (NULL, NULL, FALSE);	/* de-activate sample viewer */

  uisf_selected_uivbank = uivb;
  uisf_selected_uisfont = NULL;
  uisf_selected_elem = item;
  uisf_selected_elem_type = NODE_VBNK_MAP;

  /* check if preset map is part of a loaded virtual bank */
  node = SFTREE_LOOKUP_ITEMID (wtbl_loaded_sfbank);
  if (!node || uivb != SFTREE_UPFIND_UIVB (node))
    {
      /* preset map is not part of loaded vbank, load as a temporary audible */
      uisf_wavetable_load_sfitem (item->itemid, FALSE);

      /* point piano to new temporary audible */
      uisf_piano_follow_update (TRUE, -1, -1);
    }
  else uisf_piano_follow_update (FALSE, item->map.bank, item->map.psetnum);
}

void
uisf_wavetable_load_sfitem (SFItemID itemid, gboolean force)
{
  GtkCTreeNode *node;
  SFTreeRef *ref;
  UISFont *uisf;
  UIVBank *uivb;
  SFPreset *pset;
  GSList *p;
  gint retval;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;
  ref = SFTREE_NODE_REF (node);
  uisf = SFTREE_UPFIND_UISF (node);
  p = ref->dptr;

  if (ref->type == NODE_SFONT || ref->type == NODE_VBANK)
    {
      /* if currently loaded sfont and its not the same one to be loaded */
      if (wtbl_loaded_sfbank != SFITEMID_NONE && wtbl_loaded_sfbank != itemid)
	{
	  GtkCTreeNode *sfnode = SFTREE_LOOKUP_ITEMID (wtbl_loaded_sfbank);
	  if (sfnode)		/* clear the pixmap for old loaded sfont */
	    sftree_set_aux_pixmap (sfnode, NULL);
	}

      if (uisf)			/* ?: virtual bank? */
	retval = wtbl_load_sfont (uisf->sf); /* ?: No, sound font, load it */
      else
	{				/* ?: Yes, load virtual bank */
	  uivb = SFTREE_UPFIND_UIVB (node);
	  retval = wtbl_load_vbank (uivb->vbnk);
	}

      if (retval == OK)		/* updated loaded sfont icon, if all OK */
	sftree_set_aux_pixmap (node, loaded_xpm);
    }
  else
    {
      /* if not automatically updating temp audible and not using "force" */
      if (!uisf_auto_temp_audible && !force)
	return;

      switch (ref->type)
	{
	case NODE_PRESET:
	  wtbl_load_temp_preset ((SFPreset *)(p->data), uisf->sf);
	  break;
	case NODE_INST:
	  wtbl_load_temp_inst ((SFInst *)(p->data), uisf->sf);
	  break;
	case NODE_SAMPLE:
	  wtbl_load_temp_sam_as_inst ((SFSample *)(p->data), uisf->sf);
	  break;
	case NODE_VBNK_MAP:
	  /* find the source preset mapped by this vbank item */
	  pset = vbank_get_item_preset ((VBnkItem *)(p->data));
	  if (!pset) break;

	  /* locate its sftree node */
	  node = SFTREE_LOOKUP_ITEMID (pset->itemid);
	  if (!node) break;

	  /* find parent sound font */
	  uisf = SFTREE_UPFIND_UISF (node);

	  /* load the preset */
	  wtbl_load_temp_preset (pset, uisf->sf);
	  break;
	default:
	  break;
	}
    }
}

void
uisf_wavetable_reset (void)
{
  if (wtbl_loaded_sfbank != SFITEMID_NONE)
    {
      GtkCTreeNode *sfnode = SFTREE_LOOKUP_ITEMID (wtbl_loaded_sfbank);

      if (sfnode)		/* clear the pixmap for old loaded sfont */
	sftree_set_aux_pixmap (sfnode, NULL);

      wtbl_loaded_sfbank = SFITEMID_NONE;
    }

  wtbl_clear_samples ();
}
