/*==================================================================
 * uif_treemenu.c - sound font tree right-click menus
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <gtk/gtk.h>
#include "uif_treemenu.h"
#include "uif_sfont.h"
#include "uif_selections.h"
#include "uif_sftree.h"
#include "uif_sample.h"
#include "uif_wavegen.h"
#include "uiface.h"
#include "wavetable.h"
#include "sfont.h"
#include "util.h"
#include "i18n.h"

typedef void (*RMU_Func) ();

/* menu item enumerations (keep up to date with rmu_items[] below) */
typedef enum {
  RMU_Spacer,			/* A horizontal line menu spacer */
  RMU_Paste,
  RMU_Delete,
  RMU_Close,
  RMU_Export,
  RMU_Properties,
  RMU_Save,
  RMU_Dump,
  RMU_Find,
  RMU_Generate_Sample,
  RMU_Global_Zone,
  RMU_Goto_Instrument,
  RMU_Goto_Sample,
  RMU_New_Instrument,
  RMU_New_Preset,
  RMU_New_Sample,
  RMU_Save_As,
  RMU_Wtbl_Load,
  RMU_New_Mapping,
  RMU_END
} RMUItemType;

/* return values for RMU_ModFunc's */
enum { MODFUNC_NORMAL, MODFUNC_INSENSITIVE, MODFUNC_INVISIBLE };

/* RMU_ModFunc is called for each RMU_Item and allows for run time modification
   of menu items (sensitivity and visibility) */
typedef gint (*RMU_ModFunc) (RMUItemType type, SFItemID itemid);

typedef struct
{
  gchar flag;	  /* S = single, M = multi, R = right click multi */
  gchar *label;			/* menu label text */
  gchar *accel;			/* key accelerator */
  RMU_ModFunc modfunc;		/* menu item modify function */
  RMU_Func func;		/* function to call when item is activated */
  gpointer func_data;		/* data to pass to function */
  GtkWidget *widg;		/* the created menu item or NULL */
} RMU_Item;

static void tmenu_cb_wtbl_load_sfitem (void);

static gint rmumod_goto_zoneref (RMUItemType type, SFItemID itemid);
static gint rmumod_wavetable_load (RMUItemType type, SFItemID itemid);

/* Menu items (keep up to date with RMUItemType menu enumerations above) */
static RMU_Item rmu_items[] = {
  { 'R', N_("Paste"), NULL, NULL, uisel_paste_items_start, NULL, NULL },
  { 'M', N_("Delete"), "<control>D", NULL,
    uisel_dialog_selection_delete, NULL, NULL },
  { 'S', N_("Close"), NULL, NULL, uisf_close_sfont, NULL, NULL },
  { 'S', N_("Export"), NULL, NULL, uisam_export_sample, NULL, NULL },
  { 'S', N_("Properties"), "<control>R", NULL,
    uisf_item_properties, NULL, NULL },
  { 'S', N_("Save"), NULL, NULL, (RMU_Func) uisf_save_sfont,
    GINT_TO_POINTER (FALSE), NULL },
  { 'S', N_("Dump"), NULL, NULL, uisf_dump_sfont, NULL, NULL },
  { 'S', N_("Find"), NULL, NULL, uisf_find, GINT_TO_POINTER (TRUE), NULL },
  { 'S', N_("Generate Sample"), NULL, NULL, uiwavgen_new_sample, NULL, NULL },
  { 'S', N_("Global Zone"), "<control>G", NULL,
    uisf_item_global_zone, NULL, NULL },
  { 'S', N_("Goto Instrument"), NULL, rmumod_goto_zoneref,
    uisf_goto_zone_ref, NULL, NULL },
  { 'S', N_("Goto Sample"), NULL, rmumod_goto_zoneref,
    uisf_goto_zone_ref, NULL, NULL },
  { 'S', N_("New Instrument"), NULL, NULL,
    uisf_newmod_inst, GINT_TO_POINTER (TRUE), NULL },
  { 'S', N_("New Preset"), NULL, NULL,
    uisf_newmod_preset, GINT_TO_POINTER (TRUE), NULL },
  { 'S', N_("New Sample"), NULL, NULL, uisam_load_sample, NULL, NULL },
  { 'S', N_("Save As"), NULL, NULL,
    (RMU_Func) uisf_save_sfont, GINT_TO_POINTER(TRUE), NULL },
  { 'S', N_("Wavetable Load"), "<control>L", rmumod_wavetable_load,
    tmenu_cb_wtbl_load_sfitem, NULL, NULL },
  { 'S', N_("New Mapping"), NULL, NULL, NULL, NULL, NULL }
};

static RMUItemType rmu_sfont[] = { /* menu for sound font root nodes */
  RMU_Properties,
  RMU_Save,
  RMU_Save_As,
  RMU_Close,
  RMU_Paste,
  RMU_Find,
  RMU_Dump,
  RMU_Wtbl_Load,

  RMU_Spacer,

  RMU_New_Preset,
  RMU_New_Instrument,
  RMU_New_Sample,
  RMU_Generate_Sample,
  RMU_END
};
static RMUItemType rmu_presetroots[] = { /* menu for preset tree roots */
  RMU_New_Preset,
  RMU_Paste,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_preset[] = {	/* Menu for presets */
  RMU_Properties,
  RMU_Global_Zone,
  RMU_Paste,
  RMU_Delete,
  RMU_Wtbl_Load,

  RMU_Spacer,

  RMU_New_Preset,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_pzone[] = {	/* Menu for preset zones */
  RMU_Goto_Instrument,
  RMU_Delete,
  RMU_END
};
static RMUItemType rmu_instroot[] = {	/* Menu for instrument root tree */
  RMU_New_Instrument,
  RMU_Paste,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_inst[] = {	/* Menu for instruments */
  RMU_Properties,
  RMU_Global_Zone,
  RMU_Paste,
  RMU_Delete,
  RMU_Wtbl_Load,

  RMU_Spacer,

  RMU_New_Instrument,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_izone[] = {	/* Menu for instrument zones */
  RMU_Goto_Sample,
  RMU_Delete,
  RMU_END
};
static RMUItemType rmu_sampleroot[] = {	/* Menu for sample root tree */
  RMU_New_Sample,
  RMU_Generate_Sample,
  RMU_Paste,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_sample[] = {	/* Menu for samples */
  RMU_Properties,
  RMU_Export,
  RMU_Delete,
  RMU_Wtbl_Load,

  RMU_Spacer,

  RMU_New_Sample,
  RMU_Generate_Sample,
  RMU_Find,
  RMU_END
};
static RMUItemType rmu_vbank[] = {
  RMU_Properties,
  RMU_Save,
  RMU_Save_As,
  RMU_Close,
  RMU_Wtbl_Load,
  RMU_END
};
static RMUItemType rmu_vbnk_defbank[] = {
  RMU_Properties,
  RMU_Paste,
  RMU_END
};
static RMUItemType rmu_vbnk_maproot[] = {
  RMU_New_Mapping,
  RMU_Paste,
  RMU_END
};
static RMUItemType rmu_vbnk_map[] = {
  RMU_Properties,
  RMU_Delete,
  RMU_Wtbl_Load,

  RMU_Spacer,

  RMU_Paste,
  RMU_END
};

RMUItemType *rmu_array[NODE_LAST] = {
  NULL,				/* NODE_UNKNOWN */
  rmu_sfont,			/* NODE_SFONT */
  rmu_presetroots,		/* NODE_PRESETROOT */
  rmu_presetroots,		/* NODE_MELODIC */
  rmu_presetroots,		/* NODE_PERCUSS */
  rmu_preset,			/* NODE_PRESET */
  rmu_pzone,			/* NODE_PZONE */
  rmu_instroot,			/* NODE_INSTROOT */
  rmu_inst,			/* NODE_INST */
  rmu_izone,			/* NODE_IZONE */
  rmu_sampleroot,		/* NODE_SAMPLEROOT */
  rmu_sampleroot,		/* NODE_USER */
  rmu_sampleroot,		/* NODE_ROM */
  rmu_sample,			/* NODE_SAMPLE */
  rmu_vbank,			/* NODE_VBANK */
  rmu_vbnk_defbank,		/* NODE_VBNK_DEFBANK */
  rmu_vbnk_maproot,		/* NODE_VBNK_MAPROOT */
  rmu_vbnk_map			/* NODE_VBNK_MAP */
};

GtkWidget *treemenu_widg;
static GtkAccelGroup *treemenu_accel_group;

static GtkWidget *treemenu_get_menu_item (RMUItemType type);
static gboolean treemenu_cb_selection_done (GtkMenuShell * menushell);
static void treemenu_forcb_remove (GtkWidget *widg, gpointer data);

void
treemenu_init (void)
{
  GtkWidget *mitem;
  RMU_Item *item;
  guint key, mods;
  gint i;

  treemenu_widg = gtk_menu_new ();
  gtk_signal_connect (GTK_OBJECT (treemenu_widg), "selection-done",
		      (GtkSignalFunc)treemenu_cb_selection_done, NULL);

  /* create key accelerator group and bind to main window */
  treemenu_accel_group = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (ui_main_window),
			      treemenu_accel_group);

  i = RMU_Spacer;
  while (++i < RMU_END)		/* Loop over menu items */
    {
      item = &rmu_items[i - 1];

      mitem = gtk_menu_item_new_with_label (_(item->label));
      item->widg = mitem;

      if (item->func)
	gtk_signal_connect (GTK_OBJECT (mitem), "activate",
			    item->func, item->func_data);

      /* parse key accelerator and add it to menu item */
      if (item->accel)
	{
	  gtk_accelerator_parse (item->accel, &key, &mods);
	  gtk_widget_add_accelerator (mitem, "activate",
				      treemenu_accel_group,
				      key, mods,
				      GTK_ACCEL_VISIBLE);
	}
    }
}

void
treemenu_popup (guint button, guint32 time)
{
  GtkCTreeNode *node;
  SFTreeRef *ref;
  RMUItemType *rmu;
  GtkWidget *mitem;
  RMU_Item *item;
  gint i;

  if (!sftree_rclicked_itemid) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (sftree_rclicked_itemid))) return;
  ref = SFTREE_NODE_REF (node);

  if (!(rmu = rmu_array[ref->type])) return; /* should not happen */

  gtk_container_foreach (GTK_CONTAINER (treemenu_widg),
			 (GtkCallback)treemenu_forcb_remove,
			 treemenu_widg);

  if (!sftree_is_node_selected (node) || SFTREE_SELECTION_IS_SINGLE ())
    {				/* Single selected item */
      i = 0;
      while (rmu[i] != RMU_END)
	{
	  mitem = treemenu_get_menu_item (rmu[i]);
	  gtk_menu_append (GTK_MENU (treemenu_widg), mitem);
	  i++;
	}
    }
  else				/* Multiple selected items */
    {
      i = RMU_Spacer + 1;
      while (i < RMU_END)	/* Loop over menu items */
	{
	  item = &rmu_items[i - 1];

	  /* use type 'M'ulti or 'R'ight click items */
	  if (item->flag == 'M' || item->flag == 'R')
	    {
	      mitem = treemenu_get_menu_item (i);
	      gtk_menu_append (GTK_MENU (treemenu_widg), mitem);
	    }
	  i++;
	}
    }

  gtk_widget_show (treemenu_widg);
  gtk_menu_popup (GTK_MENU (treemenu_widg), NULL, NULL, NULL, NULL,
		  button, time);
}

static GtkWidget *
treemenu_get_menu_item (RMUItemType type)
{
  GtkWidget *mitem;
  gboolean sensitive = TRUE;
  gboolean show = TRUE;

  if (type == RMU_Spacer)
    {
      mitem = gtk_menu_item_new ();
      sensitive = FALSE;
    }
  else
    {
      mitem = rmu_items[type - 1].widg;
      
      if (rmu_items[type - 1].modfunc)
	switch ((*rmu_items[type - 1].modfunc)(type, sftree_rclicked_itemid))
	  {
	  case MODFUNC_INSENSITIVE: sensitive = FALSE; break;
	  case MODFUNC_INVISIBLE: show = FALSE; break;
	  default: break;
	  }
    }

  gtk_widget_set_sensitive (mitem, sensitive);

  if (show) gtk_widget_show (mitem);
  else gtk_widget_hide (mitem);

  return (mitem);
}

static gboolean
treemenu_cb_selection_done (GtkMenuShell * menushell)
{
  GtkCTreeNode *node;

  if (sftree_rclicked_itemid == SFITEMID_NONE) return (FALSE);

  if ((node = SFTREE_LOOKUP_ITEMID (sftree_rclicked_itemid)))
    sftree_unset_highlight (node);

  sftree_rclicked_itemid = SFITEMID_NONE;

  return (FALSE);
}

static void
treemenu_forcb_remove (GtkWidget *widg, gpointer data)
{
  gtk_container_remove (GTK_CONTAINER (data), widg);
}


static void
tmenu_cb_wtbl_load_sfitem (void)
{
  SFItemID itemid;
  GtkCTreeNode *node;

  if (!(itemid = sftree_get_selection_single ())) return;
  if (!(node = SFTREE_LOOKUP_ITEMID (itemid))) return;

  uisf_wavetable_load_sfitem (itemid, TRUE);
}

/* RMU_ModFunc's */

/* menu item modify function for "Goto Instrument" and "Goto Sample" */
static gint
rmumod_goto_zoneref (RMUItemType type, SFItemID itemid)
{
  GtkCTreeNode *node;
  GSList *p;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    return (MODFUNC_NORMAL);

  p = SFTREE_NODE_DATA (node);

  /* disable menu item if its a global zone */
  if (!((SFZone *)(p->data))->instsamp)
    return (MODFUNC_INVISIBLE);

  return (MODFUNC_NORMAL);
}

static gint
rmumod_wavetable_load (RMUItemType type, SFItemID itemid)
{
  GtkCTreeNode *node;
  SFTreeRef *ref;
  UISFont *uisf;
  UIVBank *uivb;

  if (!(node = SFTREE_LOOKUP_ITEMID (itemid)))
    return (MODFUNC_NORMAL);

  ref = SFTREE_NODE_REF (node);

  switch (ref->type)
    {
    case NODE_PRESET:
      /* disable menu item if preset is part of loaded sound font or auto
         load temporary audible mode */
      uisf = SFTREE_UPFIND_UISF (node);
      if (uisf->sf->itemid == wtbl_loaded_sfbank || uisf_auto_temp_audible)
	return (MODFUNC_INVISIBLE);
      break;
    case NODE_INST:
    case NODE_SAMPLE:
      /* disable menu item if auto load temporary audible is enabled */
      if (uisf_auto_temp_audible)
	return (MODFUNC_INVISIBLE);
      break;
    case NODE_VBNK_MAP:
      /* disable menu item if preset map is part of loaded vbank or auto
         load temporary audible mode */
      uivb = SFTREE_UPFIND_UIVB (node);
      if (uivb->vbnk->itemid == wtbl_loaded_sfbank || uisf_auto_temp_audible)
	return (MODFUNC_INVISIBLE);
      break;
    default:
      break;
    }

  return (MODFUNC_NORMAL);
}
