/*
 * ---------------------------------------------------------------------
 * diskimage.c
 *	Library for diskimages for use in nonvolatile memory device
 *	emulators like amdflash or serial eeproms and MMC/SD-Cards
 *
 * (C) 2005 Jochen Karrer
 *    Author: Jochen Karrer
 *
 * Status: Working 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */


#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/mman.h>
#include <sys/types.h>
#include "diskimage.h"

/*
 * ------------------------------------------------------------------------
 * Fill a diskimage with emptyval. This is intended for 
 * new non-sparse diskimages
 * ------------------------------------------------------------------------
 */
static int 
diskimage_fill(DiskImage *di,uint8_t emptyval) {
	int i;
	uint8_t buf[1024];
	int64_t count; /* must be signed because file might be oversized */
	off64_t pos;
	if((pos=lseek64(di->fd,0,SEEK_END))==(off_t)-1) {
		perror("lseek64 on flashfile failed");
		return -1;
	}
	count=di->size-pos;
	for(i=0;i<sizeof(buf);i++) {
		buf[i]=emptyval;
	}
	while(count>=sizeof(buf)) {
		int result;
		result = write(di->fd,buf,sizeof(buf));
		if(result<=0) {
			perror("write failed");
			break;
		} else {
			count -= result;
		}
	}
	while(count>0) {
		if(write(di->fd,buf,1)<=0) {
			perror("write failed");
			break;
		}
		count--;
	}
	return 0;
}

static int 
diskimage_sparse_fill(DiskImage *di,uint8_t emptyval) {
	off64_t pos;
	if(di->size < 1) {
		fprintf(stderr,"Sparse diskimage is < 1 Byte\n");
		return -1;
	}
	if((pos=lseek64(di->fd,0,SEEK_END))==-1) {
		perror("lseek64 on flashfile failed");
		return -1;
	}
	if(pos >= di->size) {
		return 0;
	}
	if((pos=lseek64(di->fd,di->size-1,SEEK_SET)) != (di->size-1)) {
		perror("lseek64 on sparse file failed");
		return -1;
	}
	if(write(di->fd,&emptyval,1) != 1) {
		perror("writing last byte of sparse diskimage failed");
		return -1;
	}
	return 0;
}


DiskImage *
DiskImage_Open(const char *name,uint64_t size,int flags) {
	DiskImage *di;
	uint8_t emptyval;
	di = malloc(sizeof(DiskImage));
	if(!di) {
		fprintf(stderr,"out of memory\n");
		exit(14);
	}
	memset(di,0,sizeof(DiskImage));
	di->size = size;
	if(flags & DI_RDWR) {
		if(flags & (DI_CREAT_FF | DI_CREAT_00)) {
			di->fd=open(name,O_RDWR|O_CREAT | O_LARGEFILE,0644);
		} else {
			di->fd=open(name,O_RDWR | O_LARGEFILE,0644);
		}
	} else {
		di->fd=open(name,O_RDONLY);
	}
	if(di->fd<0) {
		fprintf(stderr,"Can't open image \"%s\"",name);
		perror("");
		free(di);
		return NULL;
	}
	if(flock(di->fd,LOCK_EX|LOCK_NB)<0) {
		fprintf(stderr,"Can't get lock for diskimage \"%s\"\n",name);
		close(di->fd);
		free(di);
		return NULL;
	}
	if(flags & DI_CREAT_FF) {
		emptyval = 0xff;
	} else {
		emptyval = 0x00;
	}
	if(flags & DI_SPARSE) {
		if(diskimage_sparse_fill(di,emptyval) < 0) {
			close(di->fd);
			free(di);
			return NULL;
		}
	} else {
		if(diskimage_fill(di,emptyval) < 0) {
			close(di->fd);
			free(di);
			return NULL;
		}
	}
	return di;
}

int
DiskImage_Read(DiskImage *di,off64_t ofs,uint8_t *buf,int count) 
{
	int result;
	int cnt;
	if(lseek64(di->fd,ofs,SEEK_SET) != ofs) {
		return -EINVAL;
	}
	for(cnt=0;cnt < count;) {
		result = read(di->fd,buf+cnt,count);
		if(result<=0) {
			if(cnt) {
				return cnt;
			} else {
				return result;
			}
		} 	
		cnt += result;
	}
	return cnt;
	
}
int
DiskImage_Write(DiskImage *di,off64_t ofs,const uint8_t *buf,int count) 
{
	int result;
	int cnt;
	if(lseek64(di->fd,ofs,SEEK_SET) != ofs) {
		return -EINVAL;
	}
	for(cnt=0;cnt < count;) {
		result = write(di->fd,buf+cnt,count);
		if(result<=0) {
			if(cnt) {
				return cnt;
			} else {
				return result;
			}
		} 	
		cnt += result;
	}
	return cnt;
}

void *
DiskImage_Mmap(DiskImage *di) 
{
	di->map=mmap(0,di->size,PROT_READ|PROT_WRITE,MAP_SHARED,di->fd,0);
	if(di->map == (void*)-1) {
		perror("mmap of diskimage failed");
		close(di->fd);
		free(di);
		return NULL;
	}
	return di->map;
}

void
DiskImage_Close(DiskImage *di) 
{
	if(di->map) {
		munmap(di->map,di->size);
		di->map=NULL;
	}
	flock(di->fd,LOCK_UN);
	close(di->fd);
	free(di);
}
