/* Somaplayer - Copyright (C)>003 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include <stdio.h>
#include <string.h>
#include <ao/ao.h>
#include <sndfile.h>
#include <confuse.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#define SAVE_AND_QUIT(x) { \
                        write_config(x); \
			fprintf(stdout, "Good work :)\n\n"); \
			ao_shutdown(); \
			return 0; \
		      }

#define AUTHOR "Bakunin - <bakunin@autistici.org> Copyright GPL 2003-4"

static ao_device *device = NULL;

#ifdef WORDS_BIGENDIAN
int16_t
swap (int16_t x)
{
  return ((((u_int16_t) x & 0x00ffU) << 8) |
	  (((u_int16_t) x & 0xff00U) >> 8));
}
#endif

/* Play the file. With libsndfile open the test file and with the libao
 * test the outputs. */
int
try_audio (void)
{
  int ret;
  char buf[1024];
#ifdef WORDS_BIGENDIAN
  int16_t *buf16 = (int16_t *) buf;
#endif
  SNDFILE *sndfile;
  SF_INFO sfinfo;

  if (!(sndfile = sf_open (SOMAPLAYER_CONFIGURE_FILE, SFM_READ, &sfinfo)))
    {
      fprintf (stderr, "I can't open the test file.\n");
      exit (1);
    }

  while ((ret = sf_read_raw (sndfile, buf, 1024)))
    {
#ifdef WORDS_BIGENDIAN
      int i;
      for (i = 0; i < ret / 2; i++)
	buf16[i] = swap (buf16[i]);
#endif
      ao_play (device, buf, ret);
    }

  ao_close (device);

  sf_close (sndfile);

  fprintf (stdout, "Do you listen something? [y/n] ");
  fflush(stdin);
  fscanf (stdin, "%s", buf);

  if (*buf == 'y' || *buf == 'Y')
    return 0;

  return 1;
}

/* Write the configure file ~/.somaplayer */
void
write_config (char *audio)
{
  FILE *fl;
  char *home;
  char *file = NULL;
  int n;
  int flag = 0;
  cfg_t *cfg;

  char *buffer = NULL;
  int volume = 100;
  int random = 0;
  int repeat = 0;
  int noverbose = 0;
  int nobuffer = 0;

  struct stat st;

  cfg_opt_t opts[] = {
    CFG_STR ("buffer", 0, CFGF_NONE),
    CFG_INT ("volume", 100, CFGF_NONE),
    CFG_BOOL ("random", cfg_false, CFGF_NONE),
    CFG_BOOL ("repeat", cfg_false, CFGF_NONE),
    CFG_BOOL ("noverbose", cfg_false, CFGF_NONE),
    CFG_BOOL ("nobuffer", cfg_false, CFGF_NONE),
#ifdef ENABLE_GTK
    CFG_INT ("x", 0, CFGF_NONE),
    CFG_INT ("y", 0, CFGF_NONE),
#endif
    CFG_STR ("audio", NULL, CFGF_NONE),
    CFG_END ()
  };

  if (!getuid ())
    {
      char buf[1024];

      fprintf (stdout, "Do you want save general config file (%s) ? [y/n] ", CONFIG_FILE);
      fflush(stdin);
      fscanf (stdin, "%s", buf);

      if (*buf == 'y' || *buf == 'Y')
	file = strdup (CONFIG_FILE);
    }

  if (!file)
    {
      if (!(home = getenv ("HOME")))
	{
	  fprintf (stderr, "I can't found your home.\n");
	  exit (1);
	}

      n = strlen (home) + strlen ("/.somaplayer/config") + 1;
      file = (char *) malloc (n * sizeof (char));
      sprintf (file, "%s/.somaplayer", home);

      if(lstat(file, &st) || S_ISDIR(st.st_mode)) {
        unlink(file);
	mkdir(file, 0750);
      }

      sprintf (file, "%s/.somaplayer/config", home);
    }

  cfg = cfg_init (opts, CFGF_NOCASE);
  n = cfg_parse (cfg, file);

  switch (cfg_parse (cfg, file))
    {
    case CFG_FILE_ERROR:
      fprintf (stderr, "I can't open your config file. I create it.\n");
      flag = 1;
      break;

    case CFG_PARSE_ERROR:
      fprintf (stderr, "Syntax error in your file! I rewrite it.\n");
      unlink (file);
      flag = 1;
      break;
    }

  if (!flag)
    {
      buffer = cfg_getstr (cfg, "buffer");
      volume = cfg_getint (cfg, "volume");
      random = cfg_getbool (cfg, "random");
      noverbose = cfg_getbool (cfg, "noverbose");
      nobuffer = cfg_getbool (cfg, "nobuffer");
    }

  if (!(fl = fopen (file, "w")))
    {
      fprintf (stderr, "I can't open your config file.\n");
      exit (1);
    }

  if (buffer)
    fprintf (fl, "buffer = %s\n", buffer);
  fprintf (fl, "volume = %d\n", volume);
  fprintf (fl, "random = %s\n", random ? "true" : "false");
  fprintf (fl, "repeat = %s\n", repeat ? "true" : "false");
  fprintf (fl, "noverbose = %s\n", noverbose ? "true" : "false");
  fprintf (fl, "nobuffer = %s\n", nobuffer ? "true" : "false");
  if (audio)
    fprintf (fl, "audio = %s\n", audio);

  fclose (fl);

  free (file);
  cfg_free (cfg);

}

/* Start the game! */
int
main (int argc, char *argv[])
{
  ao_sample_format format;
  int driver_id;

  format.byte_format = AO_FMT_NATIVE;

  fprintf (stdout, "%s_configure - Version: %s - %s\n\n", PACKAGE, VERSION,
	   AUTHOR);
  fprintf (stdout, "Search your audio interface...\n\n");

  /* I insert this values becase I know the test file */
  format.bits = 16;
  format.rate = 44100;
  format.channels = 2;

  ao_initialize ();

  fprintf (stdout, "Try OSS:\n");
  driver_id = ao_driver_id ("oss");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("oss");

  fprintf (stdout, "Try ALSA:\n");
  driver_id = ao_driver_id ("alsa");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("alsa");

  fprintf (stdout, "Try ALSA09:\n");
  driver_id = ao_driver_id ("alsa09");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("alsa09");

  fprintf (stdout, "Try ESD:\n");
  driver_id = ao_driver_id ("esd");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("esd");

  fprintf (stdout, "Try ARTS:\n");
  driver_id = ao_driver_id ("esd");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("arts");

  fprintf (stdout, "Try SUN:\n");
  driver_id = ao_driver_id ("sun");
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT ("sun");

  fprintf (stdout, "Try AUTOSEARCH:\n");
  driver_id = ao_default_driver_id ();
  device = ao_open_live (driver_id, &format, NULL);
  if (device && !try_audio ())
    SAVE_AND_QUIT (NULL);

  fprintf (stdout, "I cant' found your audio interface. Sorry :/\n\n");

  ao_shutdown ();

  return 1;
}
