/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "list.h"
#include "file.h"
#include "playlist.h"
#include "other.h"
#include "format.h"

#ifdef ENABLE_GTK
#include "gtk/queue.h"
#include "gtk/graphic.h"
#endif

#define getrandom(min, max) ((rand() % (int)(((max)+1) - (min))) + (min))

/* Here there are the functions for the list of files. There is a list of 
 * files and i select the next, prev file to play. */

/* Add and element */
int
list_add (char *str, list * before)
{
  list *tmp;
  char *tmp_dir;
  struct stat st;
  int type;

  if (!str)
    return 1;

  if (!strncmp (str, "file://", 7))
    str += 7;

  type = type_file (str);

  if (type == TYPE_PLAYLIST)
    return playlist_read (str);

  if (!(tmp = (list *) malloc (sizeof (list))))
    fatal (_("Error: memory."));

  tmp->next = NULL;
  tmp->prev = NULL;
  tmp->name = NULL;

#ifdef ENABLE_GTK
  tmp->queue = 0;
#endif

  tmp->n = 0;
  if (!(tmp->filename = strdup (str)))
    fatal (_("Error: memory."));

  format_add (type);

  /* Set the name of the audio item */
  if (type)
    {
      if (play->formats[type - 1] && play->formats[type - 1]->get_name)
	tmp->name = play->formats[type - 1]->get_name (str);
    }

  /* If don't exist */
  if (!tmp->name)
    {
      int i;
      char *k;

      if (remote_file (str))
	{
	  if (!(tmp->name = strdup (str)))
	    fatal (_("Error: memory."));
	}
      else
	{
	  for (i = strlen (str); i > 0; i--)
	    if (str[i] == '/')
	      {
		i++;
		break;
	      }

	  if (!(k = strdup (str + i)))
	    fatal (_("Error: memory."));

	  for (i = strlen (k); i > 0; i--)
	    if (k[i] == '.')
	      {
		k[i] = 0;
		break;
	      }

	  if (!(tmp->name = strdup (k)))
	    fatal (_("Error: memory."));

	  free (k);
	}
    }

#ifdef ENABLE_CDAUDIO
  /* It is REMOTE, CDAUDIO, LOCAL file? */
  if (type != TYPE_CDAUDIO)
    {
#endif
      if (!remote_file (str) && lstat (str, &st))
	{
	  msg_error (_("Error Opening file %s"), str);
	  free (tmp->filename);
	  free (tmp->name);
	  free (tmp);
	  return 1;
	}
#ifdef ENABLE_CDAUDIO
    }
#endif

#ifdef ENABLE_CDAUDIO
  /* If CDAUDIO */
  if (type == TYPE_CDAUDIO)
    {
      char *n;
      int a = 0, b = 0;

      free (tmp->filename);
      free (tmp->name);
      free (tmp);

      n = str + strlen ("cdrom://");

      if (!strcmp (n, "*"))
	return list_add_cdrom (0, before);

      else
	{

	  while (*n)
	    {
	      if (*n >= '0' && *n <= '9')
		{
		  if (a)
		    a = (a * 10) + *n - '0';
		  else
		    a = *n - '0';

		}
	      else if (*n == ',')
		{
		  if (!a)
		    {
		      msg_error (_("Syntax error: %s"), str);
		      return 1;
		    }

		  if (b)
		    for (; b <= a; b++)
		      {
			if (list_add_cdrom (b, before))
			  return 1;
		      }

		  else if (list_add_cdrom (a, before))
		    return 1;

		  a = b = 0;

		}
	      else if (*n == '-')
		{
		  if (!a)
		    {
		      msg_error (_("Syntax error: %s"), str);
		      return 1;
		    }

		  b = a;
		  a = 0;
		}

	      n++;
	    }

	  if (a)
	    {
	      if (b)
		{
		  for (; b <= a; b++)
		    if (list_add_cdrom (b, before))
		      return 1;
		}
	      else if (list_add_cdrom (a, before))
		return 1;

	    }
	}
    }
  else
#endif

    /* If REMOTE of LOCALE */
  if (remote_file (str) || S_ISREG (st.st_mode))
    {

      if (!type)
	{
	  free (tmp->filename);
	  free (tmp->name);
	  free (tmp);
	  msg (_("No support for this type of file: %s"), str);
	  return 1;
	}

      pthread_mutex_lock (&play->m_list);

      if (before)
	{
	  tmp->prev = before;
	  tmp->next = before->next;

	  if (before->next)
	    before->next->prev = tmp;

	  before->next = tmp;

#ifdef ENABLE_GTK
	  if (play->graphic && play->go_graphic)
	    playlist_append_before (tmp, before);
#endif
	}
      else if (play->last)
	{
	  tmp->prev = play->last;

	  play->last->next = tmp;
	  play->last = tmp;
#ifdef ENABLE_GTK
	  if (play->graphic && play->go_graphic)
	    playlist_append (tmp);
#endif
	}
      else
	{
	  play->first = tmp;
	  play->last = tmp;
#ifdef ENABLE_GTK
	  if (play->graphic && play->go_graphic)
	    playlist_append (tmp);
#endif
	}

      play->number++;

      pthread_mutex_unlock (&play->m_list);
    }

  /* If it is a DIRECTORY...
   * This part is recursive. I read all elements of this directory
   * and run this function for insert them in the list */
  else if (S_ISDIR (st.st_mode))
    {
      DIR *d;
      struct dirent *dir;

      free (tmp->filename);
      free (tmp->name);
      free (tmp);

      if (!(d = opendir (str)))
	{
	  msg_error (_("Error opening directory %s"), str);
	  return 1;
	}

      while ((dir = readdir (d)))
	{
	  if (dir->d_name[0] == '.')
	    continue;

	  if (!
	      (tmp_dir =
	       (char *) malloc (sizeof (char) *
				(strlen (dir->d_name) + 2 + strlen (str)))))
	    fatal (_("Error: memory."));

	  sprintf (tmp_dir, "%s%s%s", str,
		   str[strlen (str) - 1] != '/' ? "/" : "", dir->d_name);

	  if (lstat (tmp_dir, &st))
	    {
	      msg_error (_("Error opening file %s"), tmp_dir);
	      free (tmp_dir);
	      return 1;
	    }

	  if (S_ISLNK (st.st_mode))
	    continue;

	  else if (S_ISDIR (st.st_mode) || S_ISREG (st.st_mode))
	    {
	      list_add (tmp_dir, before);
	    }
	  else
	    {
	      msg_error (_("Error opening file %s"), tmp_dir);
	      free (tmp_dir);
	      return 1;
	    }

	  free (tmp_dir);
	}

      closedir (d);
    }

  /* else ? */
  else
    {
      msg_error (_("Error opening file %s"), str);
      return 1;
    }

  return 0;
}

#ifdef ENABLE_CDAUDIO
/* This functions adds a CDROM track */
int
list_add_cdrom (int n, list * before)
{

  cdrom_drive *d;

  d = cdda_identify (play->cd, CDDA_MESSAGE_FORGETIT, NULL);

  if (!d)
    {
      msg_error (_("Cdaudio no present.\n"));
      return 1;
    }

  cdda_verbose_set (d, CDDA_MESSAGE_FORGETIT, CDDA_MESSAGE_FORGETIT);

  if (cdda_open (d))
    {
      msg_error (_("Cdaudio no present.\n"));
      return 1;
    }

  if (n)
    {
      if (d->tracks < n || n < 0 || !cdda_track_audiop (d, n))
	{
	  msg_error (_("Error number of cdrom track."));
	  cdda_close (d);
	  return 1;
	}

      if (list_add_cdrom_n (n, before))
	{
	  cdda_close (d);
	  return 1;
	}

    }
  else
    {

      msg (_("Cdaudio with %d track.\n"), d->tracks);

      for (n = 1; n <= d->tracks; n++)
	if (cdda_track_audiop (d, n))
	  {
	    if (list_add_cdrom_n (n, before))
	      {
		cdda_close (d);
		return 1;
	      }
	  }

	else
	  msg (_("Track %d is no audio.\n"), n);

      cdda_close (d);
    }

  return 0;
}

/* This function inserts all track */
int
list_add_cdrom_n (int n, list * before)
{

  char a[SIZE_BUFFER];
  list *tmp;

  if (!(tmp = (list *) malloc (sizeof (list))))
    fatal (_("Error: memory."));

  tmp->next = NULL;
  tmp->prev = NULL;
  tmp->n = 0;
#ifdef ENABLE_GTK
  tmp->queue = 0;
#endif

  sprintf (a, "cdrom://%d", n);

  if (!(tmp->filename = strdup (a)))
    fatal (_("Error: memory."));

  if (!(tmp->name = strdup (a)))
    fatal (_("Error: memory."));

  pthread_mutex_lock (&play->m_list);

  if (before)
    {
      tmp->prev = before;
      tmp->next = before->next;

      if (before->next)
	before->next->prev = tmp;

      before->next = tmp;
#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	playlist_append_before (tmp, before);
#endif
    }
  else if (play->last)
    {
      tmp->prev = play->last;

      play->last->next = tmp;
      play->last = tmp;

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	playlist_append (tmp);
#endif
    }
  else
    {
      play->first = tmp;
      play->last = tmp;

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	playlist_append (tmp);
#endif
    }

  play->number++;

  pthread_mutex_unlock (&play->m_list);

  return 0;
}
#endif

#ifdef ENABLE_MIC
/* Insert the mic */
list *
list_mic (void)
{
  static list *tmp = NULL;

  if (!tmp)
    {
      if (!(tmp = (list *) malloc (sizeof (list))))
	fatal (_("Error: memory."));

      tmp->next = tmp->prev = NULL;
      tmp->n = 0;
      tmp->filename = strdup ("mic://*");
      tmp->name = strdup (tmp->filename);

#ifdef ENABLE_GTK
      tmp->queue = 0;
#endif

      if (!tmp->filename || !tmp->name)
	fatal (_("Error: memory."));

      format_add (TYPE_MIC);
    }

  return tmp;
}
#endif

#ifdef ENABLE_DAEMON
/* Insert the daemon_client */
list *
list_daemon_input (void)
{
  static list *tmp = NULL;

  if (!tmp)
    {
      if (!(tmp = (list *) malloc (sizeof (list))))
	fatal (_("Error: memory."));

#ifdef ENABLE_GTK
      tmp->queue = 0;
#endif
      tmp->next = tmp->prev = NULL;
      tmp->n = 0;
      if (!
	  (tmp->filename =
	   (char *) malloc (sizeof (char) *
			    (strlen (play->daemon_input_opt) + 7))))
	fatal (_("Error: memory."));

      sprintf (tmp->filename, "sds://%s", play->daemon_input_opt);

      if (!(tmp->name = strdup (tmp->filename)))
	fatal (_("Error: memory."));

      format_add (TYPE_DAEMON_INPUT);
    }

  return tmp;
}
#endif

/* Remove and element */
void
list_remove (int id)
{
  list *tmp;

  if (!play->first)
    return;
  if (id < 0)
    return;

  pthread_mutex_lock (&play->m_list);

#ifdef ENABLE_GTK
  if (play->graphic && play->go_graphic)
    playlist_remove (id);
#endif

  tmp = play->first;

  while (tmp)
    {

      if (!id)
	{
#ifdef ENABLE_GTK
	  if (play->graphic && play->go_graphic)
	    queue_remove (tmp);
#endif

	  /* This is a big problem! If this file is the file that
	   * now i play ? */
	  if (play->this == tmp)
	    {
	      events.next = -1;
	      events.skip = 1;

	      pthread_cond_signal (&play->p_pop);

	      play->this = NULL;
	    }

	  if (!tmp->prev)
	    {
	      if (tmp->next)
		tmp->next->prev = NULL;

	      play->first = tmp->next;

	      if (play->last == tmp)
		play->last = NULL;

	    }
	  else
	    {
	      if (tmp == play->last)
		{
		  play->last = tmp->prev;
		  tmp->prev->next = NULL;
		}
	      else
		{
		  tmp->prev->next = tmp->next;

		  if (tmp->next)
		    tmp->next->prev = tmp->prev;
		}
	    }

	  play->number--;

	  free (tmp->filename);
	  free (tmp->name);
	  free (tmp);

	  break;
	}

      id--;
      tmp = tmp->next;
    }

  pthread_mutex_unlock (&play->m_list);

}

/* Next file! */
list *
list_next (void)
{
  list *tmp;
  pthread_mutex_lock (&play->m_list);

#ifdef ENABLE_MIC
  if (play->microphone)
    return list_mic ();
#endif

#ifdef ENABLE_DAEMON
  if (play->daemon_input)
    return list_daemon_input ();
#endif

  if (play->graphic && play->go_graphic)
    {
      if (play->playlist_selected != NULL)
	{
	  tmp = play->playlist_selected;
	  play->playlist_selected = NULL;
	  return tmp;
	}

      if ((tmp = queue_next ()))
	return tmp;
    }

  if (play->random)
    return list_next_random ();

  if (!play->this)
    return play->first;

  switch (events.next)
    {
    case 1:
      return play->this;

    case -1:
      return (play->this
	      && play->this->prev) ? play->this->prev : play->first;

    default:
      if (play->this && !play->this->next && play->repeat)
	return play->first;

      return play->this ? play->this->next : NULL;
    }
}

/* Next random */
list *
list_next_random (void)
{
  int l;
  list *tmp;
  int k = 0;

  tmp = play->first;
  if (!tmp)
    return NULL;

  l = getrandom (0, play->number - 1);

  if (events.next == -1 && play->this && play->this->n > 1)
    {

      while (tmp)
	{

	  if (tmp->n == play->this->n - 1)
	    {
	      play->this->n = 0;
	      return tmp;
	    }
	  tmp = tmp->next;
	}
    }

  while (tmp)
    {
      if (l)
	l--;

      else
	{
	  if (!tmp->n)
	    {

	      if (play->this)
		tmp->n = play->this->n + 1;
	      else
		tmp->n = 1;

	      return tmp;
	    }

	  k++;
	}

      tmp = tmp->next;
    }

  tmp = play->first;

  while (tmp)
    {

      if (!tmp->n)
	{

	  if (play->this)
	    tmp->n = play->this->n + 1;
	  else
	    tmp->n = 1;

	  return tmp;
	}
      tmp = tmp->next;
    }

  if (play->repeat)
    {

      tmp = play->first;
      play->this = NULL;
      while (tmp)
	{
	  tmp->n = 0;
	  tmp = tmp->next;
	}

      return list_next_random ();

    }
  else
    return NULL;
}

void
list_move (list * what, list * where)
{
  if (what == where || !what)
    return;

  pthread_mutex_lock (&play->m_list);

#ifdef ENABLE_GTK
  if (play->graphic && play->go_graphic)
    playlist_move (what, where);
#endif

  if (play->first == what)
    play->first = what->next;
  if (play->last == what)
    play->last = what->prev;

  if (what->prev)
    what->prev->next = what->next;

  if (what->next)
    what->next->prev = what->prev;

  if (where)
    {
      if (where->next)
	where->next->prev = what;
      what->next = where->next;
      what->prev = where;
      where->next = what;

      if (play->last == where)
	play->last = what;
    }

  else
    {
      what->next = play->first;
      what->prev = NULL;

      play->first = what;
    }

  pthread_mutex_unlock (&play->m_list);
}

list *
list_get (int id)
{
  list *tmp;

  tmp = play->first;

  while (tmp)
    {

      if (!id)
	return tmp;

      id--;

      tmp = tmp->next;
    }

  return NULL;
}

/* EOF */
