-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure Wf_Generic_Declaration (Node          : in STree.SyntaxNode;
                                  Current_Scope : in Dictionary.Scopes) is
   Current_Node : STree.SyntaxNode;

   procedure Process_Generic_Formal_Part
     (Node                 : in     STree.SyntaxNode;
      Generic_Unit         : in     Dictionary.Symbol;
      Error_In_Declaration : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Error_In_Declaration,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Unit,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Unit,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_formal_part;
   --# post STree.Table = STree.Table~;
   is
      Object_Or_Type_Node : STree.SyntaxNode;

      procedure Process_Formal_Object
        (Node                 : in     STree.SyntaxNode;
         Generic_Unit         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Error_In_Declaration       from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.formal_object_declaration;
      --# post STree.Table = STree.Table~;
      is
         Ident_List_Node : STree.SyntaxNode;
         Mode_Node       : STree.SyntaxNode;
         Type_Node       : STree.SyntaxNode;
         Formal_Type     : Dictionary.Symbol;

         procedure Process_Identifiers
           (Node                 : in     STree.SyntaxNode;
            Type_Sym             : in     Dictionary.Symbol;
            Generic_Unit         : in     Dictionary.Symbol;
            Error_In_Declaration : in out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     LexTokenManager.State;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict,
         --#         Error_In_Declaration       from *,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         Generic_Unit,
         --#                                         LexTokenManager.State,
         --#                                         Node,
         --#                                         STree.Table,
         --#                                         Type_Sym &
         --#         ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Generic_Unit,
         --#                                         LexTokenManager.State,
         --#                                         Node,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         Type_Sym;
         --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.identifier_list;
         is
            It        : STree.Iterator;
            Next_Node : STree.SyntaxNode;
            Ident_Str : LexTokenManager.Lex_String;
            Sym       : Dictionary.Symbol;
         begin
            It := Find_First_Node (Node_Kind    => SP_Symbols.identifier,
                                   From_Root    => Node,
                                   In_Direction => STree.Down);
            while not STree.IsNull (It) loop
               Next_Node := Get_Node (It => It);
               --# assert Syntax_Node_Type (Next_Node, STree.Table) = SP_Symbols.identifier and
               --#   Next_Node = Get_Node (It);
               Ident_Str := Node_Lex_String (Node => Next_Node);
               Sym       :=
                 Dictionary.LookupItem
                 (Name              => Ident_Str,
                  Scope             => Dictionary.GlobalScope,
                  Context           => Dictionary.ProofContext,
                  Full_Package_Name => False);
               if Sym = Dictionary.NullSymbol then
                  Dictionary.AddGenericFormalParameter
                    (Name                => Ident_Str,
                     Comp_Unit           => ContextManager.Ops.Current_Unit,
                     Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => Next_Node),
                                                                 End_Position   => Node_Position (Node => Next_Node)),
                     SubprogramOrPackage => Generic_Unit,
                     TypeMark            => Type_Sym,
                     Kind                => Dictionary.GenericObjectParameter);
               else
                  -- already exists
                  Error_In_Declaration := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 10,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Next_Node),
                     Id_Str    => Ident_Str);
               end if;
               It := STree.NextNode (It);
            end loop;
         end Process_Identifiers;

      begin -- Process_Formal_Object
         Ident_List_Node := Child_Node (Current_Node => Node);

         -- ASSUME Ident_List_Node = identifier_list
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_List_Node) = SP_Symbols.identifier_list,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_List_Node = identifier_list in Process_Formal_Object");

         Mode_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Ident_List_Node));

         -- ASSUME Mode_Node = in_mode OR inout_mode OR out_mode OR NULL
         SystemErrors.RT_Assert
           (C       => Mode_Node = STree.NullNode
              or else Syntax_Node_Type (Node => Mode_Node) = SP_Symbols.in_mode
              or else Syntax_Node_Type (Node => Mode_Node) = SP_Symbols.inout_mode
              or else Syntax_Node_Type (Node => Mode_Node) = SP_Symbols.out_mode,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Mode_Node = in_mode OR inout_mode OR out_mode OR NULL in Process_Formal_Object");

         Type_Node := Next_Sibling (Current_Node => Next_Sibling (Current_Node => Ident_List_Node));
         -- ASSUME Type_Node = type_mark
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Type_Node) = SP_Symbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Type_Node = type_mark in Process_Formal_Object");

         if Syntax_Node_Type (Node => Mode_Node) = SP_Symbols.inout_mode
           or else Syntax_Node_Type (Node => Mode_Node) = SP_Symbols.out_mode then
            Error_In_Declaration := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 639,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_List_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

         Wf_Type_Mark
           (Node          => Type_Node,
            Current_Scope => Dictionary.LocalScope (Generic_Unit),
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Formal_Type);
         if Formal_Type = Dictionary.GetUnknownTypeMark then
            Error_In_Declaration := True;
         else -- valid type mark so
            Process_Identifiers
              (Node                 => Ident_List_Node,
               Type_Sym             => Formal_Type,
               Generic_Unit         => Generic_Unit,
               Error_In_Declaration => Error_In_Declaration);
         end if;
      end Process_Formal_Object;

      -------------

      procedure Process_Formal_Type
        (Node                 : in     STree.SyntaxNode;
         Generic_Unit         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Error_In_Declaration,
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         SLI.State                  from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Generic_Unit,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.formal_type_declaration;
      --# post STree.Table = STree.Table~;
      is
         Ident_Node             : STree.SyntaxNode;
         Type_Node              : STree.SyntaxNode;
         Ident_Str              : LexTokenManager.Lex_String;
         Declaration            : Dictionary.Location;
         Type_Mark              : Dictionary.Symbol;
         Error_In_Generic_Array : Boolean;
      begin
         Ident_Node := Child_Node (Current_Node => Node);
         -- ASSUME Ident_Node = indentifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = indentifier in Process_Formal_Type");
         Ident_Str := Node_Lex_String (Node => Ident_Node);
         -- check whether name already in use
         if Dictionary.IsDefined
           (Name              => Ident_Str,
            Scope             => Dictionary.GlobalScope,
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
            Error_In_Declaration := True;
         else -- OK to add
            Type_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Ident_Node));
            -- ASSUME Type_Node = formal_private_type_definition OR formal_discrete_type_definition OR
            --                    formal_signed_integer_type_definition OR formal_modular_type_definition OR
            --                    formal_floating_point_definition OR formal_ordinary_fixed_point_definition OR
            --                    formal_array_type_definition
            Declaration :=
              Dictionary.Location'
              (Start_Position => Node_Position (Node => Ident_Node),
               End_Position   => Node_Position (Node => Ident_Node));
            case Syntax_Node_Type (Node => Type_Node) is
               when SP_Symbols.formal_private_type_definition =>
                  -- ASSUME Type_Node = formal_private_type_definition
                  Dictionary.AddGenericPrivateType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_discrete_type_definition =>
                  -- ASSUME Type_Node = formal_discrete_type_definition
                  Dictionary.AddGenericOrderedDiscreteType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_signed_integer_type_definition =>
                  -- ASSUME Type_Node = formal_signed_integer_type_definition
                  Dictionary.AddGenericIntegerType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_modular_type_definition =>
                  -- ASSUME Type_Node = formal_modular_type_definition
                  Dictionary.AddGenericModularType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_floating_point_definition =>
                  -- ASSUME Type_Node = formal_floating_point_definition
                  Dictionary.AddGenericFloatingPointType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_ordinary_fixed_point_definition =>
                  -- ASSUME Type_Node = formal_ordinary_fixed_point_definition
                  Dictionary.AddGenericFixedPointType
                    (Name        => Ident_Str,
                     Comp_Unit   => ContextManager.Ops.Current_Unit,
                     Declaration => Declaration,
                     Scope       => Dictionary.LocalScope (Generic_Unit),
                     Type_Sym    => Type_Mark);
                  if ErrorHandler.Generate_SLI then
                     SLI.Generate_Xref_Symbol
                       (Comp_Unit      => ContextManager.Ops.Current_Unit,
                        Parse_Tree     => Ident_Node,
                        Symbol         => Type_Mark,
                        Is_Declaration => True);
                  end if;
               when SP_Symbols.formal_array_type_definition =>
                  -- ASSUME Type_Node = formal_array_type_definition
                  Type_Node := Child_Node (Current_Node => Type_Node);
                  -- ASSUME Type_Node = array_type_definition
                  SystemErrors.RT_Assert
                    (C       => Syntax_Node_Type (Node => Type_Node) = SP_Symbols.array_type_definition,
                     Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Type_Node = array_type_definition in Process_Formal_Type");
                  Wf_Array_Type_Definition
                    (Node         => Type_Node,
                     Scope        => Dictionary.LocalScope (Generic_Unit),
                     Ident_Node   => Ident_Node,
                     Dec_Loc      => Node_Position (Node => Node),
                     Is_Generic   => True,
                     Errors_Found => Error_In_Generic_Array,
                     The_Array    => Type_Mark);
                  Error_In_Declaration := Error_In_Declaration or else Error_In_Generic_Array;
               when others =>
                  Type_Mark := Dictionary.GetUnknownTypeMark; -- just to avoid data flow error
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Type_Node = formal_private_type_definition OR formal_discrete_type_definition OR " &
                       "formal_signed_integer_type_definition OR formal_modular_type_definition OR " &
                       "formal_floating_point_definition OR formal_ordinary_fixed_point_definition OR " &
                       "formal_array_type_definition in Process_Formal_Type");
            end case;
            Dictionary.AddGenericFormalParameter
              (Name                => LexTokenManager.Null_String,
               Comp_Unit           => ContextManager.Ops.Current_Unit,
               Declaration         => Declaration,
               SubprogramOrPackage => Generic_Unit,
               TypeMark            => Type_Mark,
               Kind                => Dictionary.GenericTypeParameter);
         end if;
      end Process_Formal_Type;

   begin -- Process_Generic_Formal_Part
      Object_Or_Type_Node := Next_Sibling (Current_Node => Last_Child_Of (Start_Node => Node));
      while Syntax_Node_Type (Node => Object_Or_Type_Node) = SP_Symbols.formal_object_declaration
        or else Syntax_Node_Type (Node => Object_Or_Type_Node) = SP_Symbols.formal_type_declaration loop
         --# assert STree.Table = STree.Table~ and
         --#   (Syntax_Node_Type (Object_Or_Type_Node, STree.Table) = SP_Symbols.formal_object_declaration or
         --#      Syntax_Node_Type (Object_Or_Type_Node, STree.Table) = SP_Symbols.formal_type_declaration);
         case Syntax_Node_Type (Node => Object_Or_Type_Node) is
            when SP_Symbols.formal_object_declaration =>
               -- ASSUME Object_Or_Type_Node = formal_object_declaration
               Process_Formal_Object
                 (Node                 => Object_Or_Type_Node,
                  Generic_Unit         => Generic_Unit,
                  Error_In_Declaration => Error_In_Declaration);
            when SP_Symbols.formal_type_declaration =>
               -- ASSUME Object_Or_Type_Node = formal_type_declaration
               Process_Formal_Type
                 (Node                 => Object_Or_Type_Node,
                  Generic_Unit         => Generic_Unit,
                  Error_In_Declaration => Error_In_Declaration);
            when others =>
               null;
         end case;
         Object_Or_Type_Node := Next_Sibling (Current_Node => Parent_Node (Current_Node => Object_Or_Type_Node));
      end loop;
   end Process_Generic_Formal_Part;

   -----------------------------------------------------------------------------------

   procedure Process_Generic_Subprogram_Declaration
     (Node           : in     STree.SyntaxNode;
      Current_Scope  : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_subprogram_declaration;
   --# post STree.Table = STree.Table~;
   is
      type Subprogram_Kinds is (A_Function, A_Procedure, Unknown);

      Inherit_Node                : STree.SyntaxNode;
      Formal_Part_Node            : STree.SyntaxNode;
      Subprogram_Declaration_Node : STree.SyntaxNode;
      Anno_Node                   : STree.SyntaxNode;
      -- GAA Package_Sym, Subprog_Sym    : Dictionary.Symbol;
      Package_Sym          : Dictionary.Symbol;
      Subprogram_Kind      : Subprogram_Kinds;
      Error_In_Declaration : Boolean := False;

      procedure Check_And_Add_Package
        (Node            : in     STree.SyntaxNode;
         Package_Sym     :    out Dictionary.Symbol;
         Subprogram_Kind :    out Subprogram_Kinds)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict,
      --#         Package_Sym                from ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         Subprogram_Kind            from Node,
      --#                                         STree.Table;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_declaration;
      is
         Ident_Node : STree.SyntaxNode;
         Ident_Str  : LexTokenManager.Lex_String;
      begin
         Ident_Node := Child_Node (Current_Node => Node);
         -- ASSUME Ident_Node = function_specification OR procedure_specification OR proof_function_declaration
         if Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.function_specification then
            -- ASSUME Ident_Node = function_specification
            Subprogram_Kind := A_Function;
         elsif Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.procedure_specification then
            -- ASSUME Ident_Node = procedure_specification
            Subprogram_Kind := A_Procedure;
         elsif Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.proof_function_declaration then
            -- ASSUME Ident_Node = proof_function_declaration
            Subprogram_Kind := A_Function;
            Ident_Node      := Child_Node (Current_Node => Ident_Node);
            -- ASSUME Ident_Node = function_specification
         else
            Subprogram_Kind := Unknown;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = function_specification OR procedure_specification OR " &
                 "proof_function_declaration in Check_And_Add_Package");
         end if;
         -- ASSUME Ident_Node = function_specification OR procedure_specification
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.function_specification
              or else Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.procedure_specification,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect ident_Node = function_specification OR procedure_specification in Check_And_Add_Package");
         -- find identifier
         Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Ident_Node));
         -- ASSUME Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect ident_Node = identifier in Check_And_Add_Package");
         Ident_Str := Node_Lex_String (Node => Ident_Node);
         if Dictionary.IsDefined
           (Name              => Ident_Str,
            Scope             => Dictionary.GlobalScope,
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False) then
            Package_Sym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Ident_Str);
         else
            Dictionary.AddSubprogram
              (Name          => Ident_Str,
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Dictionary.GlobalScope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => Package_Sym);
            Dictionary.SetSubprogramIsGeneric (Package_Sym);
         end if;
      end Check_And_Add_Package;

      ---------------

      procedure Process_Formal_Part (Node           : in STree.SyntaxNode;
                                     The_Subprogram : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         STree.Table                from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Subprogram &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Subprogram;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_declaration;
      --# post STree.Table = STree.Table~;
      is
         Formal_Part_Node : STree.SyntaxNode;
      begin
         Formal_Part_Node := Child_Node (Current_Node => Node);
         -- ASSUME Formal_Part_Node = function_specification OR procedure_specification OR proof_function_declaration
         if Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.function_specification
           or else Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.procedure_specification then
            -- ASSUME Formal_Part_Node = function_specification OR procedure_specification
            Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Formal_Part_Node));
            -- ASSUME Formal_Part_Node = formal_part OR type_mark OR NULL
         elsif Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.proof_function_declaration then
            -- ASSUME Formal_Part_Node = proof_function_declaration
            Formal_Part_Node :=
              Next_Sibling (Current_Node => Child_Node (Current_Node => Child_Node (Current_Node => Formal_Part_Node)));
            -- ASSUME Formal_Part_Node = formal_part OR type_mark
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = function_specification OR procedure_specification OR " &
                 "proof_function_declaration in Process_Formal_Part");
         end if;
         -- ASSUME Formal_Part_Node = formal_part OR type_mark OR NULL
         if Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.formal_part then
            -- ASSUME Formal_Part_Node = formal_part
            Wf_Formal_Part
              (Node             => Formal_Part_Node,
               Current_Scope    => Dictionary.LocalScope (The_Subprogram),
               Subprog_Sym      => The_Subprogram,
               First_Occurrence => True,
               Context          => Dictionary.ProgramContext);
         elsif Formal_Part_Node /= STree.NullNode
           and then Syntax_Node_Type (Node => Formal_Part_Node) /= SP_Symbols.type_mark then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = formal_part OR type_mark OR NULL in Process_Formal_Part");
         end if;
      end Process_Formal_Part;

      ---------------

      procedure Process_Return
        (Node                 : in     STree.SyntaxNode;
         The_Function         : in     Dictionary.Symbol;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict            from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         ErrorHandler.Error_Context from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         Error_In_Declaration,
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Function;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_declaration;
      --# post STree.Table = STree.Table~;
      is
         Return_Node : STree.SyntaxNode;
         Return_Type : Dictionary.Symbol;
      begin
         Return_Node := Child_Node (Current_Node => Node);
         -- ASSUME Return_Node = function_specification OR proof_function_declaration
         if Syntax_Node_Type (Node => Return_Node) = SP_Symbols.function_specification then
            -- ASSUME Return_Node = function_specification
            Return_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Return_Node));
            -- ASSUME Return_Node = type_mark
         elsif Syntax_Node_Type (Node => Return_Node) = SP_Symbols.proof_function_declaration then
            -- ASSUME Return_Node = proof_function_declaration
            Return_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Child_Node (Current_Node => Return_Node)));
            -- ASSUME Return_Node = type_mark
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Return_Node = function_specification OR proof_function_declaration in Process_Return");
         end if;
         -- ASSUME Return_Node = type_mark
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Return_Node) = SP_Symbols.type_mark,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Return_Node = type_mark in Process_Return");
         Wf_Type_Mark
           (Node          => Return_Node,
            Current_Scope => Dictionary.LocalScope (The_Function),
            Context       => Dictionary.ProgramContext,
            Type_Sym      => Return_Type);

         -- report errors back to caller
         if Return_Type = Dictionary.GetUnknownTypeMark then
            Error_In_Declaration := True;
         end if;

         -- GenericTBD check for illegal return types here (e.g. unconstrained arrays)

         -- add return  type to Dictionary
         Dictionary.AddReturnType
           (TheFunction   => The_Function,
            TypeMark      => Return_Type,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            TypeReference => Dictionary.Location'(Start_Position => Node_Position (Node => Return_Node),
                                                  End_Position   => Node_Position (Node => Return_Node)));
      end Process_Return;

      ------------------------------

      procedure Process_Anno
        (Node                 : in     STree.SyntaxNode;
         The_Subprogram       : in     Dictionary.Symbol;
         Component_Data       : in out ComponentManager.ComponentData;
         Error_In_Declaration : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in out Aggregate_Stack.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives Aggregate_Stack.State,
      --#         Component_Data,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                    from *,
      --#                                         CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         TheHeap,
      --#                                         The_Subprogram &
      --#         ErrorHandler.Error_Context,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         TheHeap,
      --#                                         The_Subprogram &
      --#         Error_In_Declaration       from *,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         The_Subprogram;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_annotation or
      --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_annotation;
      --# post STree.Table = STree.Table~;
      is
         Global_Node     : STree.SyntaxNode;
         Dependency_Node : STree.SyntaxNode;
         Declare_Node    : STree.SyntaxNode;
         Constraint_Node : STree.SyntaxNode;
      begin
         Get_Subprogram_Anno_Key_Nodes
           (Node            => Node,
            Global_Node     => Global_Node,
            Dependency_Node => Dependency_Node,
            Declare_Node    => Declare_Node,
            Constraint_Node => Constraint_Node);

         -- ASSUME Global_Node = moded_global_definition OR NULL
         if Syntax_Node_Type (Node => Global_Node) = SP_Symbols.moded_global_definition then
            -- ASSUME Global_Node = moded_global_definition
            -- initial design disallows generic subprograms from having global anno.
            ErrorHandler.Semantic_Error
              (Err_Num   => 638,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Child_Node (Current_Node => Node)),
               Id_Str    => LexTokenManager.Null_String);
            Error_In_Declaration := True;
         elsif Global_Node = STree.NullNode then
            -- ASSUME Global_Node = NULL
            if Dictionary.IsProcedure (The_Subprogram) then
               -- ASSUME Node = procedure_annotation OR function_annotation
               if (Syntax_Node_Type (Node => Node) = SP_Symbols.procedure_annotation
                     or else Syntax_Node_Type (Node => Node) = SP_Symbols.function_annotation)
                 and then (Syntax_Node_Type (Node => Dependency_Node) = SP_Symbols.dependency_relation
                             or else Syntax_Node_Type (Node => Declare_Node) = SP_Symbols.declare_annotation) then
                  -- ASSUME Node = procedure_annotation OR function_annotation AND
                  --        Dependency_Node = dependency_relation OR Declare_Node = declare_annotation
                  Wf_Subprogram_Annotation
                    (Node          => Node,
                     Current_Scope => Dictionary.LocalScope (The_Subprogram),
                     Subprog_Sym   => The_Subprogram,
                     First_Seen    => True,
                     The_Heap      => TheHeap);
               elsif Dependency_Node = STree.NullNode then
                  -- ASSUME Dependency_Node = NULL
                  if CommandLineData.Content.Flow_Option = CommandLineData.Info_Flow then
                     -- derives is needed for info-flow, but is not required for data-flow or auto
                     Error_In_Declaration := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 154,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Node),
                        Id_Str    => Dictionary.GetSimpleName (The_Subprogram));
                  end if;
               end if;
               -- Synthesise an "all exports from all imports" dependency relation if necessary
               if Needs_Synthetic_Dependency (The_Subprogram) then
                  Create_Full_Subprog_Dependency
                    (Node_Pos    => Node_Position (Node => Node),
                     Subprog_Sym => The_Subprogram,
                     Abstraction => Dictionary.IsAbstract,
                     The_Heap    => TheHeap);
               end if;
            end if;
            Wf_Subprogram_Constraint
              (Node           => Constraint_Node,
               Scope          => Dictionary.LocalScope (The_Subprogram),
               First_Seen     => True,
               Component_Data => Component_Data,
               The_Heap       => TheHeap);
         end if;
      end Process_Anno;

   begin -- Process_Generic_Subprogram_Declaration
      Inherit_Node := Child_Node (Current_Node => Node);
      -- ASSUME Inherit_Node = inherit_clause OR generic_formal_part
      if Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.inherit_clause then
         -- ASSUME Inherit_Node = inherit_clause
         Formal_Part_Node := Next_Sibling (Current_Node => Inherit_Node);
      elsif Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.generic_formal_part then
         -- ASSUME Inherit_Node = generic_formal_part
         Formal_Part_Node := Inherit_Node;
         Inherit_Node     := STree.NullNode;
      else
         Formal_Part_Node := STree.NullNode;
         Inherit_Node     := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Inherit_Node = inherit_clause OR generic_formal_part in Process_Generic_Subprogram_Declaration");
      end if;
      -- ASSUME Formal_Part_Node = generic_formal_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Formal_Part_Node) = SP_Symbols.generic_formal_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Formal_Part_Node = generic_formal_part in Process_Generic_Subprogram_Declaration");

      Subprogram_Declaration_Node := Next_Sibling (Current_Node => Formal_Part_Node);
      -- ASSUME Subprogram_Declaration_Node = not_overriding_subprogram_declaration
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Subprogram_Declaration_Node) = SP_Symbols.not_overriding_subprogram_declaration,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Subprogram_Declaration_Node = not_overriding_subprogram_declaration " &
           "in Process_Generic_Subprogram_Declaration");

      -- First check the generic subprogram name for illegal redeclaration etc.  Add subprogram if OK.
      -- We need to do this before handling the generic formal part because we need somewhere in the
      -- Dictionary where we can attach the list of generic formal parameters.
      Check_And_Add_Package (Node            => Subprogram_Declaration_Node,
                             Package_Sym     => Package_Sym,
                             Subprogram_Kind => Subprogram_Kind);

      if Package_Sym /= Dictionary.NullSymbol then
         -- ASSUME Inherit_Node = inherit_clause OR NULL
         if Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.inherit_clause then
            -- ASSUME Inherit_Node = inherit_clause
            Dictionary.AddInheritsAnnotation
              (Package_Sym,
               Dictionary.Location'(Start_Position => Node_Position (Node => Inherit_Node),
                                    End_Position   => Node_Position (Node => Inherit_Node)));
            Wf_Inherit_Clause (Node     => Inherit_Node,
                               Comp_Sym => Package_Sym,
                               Scope    => Current_Scope);
         end if;

         Process_Generic_Formal_Part
           (Node                 => Formal_Part_Node,
            Generic_Unit         => Package_Sym,
            Error_In_Declaration => Error_In_Declaration);

         -- GAA Wf_Subprogram_Declaration (Node           => Subprogram_Declaration_Node,
         -- GAA                            Current_Scope  => Current_Scope,
         -- GAA                            Component_Data => Component_Data,
         -- GAA                            Subprog_Sym    => Subprog_Sym);

         Process_Formal_Part (Node           => Subprogram_Declaration_Node,
                              The_Subprogram => Package_Sym);
         -- note that the above does not return an erro flag; however, it does set
         -- the subprogram signature as not well formed if there is an error in the
         -- formal part

         Anno_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Subprogram_Declaration_Node));
         -- ASSUME Anno_Node = procedure_annotation OR function_annotation
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Anno_Node) = SP_Symbols.procedure_annotation
              or else Syntax_Node_Type (Node => Anno_Node) = SP_Symbols.function_annotation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation in Process_Generic_Subprogram_Declaration");

         if Subprogram_Kind = A_Function then
            Process_Return
              (Node                 => Subprogram_Declaration_Node,
               The_Function         => Package_Sym,
               Error_In_Declaration => Error_In_Declaration);
            Process_Anno
              (Node                 => Anno_Node,
               The_Subprogram       => Package_Sym,
               Component_Data       => Component_Data,
               Error_In_Declaration => Error_In_Declaration);
         elsif Subprogram_Kind = A_Procedure then
            Process_Anno
              (Node                 => Anno_Node,
               The_Subprogram       => Package_Sym,
               Component_Data       => Component_Data,
               Error_In_Declaration => Error_In_Declaration);
         end if;

         -- PNA to do:
         --    store pre/post nodes

         -- If there are any errors in the generic declaration we mark the subprogram signature as
         -- not well formed.  We can then raise a suitable error when an attempt is made to instantiate
         -- a badly formed generic unit.  Similar to the way we refuse to flow analyse a call to a malformed
         -- subprogram.
         if Error_In_Declaration then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Package_Sym);
         end if;
      end if;
   end Process_Generic_Subprogram_Declaration;

   -----------------------------------------------------------------------------------

   procedure Process_Generic_Package_Declaration (Node          : in STree.SyntaxNode;
                                                  Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_package_declaration or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.private_generic_package_declaration;
   --# post STree.Table = STree.Table~;
   is
      Private_Package_Declaration, Child_Package_Declaration, Error_In_Declaration : Boolean;
      Generic_Formal_Part_Node, Spec_Node, Ident_Node, Context_Node, Inherit_Node  : STree.SyntaxNode;
      Pack_Sym                                                                     : Dictionary.Symbol;
      Ident_Str                                                                    : LexTokenManager.Lex_String;

      ------------------------------

      procedure Find_Key_Nodes
        (Node                        : in     STree.SyntaxNode;
         Context_Node                :    out STree.SyntaxNode;
         Private_Package_Declaration :    out Boolean;
         Inherit_Node                :    out STree.SyntaxNode;
         Generic_Formal_Part_Node    :    out STree.SyntaxNode;
         Spec_Node                   :    out STree.SyntaxNode;
         Ident_Node                  :    out STree.SyntaxNode;
         Ident_Str                   :    out LexTokenManager.Lex_String;
         Child_Package_Declaration   :    out Boolean)
      --# global in STree.Table;
      --# derives Child_Package_Declaration,
      --#         Context_Node,
      --#         Generic_Formal_Part_Node,
      --#         Ident_Node,
      --#         Ident_Str,
      --#         Inherit_Node,
      --#         Private_Package_Declaration,
      --#         Spec_Node                   from Node,
      --#                                          STree.Table;
      --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_package_declaration or
      --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.private_generic_package_declaration;
      --# post (Syntax_Node_Type (Context_Node, STree.Table) = SP_Symbols.context_clause or Context_Node = STree.NullNode) and
      --#   (Syntax_Node_Type (Inherit_Node, STree.Table) = SP_Symbols.inherit_clause or Inherit_Node = STree.NullNode) and
      --#   Syntax_Node_Type (Generic_Formal_Part_Node, STree.Table) = SP_Symbols.generic_formal_part and
      --#   Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.package_specification and
      --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
      is
      begin
         if Syntax_Node_Type (Node => Node) = SP_Symbols.generic_package_declaration then
            -- ASSUME Node = generic_package_declaration
            Context_Node :=
              Child_Node
              (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node))));
         elsif Syntax_Node_Type (Node => Node) = SP_Symbols.private_generic_package_declaration then
            -- ASSUME Node = private_generic_package_declaration
            Context_Node := STree.NullNode;
         else
            Context_Node := STree.NullNode;
         end if;
         -- ASSUME Context_Node = context_clause OR library_unit OR NULL
         if Syntax_Node_Type (Node => Context_Node) = SP_Symbols.library_unit then
            -- ASSUME Context_Node = library_unit
            Context_Node := STree.NullNode;
         elsif Context_Node /= STree.NullNode and then Syntax_Node_Type (Node => Context_Node) /= SP_Symbols.context_clause then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Context_Node = context_clause OR library_unit in Find_Key_Nodes");
         end if;
         -- ASSUME Context_Node = context_clause OR NULL
         SystemErrors.RT_Assert
           (C       => Context_Node = STree.NullNode or else Syntax_Node_Type (Node => Context_Node) = SP_Symbols.context_clause,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Context_Node = context_clause OR NULL in Find_Key_Nodes");

         Private_Package_Declaration := Syntax_Node_Type (Node => Node) = SP_Symbols.private_generic_package_declaration;
         Inherit_Node                := Child_Node (Current_Node => Node);
         -- ASSUME Inherit_Node = inherit_clause OR generic_formal_part
         if Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.inherit_clause then
            -- ASSUME Inherit_Node = inherit_clause
            Generic_Formal_Part_Node := Next_Sibling (Current_Node => Inherit_Node);
         elsif Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.generic_formal_part then
            -- ASSUME Inherit_Node = generic_formal_part
            Generic_Formal_Part_Node := Inherit_Node;
            Inherit_Node             := STree.NullNode;
         else
            Generic_Formal_Part_Node := STree.NullNode;
            Inherit_Node             := STree.NullNode;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Inherit_Node = inherit_clause OR generic_formal_part in Find_Key_Nodes");
         end if;
         -- ASSUME Inherit_Node = inherit_clause OR NULL
         SystemErrors.RT_Assert
           (C       => Inherit_Node = STree.NullNode or else Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.inherit_clause,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Inherit_Node = inherit_clause OR NULL in Find_Key_Nodes");
         -- ASSUME Generic_Formal_Part_Node = generic_formal_part
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Generic_Formal_Part_Node) = SP_Symbols.generic_formal_part,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Generic_Formal_Part_Node = generic_formal_part in Find_Key_Nodes");

         Spec_Node := Next_Sibling (Current_Node => Generic_Formal_Part_Node);
         -- ASSUME Spec_Node = package_specification
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Spec_Node) = SP_Symbols.package_specification,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Spec_Node = package_specification in Find_Key_Nodes");

         Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Ident_Node = dotted_simple_name OR identifier
         if Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier then
            -- ASSUME Ident_Node = identifier
            Child_Package_Declaration := False;
         elsif Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.dotted_simple_name then
            -- ASSUME Ident_Node = dotted_simple_name
            Child_Package_Declaration := True;
            Ident_Node                := Last_Child_Of (Start_Node => Ident_Node);
         else
            Child_Package_Declaration := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Ident_Node = dotted_simple_name OR identifier in Find_Key_Nodes");
         end if;
         -- ASSUME Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Ident_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = identifier in Find_Key_Nodes");
         Ident_Str := Node_Lex_String (Node => Ident_Node);
      end Find_Key_Nodes;

      procedure Check_And_Add_Package
        (Ident_Node                  : in     STree.SyntaxNode;
         Current_Scope               : in     Dictionary.Scopes;
         Child_Package_Declaration   : in     Boolean;
         Private_Package_Declaration : in out Boolean;
         Ident_Str                   : in out LexTokenManager.Lex_String;
         Package_Sym                 :    out Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict,
      --#         Package_Sym                 from Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.Unit_Stack,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          STree.Table &
      --#         ErrorHandler.Error_Context  from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table &
      --#         Ident_Str                   from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          LexTokenManager.State,
      --#                                          STree.Table &
      --#         Private_Package_Declaration from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content &
      --#         SPARK_IO.File_Sys           from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.Unit_Stack,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.Error_Context,
      --#                                          Ident_Node,
      --#                                          Ident_Str,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration,
      --#                                          STree.Table &
      --#         STree.Table                 from *,
      --#                                          Child_Package_Declaration,
      --#                                          CommandLineData.Content,
      --#                                          Current_Scope,
      --#                                          Dictionary.Dict,
      --#                                          Ident_Node,
      --#                                          LexTokenManager.State,
      --#                                          Private_Package_Declaration;
      --# pre Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
      --# post STree.Table = STree.Table~;
      is
         Pack_Sym   : Dictionary.Symbol;
         Valid_Name : Boolean := True;
      begin
         if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 and then Child_Package_Declaration then
            Add_Child
              (Root_Id_Node => Ident_Node,
               Is_Private   => Private_Package_Declaration,
               Scope        => Current_Scope,
               Child_Sym    => Pack_Sym,
               Child_Str    => Ident_Str);
            -- if Pack_Sym is null then something went wrong when we added the child so we need to supress
            -- any further analysis of the package specification
            Valid_Name := Pack_Sym /= Dictionary.NullSymbol;
         else
            if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
               -- check that syntax conforms
               if Child_Package_Declaration or else Private_Package_Declaration then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 610,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => LexTokenManager.Null_String);
                  Private_Package_Declaration := False;
               end if;
            end if;
            Pack_Sym :=
              Dictionary.LookupItem
              (Name              => Ident_Str,
               Scope             => Current_Scope,
               Context           => Dictionary.ProofContext,
               Full_Package_Name => False);
            --# assert STree.Table = STree.Table~;
            if Pack_Sym /= Dictionary.NullSymbol then
               Valid_Name := False;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
            else
               --# assert STree.Table = STree.Table~;
               if Private_Package_Declaration then -- root level private package
                  Dictionary.AddPrivatePackage
                    (Name          => Ident_Str,
                     Comp_Unit     => ContextManager.Ops.Current_Unit,
                     Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                           End_Position   => Node_Position (Node => Ident_Node)),
                     Scope         => Current_Scope,
                     ThePackage    => Pack_Sym);
               else
                  Dictionary.AddPackage
                    (Name          => Ident_Str,
                     Comp_Unit     => ContextManager.Ops.Current_Unit,
                     Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                           End_Position   => Node_Position (Node => Ident_Node)),
                     Scope         => Current_Scope,
                     ThePackage    => Pack_Sym);
               end if;
            end if;
         end if;

         if Valid_Name then
            Package_Sym := Pack_Sym;
         else
            Package_Sym := Dictionary.NullSymbol;
         end if;
      end Check_And_Add_Package;

   begin -- Process_Generic_Package_Declaration

      --# accept Flow, 10, "Required to initialize for flow analysis";
      Error_In_Declaration := False;
      --# end accept;

      Find_Key_Nodes
        (Node                        => Node,
         Context_Node                => Context_Node,
         Private_Package_Declaration => Private_Package_Declaration,
         Inherit_Node                => Inherit_Node,
         Generic_Formal_Part_Node    => Generic_Formal_Part_Node,
         Spec_Node                   => Spec_Node,
         Ident_Node                  => Ident_Node,
         Ident_Str                   => Ident_Str,
         Child_Package_Declaration   => Child_Package_Declaration);
      if Dictionary.IsGlobalScope (Current_Scope) then
         -- First we need to check the generic package name for illegal redeclaration etc.
         --# accept Flow, 10, Private_Package_Declaration, "Code under development";
         Check_And_Add_Package
           (Ident_Node                  => Ident_Node,
            Current_Scope               => Current_Scope,
            Child_Package_Declaration   => Child_Package_Declaration,
            Private_Package_Declaration => Private_Package_Declaration,
            Ident_Str                   => Ident_Str,
            Package_Sym                 => Pack_Sym);
         --# end accept;

         if Pack_Sym /= Dictionary.NullSymbol then
            Dictionary.SetPackageIsGeneric (Pack_Sym);

            -- ASSUME Inherit_Node = inherit_clause OR NULL
            if Syntax_Node_Type (Node => Inherit_Node) = SP_Symbols.inherit_clause then
               -- ASSUME Inherit_Node = inherit_clause
               Dictionary.AddInheritsAnnotation
                 (Pack_Sym,
                  Dictionary.Location'(Start_Position => Node_Position (Node => Inherit_Node),
                                       End_Position   => Node_Position (Node => Inherit_Node)));
               Wf_Inherit_Clause (Node     => Inherit_Node,
                                  Comp_Sym => Pack_Sym,
                                  Scope    => Current_Scope);
            end if;

            --# accept Flow, 10, Error_In_Declaration, "Code under development";
            Process_Generic_Formal_Part
              (Node                 => Generic_Formal_Part_Node,
               Generic_Unit         => Pack_Sym,
               Error_In_Declaration => Error_In_Declaration);
            --# end accept;

            --# assert STree.Table = STree.Table~ and
            --#   (Syntax_Node_Type (Context_Node, STree.Table) = SP_Symbols.context_clause or Context_Node = STree.NullNode) and
            --#   Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.package_specification;

            -- ASSUME Context_Node = context_clause OR NULL
            if Syntax_Node_Type (Node => Context_Node) = SP_Symbols.context_clause then
               -- ASSUME Context_Node = context_clause
               Wf_Context_Clause (Context_Node, Pack_Sym, Dictionary.VisibleScope (Pack_Sym));
            end if;

            --# assert STree.Table = STree.Table~ and
            --#   Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.package_specification;

            Wf_Package_Specification
              (Node          => Spec_Node,
               Ident_Str     => Ident_Str,
               Pack_Sym      => Pack_Sym,
               Current_Scope => Current_Scope);
         end if;
      end if;
   end Process_Generic_Package_Declaration;

begin -- Wf_Generic_Declaration
   case CommandLineData.Content.Language_Profile is
      when CommandLineData.SPARK83 =>
         -- not allowed in SPARK 83
         ErrorHandler.Semantic_Error
           (Err_Num   => 637,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
         if Syntax_Node_Type (Node => Node) = SP_Symbols.private_generic_package_declaration then
            -- ASSUME Node = private_generic_package_declaration
            Current_Node := Node;
         elsif Syntax_Node_Type (Node => Node) = SP_Symbols.generic_declaration then
            -- ASSUME Node = generic_declaration
            Current_Node := Child_Node (Current_Node => Node);
         else
            Current_Node := Node;
         end if;
         -- ASSUME Current_Node = generic_subprogram_declaration OR generic_package_declaration OR
         --                       private_generic_package_declaration
         if Syntax_Node_Type (Node => Current_Node) = SP_Symbols.generic_subprogram_declaration then
            -- ASSUME Current_Node = generic_subprogram_declaration
            Process_Generic_Subprogram_Declaration
              (Node           => Current_Node,
               Current_Scope  => Current_Scope,
               Component_Data => GlobalComponentData);
         elsif Syntax_Node_Type (Node => Current_Node) = SP_Symbols.generic_package_declaration
           or else Syntax_Node_Type (Node => Current_Node) = SP_Symbols.private_generic_package_declaration then
            -- ASSUME Current_Node = generic_package_declaration OR private_generic_package_declaration
            Process_Generic_Package_Declaration (Node          => Current_Node,
                                                 Current_Scope => Current_Scope);
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Current_Node = generic_subprogram_declaration OR generic_package_declaration OR " &
                 "private_generic_package_declaration in Wf_Generic_Declaration");
         end if;
   end case;
end Wf_Generic_Declaration;
