#!/usr/bin/env python

__doc__ = """
SSAKE

Short Sequence Assembly by Kmer search and 3' read Extension (SSAKE)

SYNOPSIS
   Progressive clustering of millions of short DNA sequences by Kmer search and 3' read extension
"""
__author__ = "Rene L. Warren"
__version__ = '2.0'


#LICENSE
#   SSAKE Copyright (c) 2006-2007 Canada's Michael Smith Genome Science Centre.  All rights reserved.
#   Using a complete re-write of error-handling by consensus derivation (VCAKE) with its Copyright (c) 2007 University of North Carolina at Chapel Hill. All rights Reserved.

#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.

#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.

import sys, os, re, string
from datetime import datetime
from optparse import OptionParser
from UserDict import UserDict

SEQ_SLIDE = 1
MINIMUM_READ_LENGTH = 16

#-----------------------------------------------------
class AutoDictionary(UserDict):
	"""
	Filling the prefix tree using this class seems to impair SSAKE's speed
	"""
	def __getitem__(self, key):
		if key in self.data:
			return self.data[key]
		else:
			self.data[key] = AutoDictionary()
			return self.data[key]
		if hasattr(self.__class__, "__missing__"):
			return self.__class__.__missing__(self, key)

#-----------------------------------------------------
def main():
	usage = "Usage: %s --help"

	parser = OptionParser()
	parser.add_option("-f", "--file", dest="filename",
	                  help="A single fasta file containing all the sequence reads (.fa file).",)
	parser.add_option("-m", "--minimum", dest="smin", type="int", default=16,
	                  help="Minimum number of overlapping bases with the seed/contig during overhang consensus build up (default -m 16)")
	parser.add_option("-o", "--base_overlap", dest="base_overlap", type="int", default=2,
	                  help="Minimum number of reads needed to call a base during overhang consensus build up (default -o 2)",)
	parser.add_option("-r", "--base_ratio", dest="base_ratio", type="float", default=0.6,
                          help="Minimum base ratio used to accept a overhang consensus base (default -r 0.6)",)
	parser.add_option("-v", "--verbose", dest="verbose", action="store_true",
	                  help="Runs in Verbose mode.",)
	(opts, args) = parser.parse_args()

	
	try:
		f = open(opts.filename)
		fasta = f.readlines()
		f.close()
	except Exception, e:
		print "ERROR: Could not read from %s: %s" % (opts.filename, e)
		print usage % (sys.argv[0:])
		sys.exit()

        pid = "ssake_m" + str(opts.smin) + "_o" + str(opts.base_overlap) + "_r" + str(opts.base_ratio) + "_pid" + str(os.getpgrp())
        pid_filename = "%s.%s" % (opts.filename, pid)
        contig = "%s.contigs" % pid_filename
        singlet = "%s.singlets" % pid_filename
        short = "%s.short" % pid_filename
        log = "%s.log" % pid_filename
	
	CONTIG = open(contig, 'w')
	SINGLET = open(singlet, 'w')
	SHORT = open(short, 'w')

	try:
		LOG = open(log, 'w')
	except:
		print "ERROR: Can not write to %s" % log
		sys.exit()
	
	if opts.base_overlap < 1:
                print "ERROR: -o must be a number higher or equal to 1"
                sys.exit()

	if opts.base_ratio <= 0.5 or opts.base_ratio > 1.0:
                print "ERROR: -o must be a number between 0.51 and 1.00"
                sys.exit()

	if opts.smin < 11 or opts.smin > 50:
		print "ERROR: -m must be a number between 11 and 50"
		sys.exit()

        print "\nRunning %s" % (sys.argv[0:])
	LOG.write("""
Running:
%s
-f %s
-m %s
-o %s
-r %s
Contigs file: %s
Singlets file: %s
Short reads file: %s

""" % (sys.argv[0:],opts.filename, opts.smin, opts.base_overlap, opts.base_ratio, contig, singlet, short))
	
	t0 = datetime.now()
        read_message = "\nReading sequences initiated %s" % (str(t0)[:len('2006-10-05 23:04')])
        print "%s" % read_message
	LOG.write("%s" % read_message)
	(sset, bin) = parseFastaFile(fasta,opts.smin,LOG,SHORT)

	seqs_start = len(sset)

	t1 = datetime.now()
	assemble_message = "\nSequence assembly initiated %s" % (str(t1)[:len('2006-10-05 23:04')])
        print "%s" % assemble_message
	LOG.write("%s" % assemble_message)
	
	tig_count, previous_index = (0,0)

	#### Status bar 
	per = {}
	status_bar = "+"
	for i in range(1,99):
		per[i] = 1

	# sort helps cluster high coverage reads first
	sset_list = sset.items()
	sset_list.sort(cmp=lambda x, y: y[1] - x[1])

        #### Assembly starts
	for seq, reads in sset_list:

		orig_mer = len(seq)
	
		# Sequence has not been used
		if sset.has_key(seq):  ## Sequence read hasn't been used

	                if sset.has_key(seq):
	                        reads_needed = sset[seq]
        	        else:
                	        reads_needed = 0

			total_bases = (orig_mer * reads_needed) ### there are duplicates in the set, want to account for that while calculating coverage
			start_sequence = seq

                        #print "SEED:%s COPIES:%i READS LEFT:%i" % (seq,sset[seq],len(sset))
                       			
			# Remove kmer from dictionary and prefix tree
			bin = deletePrefixTreeBranch(seq, bin)
			seq_rc = reverseComplement(seq)
			bin = deletePrefixTreeBranch(seq_rc, bin)
			del sset[seq]
			
			if opts.verbose: print "\n>>> START SEED SEQUENCE :: %s <<<\n" % seq
			
			# begin 3' extension
			(seq, sset, bin, reads_needed, total_bases) = doExtension('3', orig_mer, seq, sset, bin, reads_needed,total_bases, opts)

			# end of the 3' extension
			
			# Reverse-complement the contig in order to change the 5' extension problem into a 3' extension
			seq_rc = reverseComplement(seq)

			(seq_rc, sset, bin, reads_needed, total_bases) = doExtension('5', orig_mer, seq_rc, sset, bin, reads_needed,total_bases, opts) 
                        # end of the 5' extension

			tig_count = tig_count + 1

			reverse_tig = reverseComplement(seq_rc)

			if start_sequence != seq_rc and start_sequence != reverse_tig:
				#print contigs to file
				cov = float(float(total_bases) / float(len(seq_rc)))
				CONTIG.write(">contig%i|size%i|read%i|cov%.2f\n%s\n" % (tig_count, len(seq_rc), reads_needed, cov, reverse_tig))
	      		else:
				# print singlets to file
				cov = reads_needed
				SINGLET.write(">contig%i|size%i|read%i|cov%.2f\n%s\n" % (tig_count, len(start_sequence), reads_needed, cov, start_sequence))
		
		# anything left to assemble?
		if not len(sset):
			break
		else:
			difference = int(int(seqs_start) - len(sset))
			ratio = float(float(difference) / float(seqs_start))
			index = int(ratio * 100)
			#print "%i %.2f  %i %i in" % (index, ratio, difference,len(sset))
			if per.has_key(index):
				stretch = "." * (index - previous_index)
				completion = str(index) + "%"
				print stretch,completion,
				sys.stdout.flush()	
				del per[index]
			previous_index = index
	
	t2 = datetime.now()
	finish_message = "\n\nFinished normally at %s\n\n" % str(t2)[:len('2006-10-05 23:04')]
	print "%s" % finish_message
	LOG.write("%s" % finish_message)
	
	LOG.close()
	CONTIG.close()
	SINGLET.close()
	SHORT.close()	
	return

#-----------------------------------------------------
def doExtension(direction, orig_mer, seq, sset, bin, reads_needed, total_bases, opts):
	"""
	Extension
	"""

	previous = seq
	extended = 1

	while extended:

		ct, pos, current_reads, current_bases, span = (0,0,0,0,orig_mer)
		overhang = {}
		overlapping_reads = [] 

	        for x in range(1,(orig_mer * 2)):
			if not overhang.has_key(x):
				overhang[x] = {}
			overhang[x]['A'],overhang[x]['C'],overhang[x]['G'],overhang[x]['T'] = (0,0,0,0)

		while span > opts.smin:
			span = orig_mer - ct
			pos = len(seq) - span
			subseq = seq[pos:pos + span]
			if opts.verbose:
				print "#### %s' Extension Counter: %d  Span: %d  Subseq: %s Previous: %s" % (direction, ct, span, subseq, previous)
		
                	sa = list(subseq)
	                subset = {}                
	
        	        if bin.has_key(sa[0]):
                		if bin[sa[0]].has_key(sa[1]):
					if bin[sa[0]][sa[1]].has_key(sa[2]):
						if bin[sa[0]][sa[1]][sa[2]].has_key(sa[3]):
							if bin[sa[0]][sa[1]][sa[2]][sa[3]].has_key(sa[4]):
								if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]].has_key(sa[5]):
									if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]].has_key(sa[6]):
										if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]].has_key(sa[7]):
											if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]].has_key(sa[8]):
												if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]].has_key(sa[9]):
													if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]][sa[9]].has_key(sa[10]):
														subset = bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]][sa[9]][sa[10]]


			if opts.verbose: print "\nCandidate reads in that space: \n" % subset 

			# cycle through limited kmer space
			subset_list = subset.items()
			subset_list.sort(cmp=lambda x, y: y[1] - x[1])
			for spass, count in subset_list:
				# can we align perfectly that subseq kmer to another rd start?

				if spass[:len(subseq)] == subseq:
					if opts.verbose: print "\tchosen sequence: %s\n" % spass

					dangle = spass[len(subseq):]
					overlapping_reads.append(spass)
					over = list(dangle)		
					ct_oh = 0

					for bz in over:
						ct_oh += 1
						#print "%i %s" % (ct_oh,bz)
						if sset.has_key(spass):
							overhang[ct_oh][bz] += sset[spass]
						else:
							spass_rc = reverseComplement(spass)
							overhang[ct_oh][bz] += sset[spass_rc]
	
						if opts.verbose: 
							print "%i - %s = %i\n" % (ct_oh, bz, overhang[ct_oh][bz])

				else:
	                                spass_field_regex = re.compile(spass)
       					match_spass =  spass_field_regex.search(subseq)

					if match_spass != None:                     ### spass embedded fully in subseq
					
                                       		rc_spass = reverseComplement(spass)
	                                        if sset.has_key(spass):
        	                                        current_reads = sset[spass]
                	                                current_bases = len(spass) * current_reads
                        	                        reads_needed += current_reads
                                	                total_bases += current_bases
                                        	        del sset[spass]
              					if sset.has_key(rc_spass):
                                                	current_reads = sset[rc_spass]
                                                	current_bases = len(rc_spass) * current_reads
                                                	reads_needed += current_reads
                                                	total_bases += current_bases
                                                	del sset[rc_spass]

                                        	bin = deletePrefixTreeBranch(spass, bin)
                                        	bin = deletePrefixTreeBranch(rc_spass, bin)

                        ct += SEQ_SLIDE

		consensus = ""
		if opts.verbose:
			print "Finished collecting overlapping reads - NOW BUILDING CONSENSUS OVERHANG\n"

	        # sort by overhang base order
	        overhang_list = overhang.items()
        	overhang_list.sort()
		exit_flag = 0

		for ohpos,baseorder in overhang_list:

			if exit_flag:
				break			

			if ohpos and overhang.has_key(ohpos):
				
				coverage = overhang[ohpos]['A'] + overhang[ohpos]['C'] + overhang[ohpos]['G'] + overhang[ohpos]['T']
				if opts.verbose:
					print "pos: %i cov: %i A: %i C: %i G: %i T: %i\n" % (ohpos,coverage,overhang[ohpos]['A'],overhang[ohpos]['C'],overhang[ohpos]['G'],overhang[ohpos]['T'])

				if coverage < opts.base_overlap:
					if opts.verbose:
						print "BASE COVERAGE BELOW THRESHOLD: %i < -o %i at %i - Will extend by %s\n" % (coverage,opts.base_overlap,ohpos,consensus)
					break
				
			baselist = {}
                        if overhang.has_key(ohpos):
				baselist = overhang[ohpos]
			ct_dna = 0
			previous_bz = ""

			# sort by most abundant base first
		        base_list = baselist.items()
		       	base_list.sort(cmp=lambda x, y: y[1] - x[1])

			for bz, bz_ct in base_list:
				if ct_dna:
					current_ratio = float(float(baselist[previous_bz]) / float(coverage))
					if previous_bz != "" and current_ratio >= opts.base_ratio and baselist[previous_bz] > baselist[bz]:
						consensus += previous_bz
						if opts.verbose:
							print "Added base %s (cov = %i) to %s\n" % (previous_bz,baselist[previous_bz],consensus)
						break
 					else:
						if opts.verbose:
							print "ISSUES EXTENDING: BEST base = %s (cov= %i) at %i.  Second-best = %s (cov= %i), Ratio = %.2f, RatioThreshold set to -r %.2f -- Will terminate with %s" % (previous_bz,baselist[previous_bz],ohpos,bz,baselist[bz],current_ratio,opts.base_ratio,consensus)
						exit_flag = 1

				previous_bz = bz
				ct_dna += 1

		if consensus != "" :
			if opts.verbose:
				print "Will extend %s with %s" % (seq,consensus)
			temp_sequence = seq + consensus
			integral = 0
			for ro in overlapping_reads:
				read_field_regex = re.compile(ro)
				match =  read_field_regex.search(temp_sequence)
                		if match != None:
					integral = 1
					rc_ro = reverseComplement(ro)

					if sset.has_key(ro):
						current_reads = sset[ro]
						current_bases = len(ro) * current_reads
						reads_needed += current_reads
						total_bases += current_bases
			                        del sset[ro]
					if sset.has_key(rc_ro):
                                                current_reads = sset[rc_ro]
                                                current_bases = len(rc_ro) * current_reads
                                                reads_needed += current_reads
                                                total_bases += current_bases
                                                del sset[rc_ro]

					bin = deletePrefixTreeBranch(ro, bin)
					bin = deletePrefixTreeBranch(rc_ro, bin)

			if integral == 0:
				if opts.verbose:
					print "No overlapping reads agree with the consensus sequence.  Stopping %s prime extension.\n" % direction
				extended = 0
			else:
				seq = temp_sequence
				extended = 1
				if opts.verbose:
					print "NEW CONTIG is: %s\n" % seq

			previous = seq
		else:
			extended = 0  ### stopping the extension, nothing else can be done

	if opts.verbose:
		print "\n*** NOTHING ELSE TO BE DONE IN %s - PERHAPS YOU COULD DECREASE THE MINIMUM OVERLAP -m (currently set to -m %i) and relax the -o (set to -o %i) and -r (set to -r %.2f) parameters***\n" % (direction, opts.smin, opts.base_overlap, opts.base_ratio)


	return (seq, sset, bin, reads_needed, total_bases)

#-----------------------------------------------------
def parseFastaFile(fasta,min_overlap,LOG,SHORT):
	"""
	Parse a FASTA file
	
	Return a Dictionary of sequences, with the value the count seen in the FASTA file
	and a bin of base pair counts for the first 11 bases
	"""
	sset = {}
	bin = {}
	
	fasta_sequence_field = re.compile('^[ACTG]+$')
	ctrd = 0
	for seq in fasta:
		seq = seq.rstrip('\r\n')
		seq = seq.upper()
		orig_mer  = len(seq)

		if re.match(fasta_sequence_field, seq):

			if(orig_mer >= MINIMUM_READ_LENGTH and orig_mer >= min_overlap): ## Sequence read hasn't been used, is longer than set nt and the user-difined minimum -m
	                        s10k = float(float(ctrd)/ 10000)
        	                if ctrd != 0 and s10k != 0  and (float(s10k) == float(int(s10k))):
					print ".",
					sys.stdout.flush()

				s100k = float(float(ctrd)/ 100000)
				if ctrd != 0 and s100k != 0  and (float(s100k) == float(int(s100k))):
					print "%i sequences inputted" % (ctrd)
					sys.stdout.flush()

	                        pf = list(seq)
	
				if not sset.has_key(seq):
					sset[seq] = 0
				
				sset[seq] += 1

				#if not bin[pf[0]][pf[1]][pf[2]][pf[3]][pf[4]][pf[5]][pf[6]][pf[7]][pf[8]][pf[9]][pf[10]].has_key(seq):
				#	bin[pf[0]][pf[1]][pf[2]][pf[3]][pf[4]][pf[5]][pf[6]][pf[7]][pf[8]][pf[9]][pf[10]][seq] = 0
				#bin[pf[0]][pf[1]][pf[2]][pf[3]][pf[4]][pf[5]][pf[6]][pf[7]][pf[8]][pf[9]][pf[10]][seq] += 1

		                seq_rc = reverseComplement(seq)
		                pr = list(seq_rc)

                	        bin = fillPrefixTree(pf,seq,bin)
                        	bin = fillPrefixTree(pr,seq_rc,bin)

				#if not bin[pr[0]][pr[1]][pr[2]][pr[3]][pr[4]][pr[5]][pr[6]][pr[7]][pr[8]][pr[9]][pr[10]].has_key(seq_rc):
				#	bin[pr[0]][pr[1]][pr[2]][pr[3]][pr[4]][pr[5]][pr[6]][pr[7]][pr[8]][pr[9]][pr[10]][seq_rc] = 0
				#bin[pr[0]][pr[1]][pr[2]][pr[3]][pr[4]][pr[5]][pr[6]][pr[7]][pr[8]][pr[9]][pr[10]][seq_rc] += 1                       
 
				ctrd += 1
			else:
			       	if orig_mer < MINIMUM_READ_LENGTH:
					SHORT.write("%s\tInput sequence shorter than minimum read length allowed (%i < %i nt)\n" % (seq, orig_mer, MINIMUM_READ_LENGTH))
                        	elif orig_mer < min_overlap:
                                	SHORT.write("%s\tInput sequence shorter than minimum overlap allowed (%i < %i nt)\n" % (seq, orig_mer, min_overlap))


	print "%i total sequences inputted (%i unique)" % (ctrd, len(sset.keys()))	
	LOG.write("\n\tTotal sequences read: %d\n" % ctrd)
	LOG.write("\tNumber of unique sequences: %d" % len(sset.keys()))
	
	return sset, bin

#-----------------------------------------------------
def deletePrefixTreeBranch(sequence, bin):

	sa = list(sequence)

	if bin.has_key(sa[0]):
        	if bin[sa[0]].has_key(sa[1]):
                	if bin[sa[0]][sa[1]].has_key(sa[2]):
                        	if bin[sa[0]][sa[1]][sa[2]].has_key(sa[3]):
                                	if bin[sa[0]][sa[1]][sa[2]][sa[3]].has_key(sa[4]):
                                        	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]].has_key(sa[5]):
                                                	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]].has_key(sa[6]):
                                                        	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]].has_key(sa[7]):
                                                                	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]].has_key(sa[8]):
                                                                        	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]].has_key(sa[9]):
                                                                                	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]][sa[9]].has_key(sa[10]):
                                                                                        	if bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]][sa[9]][sa[10]].has_key(sequence):
													del bin[sa[0]][sa[1]][sa[2]][sa[3]][sa[4]][sa[5]][sa[6]][sa[7]][sa[8]][sa[9]][sa[10]][sequence]

	return bin

#-----------------------------------------------------
def reverseComplement(seq):
	"Reverse complement of DNA"
	trans = string.maketrans('ACGT','TGCA')
	seq_rc = seq
	seq_rc = list( seq_rc )
	seq_rc.reverse()
	seq_rc = "".join( seq_rc )
	seq_rc = seq_rc.translate( trans )
	
	return seq_rc

#-----------------------------------------------------
def fillPrefixTree(ar,seq,bin):

        """
        that's got the be the ugliest piece of code I have ever written / but seems to lead to faster execution Vs. using the AutoDict class above
        """

        if not bin.has_key(ar[0]):
                bin[ar[0]] = {}
        if not bin[ar[0]].has_key(ar[1]):
                bin[ar[0]][ar[1]] = {}
        if not bin[ar[0]][ar[1]].has_key(ar[2]):
                bin[ar[0]][ar[1]][ar[2]] = {}
        if not bin[ar[0]][ar[1]][ar[2]].has_key(ar[3]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]].has_key(ar[4]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]].has_key(ar[5]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]].has_key(ar[6]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]].has_key(ar[7]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]].has_key(ar[8]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]].has_key(ar[9]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]].has_key(ar[10]):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]][ar[10]] = {}
        if not bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]][ar[10]].has_key(seq):
                bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]][ar[10]][seq] = 0
        bin[ar[0]][ar[1]][ar[2]][ar[3]][ar[4]][ar[5]][ar[6]][ar[7]][ar[8]][ar[9]][ar[10]][seq] +=1


        return bin

#-----------------------------------------------------
if __name__ == '__main__':
	main()
	import time
	sys.exit()
