
/* $Id */

#include "filesettings.h"
#include "settings.h"
#include "startalk.h"
#include "stdebug.h"

#include <ctype.h>
#include <string.h>
#include <stdlib.h>


struct startalk_settings *
startalk_get_settings_from_file(FILE *f)
{
  struct startalk_settings *se;
  char buf[8192];
  char *c, *vp, *sp;
  int settingmatched;
  long newval;
  int stype, snum;
  char *v;

  extern int testing;
  
  debugf(3,"startalk_get_settings_from_file called.\n");
  if (!(se = startalk_new_settings()))
    return (void *) reterr(0,"malloc error:");

  while (fgets(buf,8191,f))
  {
    buf[8191] = 0;

    if ( (buf[0] == '#') || (buf[0] == 0) || (buf[0] == '\n') )
    {
      continue;
    }

    if ( (c=strchr(buf,'\n')) == 0)
      c = strchr(buf,'\0')-1;
    while ( (c >= buf) && isspace(*c) )
      *(c--) = '\0';
    if (buf[0] == '\0')
      continue;
    
    if ( (vp = strchr(buf,':')) == 0)
    {
      warn("Couldn't parse line '%s': no colon",buf);
      continue;
    }

    /* remove colon and whitespace from variable name */
    c = vp;;
    *(c--) = '\0';
    while ( (c >= buf) && isspace(*c) )
      *(c--) = '\0';
    sp = buf;
    while ( (*sp) && isspace(*sp) )
      sp++;

    /* Get the value, and remove whitespace. */
    vp++;
    while (isspace(*vp) && *vp)
      vp++;

    debugf(3,"startalk_get_settings_from_file: got line with var='%s', val='%s'\n",sp,vp);

    settingmatched=0;
    if (!startalk_setting_find(sp, &stype, &snum))
    {
      warn("Unknown setting on line '%s'",buf);
      continue;
    }
    debugf(3,"'%s' matched!  Type %d, num %d\n",sp,stype,snum);

    if (startalk_setting_readonly(stype, snum))
    {
      if (testing)
        warn("Warning: Attempt to set setting '%s', which is readonly.  Trying anyways...",sp);
      else
      {
        warn("Warning: Attempt to set setting '%s', which is readonly.",sp);
        continue;
      }
    }
      

    switch(stype)
    {
      case STARTALK_SET_TYPE_BOOL:
        if ( (strcasecmp(vp,"on") == 0) ||
             (strcasecmp(vp,"yes") == 0) ||
             (strcasecmp(vp,"true") == 0) )
        {
          newval = 1;
        }
        else if ( (strcasecmp(vp,"off") == 0) ||
                  (strcasecmp(vp,"no") == 0) ||
                  (strcasecmp(vp,"false") == 0) )
        {
          newval = 0;
        }
        else
        {
          warn("Ignored unrecognized boolean value '%s' for setting '%s'", vp,sp);
          continue;
        }
        if ((v=startalk_validate_setting_bool(se, snum, newval)) != NULL)
        {
          warn("Ignored invalid boolean value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        startalk_set_setting_bool(se, snum, newval);

        break;

      case STARTALK_SET_TYPE_UCHAR:
        if (!isdigit(vp[0]))
        {
          warn("Ignored non-number '%s' for setting '%s'",vp,sp);
          continue;
        }
        newval = atol(vp);
        if ((v=startalk_validate_setting_uchar(se,snum,newval)) != NULL)
        {
          warn("Ignored invalid uchar value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        startalk_set_setting_uchar(se, snum, newval);

        break;

      case STARTALK_SET_TYPE_STRING:
        if ((v=startalk_validate_setting_string(se,snum,vp)) != NULL)
        {
          warn("Ignored invalid string value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        /* Quick hack for greeting */

        startalk_set_setting_string(se, snum, vp);

        break;

      case STARTALK_SET_TYPE_UINT:
        if (!isdigit(vp[0]))
        {
          warn("Ignored non-number '%s' for uint setting '%s'",vp,sp);
          continue;
        }
        newval = atol(vp);
        if ((v=startalk_validate_setting_uint(se,snum,newval)) != NULL)
        {
          warn("Ignored invalid uint value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        startalk_set_setting_uint(se, snum, newval);

        break;

      case STARTALK_SET_TYPE_LANG:
        if (strncasecmp(vp,"english",7) == 0)
          newval = 0;
        else if ( (strncasecmp(vp,"spanish",7) == 0) ||
                  (strncasecmp(vp,"espa",4) == 0) )
          newval = 5;
        else if ( (strncasecmp(vp,"french",6) == 0) ||
                  (strncasecmp(vp,"fra",3) == 0) )
          newval = 6;
        else
        {
          warn("Invalid language '%s' for setting '%s'",vp,sp);
          continue;
        }
        if ((v=startalk_validate_setting_lang(se,snum,newval)) != NULL)
        {
          warn("Ignored invalid lang value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        startalk_set_setting_lang(se, snum, newval);

        break;
        
      case STARTALK_SET_TYPE_SUINT:
        if (!isdigit(vp[0]))
        {
          warn("Ignored non-number '%s' for uint setting '%s'",vp,sp);
          continue;
        }
        newval = atol(vp);
        if ((v=startalk_validate_setting_suint(se,snum,newval)) != NULL)
        {
          warn("Ignored invalid suint value '%s' for setting '%s': %s",vp,sp,v);
          continue;
        }
        startalk_set_setting_suint(se, snum, newval);

        break;

      default:
        warn("startalk_setting_find returned invalid type value %d?!",stype);
        break;
    }
  }

  return se;
}


void
startalk_fprint_setting(FILE *f, struct startalk_settings *n, unsigned int setting_type, unsigned int setting_num)
{
  int val;
  char *sval;
  long lval;
  char *leader, *sname;

  if (startalk_setting_readonly(setting_type, setting_num))
    leader = "# (read-only) ";
  else
    leader = "";

  if ((sname=startalk_get_setting_name(setting_type, setting_num)) == NULL)
  {
    warn("Requested unknown setting: type %d, number %d\n",setting_type, setting_num);
    return;
  }
  switch(setting_type)
  {
    case STARTALK_SET_TYPE_BOOL:
      val = startalk_get_setting_bool(n, setting_num);
      if (val == -1)
        return; /* not set */
      fprintf(f, "%s%s: %s\n",leader,sname,val?"On":"Off");
      break;

    case STARTALK_SET_TYPE_UCHAR:
      val = startalk_get_setting_uchar(n, setting_num);
      if (val == -1)
        return; /* not set */
      fprintf(f, "%s%s: %d\n",leader,sname,val);
      break;

    case STARTALK_SET_TYPE_STRING:
      sval = startalk_get_setting_string(n, setting_num);
      if (sval == NULL)
        return; /* not set */
      fprintf(f, "%s%s: %s\n",leader,sname,sval);
      break;

    case STARTALK_SET_TYPE_UINT:
      lval = startalk_get_setting_uint(n, setting_num);
      if (lval == -1)
        return; /* not set */
      fprintf(f, "%s%s: %ld\n",leader,sname,lval);
      break;

    case STARTALK_SET_TYPE_SUINT:
      lval = startalk_get_setting_suint(n, setting_num);
      if (lval == -1)
        return; /* not set */
      fprintf(f, "%s%s: %ld\n",leader,sname,lval);
      break;

    case STARTALK_SET_TYPE_LANG:
      val = startalk_get_setting_lang(n, setting_num);
      if (val == -1)
        return; /* not set */
      switch(val)
      {
        case 0:
          sval = "English";
          break;
        case 5:
          sval = "Spanish";
          break;
        case 6:
          sval = "French";
          break;
        default:
          warn("Unknown language number %d",val);
          return;
      }
      fprintf(f, "%s%s: %s\n",leader,sname,sval);
      break;

    default:
      warn("Requested unknown setting: type %d, number %d\n", setting_type, setting_num);
  }
}

void
startalk_fprint_settings(FILE *f, struct startalk_settings *n)
{
  int i;
  
  for(i=0;i<=STARTALK_SET_BOOL_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_BOOL, i);
  for(i=0;i<=STARTALK_SET_UCHAR_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_UCHAR, i);
  for(i=0;i<=STARTALK_SET_STRING_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_STRING, i);
  for(i=0;i<=STARTALK_SET_UINT_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_UINT, i);
  for(i=0;i<=STARTALK_SET_SUINT_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_SUINT, i);
  for(i=0;i<=STARTALK_SET_LANG_LAST;i++)
    startalk_fprint_setting(f, n, STARTALK_SET_TYPE_LANG, i);
  fprintf(f,"\n");
}
