package net.sf.statcvs.charts;

import java.awt.Color;
import java.awt.Dimension;
import java.io.File;
import java.io.IOException;
import java.util.logging.Logger;

import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;

/**
 * An image file for a chart.
 * 
 * TODO: Better integrate all charts with ReportConfig 
 * @author jentzsch
 * @author Richard Cyganiak (richard@cyganiak.de)
 * @version $Id: ChartImage.java,v 1.6 2008/04/02 11:22:15 benoitx Exp $
 */
public class ChartImage {
    private final static Logger logger = Logger.getLogger("sf.net.statcvs");

    private final static Color BACKGROUND_COLOR = new Color(204, 204, 187);

    private final String rootDirectory;
    private JFreeChart chart;
    private final String fileName;
    private final Dimension size;
    private final String title;
    private boolean written = false;

    /**
     * Creates a new ChartFile.
     * @param rootDirectory The report root directory with trailing slash
     * @param fileName The relative file name for the chart, with .png extension
     * @param title The chart's title
     * @param chart The JFreeChart object to save as a file
     * @param size width and height of the chart in pixels
     */
    public ChartImage(final String rootDirectory, final String fileName, final String title, final JFreeChart chart, final Dimension size) {
        this.rootDirectory = rootDirectory;
        this.fileName = fileName;
        this.title = title;
        this.chart = chart;
        this.size = size;
        chart.setBackgroundPaint(BACKGROUND_COLOR);
    }

    /**
     * Writes the chart to disk as a PNG file.
     */
    public void write() {
        if (this.written) {
            return;
        }
        logger.info("writing chart '" + this.title + "' to " + this.fileName);
        try {
            ChartUtilities.saveChartAsPNG(new File(rootDirectory + fileName), chart, size.width, size.height);
        } catch (final IOException e) {
            logger.warning("could not write chart '" + fileName + "': " + e);
        }
        this.written = true;
        this.chart = null; // Free memory? Not sure if this has any effect ... 
    }

    /**
     * Returns the chart's URL, relative to the report root.
     */
    public String getURL() {
        return this.fileName;
    }

    /**
     * Returns the chart's title.
     */
    public String getFullTitle() {
        return this.title;
    }

    /**
     * Returns the chart's width in pixels.
     */
    public int getWidth() {
        return this.size.width;
    }

    /**
     * Returns the chart's height in pixels.
     */
    public int getHeight() {
        return this.size.height;
    }
}