import traceback
import sys

_base_cache = {}

def find_bases(klass):

    def _find_bases(klass, d = None):
        if d is None:
            d = {}
        for b in klass.__bases__:
            if issubclass(b, BaseSignal):
                d[b] = 1
                _find_bases(b, d)
        return tuple(d.keys())
    
    bases = _base_cache.get(klass, None)
    if bases is not None:
        return bases
    bases = _find_bases(klass)
    _base_cache[klass] = bases
    return bases

class BaseSignal:
    def __init__(self, sender):
        self.sender = sender
        return

class SignalEmitter:
    recorder = None
    
    """Base class for signal emitters"""
    def __init__(self):
        self._slots = {}

    def initialize_slots(self, *signals):
        """Initialize signal slots.
        signals should be a list of signals this emitter is going to send."""
        for s in signals:
            self._slots[s] = {}
            bs = find_bases(s)
            for k in bs:
                self._slots[k] = {}

    def signal_connect(self, signal, handler, data = None):
        """Connect a handler to a signal in this emitter, and associate
        data with the callback. This will override previous bindings of
        signal to handler."""
        self._slots[signal][handler] = data
        return

    def signal_disconnect(self, signal, handler):
        """Disconnect handler from signal."""
        del self._slots[signal][handler]
        return

    def emit_signal(self, signal):
        """Emit a signal object. Signal should be an instance of BaseSignal or
        a subclass."""
        sclasses = find_bases(signal.__class__)
        for sc in sclasses + (signal.__class__,):
            for handler, data in self._slots.get(sc, {}).items():
                if data is not None:
                    args = (signal, data)
                else:
                    args = (signal,)
                try:
                    if self.recorder is not None:
                        self.recorder.record(sc, handler)
                    apply(handler, args)
                except:
                    sys.stderr.write("Caught an exception when trying to "
                                     "call a signal handler for "
                                     "signal %s\n" %
                                     str(signal))
                    traceback.print_exc()

    def start_recording(klass, recorder):
        klass.recorder = recorder
    start_recording = classmethod(start_recording)

    def stop_recording(klass):
        if klass.recorder is not None:
            klass.recorder.stop()
            klass.recorder = None
    stop_recording = classmethod(stop_recording)


class ConfigOptionSignal(BaseSignal):
    pass

class LocalConfigOptionSignal(ConfigOptionSignal):
    pass

class FinderDoneSignal(BaseSignal):
    """
    Used by: SubscribeDialog.py

    Emitted by FeedFinder when it's finished looking for feeds
    """
    def __init__(self, sender, feeds, error=None):
        BaseSignal.__init__(self,sender)
        self.feeds = feeds
        self.error = error

class AuthNeededSignal(BaseSignal):
    """
    Used by: SubscribeDialog.py

    Emitted when feed URL needs authorization
    """
    def __init__(self, sender):
        BaseSignal.__init__(self, sender)

class FeedSummaryDoneSignal(BaseSignal):
    """
    Used by: SubscribeDialog.py

    Emitted by FeedSummaryPoller after the summary has been parsed.
    """
    def __init__(self, sender, feed, data, error=None):
        BaseSignal.__init__(self, sender)
        self.feed = (feed, data)
        self.error = error

class PollChangedSignal(BaseSignal):
    def __init__(self, sender):
        BaseSignal.__init__(self, sender)

class FeedsChangedSignal(LocalConfigOptionSignal):
    def __init__(self, sender, feed=None):
        LocalConfigOptionSignal.__init__(self, sender)
        self.feed = feed

class FeedDeletedSignal(FeedsChangedSignal):
    def __init__(self, sender, feed):
        FeedsChangedSignal.__init__(self, sender, feed)

class FeedCreatedSignal(FeedsChangedSignal):
    def __init__(self, sender, feed, category=None, index=None):
        FeedsChangedSignal.__init__(self, sender)
        self.feed = feed
        self.category = category
        self.index = index

class SaveFeedsSignal(LocalConfigOptionSignal):
    pass

class NumberOfItemsStoredChangedSignal(LocalConfigOptionSignal):
    pass

class ItemOrderChangedSignal(LocalConfigOptionSignal):
    pass

class PollFrequencyChangedSignal(PollChangedSignal):
    """ Global poll frequency """
    def __init__(self, sender, value):
        PollChangedSignal.__init__(self, sender)
        self.value = value

class BlogURLChangedSignal(LocalConfigOptionSignal):
    pass

class MainWindowSizeChangedSignal(LocalConfigOptionSignal):
    pass

class OfflineModeChangedSignal(LocalConfigOptionSignal):
    pass

class MainPanePositionChangedSignal(LocalConfigOptionSignal):
    pass

class SubPanePositionChangedSignal(LocalConfigOptionSignal):
    pass

class RefreshFeedDisplaySignal(BaseSignal):
    pass

class StatusDisplaySignal(BaseSignal):
    pass

class FindInterruptSignal(BaseSignal):
    pass

class FeedPolledSignal(BaseSignal):
    pass

class FeedStatusChangedSignal(BaseSignal):
    pass

class FeedErrorStatusChangedSignal(BaseSignal):
    pass

class FeedCategoriesChangedSignal(BaseSignal):
    pass

class FeedCategoryListChangedSignal(FeedCategoriesChangedSignal):
    pass

class FeedCategoryListLoadedSignal(FeedCategoriesChangedSignal):
    pass

class FeedCategoryAddedSignal(FeedCategoryListChangedSignal):
    def __init__(self, sender, category):
        FeedCategoryListChangedSignal.__init__(self, sender)
        self.category = category

class FeedCategoryRemovedSignal(FeedCategoryListChangedSignal):
    def __init__(self, sender, category):
        FeedCategoryListChangedSignal.__init__(self, sender)
        self.category = category

class FeedCategoryChangedSignal(FeedCategoriesChangedSignal):
    def __init__(self, sender, feed=None):
        FeedCategoriesChangedSignal.__init__(self, sender)
        self.feed = feed

class ItemsAddedSignal(BaseSignal):
    def __init__(self, sender, items):
        BaseSignal.__init__(self, sender)
        self.items = items

class ItemDeletedSignal(BaseSignal):
    def __init__(self, sender, item):
        BaseSignal.__init__(self, sender)
        self.item = item

class NewItemsSignal(BaseSignal):
    def __init__(self, sender, items):
        BaseSignal.__init__(self, sender)
        self.items = items

class ItemReadSignal(BaseSignal):
    def __init__(self, sender, item=None):
        BaseSignal.__init__(self, sender)
        if item is None:
            item = sender
        self.item = item

class ItemStickySignal(BaseSignal):
    def __init__(self, sender, item=None):
        BaseSignal.__init__(self, sender)
        if item is None:
            item = sender
        self.item = item

class AllItemsReadSignal(BaseSignal):
    def __init__(self, sender, changed):
        BaseSignal.__init__(self, sender)
        self.changed = changed

class ImageUpdatedSignal(BaseSignal):
    def __init__(self, sender, url, data):
        BaseSignal.__init__(self, sender)
        self.url = url
        self.data = data

class SubscriptionContentsUpdatedSignal(BaseSignal):
    pass

class PollingStoppedSignal(BaseSignal):
    pass

if __name__ == '__main__':
    print find_bases(ItemReadSignal)
