package Hardware;

#   $Id: Hardware.pm,v 1.2 2001/08/02 00:00:58 sdague Exp $

#   Copyright (c) 2001 International Business Machines

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#   Sean Dague <japh@us.ibm.com>

use strict;
use Carp;
use Hardware::PCI::Detect;
use Modules::ModulesConf;
use Modules::ConfModules;
use Modules::ModutilsAliases;

use vars qw(@hardtypes @conftypes $VERSION);

$VERSION = sprintf("%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/);

sub setup {
    my $config = shift;

    my @hw = get_devices($config);
    my @aliases = create_aliases(@hw);
    return build_confs($config, @aliases);
}

sub get_devices {
    my $config = shift;

    my @hw;

    my %vars = $config->varlist("^(root)");

    foreach my $htype (@hardtypes) {
        my $hwtype = $htype->new(%vars);
        if($hwtype->footprint()) {
            push @hw, $hwtype->getlist();
        }
    }
    return @hw;
}

sub create_aliases {
    my @hw = @_;
    
    my @aliases;

    my $eth = 0;
    my $tr = 0;
    my $scsi_hostadapter = "";

    for(my $i = 0; $i < scalar(@hw); $i++) {
        my $item = $hw[$i];
        if($item->{type} eq "ethernet") {
            push @aliases, ["eth$eth",$item->{module}];
            $eth++;
        }
        if($item->{type} eq "tokenring") {
            push @aliases, ["tr$tr",$item->{module}];
            $tr++;
        }
        if($item->{type} eq "scsi") {
            push @aliases, ["scsi_hostadapter$scsi_hostadapter", $item->{module}];
            $scsi_hostadapter++;
        }
    }
    return @aliases;
}

sub build_confs {
    my $config = shift;
    my @aliases = @_;
    
    my @files = ();

    my %vars = $config->varlist("^(root)");

    foreach my $conftype (@conftypes) {
        my $ctype = $conftype->new(%vars);

        if($ctype->footprint()) {
            $ctype->setup(@aliases);
            push @files, $ctype->files();
        }
    }
    
    return @files;
}

=pod

=head1 NAME

  SystemConfig::Hardware - Hardware Setup Module for System Configurator

=head1 SYNOPSIS

  use SystemConfig::Hardware;

  my @modifiedfiles = SystemConfig::Hardware::setup($config);

=head1 DESCRIPTION

The Hardware setup routine has 3 phases:

=over 4

=item get_devices($config)

In the get_devices stage pci device numbers are collected and returned.

This has only one module that it calls right now, which is SystemConfig::Hardware::DetectPCI.

=item create_aliases(@devices)

This creates a set of kernel aliases for the hardware that is found if it is
an ethernet or token ring card, or it is a scsi device.

=item build_confs(@aliases)

This modifies the appropriate conf files for the particular distribution.

=back

=head1 COPYRIGHT

Copyright 2001 International Business Machines

=head1 AUTHOR

  Sean Dague <japh@us.ibm.com>

=head1 SEE ALSO

L<SystemConfig::Kernel::ModutilsAliases>, L<SystemConfig::Kernel::ModulesConf>,
L<SystemConfig::Kernel::ConfModules>, L<SystemConfig::Hardware::DetectPCI>, L<perl>

=cut
1;





