/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE
#include <strings.h>            /* Needed for index() */

#include "paths.h"

#include "discover.h"
#include "utils.h"


typedef struct{
  unsigned long id;
  char vendor[256];		/* Vendor name                        */
  char model[256];		/* Model name                         */
  char module[32];		/* Name of the module to load         */
  DeviceType type;
  int vendorDefault;
  unsigned long vendorID;
} TableEntry;



static int read_table(FILE *tablefile, TableEntry *entry){
  char *line = NULL;
  char class[12];
  size_t len = 0;
  
  if(getline(&line, &len, tablefile) < 0) {
    if (NULL != line)
      free(line);
    return EOF;
  }/*endif*/
  
  if(line[0] == '\t'){
    sscanf(line, "%08lx\t%12s\t%32s\t%256[^\n]\n",
	   (unsigned long *)&entry->id, 
	   (char *)&class,
	   (char *)&entry->module, 
	   (char *)&entry->model); 
    entry->type = str2device(class);
    if(entry->id == 0xFFFFFFFFL) {
      entry->vendorDefault = 1;
      entry->id = entry->vendorID << 16;
    }else{
      entry->vendorDefault = 0;
    }

  }else{
    sscanf(line, "%04lx %256[^\n]\n",
           (unsigned long *)&entry->vendorID,
           (char *)&entry->vendor);
    len = 0;
    free(line);
    return read_table(tablefile, entry);
  }/*endif*/

  len = 0;
  free(line);

  return 0;
}/*endfunc read_table*/

static void load_entry(TableEntry *entry, struct cards_lst *lst, BusType bus) {
  lst->vendor = (char *)my_malloc(MAX_VENDOR);
  lst->model = (char *)my_malloc(MAX_MODEL);
  lst->modulename = (char *)my_malloc(MAX_MODULE);
  lst->long_id = entry->id;
  lst->type = entry->type;
  lst->vendorDefault = entry->vendorDefault;
  lst->vendorID = entry->vendorID;
  strcpy(lst->vendor, entry->vendor);
  strcpy(lst->model, entry->model);
  strcpy(lst->modulename, entry->module);
  lst->bus = bus;
  lst->next = (struct cards_lst *)my_malloc(sizeof(struct cards_lst));
  memset(lst->next, 0, sizeof(*lst));
} /*endfunc load_entry*/

extern struct cards_lst *init_lst(const char *pci_lst_list[],
				  const char *pcmcia_lst_list[],
				  const char *usb_lst_list[]
#ifdef USE_SBUS
				  , const char *sbus_lst_list[]
#endif /* USE_SBUS */
				  )
{
  FILE *f;
#ifdef USE_SBUS
  char *delim;
#endif
  struct cards_lst *lst = NULL, *first = NULL, *prev = NULL;
  TableEntry entry;
  const char **bkp_lst_list;
  size_t len = 0;

  first = lst = prev = (struct cards_lst *)my_malloc(sizeof(struct cards_lst));
  memset(lst, 0, sizeof(*lst));

  entry.id = 0;
  entry.vendorDefault = 0;
  entry.vendorID = 0;
  entry.vendor[0] = '\0';
  entry.model[0] = '\0';
  entry.module[0] = '\0';
  entry.type = UNKNOWN_DEVICE;

  /********************************************************************/
  /************************ PCI HARDWARE LIST *************************/
  /********************************************************************/
  bkp_lst_list = pci_lst_list;
  while (*pci_lst_list != 0) {
    const char *pci_lst = *pci_lst_list;
    if(pci_lst){  
      if(debug)
	fprintf(stdout, "Reading PCI hardware database...\n");

      /*PCI list processing*/
      if(!(f = fopen(pci_lst, "r"))){
	if(debug)
	  fprintf(stderr, FOPEN_READ_ERROR, pci_lst);
      }else {
	while(read_table(f, &entry) != EOF) {
	  load_entry(&entry, lst, PCI);
	  prev = lst;
	  lst = lst->next;
	}/*endwhile*/
	fclose(f);
      }/*endif*/
    }/*endif*/
    pci_lst_list++;
  }/*endwhile*/
  pci_lst_list = bkp_lst_list;

  /********************************************************************/
  /************************ USB HARDWARE LIST *************************/
  /********************************************************************/
  bkp_lst_list = usb_lst_list;
  while (*usb_lst_list != 0) {
    const char *usb_lst = *usb_lst_list;
    if(usb_lst) {
      if(debug)
	fprintf(stdout, "Reading USB hardware database...\n");

      /*USB list processing*/
      if(!(f = fopen(usb_lst, "r"))){
	if(debug)
	  fprintf(stderr, FOPEN_READ_ERROR, usb_lst);
      }else {
	while(read_table(f, &entry) != EOF) {
	  load_entry(&entry, lst, USB);
	  prev = lst;
	  lst = lst->next;
	}/*endwhile*/
	fclose(f);
      }/*endif*/
    }/*endif*/
    usb_lst_list++;
  }/*endwhile*/
  usb_lst_list = bkp_lst_list;

  /********************************************************************/
  /********************** PCMCIA HARDWARE LIST ************************/
  /********************************************************************/
  bkp_lst_list = pcmcia_lst_list;
  while (*pcmcia_lst_list != 0) {
    const char *pcmcia_lst = *pcmcia_lst_list;
    if(pcmcia_lst){
      if(debug)
	fprintf(stdout, "Reading PCMCIA hardware database...\n");

      /*PCMCIA list processing*/
      if(!(f = fopen(pcmcia_lst, "r"))){
	if(debug)
	  fprintf(stderr, FOPEN_READ_ERROR, pcmcia_lst);
      }else {
	while(read_table(f, &entry) != EOF) {
	  load_entry(&entry, lst, PCMCIA);
	  prev = lst;
	  lst = lst->next;
	}/*endwhile*/
	fclose(f);
      }/*endif*/
    }/*endif*/
    pcmcia_lst_list++;
  }/*endwhile*/
  pcmcia_lst_list = bkp_lst_list;

#ifdef USE_SBUS
  /***********************************************************************/
  /************************* SBUS HARDWARE LIST **************************/
  /***********************************************************************/
  while (*sbus_lst_list != 0) {
    char *sbus_lst = *sbus_lst_list;
    if (sbus_lst)
      {
	if(debug)
	  fprintf(stdout, "Reading SBUS hardware database...\n");

	if (!(f = fopen(sbus_lst, "r"))) {
	  if(debug)
	    fprintf(stderr, FOPEN_READ_ERROR, sbus_lst);
        }
	else {
	  while(read_table(f, &entry) != EOF) {
	    lst->vendor = (char *)my_malloc(MAX_VENDOR);
	    lst->model = (char *)my_malloc(MAX_MODEL);
	    lst->dev_id = (char *)my_malloc(MAX_SBUS_ID);
	    lst->modulename = (char *)my_malloc(MAX_MODULE);
	    lst->bus = SBUS;
	    lst->type = entry.type;
	    delim = index(entry.model, ':');
	    if(delim == NULL) {
	      /* Table entry is broken */
	    } else {
	      *delim++ = '\0';
	      strncpy(lst->vendor, entry.vendor, MAX_VENDOR-1);
	      strncpy(lst->dev_id, entry.model, MAX_SBUS_ID-1);
	      strncpy(lst->model, delim, MAX_MODEL-1);
	      strncpy(lst->modulename, entry.module, MAX_MODULE-1);
	    }
	    lst->next = (struct cards_lst *) my_malloc(sizeof(struct cards_lst));
	    memset(lst->next, 0, sizeof(*lst));
	    prev = lst;
	    lst = lst->next;
	    fclose(f);
	  }
	}
      }
    sbus_lst_list++;
  }/*endwhile*/
#endif
      
  len = 0;

  /* Terminate the list (without NULL entry at the end) */
  prev->next = NULL;
  lst = first;
  return lst;
}/*endfunc init_lst*/
