[comment {-*- tcl -*- doctools manpage}]
[manpage_begin units n 1.2]
[copyright {2000-2005 Mayo Foundation}]
[moddesc   {Convert and manipulate quantities with units}]
[titledesc {unit conversion}]
[require Tcl 8.1]
[require units [opt 2.1]]
[description]
[para]

This library provides a conversion facility from a variety of
scientific and engineering shorthand notations into floating point
numbers.  This allows application developers to easily convert
values with different units into uniformly scaled numbers.

[para]

The units conversion facility is also able to convert between
compatible units.  If, for example, a application is expecting a value
in [emph ohms] (Resistance), and the user specifies units of
[emph milliwebers/femtocoulomb], the conversion routine will
handle it appropriately.  An error will be generated if an incorrect
conversion is attempted.

[para]

Values are scaled from one set of units to another by dimensional
analysis.  Both the value units and the target units are reduced into
primitive units and a scale factor.  Units are checked for
compatibility, and the scale factors are applied by multiplication and
division.  This technique is extremely flexible and quite robust.

[para]

New units and new unit abbreviations can be defined in terms of
existing units and abbreviations.  It is also possible to define a new
primitive unit, although that will probably be unnecessary.  New units
will most commonly be defined to accommodate non-SI measurement
systems, such as defining the unit [emph inch] as [emph "2.54 cm"].

[section "COMMANDS"]

[list_begin definitions]

[call [cmd ::units::convert] [arg value] [arg targetUnits]]

Converts the [arg value] string into a floating point number, scaled to the
specified [arg targetUnits].  The [arg value] string may contain a
number and units.  If units are specified, then they must be
compatible with the [arg targetUnits].  If units are not specified
for the [arg value], then it will be scaled to the target units.  For
example,
[example_begin]
% ::units::convert "2.3 miles" km
3.7014912
% ::units::convert 300m/s miles/hour
671.080887616
% ::units::convert "1.0 m kg/s^2" newton
1.0
% ::units::convert 1.0 millimeter
1000.0
[example_end]


[call [cmd ::units::reduce] [arg unitString]]

Returns a unit string consisting of a scale factor followed by a space
separated list of sorted and reduced primitive units.  The reduced
unit string may include a forward-slash (separated from the
surrounding primitive subunits by spaces) indicating that the
remaining subunits are in the denominator.  Generates an error if the
[arg unitString] is invalid.

[example_begin]
% ::units::reduce pascal
1000.0 gram / meter second second
[example_end]



[call [cmd ::units::new] [arg name] [arg baseUnits]]

Creates a new unit conversion with the specified name.  The new unit
[arg name] must be only alphabetic (upper or lower case) letters.
The [arg baseUnits] string can consist of any valid units conversion
string, including constant factors, numerator and denominator parts,
units with prefixes, and exponents.  The baseUnits may contain any
number of subunits, but it must reduce to primitive units.  BaseUnits
could also be the string [emph "-primitive"] to represent a new
kind of quantity which cannot be derived from other units.  But you
probably would not do that unless you have discovered some kind of new
universal property.

[example_begin]
% ::units::new furlong "220 yards"
% ::units::new fortnight "14 days"
% ::units::convert 100m/s furlongs/fortnight
601288.475303
[example_end]

[list_end]



[section "UNIT STRING FORMAT"]

Value and unit string format is quite flexible.  It is possible to
define virtually any combination of units, prefixes, and powers.
Valid unit strings must conform to these rules.


[list_begin bullet]

[bullet]
A unit string consists of an optional scale factor followed by zero or
more subunits.  The scale factor must be a valid floating point
number, and may or may not be separated from the subunits.  The scale
factor could be negative.

[bullet]
Subunits are separated form each other by one or more separator
characters, which are space (" "), hyphen ("-"), asterisk ("*"), and
forward-slash ("/").  Sure, go ahead and complain about using a minus
sign ("-") to represent multiplication.  It just isn't sound
mathematics, and, by rights, we should require everyone to use the
asterisk ("*") to separate all units.  But the bottom line is that
complex unit strings like [emph "m-kg/s^2"] are pleasantly
readable.

[bullet]
The forward-slash seperator ("/") indicates that following subunits are
in the denominator.  There can be at most one forward-slash separator.

[bullet]
Subunits can be floating point scale factors, but with the exception
of the leading scale factor, they must be surrounded by valid
separators.  Subunit scale factors cannot be negative. (Remember that
the hyphen is a unit separator.)

[bullet]
Subunits can be valid units or abbreviations.  They may include a
prefix.  They may include a plural suffix "s" or "es".  They may also
include a power string denoted by a circumflex ("^"), followed by a
integer, after the unit name (or plural suffix, if there is one).
Negative exponents are not allowed.  (Remember that the hyphen is a
unit separator.)

[list_end]

[subsection "Example Valid Unit Strings"]

[example_begin]
Unit String              Reduced Unit String
------------------------------------------------------------
meter                    1.0 meter
kilometer                1000.0 meter
km                       1000.0 meter
km/s                     1000.0 meter / second
/microsecond             1000000.0 / second
/us                      1000000.0 / second
kg-m/s^2                 1000.0 gram meter / second second
30second                 30.0 second
30 second                30.0 second
30 seconds               30.0 second
200*meter/20.5*second    9.75609756098 meter / second
[example_end]





[section "SI UNITS"]

[para]

The standard SI units are predefined according to [emph "NIST Special
Publication 330"].  Standard units for both SI Base Units (Table
1) and SI Derived Units with Special Names (Tables 3a and 3b) are
included here for reference.  Each standard unit name and abbreviation
are included in this package.


[subsection "SI Base Units"]

[example_begin]
Quantity                Unit Name    Abbr.
---------------------------------------------
Length                  meter        m
Mass                    kilogram     kg
Time                    second       s
Current                 ampere       A
Temperature             kelvin       K
Amount                  mole         mol
Luminous Intensity      candela      cd
[example_end]


[subsection "SI Derived Units with Special Names"]

[example_begin]
Quantity                Unit Name    Abbr.   Units     Base Units
--------------------------------------------------------------------
plane angle             radian      rad     m/m       m/m
solid angle             steradian   sr      m^2/m^2   m^2/m^2
frequency               hertz       Hz                /s
force                   newton      N                 m-kg/s^2
pressure                pascal      Pa      N/m^2     kg/m-s^2
energy, work            joule       J       N-m       m^2-kg/s^2
power, radiant flux     watt        W       J/s       m^2-kg/s^3
electric charge         coulomb     C                 s-A
electric potential      volt        V       W/A       m^2-kg/s^3-A
capacitance             farad       F       C/V       s^4-A^2/m^2-kg
electric resistance     ohm                 V/A       m^2-kg/s^3-A^2
electric conductance    siemens     S       A/V       s^3-A^2/m^2-kg
magnetic flux           weber       Wb      V-s       m^2-kg/s^2-A
magnetic flux density   tesla       T       Wb/m^2    kg/s^2-A
inductance              henry       H       Wb/A      m^2-kg/s^2-A^2
luminous flux           lumen       lm                cd-sr
illuminance             lux         lx      lm/m^2    cd-sr/m^2
activity (of a
radionuclide)           becquerel   Bq                /s
absorbed dose           gray        Gy      J/kg      m^2/s^2
dose equivalent         sievert     Sv      J/kg      m^2/s^2
[example_end]

[para]

Note that the SI unit kilograms is actually implemented as grams
because 1e-6 kilogram = 1 milligram, not 1 microkilogram.  The
abbreviation for Electric Resistance (ohms), which is the omega
character, is not supported.  

[para]

Also note that there is no support for Celsius or Farenheit
temperature.  The units conversion routines can only scale values
with multiplication and division, so it is not possible to convert
from thermodynamic temperature (kelvins) to absolute degrees Celsius
or Farenheit.  Conversion of thermodynamic quantities, such as
thermal expansion (per unit temperature), however, are easy to add
to the units library.

[para]

SI Units can have a multiple or sub-multiple prefix.  The prefix or its
abbreviation should appear before the unit, without spaces.  Compound
prefixes are not allowed, and a prefix should never be used alone.
These prefixes are defined in Table 5 of [emph "Special Publication
330"].


[subsection "SI Prefixes"]

[example_begin]
Prefix Name     Abbr.   Factor
---------------------------------------
yotta           Y       1e24
zetta           Z       1e21
exa             E       1e18
peta            P       1e15
tera            T       1e12
giga            G       1e9
mega            M       1e6
kilo            k       1e3
hecto           h       1e2
deka            da      1e1
deca                    1e1

deci            d       1e-1
centi           c       1e-2
milli           m       1e-3
micro           u       1e-6
nano            n       1e-9
pico            p       1e-12
femto           f       1e-15
atto            a       1e-18
zepto           z       1e-21
yocto           y       1e-24
[example_end]

[para]

Note that we define the same prefix with both the USA ("deka") and
non-USA ("deca") spellings.  Also note that we take the liberty of
allowing "micro" to be typed as a "u" instead of the Greek character
mu.

[para]

Many non-SI units are commonly used in applications.  Appendix B.8 of
[emph "NIST Special Publication 811"] lists many non-SI conversion
factors.  It is not possible to include all possible unit definitions
in this package.  In some cases, many different conversion factors
exist for a given unit, depending on the context. (The appendix lists
over 40 conversions for British thermal units!)  Application specific
conversions can always be added using the [cmd new]
command, but some well known and often used conversions are included
in this package.


[subsection "Non-SI Units"]

[example_begin]
Unit Name            Abbr.    Base Units
--------------------------------------------------
angstrom                      1.0E-10 m
astronomicalUnit     AU       1.495979E11 m
atmosphere                    1.01325E5 Pa
bar                           1.0E5 Pa
calorie                       4.1868 J
curie                         3.7E10 Bq
day                           8.64E4 s
degree                        1.745329E-2 rad
erg                           1.0E-7 J
faraday                       9.648531 C
fermi                         1.0E-15 m
foot                 ft       3.048E-1 m
gauss                         1.0E-4 T
gilbert                       7.957747E-1 A
grain                gr       6.479891E-5 kg
hectare              ha       1.0E4 m^2
hour                 h        3.6E3 s
inch                 in       2.54E-2 m
lightYear                     9.46073E15 m
liter                L        1.0E-3 m^3
maxwell              Mx       1.0E-8 Wb
mho                           1.0 S
micron                        1.0E-6 m
mil                           2.54E-5 m
mile                 mi       1.609344E3 m
minute               min      6.0E1 s
parsec               pc       3.085E16 m
pica                          4.233333E-3 m
pound                lb       4.535924E-1 kg
revolution                    6.283185 rad
revolutionPerMinute  rpm      1.047198E-1 rad/s
yard                 yd       9.144E-1 m
year                          3.1536E7 s
[example_end]


[subsection "Quantities and Derived Units with Special Names"]

This units conversion package is limited specifically to unit
reduction, comparison, and scaling.  This package does not consider
any of the quantity names for either base or derived units.  A similar
implementation or an extension in a typed or object-oriented language
might introduce user defined types for the quantities.  Quantity type
checking could be used, for example, to ensure that all
[emph "length"] values properly reduced to [emph "meters"], or
that all [emph "velocity"] values properly reduced to
[emph "meters/second"].

[para]

A C implementation of this package has been created to work in
conjunction with the Simplified Wrapper Interface Generator
([uri http://www.swig.org/]).  That package (units.i) exploits SWIG's typemap
system to automatically convert script quantity strings into floating
point quantities.  Function arguments are specified as quantity types
(e.g., [emph "typedef float Length"]), and target units (expected
by the C application code) are specified in an associative array.
Default units are also defined for each quantity type, and are applied
to any unit-less quantity strings.

[para]

A units system enhanced with quantity type checking might benefit from
inclusion of other derived types which are expressed in terms of
special units, as illustrated in Table 2 of [emph "NIST Publication
330"].  The quantity [emph "area"], for example, could be
defined as units properly reducing to [emph "meter^2"], although
the utility of defining a unit named [emph "square meter"] is
arguable.


[section "REFERENCES"]

The unit names, abbreviations, and conversion values are derived from
those published by the United States Department of Commerce Technology
Administration, National Institute of Standards and Technology (NIST)
in [emph "NIST Special Publication 330: The International System of
Units (SI)"] and [emph "NIST Special Publication 811: Guide for
the Use of the International System of Units (SI)"].  Both of
these publications are available (as of December 2000) from
[uri http://physics.nist.gov/cuu/Reference/contents.html]

[para]

The ideas behind implementation of this package is based in part on
code written in 1993 by Adrian Mariano which performed dimensional
analysis of unit strings using fixed size tables of C structs.  After
going missing in the late 1990's, Adrian's code has reappeared in the
GNU Units program at [uri http://www.gnu.org/software/units/]


[section "AUTHORS"]

Robert W. Techentin


[keywords unit conversion constants distance radians angle]

[manpage_end]
