#   include	"tedConfig.h"

#   include	<stdlib.h>

#   include	"docDraw.h"
#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  For dividing table rows over pages.					*/
/*									*/
/************************************************************************/

typedef struct CellDrawingProgress
    {
    int			cdpAdvanced;
    int			cdpPara;
    int			cdpLinesDone;

    int			cdpX0Twips;
    int			cdpX1Twips;
    int			cdpX11Twips;	/*  Includes right border	*/

    int			cdpX0Pixels;
    int			cdpX1Pixels;
    int			cdpX11Pixels;	/*  Includes right border	*/

    ParagraphFrame	cdpParagraphFrame;
    } CellDrawingProgress;

/************************************************************************/
/*									*/
/*  Initialise a DrawingContext						*/
/*									*/
/************************************************************************/

void docInitDrawingContext(	DrawingContext *	dc )
    {
    dc->dcCurrentPhysicalFont= -1;

    dc->dcDrawingData= (AppDrawingData *)0;
    dc->dcDocument= (BufferDocument *)0;
    dc->dcClipRect= (DocumentRectangle *)0;
    dc->dcDocumentSelection= (DocumentSelection *)0;
    dc->dcSelectionGeometry= (SelectionGeometry *)0;
    dc->dcFirstPage= -1;
    dc->dcLastPage= -1;
    dc->dcDrawHeadersFooters= 0;

    dc->dcDrawTableGrid= 0;

    dc->dcDrawTextLine= (DRAW_TEXT_LINE)0;
    dc->dcDrawParaTop= (DRAW_PARA_TOP)0;
    dc->dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
    dc->dcDrawCellTop= (DRAW_CELL_TOP)0;
    dc->dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
    dc->dcDrawCellLeft= (DRAW_CELL_LEFT)0;
    dc->dcDrawCellRight= (DRAW_CELL_RIGHT)0;
    dc->dcFinishPage= (FINISH_PAGE)0;
    dc->dcStartPage= (START_PAGE)0;
    dc->dcParaFramePixels= (PARA_SIDES_PIXELS)0;

    dc->dcLayoutExternal= (LAYOUT_EXTERNAL)0;
    dc->dcCloseObject= (DOC_CLOSE_OBJECT)0;

    return;
    }

/************************************************************************/
/*									*/
/*  Generic Document drawing machinery.					*/
/*									*/
/************************************************************************/

static int docDrawOneLine(	void *				through,
				const BufferItem *		bi,
				int				part,
				const ParagraphFrame *		pf,
				int				line,
				DrawingContext *		dc )
    {
    const TextLine *		tl= bi->biParaLines+ line;

    int				accepted;

    if  ( part == 0				&&
	  bi->biParaBorderAboveParagraph	&&
	  dc->dcDrawParaTop			)
	{
	if  ( (*dc->dcDrawParaTop)( bi, bi->biParaBorderAboveParagraph,
							pf, through, dc ) )
	    { LDEB(1); return -1;	}
	}

    if  ( dc->dcDrawTextLine )
	{
	accepted= (*dc->dcDrawTextLine)( bi, line, pf, through, dc );
	}
    else{ accepted= tl->tlParticuleCount;	}

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( part+ accepted == bi->biParaParticuleCount )
	{
	if  ( bi->biParaBorderBelowParagraph	&&
	      dc->dcDrawParaBottom			)
	    {
	    if  ( (*dc->dcDrawParaBottom)( bi,
					bi->biParaBorderBelowParagraph,
							pf, through, dc ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return accepted;
    }

static int docDrawTextLines(	void *				through,
				const BufferItem *		bi,
				const LayoutPosition *		lpHere,
				const ParagraphFrame *		pf,
				int *				pLinesDone,
				DrawingContext *		dc )
    {
    int				done= 0;
    int				pastSelectionEnd= 0;

    int				line= *pLinesDone;
    const TextLine *		tl= bi->biParaLines+ line;

    while( line < bi->biParaLineCount )
	{
	int	accepted;
	int	beforeSelectionBegin= 0;

	if  ( tl->tlTopPosition.lpPage > lpHere->lpPage )
	    { break;	}

	if  ( tl->tlTopPosition.lpPage < lpHere->lpPage )
	    { beforeSelectionBegin= 1;	}

	if  ( dc->dcClipRect )
	    {
	    AppDrawingData *		add= dc->dcDrawingData;

	    int	y1= TL_BELOW_PIXELS( add, tl );

	    if  ( TL_TOP_PIXELS( add, tl ) > dc->dcClipRect->drY1 )
		{ pastSelectionEnd= 1;	}
	    if  ( y1 < dc->dcClipRect->drY0 )
		{ beforeSelectionBegin= 1;	}
	    }

	if  ( dc->dcFirstPage >= 0			&&
	      lpHere->lpPage < dc->dcFirstPage	)
	    { beforeSelectionBegin= 1;	}

	if  ( dc->dcDocumentSelection					&&
	      ! beforeSelectionBegin					&&
	      ! pastSelectionEnd					&&
	      ( bi == dc->dcDocumentSelection->dsBegin.dpBi	||
	        bi == dc->dcDocumentSelection->dsEnd.dpBi	)	)
	    {
	    const DocumentSelection *	ds= dc->dcDocumentSelection;

	    DocumentSelection		dsLine;
	    const int			mindPart= 1;

	    docLineSelection( &dsLine, bi, line );

	    if  ( bi == ds->dsEnd.dpBi					&&
		  docComparePositions( &(dsLine.dsBegin),
				    &(ds->dsEnd), mindPart ) > 0	)
		{ pastSelectionEnd= 1;	}

	    if  ( bi == ds->dsBegin.dpBi				&&
		  docComparePositions( &(dsLine.dsEnd),
				    &(ds->dsBegin), mindPart ) < 0	)
		{ beforeSelectionBegin= 1;	}
	    }

	if  ( beforeSelectionBegin || pastSelectionEnd )
	    { accepted= tl->tlParticuleCount;		}
	else{
	    accepted= docDrawOneLine( through, bi, tl->tlFirstParticule,
							    pf, line, dc );
	    }

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1; }

	done += accepted; line++; tl++;
	}

    *pLinesDone= line;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/*  1)  Just to make the compiler happy, if nothing is to be printed	*/
/*	anymore, the current position is not going to be used.		*/
/*									*/
/************************************************************************/

static int docDrawToNextColumn(	BufferItem *		thisBodyBi,
				BufferItem *		nextBodyBi,
				void *			through,
				LayoutPosition *	lpHere,
				BlockFrame *		bf,
				DrawingContext *	dc )
    {
    BufferDocument *		bd= dc->dcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);

    const int			asLast= 0;

    while( thisBodyBi && thisBodyBi->biLevel != DOClevSECT )
	{ thisBodyBi= thisBodyBi->biParent; }
    while( nextBodyBi && nextBodyBi->biLevel != DOClevSECT )
	{ nextBodyBi= nextBodyBi->biParent; }

    if  ( dc->dcFirstPage < 0				||
	  lpHere->lpPage >= dc->dcFirstPage		)
	{
	if  ( thisBodyBi						&&
	      dc->dcDrawHeadersFooters					)
	    {
	    if  ( docDrawFootnotesForColumn( lpHere->lpPage, through, dc ) )
		{ LDEB(lpHere->lpPage); return -1;	}

	    if  ( docDrawPageFooter( thisBodyBi, through, dc, lpHere->lpPage ) )
		{ LDEB(lpHere->lpPage); return -1;	}
	    }

	if  ( dc->dcFinishPage						&&
	      (*dc->dcFinishPage)( through, dc, lpHere->lpPage,
							asLast )	)
	    { LDEB(1); return -1;	}
	}

    if  ( dc->dcLastPage < 0				||
	  lpHere->lpPage+ 1 <= dc->dcLastPage		)
	{
	lpHere->lpPage++;

	if  ( nextBodyBi )
	    {
	    docLayoutSectColumnTop( nextBodyBi, bd, lpHere, bf );

	    if  ( dc->dcStartPage					&&
		  (*dc->dcStartPage)( through, dg, dc, lpHere->lpPage )	)
		{ LDEB(1); return -1;	}

	    if  ( dc->dcDrawHeadersFooters				&&
		  docDrawPageHeader( nextBodyBi, through, dc,
						    lpHere->lpPage )	)
		{ LDEB(lpHere->lpPage); return -1;	}

	    }
	}

    return 0;
    }

int docDrawToPageOfItem(		BufferItem *		prevBodyBi,
					BufferItem *		thisBodyBi,
					BufferItem *		thisBi,
					void *			through,
					LayoutPosition *	lpHere,
					DrawingContext *	dc )
    {
    BlockFrame	bf;

    while( thisBi->biTopPosition.lpPage > lpHere->lpPage	&&
	   thisBi->biTopPosition.lpPage > dc->dcFirstPage	)
	{
	if  ( docDrawToNextColumn( prevBodyBi, thisBodyBi, through,
						    lpHere, &bf, dc )	)
	    { SDEB(docLevelStr(thisBi->biLevel)); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print a node in the BufferItem hierarchy.				*/
/*									*/
/*  1)  The last page is finished by the caller.			*/
/*									*/
/************************************************************************/

static int docDrawParaItem(	BufferItem *			paraBi,
				void *				through,
				DrawingContext *		dc )
    {
    ParagraphFrame		pf;
    int				line= 0;

    int				bottomTwips= -1;
    int				stripHigh= -1;

    BufferDocument *		bd= dc->dcDocument;

    BlockFrame			bf;
    LayoutPosition		lpTop;

    docBlockFrameTwips( &bf, paraBi, bd, paraBi->biTopPosition.lpPage,
					    paraBi->biTopPosition.lpColumn );

    docParagraphFrameTwips( &pf, &bf, bottomTwips, stripHigh, paraBi );

    if  ( dc->dcParaFramePixels )
	{ (*dc->dcParaFramePixels) ( &pf, dc->dcDrawingData, paraBi ); }

    lpTop= paraBi->biTopPosition;
    while( line < paraBi->biParaLineCount )
	{
	if  ( docDrawTextLines( through, paraBi, &lpTop, &pf, &line, dc ) )
	    { LDEB(lpTop.lpPage); return -1;	}

	/*  1  */
	if  ( dc->dcLastPage >= 0		&&
	      lpTop.lpPage >= dc->dcLastPage	)
	    { break;	}

	if  ( line < paraBi->biParaLineCount )
	    {
	    if  ( docDrawToNextColumn( paraBi, paraBi,
						through, &lpTop, &bf, dc ) )
		{ SDEB(docLevelStr(paraBi->biLevel));return -1;	}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw part of the grid originating from a table row.			*/
/*									*/
/************************************************************************/

static int docDrawCellSides(	const BufferItem *		rowBi,
				const CellDrawingProgress *	cdp,
				int				drawTop,
				int				drawSides,
				int				drawBottom,
				void *				through,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    const CellProperties *	cp= rowBi->biRowCells;

    int				col;

    const BufferItem *		prevRow= (const BufferItem *)0;
    const BufferItem *		nextRow= (const BufferItem *)0;

    const CellProperties *	prevCp= (const CellProperties *)0;
    const CellProperties *	nextCp= (const CellProperties *)0;

    if  ( dc->dcDrawTableGrid )
	{
	const BufferItem *	sectBi= rowBi->biParent;

	if  ( drawTop && rowBi->biNumberInParent )
	    {
	    prevRow= sectBi->biChildren[rowBi->biNumberInParent- 1];

	    if  ( ! prevRow->biRowHasTableParagraphs			||
		  ! docAlignedColumns( &(prevRow->biRowProperties),
					    &(rowBi->biRowProperties) )	)
		{ prevRow= (const BufferItem *)0;	}
	    else{ prevCp= prevRow->biRowCells;		}
	    }

	if  ( rowBi->biNumberInParent < sectBi->biChildCount- 1 )
	    {
	    nextRow= sectBi->biChildren[rowBi->biNumberInParent+ 1];

	    if  ( ! nextRow->biRowHasTableParagraphs			||
		  ! docAlignedColumns( &(nextRow->biRowProperties),
					    &(rowBi->biRowProperties) )	)
		{ nextRow= (const BufferItem *)0;	}
	    else{ nextCp= nextRow->biRowCells;	}
	    }
	}

    for ( col= 0; col < rowBi->biChildCount; cdp++, cp++, col++ )
	{
	int		topAsGrid= 0;
	int		leftAsGrid= 0;
	int		rightAsGrid= 0;
	int		bottomAsGrid= 0;

	int		drawLeft= drawSides;
	int		drawRight= drawSides;

	if  ( cp->cpMergedWithLeft )
	    { drawLeft= 0;	}
	if  ( col < rowBi->biChildCount- 1	&&
	      cp[1].cpMergedWithLeft		)
	    { drawRight= 0;	}

	if  ( dc->dcDrawTableGrid )
	    {
	    if  ( cp->cpTopBorder.bpStyle == DOCbsNONE			&&
		  ( ! prevRow					||
		    prevCp->cpBottomBorder.bpStyle == DOCbsNONE	)	)
		{ topAsGrid= 1;	}

	    if  ( col == 0 && cp->cpLeftBorder.bpStyle == DOCbsNONE )
		{ leftAsGrid= 1;	}

	    if  ( col == rowBi->biChildCount- 1 )
		{
		if  ( cp->cpRightBorder.bpStyle == DOCbsNONE )
		    { rightAsGrid= 1;	}
		}
	    else{
		if  ( cp->cpRightBorder.bpStyle == DOCbsNONE	&&
		      cp[1].cpLeftBorder.bpStyle == DOCbsNONE	)
		    { rightAsGrid= 1;	}
		}

	    if  ( cp->cpBottomBorder.bpStyle == DOCbsNONE		&&
		  ( ! nextRow					||
		    nextCp->cpTopBorder.bpStyle == DOCbsNONE	)	)
		{ bottomAsGrid= 1;	}
	    }

	if  ( drawTop							&&
	      dc->dcDrawCellTop						&&
	      ( cp->cpTopBorder.bpStyle != DOCbsNONE || topAsGrid )	)
	    {
	    if  ( (*dc->dcDrawCellTop)(		&(cp->cpTopBorder),
						topAsGrid,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						cdp->cdpX11Twips,
						cdp->cdpX11Pixels,
						through, dc,
						&(rowBi->biTopPosition) ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawLeft							&&
	      dc->dcDrawCellLeft					&&
	      ( cp->cpLeftBorder.bpStyle != DOCbsNONE || leftAsGrid )	)
	    {
	    if  ( (*dc->dcDrawCellLeft)(	&(cp->cpLeftBorder),
						leftAsGrid,
						through, dc,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						lpTop, lpBelow ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawRight							&&
	      dc->dcDrawCellRight					&&
	      ( cp->cpRightBorder.bpStyle != DOCbsNONE || rightAsGrid )	)
	    {
	    if  ( (*dc->dcDrawCellRight)(	&(cp->cpRightBorder),
						rightAsGrid,
						through, dc,
						cdp->cdpX1Twips,
						cdp->cdpX1Pixels,
						lpTop, lpBelow ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawBottom						&&
	      dc->dcDrawCellBottom					&&
	      ( cp->cpBottomBorder.bpStyle != DOCbsNONE || bottomAsGrid ) )
	    {
	    if  ( (*dc->dcDrawCellBottom)(	&(cp->cpBottomBorder),
						bottomAsGrid,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						cdp->cdpX11Twips,
						cdp->cdpX11Pixels,
						through, dc,
						&(rowBi->biBelowPosition) ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( prevRow )
	    { prevCp++;	}
	if  ( nextRow )
	    { nextCp++;	}
	}

    return 0;
    }


/************************************************************************/
/*									*/
/*  Draw the lines in a table cell that appear on one printed page.	*/
/*									*/
/************************************************************************/

static int docDrawAdvanceCell(	void *				through,
				int *				pAdvanced,
				BufferItem *			cellBi,
				const LayoutPosition *		lpHere,
				CellDrawingProgress *		cdp,
				BlockFrame *			bf,
				DrawingContext *		dc,
				LayoutPosition *		lpBelow )
				
    {
    int				bottomTwips= -1;
    int				stripHigh= -1;

    int				advanced= 0;

    LayoutPosition		lp;

    while( cdp->cdpPara < cellBi->biChildCount )
	{
	BufferItem *		paraBi= cellBi->biChildren[cdp->cdpPara];
	int			linesDone= cdp->cdpLinesDone;

	if  ( paraBi->biTopPosition.lpPage > lpHere->lpPage )
	    { break;	}

	if  ( cdp->cdpLinesDone == 0 )
	    {
	    docParagraphFrameTwips( &(cdp->cdpParagraphFrame), bf,
					    bottomTwips, stripHigh, paraBi );

	    if  ( dc->dcParaFramePixels )
		{
		(*dc->dcParaFramePixels) ( &(cdp->cdpParagraphFrame),
						dc->dcDrawingData, paraBi );
		}
	    }

	if  ( docDrawTextLines( through, paraBi, lpHere,
				&(cdp->cdpParagraphFrame), &linesDone, dc ) )
	    { LDEB(lpHere->lpPage); return -1;	}

	if  ( linesDone > cdp->cdpLinesDone )
	    { cdp->cdpAdvanced= advanced= 1;	}

	cdp->cdpLinesDone= linesDone;

	if  ( cdp->cdpLinesDone < paraBi->biParaLineCount )
	    {
	    if  ( cdp->cdpLinesDone > 0 )
		{
		TextLine *		tl;

		tl= paraBi->biParaLines+ cdp->cdpLinesDone- 1;

		lp.lpPage= tl->tlTopPosition.lpPage;
		lp.lpPageYTwips= tl->tlTopPosition.lpPageYTwips+
						    tl->tlLineSpacingTwips;
		lp.lpAtTopOfColumn= 0;
		}
	    else{ lp= paraBi->biTopPosition;	}
	    }
	else{ lp= paraBi->biBelowPosition;	}

	docPsLayoutPushBottomDown( lpBelow, &lp );

	if  ( cdp->cdpLinesDone < paraBi->biParaLineCount )
	    { break;	}

	cdp->cdpLinesDone= 0;
	cdp->cdpPara++;
	}

    if  ( advanced )
	{ *pAdvanced= 1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print as much of a table row as fits on the current page.		*/
/*									*/
/************************************************************************/

static int docDrawRowPageStrip(	BufferItem *			rowBi,
				const LayoutPosition *		lpHere,
				int *				pToNextPage,
				CellDrawingProgress *		cdpRow,
				void *				through,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    BufferDocument *		bd= dc->dcDocument;
    int				col;

    int				toNextPage= 0;

    int				advanced;
    int				atTop= 1;

    LayoutPosition		lpBottom;

    CellDrawingProgress *	cdp;

    BlockFrame			bf;

    docBlockFrameTwips( &bf, rowBi, bd, lpHere->lpPage, lpHere->lpColumn );

    lpBottom= *lpTop;

    advanced= 0;
    cdp= cdpRow;
    for ( col= 0; col < rowBi->biChildCount; cdp++, col++ )
	{
	BufferItem *	cellBi= rowBi->biChildren[col];

	if  ( dc->dcDocumentSelection )
	    {
	    const DocumentSelection *	ds= dc->dcDocumentSelection;

	    if  ( docCompareItemPositions( cellBi, ds->dsBegin.dpBi ) < 0 )
		{ continue;	}

	    if  ( docCompareItemPositions( cellBi, ds->dsEnd.dpBi ) > 0	)
		{ continue;	}

	    if  ( ( ds->dsCol0 >= 0 && col < ds->dsCol0	)	||
	          ( ds->dsCol1 >= 0 && col > ds->dsCol1	)	)
		{ continue;	}
	    }

	if  ( cdp->cdpAdvanced )
	    { atTop= 0;	}

	if  ( docDrawAdvanceCell( through, &advanced, cellBi, lpHere,
						    cdp, &bf, dc, &lpBottom ) )
	    { LDEB(1); return -1;	}

	if  ( cdp->cdpPara < cellBi->biChildCount	)
	    { toNextPage= 1;	}
	}

    if  ( advanced )
	{
	const int	drawTop= atTop != 0;
	const int	drawSides= toNextPage != 0;
	const int	drawBottom= 0;

	if  ( drawTop || drawSides || drawBottom )
	    {
	    if  ( docDrawCellSides( rowBi, cdpRow,
					drawTop, drawSides, drawBottom,
					through, dc, lpTop, &lpBottom ) )
		{ LDEB(1); return -1;	}
	    }
	}

    *pToNextPage= toNextPage; return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a table row by successively drawing the portions that fit on	*/
/*  the current page and moving to the next page.			*/
/*									*/
/*  1)  The last page is finished by the caller.			*/
/*									*/
/************************************************************************/

static int docDrawRowItem(	BufferItem *			rowBi,
				void *				through,
				DrawingContext *		dc )
    {
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;

    int				col;

    const RowProperties *	rp= &(rowBi->biRowProperties);
    BufferItem *		sectBi= rowBi->biParent;
    const DocumentGeometry *	dg= &(sectBi->biSectDocumentGeometry);
    CellProperties *		cp;

    int				toNextPage;

    const int			drawTop= 0;
    const int			drawSides= 1;
    const int			drawBottom= 1;

    static CellDrawingProgress *	colP;
    CellDrawingProgress *		cdp;

    int				x00Pixels;

    int				x0Twips;
    int				x1Twips;
    int				x0Pixels;
    int				x1Pixels;

    LayoutPosition		lpTop;
    LayoutPosition		lpHere;

    if  ( rp->rpCellCount < rowBi->biChildCount )
	{
	LLDEB(rp->rpCellCount,rowBi->biChildCount);
	docListItem( 0, rowBi );
	return -1;
	}

    cdp= (CellDrawingProgress *)realloc( colP,
				rp->rpCellCount* sizeof(CellDrawingProgress) );
    if  ( ! cdp )
	{ LXDEB(rp->rpCellCount,cdp); return -1;	}
    colP= cdp;

    toNextPage= 0;

    x00Pixels= TWIPStoPIXELS( xfac, dg->dgLeftMarginTwips );

    x1Twips= x0Twips= dg->dgLeftMarginTwips+ rowBi->biRowLeftIndentTwips;
    x1Pixels= x0Pixels= x00Pixels+ rowBi->biRowLeftIndentPixels;

    cp= rp->rpCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, col++ )
	{
	x1Twips= dg->dgLeftMarginTwips+ cp->cpRightBoundaryTwips;
	x1Pixels= x00Pixels+ cp->cpRightBoundaryPixels;

	cdp[col].cdpAdvanced= 0;

	cdp[col].cdpPara= 0;
	cdp[col].cdpLinesDone= 0;

	cdp[col].cdpX0Twips= x0Twips;
	cdp[col].cdpX1Twips= x1Twips;
	cdp[col].cdpX11Twips= x1Twips;

	cdp[col].cdpX0Pixels= x0Pixels;
	cdp[col].cdpX1Pixels= x1Pixels;
	cdp[col].cdpX11Pixels= x1Pixels;

	if  ( cp->cpRightBorder.bpStyle != DOCbsNONE )
	    { cdp[col].cdpX11Twips +=  cp->cpRightBorder.bpPenWideTwips; }

	x0Twips= x1Twips;
	x0Pixels= x1Pixels;
	}

    lpTop= rowBi->biTopPosition;
    lpHere= rowBi->biTopPosition;

    if  ( docDrawRowPageStrip( rowBi, &lpHere, &toNextPage, cdp,
						    through, dc, &lpTop ) )
	{ LDEB(1); return -1;	}

    while( toNextPage )
	{
	BlockFrame		bf;

	/*  1  */
	if  ( dc->dcLastPage >= 0		&&
	      lpHere.lpPage >= dc->dcLastPage	)
	    { break;	}

	if  ( docDrawToNextColumn( rowBi, rowBi, through, &lpTop, &bf, dc ) )
	    { SDEB(docLevelStr(rowBi->biLevel)); return -1;	}

	lpHere= lpTop;

	if  ( docDrawRowPageStrip( rowBi, &lpHere, &toNextPage, cdp,
						    through, dc, &lpTop ) )
	    { LDEB(1); return -1;	}
	}

    if  ( docDrawCellSides( rowBi, cdp,
			    drawTop, drawSides, drawBottom,
			    through, dc, &lpTop, &(rowBi->biBelowPosition) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the children of a group item.					*/
/*									*/
/************************************************************************/

static int docDrawGroupItem(	const BufferItem *		bi,
				void *				through,
				DrawingContext *		dc )
    {
    AppDrawingData *		add= dc->dcDrawingData;

    int				i;
    LayoutPosition		lpHere;

    BufferItem *		prevBi= (BufferItem *)0;

    if  ( bi->biChildCount > 0 )
	{ prevBi= bi->biChildren[0];	}

    lpHere= bi->biTopPosition;
    for ( i= 0; i < bi->biChildCount; i++ )
	{
	BufferItem *	thisBi= bi->biChildren[i];

	if  ( dc->dcClipRect						&&
	      BI_TOP_PIXELS( add, thisBi ) > dc->dcClipRect->drY1	)
	    { break;	}

	if  ( dc->dcDocumentSelection					&&
	      docCompareItemPositions( thisBi,
			    dc->dcDocumentSelection->dsEnd.dpBi ) > 0	)
	    { break;	}

	if  ( dc->dcLastPage >= 0				&&
	      thisBi->biTopPosition.lpPage > dc->dcLastPage	)
	    { break;	}

	if  ( dc->dcFirstPage < 0				||
	      thisBi->biBelowPosition.lpPage >= dc->dcFirstPage	)
	    {
	    if  ( docDrawToPageOfItem( prevBi, thisBi, thisBi,
						    through, &lpHere, dc ) )
		{ SDEB(docLevelStr(thisBi->biLevel)); return -1;	}

	    if  ( docDrawItem( thisBi, through, dc ) )
		{ LDEB(i); return -1;	}

	    lpHere= thisBi->biBelowPosition;
	    }

	prevBi= thisBi;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the document root item.					*/
/*									*/
/************************************************************************/

static int docDrawDocItem(	const BufferItem *		docBi,
				void *				through,
				DrawingContext *		dc )
    {
    BufferDocument *		bd= dc->dcDocument;
    DocumentProperties *	dp= &(bd->bdProperties);

    if  ( docDrawGroupItem( docBi, through, dc ) )
	{ LDEB(1); return -1;	}

    if  ( docBi->biInExternalItem == DOCinBODY				&&
	  dp->dpEndnoteProperties.npPosition == DPftnPOS_DOC_END	)
	{
	if  ( docDrawEndnotesForDocument( through, dc ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a section and its dependants.					*/
/*									*/
/************************************************************************/

static int docDrawSectItem(	const BufferItem *		sectBi,
				void *				through,
				DrawingContext *		dc )
    {
    BufferDocument *		bd= dc->dcDocument;
    DocumentProperties *	dp= &(bd->bdProperties);

    if  ( docDrawGroupItem( sectBi, through, dc ) )
	{ LDEB(1); return -1;	}

    if  ( sectBi->biInExternalItem == DOCinBODY				&&
	  dp->dpEndnoteProperties.npPosition == DPftnPOS_SECT_END	)
	{
	if  ( docDrawEndnotesForSection( sectBi->biNumberInParent,
							    through, dc ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw an arbitrary item.						*/
/*									*/
/************************************************************************/

int docDrawItem(	BufferItem *			bi,
			void *				through,
			DrawingContext *		dc )
    {
    AppDrawingData *		add= dc->dcDrawingData;
    const DocumentSelection *	ds= dc->dcDocumentSelection;

    if  ( dc->dcClipRect )
	{
	if  ( BI_TOP_PIXELS( add, bi ) > dc->dcClipRect->drY1 )
	    { return 0;	}

	if  ( BI_BELOW_PIXELS( add, bi ) < dc->dcClipRect->drY0 )
	    { return 0; }
	}

    if  ( ds )
	{
	if  ( docCompareItemPositions( bi, ds->dsBegin.dpBi ) < 0 )
	    { return 0;	}

	if  ( docCompareItemPositions( bi, ds->dsEnd.dpBi ) > 0 )
	    { return 0; }
	}

    if  ( dc->dcFirstPage >= 0					&&
	  bi->biBelowPosition.lpPage < dc->dcFirstPage		)
	{ return 0;	}

    if  ( dc->dcLastPage >= 0					&&
	  bi->biTopPosition.lpPage > dc->dcLastPage		)
	{ return 0;	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	    if  ( docDrawDocItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevSECT:
	    if  ( docDrawSectItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevCELL:
	rowAsGroup:
	    if  ( docDrawGroupItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    if  ( docDrawRowItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevPARA:
	    if  ( docDrawParaItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }
