/************************************************************************/
/*									*/
/*  Ted: Changing font and text attributes.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"
#   include	<appFontTool.h>
#   include	<appSymbolPicker.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Change to font attributes of the current selection.			*/
/*									*/
/************************************************************************/

static void tedSetCurrentAttribute(	TedDocument *		td,
					AppDrawingData *	add,
					const PropertyMask *	updMask,
					TextAttribute		taNew )
    {
    BufferDocument *		bd= td->tdDocument;
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentFontList *		dfl= &(dp->dpFontList);

    PropertyMask		changeMask;

    PROPmaskCLEAR( &changeMask );

    docUpdateTextAttribute( &changeMask,
			    &(td->tdCurrentTextAttribute), taNew, updMask );

    if  ( ! PROPmaskISEMPTY( &changeMask )	||
	  td->tdCurrentPhysicalFont < 0		)
	{
	td->tdCurrentPhysicalFont= appOpenDocumentFont( add, dfl,
						td->tdCurrentTextAttribute );
	}

    return;
    }

static void tedAppChangeSelectionAttribute(
					EditDocument *		ed,
					const PropertyMask *	taUpdMask,
					TextAttribute		taNew )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    PropertyMask		ppUpdMask;
    PropertyMask		spUpdMask;

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskCLEAR( &spUpdMask );

    tedSetCurrentAttribute( td, add, taUpdMask, taNew );

    tedChangeSelectionProperties( ed,
			    taUpdMask, taNew,
			    &ppUpdMask, (const ParagraphProperties *)0,
			    &spUpdMask, (const SectionProperties *)0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Adapt fonts tool to the current document.				*/
/*									*/
/************************************************************************/

void tedAdaptFontsToolToDocument(	EditApplication *	ea,
					EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    AppDrawingData *	add= &(ed->edDrawingData);

    const char *	familyName= (char *)0;
    int			psFamilyNumber= -1;
    int			enc= -1;

    PropertyMask	updMask;
    TextAttribute	taNew;

    PROPmaskCLEAR( &updMask );

    if  ( tedGetDocumentAttributes( td, &familyName, &psFamilyNumber, &enc,
							&updMask, &taNew ) )
	{ return;	}

    appFontSetCurrentFont( ea->eaFontsTool, add->addMagnifiedPixelsPerTwip,
					psFamilyNumber, enc, &updMask, taNew );

    if  ( ea->eaSymbolPicker )
	{
	if  ( appAdaptSymbolPickerToFontFamily( ea->eaSymbolPicker,
						psFamilyNumber, enc ) )
	    { LLDEB(psFamilyNumber,enc);	}
	}

    appFontToolTextReadonly( ea->eaFontsTool, ed->edIsReadonly );

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Adapt fonts tool &c to the current values.			*/
/*  2)  Adapt fonts tool &c to the current selection.			*/
/*									*/
/************************************************************************/

/*  1  */
static void tedAdaptFontIndicatorsToValues(
					EditDocument *		ed,
					int			psFamilyNumber,
					int			enc,
					const PropertyMask *	updMask,
					TextAttribute		taNew )
    {
    EditApplication *	ea= ed->edApplication;
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    if  ( ea->eaFontsTool )
	{
	appFontSetCurrentFont( ea->eaFontsTool, add->addMagnifiedPixelsPerTwip,
					psFamilyNumber, enc, updMask, taNew );
	}

    if  ( ea->eaSymbolPicker )
	{
	if  ( appAdaptSymbolPickerToFontFamily( ea->eaSymbolPicker,
						psFamilyNumber, enc ) )
	    { LLDEB(psFamilyNumber,enc);	}
	}

    appGuiSetToggleItemState( td->tdFontBoldOption,
				PROPmaskISSET( updMask, TApropFONTBOLD ) &&
				taNew.taFontIsBold );

    appGuiSetToggleItemState( td->tdFontItalicOption,
				PROPmaskISSET( updMask, TApropFONTSLANTED ) &&
				taNew.taFontIsSlanted );

    appGuiSetToggleItemState( td->tdFontUnderlinedOption,
				PROPmaskISSET( updMask, TApropTEXTUNDERLINED )
				&& taNew.taIsUnderlined );

    appGuiSetToggleItemState( td->tdFontSuperscriptOption,
				PROPmaskISSET( updMask, TApropSUPERSUB ) &&
				taNew.taSuperSub == DOCfontSUPERSCRIPT );

    appGuiSetToggleItemState( td->tdFontSubscriptOption,
				PROPmaskISSET( updMask, TApropSUPERSUB ) &&
				taNew.taSuperSub == DOCfontSUBSCRIPT );

    return;
    }

/*  2  */
void tedAdaptFontIndicatorsToSelection(	EditApplication *	ea,
					EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    const char *	familyName= (char *)0;
    int			psFamilyNumber= -1;
    int			enc= -1;

    PropertyMask	updMask;
    TextAttribute	taNew;

    PROPmaskCLEAR( &updMask );

    if  ( tedGetDocumentAttributes( td, &familyName, &psFamilyNumber, &enc,
							&updMask, &taNew ) )
	{ return;	}

    tedAdaptFontIndicatorsToValues( ed, psFamilyNumber, enc, &updMask, taNew );

    return;
    }


/************************************************************************/
/*									*/
/*  One of the text attribute options has been pushed.			*/
/*									*/
/************************************************************************/

static void tedTextAttributePushed(	APP_WIDGET	attributeOption,
					void *		voided,
					void *		voidcbs,
					int		prop )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;
    int			set;
    TextAttribute	taNew;

    PropertyMask	updMask;

    PROPmaskCLEAR( &updMask );
    docInitTextAttribute( &taNew );

    set= appGuiGetMenuToggleStateFromCallback( attributeOption, voidcbs );

    switch( prop )
	{
	case TApropFONTBOLD:
	    taNew.taFontIsBold= ( set != 0 );
	    PROPmaskADD( &updMask, prop );
	    break;
	case TApropFONTSLANTED:
	    taNew.taFontIsSlanted= ( set != 0 );
	    PROPmaskADD( &updMask, prop );
	    break;
	case TApropTEXTUNDERLINED:
	    taNew.taIsUnderlined= ( set != 0 );
	    PROPmaskADD( &updMask, prop );
	    break;
	default:
	    LDEB(prop);
	    return;
	}

    tedAppChangeSelectionAttribute( ed, &updMask, taNew );

    if  ( ea->eaFontsTool )
	{ tedAdaptFontsToolToDocument( ea, ed );	}

    return;
    }

int tedAppChangeSelectionAttributeString(	EditDocument *	ed,
						char *		attrString )
    {
    char *		familyName;

    TextAttribute	taNew;
    PropertyMask	updMask;

    EditApplication *	ea= ed->edApplication;

    if  ( appFontGetAttributes( attrString, &familyName, &updMask, &taNew ) )
	{ SDEB( attrString ); return -1;	}

    if  ( PROPmaskISSET( &updMask, TApropFONTFAMILY ) )
	{
	TedDocument *		td= (TedDocument *)ed->edPrivateData;
	BufferDocument *	bd= td->tdDocument;
	DocumentProperties *	dp= &(bd->bdProperties);
	DocumentFontList *	dfl= &(dp->dpFontList);

	taNew.taFontNumber= docGetFontByName( dfl, familyName, -1, 0 );
	if  ( taNew.taFontNumber < 0 )
	    { SLDEB(familyName,taNew.taFontNumber); return -1;	}
	}

    tedAppChangeSelectionAttribute( ed, &updMask, taNew );

    tedAdaptFontIndicatorsToSelection( ea, ed );

    return 0;
    }

void tedDocFontBold(		APP_WIDGET	option,
				void *		voided,
				void *		voidcbs	 )
    { tedTextAttributePushed( option, voided, voidcbs, TApropFONTBOLD ); }

void tedDocFontItalic(	APP_WIDGET	option,
			void *		voided,
			void *		voidcbs	 )
    {
    tedTextAttributePushed( option, voided, voidcbs, TApropFONTSLANTED );
    }

void tedDocFontUnderlined(	APP_WIDGET	option,
				void *		voided,
				void *		voidcbs	 )
    {
    tedTextAttributePushed( option, voided, voidcbs, TApropTEXTUNDERLINED );
    }

void tedDocFontSupersub(	APP_WIDGET		option,
				void *			voided,
				void *			voidcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    PropertyMask	updMask;
    TextAttribute	taNew;
    int			set;

    PROPmaskCLEAR( &updMask );
    PROPmaskADD( &updMask, TApropSUPERSUB );

    docInitTextAttribute( &taNew );

    set= appGuiGetToggleStateFromCallback( option, voidcbs );

    if  ( set )
	{
	if  ( option == td->tdFontSubscriptOption )
	    { taNew.taSuperSub= DOCfontSUBSCRIPT;	}

	if  ( option == td->tdFontSuperscriptOption )
	    { taNew.taSuperSub= DOCfontSUPERSCRIPT;	}
	}
    else{ taNew.taSuperSub= DOCfontREGULAR;	}

    tedAppChangeSelectionAttribute( ed, &updMask, taNew );

    appGuiSetToggleState( td->tdFontSuperscriptOption,
				PROPmaskISSET( &updMask, TApropSUPERSUB ) &&
				taNew.taSuperSub == DOCfontSUPERSCRIPT );

    appGuiSetToggleState( td->tdFontSubscriptOption,
				PROPmaskISSET( &updMask, TApropSUPERSUB ) &&
				taNew.taSuperSub == DOCfontSUBSCRIPT );

    return;
    }

/************************************************************************/
/*									*/
/*  Ted, cope with the fonts tool.					*/
/*									*/
/************************************************************************/

static void tedFontsToolClosed(		void *		voidea	)
    {
    EditApplication *	ea=	(EditApplication *)voidea;

    ea->eaFontsTool= (void *)0;
    }

static void tedFontToolSet(		void *		voidea,
					const char *	familyName,
					int		psFamilyNumber,
					int		enc,
					int		encodingIsDefault,
					int		fontBold,
					int		fontSlanted,
					int		fontSizeHalfPoints )
    {
    EditApplication *	ea= (EditApplication *)voidea;
    EditDocument *	ed= ea->eaCurrentDocument;
    TedDocument *	td;

    PropertyMask	updMask;
    TextAttribute	taNew;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    docInitTextAttribute( &taNew );
    PROPmaskCLEAR( &updMask );

    if  ( familyName )
	{
	BufferDocument *	bd= td->tdDocument;
	DocumentProperties *	dp= &(bd->bdProperties);
	DocumentFontList *	dfl= &(dp->dpFontList);

	taNew.taFontNumber= docGetFontByName( dfl, familyName,
						    enc, encodingIsDefault );
	if  ( taNew.taFontNumber < 0 )
	    { SLDEB(familyName,taNew.taFontNumber); return;	}

	PROPmaskADD( &updMask, TApropFONTFAMILY );
	}

    if  ( fontBold >= 0 )
	{
	taNew.taFontIsBold= ( fontBold != 0 );
	PROPmaskADD( &updMask, TApropFONTBOLD );
	}

    if  ( fontSlanted >= 0 )
	{
	taNew.taFontIsSlanted= ( fontSlanted != 0 );
	PROPmaskADD( &updMask, TApropFONTSLANTED );
	}

    if  ( fontSizeHalfPoints > 0 )
	{
	taNew.taFontSizeHalfPoints= fontSizeHalfPoints;
	PROPmaskADD( &updMask, TApropFONTSIZE );
	}

    tedAppChangeSelectionAttribute( ed, &updMask, taNew );

    if  ( fontBold < 0 || fontSlanted < 0 )
	{
	const char *	psFamilyNameGot;

	if  ( tedGetDocumentAttributes( td,
		&psFamilyNameGot, &psFamilyNumber, &enc, &updMask, &taNew ) )
	    { LLDEB(fontBold,fontSlanted);	}
	}

    appGuiSetToggleItemState( td->tdFontBoldOption,
			    PROPmaskISSET( &updMask, TApropFONTBOLD ) &&
			    taNew.taFontIsBold );

    appGuiSetToggleItemState( td->tdFontItalicOption,
			    PROPmaskISSET( &updMask, TApropFONTSLANTED ) &&
			    taNew.taFontIsSlanted );

    if  ( ea->eaSymbolPicker )
	{
	if  ( appAdaptSymbolPickerToFontFamily( ea->eaSymbolPicker,
						psFamilyNumber, enc ) )
	    { LLDEB(psFamilyNumber,enc);	}
	}

    return;
    }


void tedShowFontsTool(		APP_WIDGET		fontsOption,
				EditApplication *	ea,
				void *			voidcbs	 )
    {
    EditDocument *	ed= ea->eaCurrentDocument;

    if  ( ! ea->eaFontsTool )
	{
	char *			pixmapName= "tedfont";
	APP_BITMAP_IMAGE	iconPixmap= (APP_BITMAP_IMAGE)0;
	APP_BITMAP_MASK		iconMask= (APP_BITMAP_MASK)0;

	if  ( ! ed )
	    { XDEB(ed); return;	}

	if  ( appGetImagePixmap( ea, pixmapName, &iconPixmap, &iconMask )  )
	    { SDEB(pixmapName); return;	}

	ea->eaFontsTool= appMakeFontsTool( fontsOption, "tedFontTool",
		    iconPixmap, iconMask,
		    tedFontsToolClosed, tedFontToolSet, ea );
	}
    else{ appShowFontsTool( ea->eaFontsTool );	}

    if  ( ed )
	{ tedAdaptFontsToolToDocument( ea, ed );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Show or create a symbol picker.					*/
/*									*/
/************************************************************************/

static void tedSymbolPickerInsert(	void *		voidea,
					int		symbol,
					int		psFamilyNumberSet,
					char *		familyNameSet,
					int		encodingSet,
					int		encodingIsDefault )
    {
    unsigned char	fresh[2];

    EditApplication *	ea= (EditApplication *)voidea;
    EditDocument *	ed= ea->eaCurrentDocument;
    AppDrawingData *	add;
    TedDocument *	td;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;
    add= &(ed->edDrawingData);

    fresh[0]= symbol;
    fresh[1]= '\0';

    if  ( tedHasSelection( td )		&&
	  td->tdCanReplaceSelection	)
	{
	const char *	familyNameGot= (char *)0;
	int		psFamilyNumberGot= -1;
	int		encodingGot= -1;

	PropertyMask	updMask;
	TextAttribute	taNew;

	if  ( tedGetDocumentAttributes( td, &familyNameGot,
			&psFamilyNumberGot, &encodingGot, &updMask, &taNew ) )
	    { LDEB(1); return;	}

	if  ( ! PROPmaskISSET( &updMask, TApropFONTFAMILY )	||
	      psFamilyNumberSet != psFamilyNumberGot		||
	      encodingSet != encodingGot			)
	    {
	    BufferDocument *		bd= td->tdDocument;
	    DocumentProperties *	dp= &(bd->bdProperties);
	    DocumentFontList *		dfl= &(dp->dpFontList);

	    taNew.taFontNumber= docGetFontByName( dfl, familyNameSet,
					    encodingSet, encodingIsDefault );
	    if  ( taNew.taFontNumber < 0 )
		{ SLDEB(familyNameSet,taNew.taFontNumber); return;	}

	    td->tdCurrentTextAttribute.taFontNumber= taNew.taFontNumber;
	    td->tdCurrentPhysicalFont= appOpenDocumentFont( add, dfl,
						td->tdCurrentTextAttribute );
	    }

	tedAppReplaceSelection( ed, fresh, 1 );
	}

    return;
    }

static void tedSymbolPickerClosed(	void *		voidea	)
    {
    EditApplication *	ea=	(EditApplication *)voidea;

    ea->eaSymbolPicker= (void *)0;
    }

void tedDocToolInsertSymbol(	APP_WIDGET	symbolOption,
				void *		voided,
				void *		voidcbs	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    EditApplication *	ea= ed->edApplication;

    if  ( ! td->tdCanReplaceSelection )
	{ LDEB(td->tdCanReplaceSelection); return;	}

    if  ( ! ea->eaSymbolPicker )
	{
	char *			pixmapName= "tedsymbol";
	APP_BITMAP_IMAGE	iconPixmap= (APP_BITMAP_IMAGE)0;
	APP_BITMAP_MASK		iconMask= (APP_BITMAP_MASK)0;

	const char *	familyName= (char *)0;
	int		psFamilyNumber= -1;
	int		enc= -1;

	PropertyMask	updMask;
	TextAttribute	taNew;

	if  ( tedGetDocumentAttributes( td, &familyName, &psFamilyNumber, &enc,
							&updMask, &taNew ) )
	    { return;	}

	if  ( ! PROPmaskISSET( &updMask, TApropFONTFAMILY ) )
	    { XDEB(TApropFONTFAMILY); return;	}

	if  ( appGetImagePixmap( ea, pixmapName, &iconPixmap, &iconMask )  )
	    { SDEB(pixmapName); return;	}

	ea->eaSymbolPicker= appMakeSymbolPicker( symbolOption, ea,
			    "tedSymbolPicker",
			    iconPixmap, iconMask,
			    tedSymbolPickerInsert,
			    tedSymbolPickerClosed,
			    (void *)ea );

	if  ( appAdaptSymbolPickerToFontFamily( ea->eaSymbolPicker,
						psFamilyNumber, enc ) )
	    { LLDEB(psFamilyNumber,enc);	}
	}
    else{ appShowSymbolPicker( ea->eaSymbolPicker ); }

    return;
    }
