/************************************************************************/
/*									*/
/*  Ted: Table manipulation.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedEdit.h"
#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Insert a table in the document.					*/
/*									*/
/*  0)  Also expose the table borders.					*/
/*  1)  If a range of data was selected, first discard it.		*/
/*  2)  If the selection was at the end of a paragraph, try to move to	*/
/*	the beginning of the next one.					*/
/*  3)  If the IBar is in the middle of a paragraph, split it.		*/
/*	tedSplitParaContents() does the layout of both halves.		*/
/*  4)  If the paragraph is not the first one in the 'cell' that	*/
/*	contains it, split the cell.					*/
/*  5)  If the 'cell' is not the first one in the 'row' that contains	*/
/*	it, split the row.						*/
/*									*/
/************************************************************************/

static int tedInsertRows(	BufferItem *		sectBi,
				BufferItem **		pNewBi,
				const BufferItem *	refRowBi,
				const RowProperties *	rp,
				TextAttribute		ta,
				AppDrawingData *	add,
				int			n,
				int			paraNr,
				int			rows,
				EditOperation *		eo )
    {
    DocumentPosition	dp;

    int			row;
    int			firstPos= n;

    for ( row= 0; row < rows; n++, row++ )
	{
	BufferItem *		rowBi= (BufferItem *)0;

	const int		stroffFrom= 0;
	const int		sectShift= 0;
	const int		stroffShift= 0;
	const int		paraShift= rp->rpCellCount;

	rowBi= docInsertRowItem( eo->eoBd, sectBi, n, rp, ta );
	if  ( ! rowBi )
	    { XDEB(rowBi); return -1;	}

	if  ( sectBi->biInExternalItem == DOCinBODY )
	    {
	    docShiftNoteReferences( eo->eoBd, paraNr, stroffFrom,
				    sectShift, paraShift, stroffShift );
	    }

	docEditShiftReformatRangeParaNr( eo, paraNr+ 1, paraShift );

	paraNr += rp->rpCellCount;

	if  ( refRowBi						&&
	      docCopyRowColumnAttributes( rowBi, refRowBi )	)
	    { LDEB(1); return -1;	}

	docEditIncludeItemInReformatRange( eo, rowBi );
	}

    if  ( docFirstPosition( &dp, sectBi->biChildren[firstPos] ) )
	{ LDEB(firstPos);	}

    *pNewBi= dp.dpBi; return 0;
    }

int tedInsertTable(		EditDocument *		ed,
				int			rows,
				int			columns )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentProperties *	dp= &(bd->bdProperties);

    BufferItem *		newBi;
    BufferItem *		sectBi;
    BufferItem *		refRowBi;
    int				col;
    int				paraNr;

    TextAttribute		ta;

    int				wide;

    RowProperties		rp;

    EditOperation		eo;
    DocumentSelection		dsOld;
    SelectionGeometry		sg;

    BufferItem *		bi;

    DocumentSelection		dsRep;

    tedStartEditOperation( &eo, &dsOld, &sg, ed, 1 );
    bi= dsOld.dsBegin.dpBi;

    /*  0  */
    eo.eoChangedRectangle.drY0--;

    /*  1  */
    if  ( tedEditReplaceSelection( &eo, &dsRep, ed,
					    (const unsigned char *)0, 0 ) )
	{ LDEB(1); return -1;	}

    /*  2  */
    bi= dsRep.dsBegin.dpBi;
    if  ( dsRep.dsBegin.dpStroff == bi->biParaStrlen )
	{
	docNextPosition( &(dsRep.dsBegin) );
	bi= dsRep.dsBegin.dpBi;
	}

    /*  3  */
    if  ( dsRep.dsBegin.dpStroff != 0 )
	{
	int		onNewPage= 0;

	if  ( tedSplitParaContents( &eo, &newBi, bi, dsRep.dsBegin.dpStroff,
						ed, DOClevROW, onNewPage ) )
	    { LDEB(1); return -1;	}

	bi= newBi;
	}
    else{
	BufferItem *		aftBi;

	/* 4,5 */
	if  ( bi->biNumberInParent > 0			||
	      bi->biParent->biNumberInParent > 0	)
	    {
	    if  ( docSplitGroupItem( bd, &newBi, &aftBi, bi->biParent,
					bi->biNumberInParent, DOClevROW ) )
		{ LDEB(bi->biNumberInParent); return -1;	}

	    if  ( aftBi && aftBi->biParent )
		{
		docEditIncludeItemInReformatRange( &eo, aftBi->biParent );
		}
	    else{ XDEB(aftBi);	}
	    }
	}

    ta= bi->biParaParticules[0].tpTextAttribute;

    wide=   dp->dpGeometry.dgPageWideTwips;
    wide -= dp->dpGeometry.dgLeftMarginTwips;
    wide -= dp->dpGeometry.dgRightMarginTwips;

    docInitRowProperties( &rp );
    rp.rpHalfGapWidthTwips= 5* ta.taFontSizeHalfPoints;
    rp.rpLeftIndentTwips=  -5* ta.taFontSizeHalfPoints;

    for ( col= 0; col < columns; col++ )
	{
	CellProperties	cp;

	docInitCellProperties( &cp );
	cp.cpRightBoundaryTwips= ( ( col+ 1 )* wide )/ columns;

	if  ( docInsertRowColumn( &rp, col, &cp ) )
	    { LDEB(col); return -1;	}

	rp.rpHasTableParagraphs= 1;
	}

    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;
    sectBi= bi->biParent->biParent->biParent;
    paraNr= docNumberOfParagraph( bi );

    if  ( tedInsertRows( sectBi, &newBi, refRowBi,
			    &rp, ta, add,
			    bi->biParent->biParent->biNumberInParent, paraNr,
			    rows, &eo ) )
	{ LDEB(rows); return -1;	}

    docCleanRowProperties( &rp );

    tedEditFinishIBarSelection( ed, &eo, newBi, 0 );

    return 0;
    }

static int tedInsertRowsInTable(
				BufferItem *			sectBi,
				EditOperation *			eo,
				const DocumentSelection *	ds,
				const RowProperties *		rp,
				EditDocument *			ed,
				int				row,
				int				rows )
	    {
    AppDrawingData *		add= &(ed->edDrawingData);
    BufferItem *		refParaBi;

    BufferItem *		newBi;
    BufferItem *		refRowBi;
    int				paraNr;

    TextAttribute		ta;

    DocumentPosition		dpRef;

    if  ( sectBi->biChildCount == 0 )
	{ LDEB(sectBi->biChildCount); return -1;	}

    if  ( row < sectBi->biChildCount )
	{ refRowBi= sectBi->biChildren[row];				}
    else{ refRowBi= sectBi->biChildren[sectBi->biChildCount- 1];	}

    if  ( docLastPosition(  &dpRef, refRowBi ) )
	{ LDEB(1); return -1;	}

    refParaBi= ds->dsBegin.dpBi;
    ta= refParaBi->biParaParticules[0].tpTextAttribute;
    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;
    paraNr= docNumberOfParagraph( dpRef.dpBi )+ 1;

    if  ( refParaBi->biParaInTable )
	{ refRowBi= refParaBi->biParent->biParent;	}

    if  ( tedInsertRows( sectBi, &newBi, refRowBi, rp, ta, add,
						    row, paraNr, rows, eo ) )
	{ LDEB(rows); return -1;	}

    tedEditFinishIBarSelection( ed, eo, newBi, 0 );

    return 0;
    }

int tedInsertRowInTable(	EditDocument *		ed )
    {
    const RowProperties *	rp;

    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    const int			rows= 1;

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &sectBi,
						&col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, &eo, &ds, rp, ed, row, rows ) )
	{ LDEB(row); return -1;	}

    return 0;
    }

int tedAppendRowToTable(	EditDocument *		ed )
    {
    const RowProperties *	rp;

    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    const int			rows= 1;

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &sectBi,
						&col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, &eo, &ds, rp, ed, row+ 1, rows ) )
	{ LDEB(row); return -1;	}
    
    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete rows from a table.						*/
/*									*/
/*  4)  Sanity check.							*/
/*									*/
/************************************************************************/

int tedDeleteRowsFromTable(	EditDocument *		ed,
				int			delRow0,
				int			delRow1 )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		selSectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    DocumentPosition		dpNew;

    EditOperation		eo;

    int				paraNr;

    const int			stroffFrom= 0;
    const int			sectShift= 0;
    const int			stroffShift= 0;

    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &selSectBi,
						&col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    if  ( delRow0 > delRow1 )
	{ LLDEB(delRow0,delRow1); return -1;	}
    if  ( delRow0 < row0 || delRow0 > row1 )
	{ LLLDEB(row0,delRow0,row1); return -1;	}
    if  ( delRow1 < row0 || delRow1 > row1 )
	{ LLLDEB(row0,delRow1,row1); return -1;	}

    docInitDocumentPosition( &dpNew );
    if  ( docLastPosition( &dpNew, selSectBi->biChildren[delRow1] ) )
	{ LLDEB(delRow0,delRow1); return -1;	}

    paraNr= docNumberOfParagraph( dpNew.dpBi )+ 1;

    docInitDocumentPosition( &dpNew );
    if  ( docLastPosition( &dpNew, selSectBi->biChildren[delRow1] )	||
	  docNextPosition( &dpNew )					)
	{
	docInitDocumentPosition( &dpNew );
	if  ( docFirstPosition( &dpNew,
				selSectBi->biChildren[delRow0] )	||
	      docPrevPosition( &dpNew, /*lastOne=*/ 0 )			)
	    {
	    docInitDocumentPosition( &dpNew );
	    }
	}

    tedEditIncludeRowsInRedraw( &eo, ed, selSectBi, delRow0, delRow1 );

    docEditDeleteItems( &eo, selSectBi, delRow0, delRow1- delRow0+ 1 );

    /* NO! done by docEditDeleteItems()
    docEditShiftReformatRangeParaNr( eo, paraNr+ 1, -eo.eoParagraphsDeleted );
    */

    if  ( selSectBi->biInExternalItem == DOCinBODY )
	{
	docShiftNoteReferences( eo.eoBd, paraNr, stroffFrom,
				    sectShift, -eo.eoParagraphsDeleted,
				    stroffShift );
	}

    docEditIncludeItemInReformatRange( &eo, selSectBi );

    if  ( ! dpNew.dpBi )
	{
	dpNew.dpBi= docInsertEmptyParagraph( bd, selSectBi,
						td->tdCurrentTextAttribute );

	docEditIncludeItemInReformatRange( &eo, selSectBi );

	if  ( docFirstPosition( &dpNew, dpNew.dpBi ) )
	    { LDEB(1);	}
	}

    tedEditFinishIBarSelection( ed, &eo, dpNew.dpBi, dpNew.dpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert cells/columns in tables.					*/
/*									*/
/************************************************************************/

static BufferItem * tedInsertCellInRow(	BufferDocument *	bd,
					BufferItem *		rowBi,
					int			col,
					const CellProperties *	cp,
					TextAttribute		ta )
    {
    RowProperties *	rp;

    BufferItem *	cellBi;
    BufferItem *	paraBi;

    rp= &(rowBi->biRowProperties);

    if  ( docInsertRowColumn( rp, col, cp ) )
	{ LDEB(col); return (BufferItem *)0;	}

    cellBi= docInsertItem( bd, rowBi, col, DOClevCELL );
    if  ( ! cellBi )
	{ LXDEB(col,cellBi); return (BufferItem *)0;	}

    paraBi= docInsertEmptyParagraph( bd, cellBi, ta );
    if  ( ! paraBi )
	{ XDEB(paraBi); return (BufferItem *)0;	}

    paraBi->biParaInTable= 1;

    return paraBi;
    }

static int tedSplitColumnInRows(	EditDocument *	ed,
					EditOperation *	eo,
					BufferItem *	sectBi,
					int		row0,
					int		row,
					int		row1,
					int		col,
					int		after )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		newBi= (BufferItem *)0;

    BufferItem *		rowBi;

    TextAttribute		ta;

    tedEditIncludeRowsInRedraw( eo, ed, sectBi, row0, row1 );

    if  ( after )
	{ after= 1;	}

    while( row0 <= row1 )
	{
	RowProperties *		rp;

	CellProperties *	oldCp;
	CellProperties		newCp;

	int			left;
	int			middle;

	BufferItem *		cellBi;
	BufferItem *		paraBi;

	rowBi= sectBi->biChildren[row0];
	rp= &(rowBi->biRowProperties);

	oldCp= rp->rpCells+ col;

	if  ( col == 0 )
	    { left= rp->rpLeftIndentTwips;			}
	else{ left= rp->rpCells[col-1].cpRightBoundaryTwips;	}

	middle= ( rp->rpCells[col].cpRightBoundaryTwips+ left )/ 2;

	newCp= *oldCp;
	if  ( after )
	    { oldCp->cpRightBoundaryTwips= middle;	}
	else{ newCp. cpRightBoundaryTwips= middle;	}

	cellBi= rowBi->biChildren[col];
	paraBi= cellBi->biChildren[0];
	ta= paraBi->biParaParticules[0].tpTextAttribute;

	paraBi= tedInsertCellInRow( bd, rowBi, col+ after, &newCp, ta );
	if  ( row0 == row )
	    { newBi= paraBi;	}

	docEditIncludeItemInReformatRange( eo, rowBi );

	row0++;
	}

    tedEditFinishIBarSelection( ed, eo, newBi, 0 );

    return 0;
    }

int tedInsertColumnInTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    int				after= 0;

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( ed, &eo, sectBi, row0, row, row1, col, after );
    }

int tedAppendColumnToTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    int				after= 1;

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( ed, &eo, sectBi, row0, row, row1, col, after );
    }

int tedDeleteColumnsFromRows(	EditDocument *	ed,
				int		delRow0,
				int		delRow1,
				int		delCol0,
				int		delCol1 )
    {
    BufferItem *		sectBi;
    BufferItem *		rowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    DocumentPosition		dpNew;

    int				count= delCol1- delCol0+ 1;

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    if  ( docDelimitTable( ds.dsBegin.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )

    tedEditIncludeRowsInRedraw( &eo, ed, sectBi, row0, row1 );

    docInitDocumentPosition( &dpNew );
    rowBi= sectBi->biChildren[delRow1];
    if  ( delCol1 >= rowBi->biChildCount- 1			||
	  docLastPosition( &dpNew, rowBi->biChildren[delCol1] )	||
	  docNextPosition( &dpNew )				)
	{
	docInitDocumentPosition( &dpNew );
	rowBi= sectBi->biChildren[delRow0];
	if  ( delCol0 <= 0						||
	      docFirstPosition( &dpNew, rowBi->biChildren[delCol0] )	||
	      docPrevPosition( &dpNew, /*lastOne=*/ 0 )			)
	    { docInitDocumentPosition( &dpNew ); }
	}

    while( delRow0 <= delRow1 )
	{
	int		wide;
	RowProperties *	rp;

	rowBi= sectBi->biChildren[delRow0];
	rp= &(rowBi->biRowProperties);

	if  ( delCol0 == 0 )
	    { wide= rp->rpLeftIndentTwips;				}
	else{ wide= rp->rpCells[delCol0-1].cpRightBoundaryTwips;	}
	wide= rp->rpCells[delCol1].cpRightBoundaryTwips- wide;

	if  ( rp->rpCellCount <= count )
	    { LLDEB(rp->rpCellCount,count);	}
	else{
	    rp->rpCellCount -= count;

	    for ( col= delCol0; col < rp->rpCellCount; col++ )
		{
		docCleanCellProperties( &(rp->rpCells[col]) );
		rp->rpCells[col]= rp->rpCells[col+ count];
		rp->rpCells[col].cpRightBoundaryTwips -= wide;
		}

	    docEditDeleteItems( &eo, rowBi, delCol0, count );

	    docEditIncludeItemInReformatRange( &eo, rowBi );
	    }

	delRow0++;
	}

    tedEditFinishIBarSelection( ed, &eo, dpNew.dpBi, dpNew.dpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  'Set' event handler of the table tool.				*/
/*									*/
/************************************************************************/

void tedAppSetTableProperties(	EditApplication *	ea,
				const TableRectangle *	trChange,
				const RowProperties *	rpFrom )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td;
    BufferDocument *		bd;

    BufferItem *		selSectBi;

    RowProperties *		rpTo;
    const CellProperties *	cpFrom;
    CellProperties *		cpTo;

    int				rowPropertiesAlso= 0;

    DocumentSelection		dsNew;
    SelectionGeometry		sg;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;
    bd= td->tdDocument;

    docInitDocumentSelection( &dsNew );
    if  ( tedGetSelection( &dsNew, &sg, td ) )
	{ LDEB(1); return;	}

    if  ( docTableRectangleSelection( &dsNew, &selSectBi, bd, trChange ) )
	{ LDEB(1); return;	}

    if  ( trChange->trCol0 >= 0 && trChange->trCol0 == trChange->trCol1 )
	{
	int			row;

	cpFrom= rpFrom->rpCells+ trChange->trCol0;

	for ( row= trChange->trRow0; row <= trChange->trRow1; row++ )
	    {
	    rpTo= &(selSectBi->biChildren[row]->biRowProperties);
	    cpTo= rpTo->rpCells+ trChange->trCol0;

	    cpTo->cpLeftBorder= cpFrom->cpLeftBorder;
	    cpTo->cpRightBorder= cpFrom->cpRightBorder;
	    }
	}

    if  ( trChange->trRow0 >= 0 && trChange->trRow0 == trChange->trRow1 )
	{
	int			col;

	rpTo= &(selSectBi->biChildren[trChange->trRow0]->biRowProperties);
	cpFrom= rpFrom->rpCells;
	cpTo= rpTo->rpCells;

	for ( col= trChange->trCol0;
	      col <= trChange->trCol1;
	      cpFrom++, cpTo++, col++ )
	    {
	    cpTo->cpTopBorder= cpFrom->cpTopBorder;
	    cpTo->cpBottomBorder= cpFrom->cpBottomBorder;
	    }

	rowPropertiesAlso= 1;
	}

    tedChangeTableLayout( ed, selSectBi,
	    rpFrom, rowPropertiesAlso, trChange->trRow0, trChange->trRow1 );

    tedAdaptToolsToSelection( ed );

    appDocumentChanged( ed, 1 );

    return;
    }

void tedAppSetTableSelection(	EditDocument *		ed,
				const TableRectangle *	trSet )
    {
    TedDocument *		td;

    TableRectangle		tr;

    int				tableRectangle= 0;

    DocumentSelection		ds;
    SelectionGeometry		sg;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    tedGetSelection( &ds, &sg, td );

    if  ( ! tedGetTableRectangle( &tr, &ds ) )
	{ tableRectangle= 1;	}

    if  ( ! tableRectangle )
	{ LDEB(tableRectangle); return;	}

    docExpandTableRectangleToWholeTable( &tr );

    if  ( ! docIntersectTableRectangle( &tr, &tr, trSet ) )
	{ return;	}

    tedDocTableSelectTableRectangle( ed, &tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Update the of a table in response to a user action.			*/
/*									*/
/************************************************************************/

void tedChangeTableLayout(	EditDocument *		ed,
				BufferItem *		sectBi,
				const RowProperties *	rp,
				int			rowPropertiesAlso,
				int			row0,
				int			row1 )
    {
    BufferItem *		rowBi= sectBi->biChildren[row0];

    EditOperation		eo;
    DocumentSelection		ds;
    SelectionGeometry		sg;

    tedStartEditOperation( &eo, &ds, &sg, ed, 1 );

    tedEditIncludeRowsInRedraw( &eo, ed, sectBi, row0, row1 );

    while( row0 <= row1 )
	{
	CellProperties *	cp;
	int			col;

	rowBi= sectBi->biChildren[row0];

	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	if  ( rowPropertiesAlso )
	    {
	    rowBi->biRowHeightTwips= rp->rpHeightTwips;
	    rowBi->biRowKeepOnOnePage= rp->rpKeepOnOnePage;
	    rowBi->biRowIsTableHeader= rp->rpIsTableHeader;
	    }

	rowBi->biRowHalfGapWidthTwips= rp->rpHalfGapWidthTwips;
	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	cp= rp->rpCells;
	for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	    {
	    rowBi->biRowCells[col].cpRightBoundaryTwips=
						cp->cpRightBoundaryTwips;
	    }

	docEditIncludeItemInReformatRange( &eo, rowBi );

	row0++;
	}

    if  ( tedEditFinishRangeSelection( ed, &eo, &ds ) )
	{ LDEB(1);	}

    return;
    }

/************************************************************************/
/*									*/
/*  Callback for the 'DrawTableGrid' toggle.				*/
/*									*/
/************************************************************************/

void tedDocTableDrawGrid(	APP_WIDGET	option,
				void *		voided,
				void *		voidcbs )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    int				set;

    set= appGuiGetToggleStateFromCallback( option, voidcbs );

    td->tdDrawTableGrid= set;

    appDocExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    return;
    }

