// Copyright (c) 1999, 2000 David A. Bartold
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#if (TF_REAL_LIGHTING)
// Earthly Lighting (sun, moon, stars)

// Noon:
#local noon_sun_color   = <1.00, 0.95, 0.80>;
#local noon_sky_color_1 = <0.98, 0.98, 1.00>; // Low in sky
#local noon_sky_color_2 = <0.75, 0.75, 0.90>; // Mid of sky
#local noon_sky_color_3 = <0.30, 0.40, 0.60>; // High in sky
#local noon_light_color = <1.00, 0.95, 0.80>;
#local noon_apparent_brightness = 1.00;


// Sunrise/Sunset:
#local srise_sun_color   = <1.00, 0.70, 0.33>;
#local srise_sky_color_1 = <1.00, 0.70, 0.40>; // Low
#local srise_sky_color_2 = <0.85, 0.85, 0.95>; // Mid
#local srise_sky_color_3 = <0.70, 0.70, 0.95>; // High
#local srise_light_color = <1.00, 0.80, 0.70>;
#local srise_apparent_brightness = 0.90;


// Night time:
// The sun isn't visible then, but the reason why the sun's color at night
// is reddish is so that we can blend the color for sunrise/sunset.
#local night_sun_color   = <1.0000, 0.6000, 0.2330>;
#local night_sky_color_1 = <0.0010, 0.0010, 0.0010>;
#local night_sky_color_2 = <0.0005, 0.0005, 0.0005>;
#local night_sky_color_3 = <0.0005, 0.0005, 0.0005>;
#local night_light_color = <0.0000, 0.0000, 0.0000>;
#local night_apparent_brightness = 0.70;


// Other lights (These are always constant):
#local moon_color  = <0.0040, 0.0040, 0.0040>;
#local stars_color = <0.0032, 0.0032, 0.0032>;


// Sky color blender:
#local PURE_DAY_START   =  7.00;
#local PURE_DAY_STOP    = 17.00;
#local PURE_NIGHT_START = 19.00;
#local PURE_NIGHT_STOP  =  5.00;
#local SUN_RISE         =  6.00;
#local SUN_SET          = 18.00;
 

#macro Blend(MinPos,MinClr,MaxPos,MaxClr,Pos)
  MaxClr * ((Pos - MinPos) / (MaxPos - MinPos)) +
  MinClr * ((MaxPos - Pos) / (MaxPos - MinPos))
#end


#if (TF_TIME_OF_DAY >= PURE_DAY_START & TF_TIME_OF_DAY <= PURE_DAY_STOP)
#local sun_color   = noon_sun_color;
#local sky_color_1 = noon_sky_color_1;
#local sky_color_2 = noon_sky_color_2;
#local sky_color_3 = noon_sky_color_3;
#local light_color = noon_light_color;
#local apparent_brightness = noon_apparent_brightness;
#else

#if (TF_TIME_OF_DAY >= PURE_NIGHT_START | TF_TIME_OF_DAY < PURE_NIGHT_STOP)
#local sun_color   = night_sun_color;
#local sky_color_1 = night_sky_color_1;
#local sky_color_2 = night_sky_color_2;
#local sky_color_3 = night_sky_color_3;
#local light_color = night_light_color;
#local apparent_brightness = night_apparent_brightness;
#else

#if (TF_TIME_OF_DAY >= PURE_NIGHT_STOP & TF_TIME_OF_DAY <= PURE_DAY_START)

#if (TF_TIME_OF_DAY >= SUN_RISE)
#local sun_color   = Blend (SUN_RISE, srise_sun_color,   PURE_DAY_START, noon_sun_color, TF_TIME_OF_DAY);
#local sky_color_1 = Blend (SUN_RISE, srise_sky_color_1, PURE_DAY_START, noon_sky_color_1, TF_TIME_OF_DAY);
#local sky_color_2 = Blend (SUN_RISE, srise_sky_color_2, PURE_DAY_START, noon_sky_color_2, TF_TIME_OF_DAY);
#local sky_color_3 = Blend (SUN_RISE, srise_sky_color_3, PURE_DAY_START, noon_sky_color_3, TF_TIME_OF_DAY);
#local light_color = Blend (SUN_RISE, srise_light_color, PURE_DAY_START, noon_light_color, TF_TIME_OF_DAY);
#local apparent_brightness = Blend (SUN_RISE, srise_apparent_brightness, PURE_DAY_START, noon_apparent_brightness, TF_TIME_OF_DAY);
#else
// Time is between night and sunrise
#local sun_color   = Blend (PURE_NIGHT_STOP, night_sun_color, SUN_RISE, srise_sun_color, TF_TIME_OF_DAY);
#local sky_color_1 = Blend (PURE_NIGHT_STOP, night_sky_color_1, SUN_RISE, srise_sky_color_1, TF_TIME_OF_DAY);
#local sky_color_2 = Blend (PURE_NIGHT_STOP, night_sky_color_2, SUN_RISE, srise_sky_color_2, TF_TIME_OF_DAY);
#local sky_color_3 = Blend (PURE_NIGHT_STOP, night_sky_color_3, SUN_RISE, srise_sky_color_3, TF_TIME_OF_DAY);
#local light_color = Blend (PURE_NIGHT_STOP, night_light_color, SUN_RISE, srise_light_color, TF_TIME_OF_DAY);
#local apparent_brightness = Blend (PURE_NIGHT_STOP, night_apparent_brightness, SUN_RISE, srise_apparent_brightness, TF_TIME_OF_DAY);

#end // Is between sunrise and day

#else
// Then time is during sunset

#if (TF_TIME_OF_DAY >= SUN_SET)
#local sun_color   = Blend (SUN_SET, srise_sun_color, PURE_NIGHT_START, night_sun_color, TF_TIME_OF_DAY);
#local sky_color_1 = Blend (SUN_SET, srise_sky_color_1, PURE_NIGHT_START, night_sky_color_1, TF_TIME_OF_DAY);
#local sky_color_2 = Blend (SUN_SET, srise_sky_color_2, PURE_NIGHT_START, night_sky_color_2, TF_TIME_OF_DAY);
#local sky_color_3 = Blend (SUN_SET, srise_sky_color_3, PURE_NIGHT_START, night_sky_color_3, TF_TIME_OF_DAY);
#local light_color = Blend (SUN_SET, srise_light_color, PURE_NIGHT_START, night_light_color, TF_TIME_OF_DAY);
#local apparent_brightness = Blend (SUN_SET, srise_apparent_brightness, PURE_NIGHT_START, night_apparent_brightness, TF_TIME_OF_DAY);
#else
// Is between day and sunset
#local sun_color   = Blend (PURE_DAY_STOP, noon_sun_color, SUN_SET, srise_sun_color, TF_TIME_OF_DAY);
#local sky_color_1 = Blend (PURE_DAY_STOP, noon_sky_color_1, SUN_SET, srise_sky_color_1, TF_TIME_OF_DAY);
#local sky_color_2 = Blend (PURE_DAY_STOP, noon_sky_color_2, SUN_SET, srise_sky_color_2, TF_TIME_OF_DAY);
#local sky_color_3 = Blend (PURE_DAY_STOP, noon_sky_color_3, SUN_SET, srise_sky_color_3, TF_TIME_OF_DAY);
#local light_color = Blend (PURE_DAY_STOP, noon_light_color, SUN_SET, srise_light_color, TF_TIME_OF_DAY);
#local apparent_brightness = Blend (PURE_DAY_STOP, noon_apparent_brightness, SUN_SET, srise_apparent_brightness, TF_TIME_OF_DAY);

#end // Is between sunset and night

#end // Is time during sunrise

#end // Is time during pure night

#end // Is time during pure day


#macro MaxStretch(Vec)
  max (vlength (Vec * x), max (vlength (Vec * y), vlength (Vec * z)))
#end


#local eye_sensitivity = apparent_brightness / MaxStretch (stars_color + sky_color_1);


light_source
{
  <-0.75, 0.75, -0.35> * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>)

  color (sky_color_1 * 0.20 * eye_sensitivity)
  rotate (y * TF_NORTH_DIR)
  shadowless
}


#macro SphereSizeFactor()
  (abs (sin (TF_TIME_OF_DAY * 2.0 * pi / 24.0)) * 0.6 + 0.4)
#end


// Moon:
#if (TF_TIME_OF_DAY<PURE_DAY_START | TF_TIME_OF_DAY>PURE_DAY_STOP)
light_source
{
  <0.0, 4.8, 0.0> * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>)

  color  (moon_color * eye_sensitivity)
  rotate (z * TF_TIME_OF_DAY * 360.0 / 24.0)
  rotate (y * TF_NORTH_DIR)

  looks_like
  {
    sphere
    {
      <0.0, 0.0, 0.0>, 1.0

      pigment
      {
        bozo
        turbulence 0.5

        pigment_map
        {
          [0.0 color moon_color*eye_sensitivity]
          [1.0 color moon_color*0.8*eye_sensitivity]
        }
        scale 0.2
      }

      finish
      {
        ambient 1.0
        diffuse 0.0
      }

      scale (0.2 * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>) * SphereSizeFactor())
    }
  }
}


light_source
{
  <0.0, 4.8, 0.0> * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>)
  color (moon_color * 0.2 * eye_sensitivity)
}
#end


// Sun:
#if (TF_TIME_OF_DAY > PURE_NIGHT_STOP & TF_TIME_OF_DAY < PURE_NIGHT_START)

light_source
{
  <0.0, 0.75, 0.0> * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>)

  color  ((sky_color_1 * 0.15 + light_color * 0.4) * eye_sensitivity)
  rotate (y * TF_NORTH_DIR)
  shadowless
}


#local Sun_Location = vrotate (vrotate (<0.0, -5.5, 0.0> * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>), z * (TF_TIME_OF_DAY * 360.0 / 24.0)), y * TF_NORTH_DIR);


light_source
{
  Sun_Location
  color (sun_color * 0.6 * eye_sensitivity)

  spotlight
  radius    15
  falloff   20
  tightness 10

  point_at <0.0, 0.0, 0.0>

  looks_like
  {
    sphere
    {
      <0.0, 0.0, 0.0>, 1.0 * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>)
      pigment
      {
        color sun_color*eye_sensitivity
      }

      finish
      {
        ambient 1.0
        diffuse 0.0
      } 

      scale 0.5 * SphereSizeFactor()
    }
  }

  media_attenuation on
  media_interaction on
}
#end


#local P_cloud_gradient =
pigment
{
  planar
  turbulence 0.2
  omega 0.2
  pigment_map
  {
    [0.0 color sky_color_2 filter 0.8] // Mid in sky
    [1.0 color sky_color_1 filter 0.8] // Low in sky
  }
  scale 8
}


#if (TF_HAVE_SUNBEAMS)
media
{
#if (TF_FAST_SUNBEAMS)
  intervals 4
#else
  intervals 10
#end

  scattering { 2, rgb 0.045/vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>) }

  samples    1, 5
  confidence 0.99
  variance   0.001
  ratio      0.9
}
#end // TF_HAVE_SUNBEAMS


// Create some clouds:
#if (TF_HAVE_CLOUDS)
sphere
{
  <0.0, 0.0, 0.0>, 1.0

  hollow

  texture
  {
    pigment
    {
      bozo
      turbulence 0.5
      omega      0.5
      scale      0.05

      pigment_map
      {
        [0.0 P_cloud_gradient]
        [0.7 color rgbf 1.0]
        [1.0 color rgbf 1.0]
      }
    }

    finish
    {
      ambient 0.1
      diffuse 0.7
      reflection 0.0
    }

    scale <2.0, 0.2, 1.0>
    translate <clock * 10.0, 0.0, 0.0>
  }

  scale vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>) * 4.0
}
#end // TF_HAVE_CLOUDS


// Model atmosphere:
#local P_atmosphere =
pigment
{
  gradient y

  pigment_map
  {
    [0.0 color sky_color_1*eye_sensitivity]
    [0.3 color sky_color_2*eye_sensitivity]
    [1.0 color sky_color_3*eye_sensitivity]
  }

  scale 1000.0
}


// Model starred atmosphere:
#local P_starred_atmosphere =
pigment
{
  gradient y

  pigment_map
  {
    [0.0 color (stars_color + sky_color_1) * eye_sensitivity]
    [0.3 color (stars_color + sky_color_2) * eye_sensitivity]
    [1.0 color (stars_color + sky_color_3) * eye_sensitivity]
  }

  scale 1000.0
}


// Model stars and sky:
sphere
{
  <0.0, 0.0, 0.0>, 1.0
  hollow

  texture
  {
    pigment
    {
      bozo
      pigment_map
      {
        [0.00 P_starred_atmosphere]
        [0.01 P_atmosphere]
        [1.00 P_atmosphere]
      }
    }

    scale 0.001
  }

  finish
  {
    ambient 1.0
    diffuse 0.0
    reflection 0.0
  }

  scale vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>) * 7.0
}

#else /* TF_REAL_LIGHTING */


// Turn off realistic light sources to give a "toy model" look.
light_source
{
  <-0.5, 3.0, -0.5> * <TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>

  color  <1.0, 0.95, 0.8>
  rotate (y * TF_NORTH_DIR)
  shadowless
}

light_source
{
  <0.5, 3.0, 0.5> * <TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>

  color  <1.0, 0.95, 0.8>
  rotate (y * TF_NORTH_DIR)
  shadowless
}

#if (TF_HAVE_CLOUDS)
sky_sphere
{
  S_Cloud1
}
#end // TF_HAVE_CLOUDS

#end // TF_REAL_LIGHTING


#if (TF_HAVE_FOG)
fog
{
  distance   (TF_FOG_DISTANCE * vlength (<TF_X_SCALE, TF_Y_SCALE, TF_Z_SCALE>))
  color      <0.75, 0.75, 0.75 1.0>
  fog_type   2
  //fog_offset 0.072
  //fog_alt    0.050
  fog_offset TF_FOG_OFFSET
  fog_alt    (1-TF_FOG_OFFSET)*TF_Y_SCALE/(1/TF_FOG_ALT)
  turbulence 0.200
}
#end // TF_HAVE_FOG
