/*

    File: file_ogg.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <stdio.h>
#include "types.h"
#include "photorec.h"

static const char* header_check_ogg(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery);
static int data_check_ogg(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery);

const t_file_hint file_hint_ogg= {
  .extension="ogg",
  .description="OGG audio",
  .min_header_distance=0,
  .min_filesize=0,
  .max_filesize=PHOTOREC_MAX_FILE_SIZE,
  .recover=1,
  .header_check=&header_check_ogg,
  .data_check=&data_check_ogg,
  .file_check=NULL
};
static unsigned int frame_offset=0;

static const char* header_check_ogg(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery)
{
  const unsigned char ogg_header[4]= {'O','g','g','S'};
  if(file_recovery!=NULL && file_recovery->file_stat!=NULL && file_recovery->file_stat->file_hint==&file_hint_ogg)
    return NULL;
  if(memcmp(buffer,ogg_header,sizeof(ogg_header))==0)
  {
    frame_offset=0;
    return file_hint_ogg.extension;
  }
  return NULL;
}

/* http://www.ietf.org/rfc/rfc3533.txt */
static int data_check_ogg(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery)
{
  const unsigned char ogg_header[4]= {'O','g','g','S'};
  if(file_recovery->file_size==0)
    frame_offset+=buffer_size/2;
  else
    frame_offset-=buffer_size/2;
  /*
     ecrit_rapport("frame_offset=%u\n",frame_offset);
  */
  while(frame_offset+27+255<buffer_size)
  {
    if(memcmp(&buffer[frame_offset],ogg_header,sizeof(ogg_header))==0)
    {
      unsigned int number_page_segments;
      unsigned int header_size;
      unsigned int page_size;
      unsigned int i;
      number_page_segments=buffer[frame_offset+26];
      header_size = number_page_segments + 27;
      page_size=header_size;
      for(i=0;i<number_page_segments;i++)
	page_size+=buffer[frame_offset+27+i];
      /* By definition, page_size<=27+255+255*255=65307 */
      frame_offset+=page_size;
      /*
	 ecrit_rapport("+0x%x=0x%x\n",page_size,frame_offset);
      */
    }
    else
    {
      if(frame_offset>(buffer_size/2))
	frame_offset-=buffer_size/2;
      file_recovery->file_size+=frame_offset;
      frame_offset=0;
      return 2;
    }
  }
  return 1;
}

