/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue.c -- Network queue handling functions.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file nqueue.c
 * Network queue handling functions.
 *
 * This module contains routines to manage the network I/O
 * queue. Utilities to frob what events are expected on which file
 * descriptor, and to actually wait for events (or a timeout).
 */

#include "system.h"

#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <unistd.h>

#include "compat/compat.h"

#include "config.h"
#include "nqueue.h"
#include "nqueue_devpoll.h"
#include "nqueue_epoll.h"
#include "nqueue_kqueue.h"
#include "nqueue_poll.h"
#include "nqueue_select.h"
#include "options.h"
#include "types.h"
#include "thy.h"

/** Type of a function that fiddles with FDs and events.
 * See thy_nq_fd_control() for more details.
 */
typedef void (*thy_nq_fd_control_f) (int fd, thy_nq_event_t event, int set);
/** Type of a function that initialises a specific event system.
 * See thy_nq_fd_init() for more details.
 */
typedef int (*thy_nq_fd_init_f) (void);
/** Type of a function that checks wether an event has occurred on an FD.
 * See thy_nq_fd_check() for more details.
 */
typedef int (*thy_nq_fd_check_f) (int fd, thy_nq_event_t event);
/** Type of a function that waits for events.
 * See thy_nq_wait() for more details.
 */
typedef int (*thy_nq_wait_f) (long int timeout);

/** Highest numbered file descriptor in any of our sets. */
long int thy_nq_max_fd = 0;
/** Number of active FDs in our sets. */
long int thy_nq_act_fds = 0;
int *trigger; /**< FDs that are always set. */
long trigger_count = 0; /**< Pending session counter. */
int *fdstates; /**< FD status array. */

/** Pointer to a #thy_nq_fd_control_f implementation. */
static thy_nq_fd_control_f fd_control;
/** Pointer to a #thy_nq_fd_check_f implementation. */
static thy_nq_fd_check_f fd_check;
/** Pointer to a #thy_nq_wait_f implementation. */
static thy_nq_wait_f fd_wait;

long int
thy_nq_max (void)
{
  return thy_nq_max_fd;
}

long int
thy_nq_act (void)
{
  return thy_nq_act_fds;
}

int
thy_nq_has_pending (void)
{
  return (trigger_count > 0);
}

void
thy_nq_reset (void)
{
  thy_nq_max_fd = 0;
}

/** Try to initialise a given event system.
 * This function tries to initialise a given event system (aka,
 * waiter).
 *
 * @param waiter specifies which system to try.
 *
 * @returns Zero on success, -1 otherwise.
 */
static int
_thy_nq_init_try (thy_nqueue_waiter_t waiter)
{
  int i;

  switch (waiter)
    {
#if HAVE_EPOLL_CREATE
    case THY_NQUEUE_WAITER_EPOLL:
      return -1; /* epoll() is disabled for now. */
      i = (*thy_nq_init_epoll) ();
      if (i == 0)
	{
	  fd_control = thy_nq_fd_control_epoll;
	  fd_check = thy_nq_fd_check_epoll;
	  fd_wait = thy_nq_wait_epoll;
	  return 0;
	}
      break;
#endif
#if HAVE_KQUEUE
    case THY_NQUEUE_WAITER_KQUEUE:
      return -1; /* kqueue() is disabled for now. */
      i = (*thy_nq_init_kqueue) ();
      if (i == 0)
	{
	  fd_control = thy_nq_fd_control_kqueue;
	  fd_check = thy_nq_fd_check_kqueue;
	  fd_wait = thy_nq_wait_kqueue;
	  return 0;
	}
      break;
#endif
#if defined(HAVE_SYS_DEVPOLL_H) || defined(HAVE_LINUX_DEVPOLL_H)
    case THY_NQUEUE_WAITER_DEVPOLL:
      i = (*thy_nq_init_devpoll) ();
      if (i == 0)
	{
	  fd_control = thy_nq_fd_control_devpoll;
	  fd_check = thy_nq_fd_check_devpoll;
	  fd_wait = thy_nq_wait_devpoll;
	  return 0;
	}
      break;
#endif
#if HAVE_SYS_POLL_H
    case THY_NQUEUE_WAITER_POLL:
      i = (*thy_nq_init_poll) ();
      if (i == 0)
	{
	  fd_control = thy_nq_fd_control_poll;
	  fd_check = thy_nq_fd_check_poll;
	  fd_wait = thy_nq_wait_poll;
	  return 0;
	}
      break;
#endif
#if HAVE_SYS_SELECT_H
    case THY_NQUEUE_WAITER_SELECT:
      return -1; /* select() is disabled for now. */
      i = (*thy_nq_init_select) ();
      if (i == 0)
	{
	  fd_control = thy_nq_fd_control_select;
	  fd_check = thy_nq_fd_check_select;
	  fd_wait = thy_nq_wait_select;
	  return 0;
	}
      break;
#endif
    default:
      return -1;
    }
  return -1;
}

int
thy_nq_init (void)
{
  const config_t *config = config_get ();
  int i = -1;

  trigger = (int *)bhc_calloc (_THY_MAXCONN + 1, sizeof (int));
  fdstates = (int *)bhc_calloc (_THY_MAXCONN + 1, sizeof (int));

  if (config->options.waiter != THY_NQUEUE_WAITER_NONE)
    return _thy_nq_init_try (config->options.waiter);

  for (i = 0; i < THY_NQUEUE_WAITER_NONE; i++)
    if (_thy_nq_init_try ((thy_nqueue_waiter_t)i) == 0)
      return 0;

  return -1;
}

void
thy_nq_fd_control (int fd, thy_nq_event_t event, int set)
{
  (*fd_control) (fd, event, set);
}

int
thy_nq_fd_check (int fd, thy_nq_event_t event)
{
  return (*fd_check) (fd, event);
}

int
thy_nq_wait (long int timeout)
{
  return (*fd_wait) (timeout);
}
