#ifndef INDII_TINT_CLUSTERMODEL_HPP
#define INDII_TINT_CLUSTERMODEL_HPP

#include "Observable.hpp"
#include "ImageResource.hpp"
#include "ImageManipulation.hpp"
#include "ColourSpace.hpp"
#include "../cluster/DataSet.hpp"
#include "../cluster/KMeansClusterer.hpp"

#include "wx/colour.h"

#include <vector>

namespace indii {
  namespace tint {

class ClusterModelObserver;
  
/**
 * Cluster model.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * @version $Rev: 125 $
 * @date $Date: 2010-01-24 13:22:08 +0800 (Sun, 24 Jan 2010) $
 */
class ClusterModel : public Observable<ClusterModel> {
public:
  /**
   * Constructor.
   *
   * @param res Image resource.
   */
  ClusterModel(ImageResource* res);

  /**
   * Destructor.
   */
  virtual ~ClusterModel();

  /**
   * Set default parameters.
   */
  void setDefaults();

  /**
   * Set parameters for dialog.
   */
  void setForDialog();
  
  /**
   * Get number of clusters.
   *
   * @return Number of clusters.
   */
  unsigned getNumClusters();

  /**
   * Set number of clusters.
   *
   * @param k Number of clusters.
   */
  void setNumClusters(const unsigned k);

  /**
   * Is cluster assignment hard?
   *
   * @return True if clustering is hard, false otherwise.
   */
  bool isHard();

  /**
   * Set whether cluster assignment is hard or soft.
   *
   * @param hard Is cluster assignment hard?
   */
  void setHard(const bool hard = true);

  /**
   * Get number of repetitions.
   *
   * @return Number of repetitions.
   */
  unsigned getNumRepetitions();

  /**
   * Set number of clusters.
   *
   * @param reps Number of repetitions.
   */
  void setNumRepetitions(const unsigned reps);

  /**
   * Get saturation threshold.
   *
   * @return Saturation threshold.
   */
  unsigned char getSaturationThreshold();

  /**
   * Set saturation threshold.
   *
   * @param x Saturation threshold.
   */
  void setSaturationThreshold(const unsigned char x);

  /**
   * Get maximum pixels for clustering.
   *
   * @return Maximum pixels for clustering.
   */
  unsigned getMaxPixels();
  
  /**
   * Set maximum pixels for clustering.
   *
   * @param n Maximum pixels for clustering.
   */
  void setMaxPixels(const unsigned n);

  /**
   * Get saturation decay.
   *
   * @return Saturation decay.
   */
  float getSaturationDecay();

  /**
   * Set saturation decay.
   *
   * @param x Saturation decay.
   */
  void setSaturationDecay(const float x);

  /**
   * Get centroid distance decay.
   *
   * @return Centroid distance decay.
   */
  float getCentroidDecay();

  /**
   * Set centroid distance decay.
   *
   * @param x Centroid distance decay.
   */
  void setCentroidDecay(float x);

  /**
   * Get saturation softness.
   *
   * @return Saturation softness.
   */
  float getSaturationSoftness();

  /**
   * Set saturation softness.
   *
   * @param x Saturation softness.
   */
  void setSaturationSoftness(const float x);

  /**
   * Get centroid distance softness.
   *
   * @return Centroid distance softness.
   */
  float getCentroidSoftness();

  /**
   * Set centroid distance softness.
   *
   * @param x Centroid distance softness.
   */
  void setCentroidSoftness(const float x);

  /**
   * Get red channel proportion for conversion to greyscale.
   *
   * @return Red channel proportion.
   */
  float getRedMix();

  /**
   * Get green channel proportion for conversion to greyscale.
   *
   * @return Green channel proportion.
   */
  float getGreenMix();

  /**
   * Get blue channel proportion for conversion to greyscale.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix();

  /**
   * Set channel proportions for conversion to greyscale. Values needn't be
   * normalised.
   * 
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setGreyscale(const float r, const float g, const float b);

  /**
   * Is cluster shown in colour?
   *
   * @param i Cluster number.
   */
  bool isShown(const unsigned i);

  /**
   * Show (or hide) cluster.
   *
   * @param i Cluster number.
   * @param on True to show cluster in colour, false for b&w.
   */
  void show(const unsigned i, const bool on = true);

  /**
   * Show (or hide) all clusters.
   *
   * @param on True to show all clusters in colour, false for b&w.
   */
  void showAll(const bool on = true);

  /**
   * Get cluster colour.
   *
   * @param i Cluster number.
   *
   * @return Colour representing given cluster.
   */
  const wxColour& getColour(const unsigned i);
  
  /**
   * Set cluster colour.
   *
   * @param i Cluster number.
   * @param col Cluster colour.
   */
  void setColour(const unsigned i, const wxColour& col);

  /**
   * Get cluster hue rotation.
   *
   * @param i Cluster number.
   *
   * @return Hue rotation of given cluster.
   */
  float getHue(const unsigned i);
  
  /**
   * Set cluster hue rotation.
   *
   * @param i Cluster number.
   * @param x Hue rotation to set.
   */
  void setHue(const unsigned i, const float x);

  /**
   * Get cluster saturation adjustment.
   *
   * @param i Cluster number.
   *
   * @return Saturation adjustment of given cluster.
   */
  float getSat(const unsigned i);
  
  /**
   * Set cluster saturation adjustment.
   *
   * @param i Cluster number.
   * @param x Saturation adjustment to set.
   */
  void setSat(const unsigned i, const float x);

  /**
   * Get cluster lightness adjustment.
   *
   * @param i Cluster number.
   *
   * @return Lightness adjustment of given cluster.
   */
  float getLight(const unsigned i);
  
  /**
   * Set cluster lightness adjustment.
   *
   * @param i Cluster number.
   * @param x Lightness adjustment to set.
   */
  void setLight(const unsigned i, const float x);

  /**
   * Calculate foreground for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] o Output image.
   */
  void calcFg(const unsigned i, const wxRect& rect,
      const unsigned width, const unsigned height, wxImage& o);

  /**
   * Calculate foreground of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] o Output image.
   */
  void calcFg(const wxRect& rect,
      const unsigned width, const unsigned height, wxImage& o);

  /**
   * Calculate foreground of whole image.
   *
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] o Output image.
   */
  void calcFg(const unsigned width, const unsigned height,
      wxImage& o);

  /**
   * Calculate alpha channel of subimage for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] c Output channel.
   */
  void calcAlpha(const unsigned i, const wxRect& rect,
      const unsigned width, const unsigned height, channel& c);

  /**
   * Calculate alpha channel of subimage.
   *
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] c Output channel.
   */
  void calcAlpha(const wxRect& rect,
      const unsigned width, const unsigned height, channel& c);

  /**
   * Calculate alpha channel of whole image.
   *
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   * @param[out] c Output channel.
   */
  void calcAlpha(const unsigned width, const unsigned height,
      channel& c);

  /**
   * Calculate mask for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   *
   * @return Mask for the given cluster and region.
   */
  sparse_mask calcMask(const unsigned i, const wxRect& rect,
      const unsigned width = 0, const unsigned height = 0);

  /**
   * Prepare data set for clustering.
   */
  void prepare();

  /**
   * Cluster image.
   */
  void cluster();

  /**
   * Lock model. Ignore any calls to prepare() or cluster() until unlocked.
   * This is basically a hack to stop them being overcalled, lock(), change
   * a bunch of settings, unlock(), then prepare() and cluster() to keep the
   * model consistent.
   */
  void lock();

  /**
   * Unlock model.
   */
  void unlock();

private:
  /**
   * Image resources.
   */
  ImageResource* res;

  /**
   * Number of clusters.
   */
  unsigned k;

  /**
   * Is clustering hard?
   */
  bool hard;
  
  /**
   * Number of repetitions.
   */
  unsigned reps;
  
  /**
   * Saturation threshold.
   */
  unsigned char saturationThreshold;
  
  /**
   * Maximum number of pixels for clustering.
   */
  unsigned maxPixels;
  
  /**
   * Saturation decay.
   */
  float saturationDecay;

  /**
   * Centroid decay.
   */
  float centroidDecay;

  /**
   * Saturation softness.
   */
  float saturationSoftness;

  /**
   * Centroid softness.
   */
  float centroidSoftness;

  /**
   * Colour space model.
   */
  ColourSpace cs;

  /**
   * Greyscale red mixing.
   */
  float greyR;

  /**
   * Greyscale green mixing.
   */
  float greyG;

  /**
   * Greyscale green mixing.
   */
  float greyB;

  /**
   * Cluster visibility.
   */
  std::vector<bool> visible;

  /**
   * Cluster hue rotation.
   */
  std::vector<float> hues;

  /**
   * Cluster saturation adjustment.
   */
  std::vector<float> sats;

  /**
   * Cluster light adjustment.
   */
  std::vector<float> lights;

  /**
   * Cluster colours.
   */
  std::vector<wxColour> colours;

  /**
   * Data set for clustering.
   */
  indii::cluster::DataSet<float,unsigned> data;

  /**
   * Clusterer used for last clustering.
   */
  indii::cluster::KMeansClusterer<>* minClusterer;

  /**
   * Is model locked?
   */
  bool locked;

};

  }
}

inline unsigned indii::tint::ClusterModel::getNumClusters() {
  return k;
}

inline bool indii::tint::ClusterModel::isHard() {
  return hard;
}

inline unsigned indii::tint::ClusterModel::getNumRepetitions() {
  return reps;
}

inline unsigned char indii::tint::ClusterModel::getSaturationThreshold() {
  return saturationThreshold;
}

inline unsigned indii::tint::ClusterModel::getMaxPixels() {
  return maxPixels;
}

inline float indii::tint::ClusterModel::getSaturationDecay() {
  return saturationDecay;
}

inline float indii::tint::ClusterModel::getCentroidDecay() {
  return centroidDecay;
}

inline float indii::tint::ClusterModel::getSaturationSoftness() {
  return saturationSoftness;
}

inline float indii::tint::ClusterModel::getCentroidSoftness() {
  return centroidSoftness;
}

inline float indii::tint::ClusterModel::getRedMix() {
  return greyR;
}

inline float indii::tint::ClusterModel::getGreenMix() {
  return greyG;
}

inline float indii::tint::ClusterModel::getBlueMix() {
  return greyB;
}

inline bool indii::tint::ClusterModel::isShown(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return visible[i];
}

inline const wxColour& indii::tint::ClusterModel::getColour(
    const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return colours[i];
}

inline float indii::tint::ClusterModel::getHue(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return hues[i];
}

inline float indii::tint::ClusterModel::getSat(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return sats[i];
}

inline float indii::tint::ClusterModel::getLight(const unsigned i) {
  /* pre-condition */
  assert (i < k);

  return lights[i];
}

inline void indii::tint::ClusterModel::lock() {
  locked = true;
}

inline void indii::tint::ClusterModel::unlock() {
  locked = false;
}

#endif

