/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	 Department of Computer Science,
 *       The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#if __GNUC__
#define UNUSED __attribute__ ((unused)) 
#else
#define UNUSED
#endif

static char rcsid[] UNUSED = "$Id: iu.cpp 1.2 Fri, 27 Dec 1996 14:28:13 -0700 maccabe $";

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "sizedefs.h"
#include "globals.h"
#include "Instruct.h"
#include "MMU.h"
#include "sys_bus.h"
#include "RegBlock.h"
#include "IU.h"
#include "FPU.h"

extern char ErrMsg[];

// Constructor
IntegerUnit::IntegerUnit(UInt32 n_of_wins, SystemBus& s,
			 MMU& m, RegisterBlock& r)
    : mode(RESET), NWINDOWS(n_of_wins), sbus(s), mmu(m), reg(r)
{
    initialize();
}


// ** public Member functions **
	 
IntegerUnit::State IntegerUnit::IU_state(){
    return mode; 
}

void IntegerUnit::reset() {
    mode = RESET; 
    initialize();
}

// Integer Unit Register Query
//
UInt32 IntegerUnit::PC() {
    return IU_PC; 
}

UInt32 IntegerUnit::nPC() {
    return IU_nPC; 
}

UInt32 IntegerUnit::WIM() {
    //
    // unimplemented register sets always return 0's
    //  -- see page 30 of the SPARC Architecture Manual, Version 8
    //

    UInt32 mask = 0;
    mask = ~mask;	// all 1's
    mask <<= NWINDOWS;	// 0's in the NWINDOWS least significant bits
    return IU_WIM & ~mask; 
}

UInt32 IntegerUnit::Y() {
    return IU_Y; 
}

UInt32 IntegerUnit::PSR() {
    return (IU_N << 23) |  (IU_Z << 22)  |  (IU_V << 21) | (IU_C << 20)
      | (IU_EC << 13) | (IU_EF << 12) | (IU_PIL << 8) | (IU_S << 7)
      | (IU_PS << 6) | (IU_ET << 5) | (reg.CWP() & 0x1F);
}

UInt32 IntegerUnit::TBR() {
    return (TBA << 12) | (tt << 4);
}

// Integer Unit Register Modify
void IntegerUnit::PC(UInt32 value) {
    IU_PC = value;
    IU_nPC = IU_PC + 4;
}

void IntegerUnit::WIM(UInt32 value) {
    IU_WIM = value;
}

void IntegerUnit::Y(UInt32 value) {
    IU_Y = value;
}

void IntegerUnit::PSR(UInt32 value) {
    IU_N   = (value >> 23) & 1 ;
    IU_Z   = (value >> 22) & 1 ;
    IU_V   = (value >> 21) & 1 ;
    IU_C   = (value >> 20) & 1 ;
    IU_EC  = (value >> 13) & 1 ;
    IU_EF  = (value >> 12) & 1 ;
    IU_PIL = (value >>  8) & 0xF ;
    IU_S   = (value >>  7) & 1 ;
    IU_PS  = (value >>  6) & 1 ;
    IU_ET  = (value >>  5) & 1 ;
    reg.CWP(value & 0x1F) ;
}

void IntegerUnit::TBR(UInt32 value) {
    TBA = (value >> 12) & 0xfffff;
    tt  = (value >>  4) & 0xff;
}

// PSR fields query
UInt32  IntegerUnit::N() {
    return  IU_N ;
}

UInt32  IntegerUnit::Z() {
    return  IU_Z ;
}

UInt32  IntegerUnit::V() {
    return  IU_V ;
}

UInt32  IntegerUnit::C() {
    return  IU_C ;
}

UInt32  IntegerUnit::EC() {
    return  IU_EC ;
}

UInt32  IntegerUnit::EF() {
    return  IU_EF ;
}

UInt32  IntegerUnit::PIL() {
    return  IU_PIL ;
}

UInt32  IntegerUnit::S() {
    return  IU_S ;
}

UInt32  IntegerUnit::PS() {
    return  IU_PS ;
}

UInt32  IntegerUnit::ET() {
    return  IU_ET ;
}

// ** ISP functions **
int IntegerUnit::execute() {
    int mem_ref = 0;
    static int beenHere = 0;

    if(!beenHere++)
    	mode = EXECUTE;

#ifdef NOT_YET
    if(bp_reset_in == 1) {
	mode = RESET;
	return;
    } else
#endif
    if((IU_ET == 1U) &&
       ((sbus.bp_IRL() == 15) || ((unsigned)sbus.bp_IRL() > IU_PIL))) {
	trap = 1;
	interrupt_level = sbus.bp_IRL();
    }
    // next
    
    if(trap) execute_trap();

    if(mode == EXECUTE) {
	// delayed write of Integer Unit registers
	if(delayed_write_count > 0) {
	    do_pipelined_write_of_state_regs() ;
	    delayed_write_count-- ;
	}
	else {
	    do_parallel_load_of_state_regs() ;
	}
	// next

	int addr_space = (IU_S == 0) ? USER_TEXT : SUPERVISOR_TEXT ;
	Instruction instruction(mmu.read(addr_space,IU_PC));
	// next

	if((sbus.bp_memory_exception() == 1) && (IU_annul == 0)) {
	    trap = 1;
	    trapFlag[instruction_access_exception] = 1;
	} else {
	    if(IU_annul == 0) {
		mem_ref = dispatch_instruction(instruction);
		// next

		//
		// dispatch has been modified to include possible
		//   update of PC as per ISP that would normally
		//   appear here.
		//
	    } else /* IU_annul != 0 */ { 
		IU_annul = 0;
		IU_PC  = IU_nPC;
		IU_nPC = IU_nPC + 4;
	    }
	}
    }

    //
    // HACK ALERT - this way we don't have to call execute get the
    //   behavior we expect.  Does this make a difference ?
    //
    if(trap) execute_trap();
    return mem_ref;
} // end of execute


void IntegerUnit::trapFunction(int i, void (*tf)(int)) {
    Assert(i >= 0, "trap index out of range");
    Assert(i <= LAST_TRAP, "trap index out of range");

    trapFunction_[i] = tf;
}


int IntegerUnit::dispatch_instruction(const Instruction& inst) {
    int updatePC = 1;
    int trapNum;
    int mem_ref = 0;

    switch (inst.op()) {
    //
    // Instruction Format 1
    //
    case  Instruction::FORMAT1:
	call(inst);
	updatePC = 0;
	break;

    //
    // Instruction Format 2
    //
    case  Instruction::FORMAT2:
	switch (inst.op2()){
	case Instruction::UNIMP:
	    unimplemented();
	    break;

	case Instruction::Bicc:
	    branch_icc(inst);
	    updatePC = 0;
	    break;

	case Instruction::SETHI:
	    sethi(inst);
	    break;

	case Instruction::FBfcc: 
	case Instruction::CBccc:
	    unimplemented();
	    updatePC = 0;
	    break;

	default:
	    Assert(0, "unexpected format 2 value: PLEASE EMAIL isem@cs.unm.edu with this error!");
	}
    break;

    //
    // Instruction Format 3
    //
    case Instruction::FORMAT3:
    switch(inst.op3()) {
	case Instruction::ADD:
	case Instruction::ADDX:
	case Instruction::ADDcc:
	case Instruction::ADDXcc:
	    add(inst);
	    break;

	case Instruction::SUB:
	case Instruction::SUBX:
	case Instruction::SUBcc:
	case Instruction::SUBXcc:
	    subtract(inst);
	    break;

	case Instruction::OR:
	case Instruction::ORcc:
	case Instruction::ORN:
	case Instruction::ORNcc:
	case Instruction::XOR:
	case Instruction::XORcc:
	case Instruction::XNOR:
	case Instruction::XNORcc:
	case Instruction::AND:
	case Instruction::ANDcc:
	case Instruction::ANDN:
	case Instruction::ANDNcc:
	    logical(inst);
	    break;
	                    
	case Instruction::TADDcc:
	case Instruction::TADDccTV:
	    tagged_add(inst);
	    break;

	case Instruction::TSUBcc:
	case Instruction::TSUBccTV:
	    tagged_subtract(inst);
	    break;

	case Instruction::MULScc:
	    multiply_step(inst);
	    break;

	case Instruction::UMUL:
	case Instruction::UMULcc:
	case Instruction::SMUL:
	case Instruction::SMULcc:
	    multiply(inst);
	    break;

	case Instruction::UDIV:
	case Instruction::UDIVcc:
	case Instruction::SDIV:
	case Instruction::SDIVcc:
	    divide(inst);
	    break;

	case Instruction::SLL:
	case Instruction::SRL:
	case Instruction::SRA:
	    shift(inst);
	    break;

	case Instruction::RDY:
	case Instruction::RDPSR:
	case Instruction::RDWIM:
	case Instruction::RDTBR: 
	    read_state_reg(inst);
	    break;

	case Instruction::WRY:
	case Instruction::WRPSR:
	case Instruction::WRWIM:
	case Instruction::WRTBR: 
	    write_state_reg(inst);
	    break;

	case Instruction::FPop1:
	case Instruction::FPop2:
	    trapNum = fpu->dispatch_instruction(inst);
	    if( trapNum ) {
		trap = 1;
		trapFlag[ trapNum ] = 1;
	    }
	    break;

	case Instruction::CPop1:
	case Instruction::CPop2:
	    unimplemented();
	    break;

	case Instruction::JMPL:
	    jump_link(inst);
	    updatePC = 0;
	    break;

	case Instruction::RETT:
	    return_from_trap(inst);
	    updatePC = 0;
	    break;

	case Instruction::Ticc:
	    trap_icc(inst);
	    updatePC = 0;
	    break;

	case Instruction::IFLUSH:
	    unimplemented();
	    break;

	case Instruction::SAVE:
	case Instruction::RESTORE:
	    save_restore(inst);
	    break;

	default:
	    unimplemented();
	    break;
	}
    break;

    //
    // Instruction Format 3LS
    //
    case Instruction::FORMAT3LS:
	switch (inst.op3()) {
	case Instruction::LD:
	case Instruction::LDUB:
	case Instruction::LDUH:
	case Instruction::LDD:
	case Instruction::LDSB:
	case Instruction::LDSH:
	case Instruction::LDA:
	case Instruction::LDUBA:
	case Instruction::LDUHA:
	case Instruction::LDDA:
	case Instruction::LDSBA:
	case Instruction::LDSHA:  
	case Instruction::LDF:
	case Instruction::LDFSR:
	case Instruction::LDDF:
	case Instruction::LDC:
	case Instruction::LDCSR:
	case Instruction::LDDC:
	    mem_ref = 1;
	    load(inst);
	    break;

	case Instruction::ST:
	case Instruction::STB:
	case Instruction::STH:
	case Instruction::STD:
	case Instruction::STA:
	case Instruction::STBA:
	case Instruction::STHA:      
	case Instruction::STDA:
	case Instruction::STF:
	case Instruction::STFSR:
	case Instruction::STDFQ:
	case Instruction::STDF:
	case Instruction::STC:
	case Instruction::STCSR:
	case Instruction::STDCQ:
	case Instruction::STDC:  
	    mem_ref = 1;
	    store(inst); 
	    break;

	case Instruction::LDSTUB:
	case Instruction::LDSTUBA:
	    mem_ref = 1;
	    atomic_load_store(inst);
	    break;

	case Instruction::SWAP:
	case Instruction::SWAPA:
	    mem_ref = 1;
	    swap(inst);
	    break;

	default:
	    unimplemented();
	    break;
        }
    }

    if(trap == 0 && updatePC) {
	IU_PC = IU_nPC;
	IU_nPC += 4;
    }
    return mem_ref;
}


//
// see page 161 of SPARC ARM V 8
//
void IntegerUnit::execute_trap() {
    select_trap();

    if(mode != ERROR) {
	while( delayed_write_count > 0 ) {
	    do_pipelined_write_of_state_regs() ;
	    delayed_write_count-- ;
	}
	IU_ET = 0;
	IU_PS = IU_S;
	reg.CWP( (reg.CWP() - 1 + NWINDOWS) % NWINDOWS );

	// next
	if(IU_annul == 0) {
	    reg[17] = IU_PC;
	    reg[18] = IU_nPC;
	} else /* IU_annul != 0 */ {
	    reg[17] = IU_nPC;
	    reg[18] = IU_nPC + 4;
	    IU_annul = 0;
	}

	// next
	IU_S = 1;
	if(reset_trap == 0) {
	    if(sbus.bp_IRL() > 0)
		tt = 0x10 | sbus.bp_IRL();

	    IU_PC  = TBR();
	    IU_nPC = TBR() + 4;
	} else /* reset_trap == 1 */ {
	    IU_PC  = 0;
	    IU_nPC = 4;
	    reset_trap = 0;
	}
    }
}


//
// select trap -- set tt to reflect the highest priority trap 
//   see page 161 of SPARC Arch. Ref. Manual (Version 8)
//
void IntegerUnit::select_trap() {
    if(reset_trap == 1) {
	// ignore ET and leave tt unchanged
    } else if( IU_ET == 0) {
        sprintf( ErrMsg, "Trap instruction when traps are disabled" );
	mode = ERROR;
    // priority level 2
    } else if( trapFlag[data_store_error] == 1 ) tt = data_store_error;
    else if( trapFlag[instruction_access_MMU_miss] == 1 )
        tt = instruction_access_MMU_miss;
    // priority level 3
    else if( trapFlag[instruction_access_exception] == 1 ) 
        tt = instruction_access_exception;
    // priotiry level 4
    else if( trapFlag[r_register_access_error] == 1 ) 
        tt = r_register_access_error;
    // priority level 5
    else if( trapFlag[instruction_access_exception] == 1 )
        tt = instruction_access_exception;
    // priority level 6
    else if( trapFlag[privileged_instruction] == 1 )
        tt = privileged_instruction;
    // priority level 7
    else if( trapFlag[illegal_instruction] == 1 ) tt = illegal_instruction;
    // priority level 8
    else if( trapFlag[fp_disabled] == 1 ) tt = fp_disabled;
    else if( trapFlag[cp_disabled] == 1 ) tt = cp_disabled;
    else if( trapFlag[unimplemented_flush] == 1 ) tt = unimplemented_flush;
    else if( trapFlag[watch_point_detected] == 1 ) tt = watch_point_detected;
    // priority level 9
    else if( trapFlag[window_overflow] == 1 ) tt = window_overflow;
    else if( trapFlag[window_underflow] == 1 ) tt = window_underflow;
    // priority level 10
    else if( trapFlag[mem_address_not_aligned] == 1 )
        tt = mem_address_not_aligned;
    // priority level 11
    else if( trapFlag[fp_exception] == 1 ) tt = fp_exception;
    else if( trapFlag[cp_exception] == 1 ) tt = cp_exception;
    // priority level 12
    else if( trapFlag[data_access_error] == 1 ) tt = data_access_error;
    else if( trapFlag[data_access_MMU_miss] == 1 ) tt = data_access_MMU_miss;
    // priority level 13
    else if( trapFlag[data_access_exception] == 1 )
        tt = data_access_exception;
    // priority level 14
    else if( trapFlag[tag_overflow] == 1 ) tt = tag_overflow;
    // priority level 15
    else if( trapFlag[division_by_zero] == 1 ) tt = division_by_zero;
    else {
	int trap_found = 0;
	int i;

	// check for trap instructions
	for( i = trap_instruction ; i <= LAST_TRAP ; i++) {
	    if( trapFlag[i] == 1 ) {
		tt = i;
		trap_found = 1;
	    }
	}

	if( !trap_found ) {
	    // check for external interrupts
	    for( i = interrupt_level_15 ; i >= interrupt_level_1 ; i-- ) {
		if( trapFlag[i] == 1 ) {
		    tt = i;
		    trap_found = 1;
		}
	    }
	}

	if( !trap_found ) {
	    // check for implementation defined traps
	    for( i = impl_dependent_exception ; i < trap_instruction ; i++ ) {
		if( trapFlag[i] == 1 ) tt = i;
	    }
	}
    }

    // next
    trap = 0;
    for (int i = 0; i <= LAST_TRAP; i++) {
	trapFlag[i] = 0;
    }
}


void IntegerUnit::initialize() {
    IU_PC     = 0;
    IU_nPC    = IU_PC + 4;
    IU_WIM    = 1 << (NWINDOWS - 1);
    IU_Y      = 0;

    IU_EC      = 0;
    IU_EF      = 0;
    IU_PIL     = 0;
    IU_S       = 1;
    IU_PS      = 0;
    IU_ET      = 1;

    TBA     = 0;
    tt      = 0;

    do_parallel_load_of_state_regs(); // set all prime regs to initial register
	                              // value

    IU_annul		= 0;
    delayed_write_count = 0;
    interrupt_level     = IU_PIL;		/**** DO WE NEED BOTH? ****/
    reset_trap		= 0;

    // setup the Exception and Interrupt Table (this is ugly)
    //
    trap = 0;
    int i;
    for( i = 0; i <= LAST_TRAP; i++) {
	trapFlag[i] = 0;
    }

    // place instructions for clean return of C function
    //
    UInt32 jmpl = 0x81c48000;	// jmpl %r18, %r0	! old nPc
    UInt32 rett = 0x81cca004;	// rett %r18+4		! old nPc + 4

    UInt32 mem = 0;
    for (i = 0; i <= LAST_TRAP; i++) {
	mmu.write(SUPERVISOR_TEXT, mem, 0xf, jmpl);	mem += 4;
	mmu.write(SUPERVISOR_TEXT, mem, 0xf, rett);	mem += 4;
	mem += 8;
    }

    // device interrupt handlers get called before the instruction
    //   gets executed, therefore they're a little different

    jmpl = 0x81c44000;		// jmpl %r17, %r0	! old nPc
    rett = 0x81cc8000;		// rett %r18		! old nPc + 4

    mem = 0x11 << 4;
    for (i = 0x11; i <= 0x1f; i++) {
	mmu.write(SUPERVISOR_TEXT, mem, 0xf, jmpl);	mem += 4;
	mmu.write(SUPERVISOR_TEXT, mem, 0xf, rett);	mem += 4;
	mem += 8;
    }

} // end of initialize

void IntegerUnit::do_pipelined_write_of_state_regs() {
    PSR(PSRp);
    PSRp = PSRpp;
    PSRpp = PSRppp;
    PSRppp = PSRpppp;
    
    TBR(TBRp);
    TBRp = TBRpp;
    TBRpp = TBRppp;
    TBRppp = TBRpppp;
    
    WIM(WIMp);
    WIMp = WIMpp;
    WIMpp = WIMppp;
    WIMppp = WIMpppp;
    
    Y(Yp);
    Yp = Ypp;
    Ypp = Yppp;
    Yppp = Ypppp;


} // end of pipelined_write

void IntegerUnit::do_parallel_load_of_state_regs() {
    PSRp    = PSR();
    PSRpp   = PSR();
    PSRppp  = PSR();
    PSRpppp = PSR();

    WIMp    = WIM();
    WIMpp   = WIM();
    WIMppp  = WIM();
    WIMpppp = WIM();

    TBRp    = TBR();
    TBRpp   = TBR();
    TBRppp  = TBR();
    TBRpppp = TBR();

    Yp      = Y();
    Ypp     = Y();
    Yppp    = Y();
    Ypppp   = Y();


} // end of parallel_load
