// Copyright (C) 2002 Johnny Mariethoz (Johnny.Mariethoz@idiap.ch)
//                and Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "IOHtk.h"
#include "string_utils.h"

namespace Torch {

//several kind of features
static const char *pmkmap[] = {"WAVEFORM", "LPC", "LPREFC", "LPCEPSTRA", 
  "LPDELCEP", "IREFC", 
  "MFCC", "FBANK", "MELSPEC",
  "USER", "DISCRETE", 
  "ANON"};

IOHtk::IOHtk(char* file,int max_load)
{
	//header of binary file
  HTKhdr header;
	
  FILE* f=fopen(file,"r");
  if(f == NULL)
    error("Cannot open file: %s\n",file);

	//read the header
  xfread(&header.nSamples,sizeof(long),1,f);
  if (max_load > 0 && max_load < header.nSamples)
    header.nSamples = (long)max_load;
  xfread(&header.sampPeriod,sizeof(long),1,f);
  xfread(&header.sampSize,sizeof(short),1,f);
  xfread(&header.sampKind,sizeof(short),1,f);

	//set the kind of features
  kind = (char*)xalloc(64 * sizeof(char));
  kind = parmKind2Str(header.sampKind,kind);

	//vector size and number of frames
  n_lines = header.nSamples;
  n_cols =  header.sampSize/4;
	samp_period = header.sampPeriod;
	//some error check
  if(!(n_lines >= 0)||!(n_cols>0))
    error("File %s is probably not an htk file format\n",file);

	//load all or only a part of datas??
  n_lines = max_load > 0 && max_load < n_lines ? max_load : n_lines;
  int n_data = n_lines*n_cols;
  data = (real*)xalloc(n_data*sizeof(real));

	//read datas on float or double precision
#ifdef USEDOUBLE
  float* temp = (float*)xalloc(n_data*sizeof(float));
  xfread(temp, sizeof(float), n_data,f);
  for(int i=0;i<n_data;i++)
	  data[i]=(double)temp[i];
  free(temp);
#else
  xfread(data, sizeof(float), n_data,f);
#endif
  
	//get file name
  char* start = strBaseName(file);
  file_name = strRemoveSuffix(start);
  
  fclose(f);
}

void IOHtk::write(char* dir_to_save)
{
	//header of binary file
  HTKhdr header;
	
	char save_file_name[100];
	if(dir_to_save){
		sprintf(save_file_name,"%s/%s",dir_to_save,file_name);
	}else{
		sprintf(save_file_name,"./%s",file_name);
	}

  FILE* f=fopen(save_file_name,"w");
  if(f == NULL)
    error("Cannot open file in write mode: %s\n",save_file_name);

  //HTK header
	header.nSamples = n_lines;
  header.sampSize = n_cols * 4;
	header.sampPeriod = samp_period;
	header.sampKind =  str2ParmKind(kind);
  xfwrite(&header.nSamples,sizeof(long),1,f);
  xfwrite(&header.sampPeriod,sizeof(long),1,f);
  xfwrite(&header.sampSize,sizeof(short),1,f);
  xfwrite(&header.sampKind,sizeof(short),1,f);


	//load all or only a part of datas??
  int n_data = n_lines*n_cols;
	//read datas on float or double precision
#ifdef USEDOUBLE
  float* temp = (float*)xalloc(n_data*sizeof(float));
  for(int i=0;i<n_data;i++)
		temp[i] = (float)data[i];
	xfwrite(temp, sizeof(float), n_data, f);
  free(temp);
#else
	xfwrite(data, sizeof(real), n_data, f);
#endif
  
  fclose(f);
}


void IOHtk::createMaskFromParam(bool* mask){
   int nbre_coeff = 0;
   int nbre_delta = 0;
   int offset = 0;

	 //paramters name
   short param =  str2ParmKind(kind);

	 //check if this file has energy
   if(!hasEnergy(param)){
      warning("flag -rem-e without energy\n");
   }
   offset++;
   nbre_delta++;

   if(hasDelta(param)){
      offset++;
      nbre_delta++;

      if(hasAccs(param)){
         offset++;
         nbre_delta++;
      }
   }

   if(hasNulle(param))
      offset--;
   nbre_coeff = (n_cols - offset)/nbre_delta;

	 //set parmeters to remove
   if(!hasNulle(param))
      mask[nbre_coeff] = 1;

   /*if(hasDelta(param)){
      mask[2*nbre_coeff + 1] = 1;
      if(hasAccs(param))
         mask[3*nbre_coeff + 2] = 1;
   }*/
}


IOHtk::~IOHtk()
{
  free(data);
  free(file_name);
  free(kind);
}

/******************* HTK source code **********************/

char* IOHtk::parmKind2Str(ParmKind the_kind, char* buf)
{
  strcpy(buf,pmkmap[baseParmKind(the_kind)]);
  if (hasEnergy(the_kind))    strcat(buf,"_E");
  if (hasDelta(the_kind))     strcat(buf,"_D");
  if (hasNulle(the_kind))     strcat(buf,"_N");
  if (hasAccs(the_kind))      strcat(buf,"_A");
  if (hasCompx(the_kind))     strcat(buf,"_C");
  if (hasCrcc(the_kind))      strcat(buf,"_K");
  if (hasZerom(the_kind))     strcat(buf,"_Z");
  if (hasZeroc(the_kind))     strcat(buf,"_0");
  if (hasVQ(the_kind))        strcat(buf,"_V");
  return buf;
}

ParmKind IOHtk::str2ParmKind(char *str)
{
  ParmKind i = -1;
  char *s,buf[255];
  bool hasE,hasD,hasN,hasA,hasC,hasK,hasZ,has0,hasV,found;
  int len;

  hasV=hasE=hasD=hasN=hasA=hasC=hasK=hasZ=has0=false;
  strcpy(buf,str);len=strlen(buf);
  s=buf+len-2;
  while (len>2 && *s=='_') {
    switch(*(s+1)){
      case 'E': hasE = true;break;
      case 'D': hasD = true;break;
      case 'N': hasN = true;break;
      case 'A': hasA = true;break;
      case 'C': hasC = true;break;
      case 'K': hasK = true;break;
      case 'Z': hasZ = true;break;
      case '0': has0 = true;break;
      case 'V': hasV = true;break;
      default: error("str2ParmKind: unknown ParmKind qualifier %s",str);
               exit (-1);
    }
    *s = '\0';len -= 2;s -= 2;
  }
  found = false;
  do {
    s=(char*)pmkmap[++i];
    if (strcmp(buf,s) == 0) {
      found = true;
      break;
    }
  } while (strcmp("ANON",s)!=0);
  if (!found)
    return ANON;
  if (i == LPDELCEP)         /* for backward compatibility with V1.2 */
    i = LPCEPSTRA | HASDELTA;
  if (hasE) i |= HASENERGY;
  if (hasD) i |= HASDELTA;
  if (hasN) i |= HASNULLE;
  if (hasA) i |= HASACCS;
  if (hasK) i |= HASCRCC;
  if (hasC) i |= HASCOMPX;
  if (hasZ) i |= HASZEROM;
  if (has0) i |= HASZEROC;
  if (hasV) i |= HASVQ;
  return i;
}

ParmKind IOHtk::baseParmKind(ParmKind k) { return k & BASEMASK;}

/* EXPORT->HasXXXX: returns true if XXXX included in ParmKind */
bool IOHtk::hasEnergy(ParmKind k){return (k & HASENERGY) != 0;}
bool IOHtk::hasDelta(ParmKind k) {return (k & HASDELTA) != 0;}
bool IOHtk::hasAccs(ParmKind k)  {return (k & HASACCS) != 0;}
bool IOHtk::hasNulle(ParmKind k) {return (k & HASNULLE) != 0;}
bool IOHtk::hasCompx(ParmKind k) {return (k & HASCOMPX) != 0;}
bool IOHtk::hasCrcc(ParmKind k)  {return (k & HASCRCC) != 0;}
bool IOHtk::hasZerom(ParmKind k) {return (k & HASZEROM) != 0;}
bool IOHtk::hasZeroc(ParmKind k) {return (k & HASZEROC) != 0;}
bool IOHtk::hasVQ(ParmKind k)    {return (k & HASVQ) != 0;}


}

