// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef LIST_INC
#define LIST_INC

#include "general.h"

namespace Torch {

/** List structure used in all the library.
   @author Ronan Collobert (collober@iro.umontreal.ca)
 */
struct List
{
    /** Integer, which is not always used.
        Usually, if #ptr# is a table, it contains
        the number of elements in this table
    */
    int n;

    /// Pointer which can be used as you want
    void *ptr;

    /// Next node, NULL if nothing.
    List *next;
};

/// List functions
//@{

/** Add  #y# to at the end of #*x#.
    Note: if #*x# is empty, #*x# \emph{must} be NULL.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void addToList(List **x, List *y);

/** Add the node {#n#, #ptr#} at the end of #*x#.
    Note: if #*x# is empty, #*x# \emph{must} be NULL.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void addToList(List **x, int n, void *ptr);

/** Copy the content of list #from# into the list #to#.
    This supposes that each list contains a vector of real
    in each node. The number of real which will be copied
    is given by #to#.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void copyList(List* to, List* from);

/** Copy the content of list #from# into the vector #to#.
    This supposes the list contains a vector of real
    in each node. The number of real which will be copied
    is given by #from#.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void copyList(real* to, List* from);

/** Destroy the list #*x#.
    It can free the memory inside each node (in #ptr#)
    if #free_ptrs# == #true#.
    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
void freeList(List **x, bool free_ptrs=false);

//@}


}

#endif
