// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "Mat.h"

namespace Torch {

Mat::Mat(real ** ptr_, int n_rows, int n_cols)
{
  ptr = ptr_;
  m = n_rows;
  n = n_cols;
  ptr_is_allocated = false;
  base = NULL;
}

Mat::Mat(int n_rows, int n_cols)
{
  m = n_rows;
  n = n_cols;
  base = (real *) xalloc(sizeof(real) * m * n);
  ptr = (real **) xalloc(sizeof(real *) * m);
  for (int i = 0; i < m; i++)
    ptr[i] = base + i * n;
  ptr_is_allocated = true;
}

void Mat::copy(Mat * mat)
{
  if (mat == this)
    return;

  for (int i = 0; i < m; i++)
  {
    real *ptr_r = mat->ptr[i];
    real *ptr_w = ptr[i];
    for (int j = 0; j < n; j++)
      *ptr_w++ = *ptr_r++;
  }
}

void Mat::zero()
{
  for (int i = 0; i < m; i++)
  {
    real *ptr_w = ptr[i];
    for (int j = 0; j < n; j++)
      *ptr_w++ = 0.;
  }
}

real Mat::norm1()
{
  real max_val = 0.;
  for (int j = 0; j < n; j++)
  {
    real sum = 0.0;
    for (int i = 0; i < m; i++)
      sum += fabs(ptr[i][j]);

    if (max_val < sum)
      max_val = sum;
  }

  return max_val;
}

real Mat::normFrobenius()
{
  real sum = 0.;
  for (int i = 0; i < m; i++)
  {
    real *ptr_x = ptr[i];
    for (int j = 0; j < n; j++)
    {
      real z = *ptr_x++;
      sum += z * z;
    }
  }

  return sqrt(sum);
}

real Mat::normInf()
{
  real max_val = 0.;
  for (int i = 0; i < m; i++)
  {
    real sum = 0.0;
    real *ptr_x = ptr[i];
    for (int j = 0; j < n; j++)
      sum += fabs(*ptr_x++);

    if (max_val < sum)
      max_val = sum;
  }

  return max_val;
}

Vec *Mat::getRow(int row, Vec * vec)
{
  if (!vec)
    vec = new Vec(n);

  real *ptr_w = vec->ptr;
  real *ptr_r = ptr[row];
  for (int i = 0; i < n; i++)
    *ptr_w++ = *ptr_r++;

  return (vec);
}

Vec *Mat::getCol(int col, Vec * vec)
{
  if (!vec)
    vec = new Vec(m);

  real *ptr_w = vec->ptr;
  for (int i = 0; i < m; i++)
    *ptr_w++ = ptr[i][col];

  return (vec);
}

void Mat::setRow(int row, Vec * vec)
{
  real *ptr_w = ptr[row];
  real *ptr_r = vec->ptr;
  for (int i = 0; i < n; i++)
    *ptr_w++ = *ptr_r++;
}

void Mat::setCol(int col, Vec * vec)
{
  real *ptr_r = vec->ptr;
  for (int i = 0; i < m; i++)
    ptr[i][col] = *ptr_r++;
}

Mat *Mat::subMat(int row1, int col1, int row2, int col2)
{
  real **ptr_ = (real **) xalloc(sizeof(real *) * (row2 - row1 + 1));
  for (int i = row1; i <= row2; i++)
    ptr_[i - row1] = &ptr[i][col1];

  Mat *mat = new Mat(ptr_, row2 - row1 + 1, col2 - col1 + 1);
  mat->ptr_is_allocated = true;

  return (mat);
}

Mat::~Mat()
{
  if (ptr_is_allocated)
    free(ptr);
  free(base);
}

}

