#include "tra.h"

Replica *repl;

static Path*
strtopath(char *s)
{
	char *f[20];
	int i, nf;
	Path *p;

	nf = getfields(s, f, nelem(f), 1, "/");
	p = nil;
	for(i=0; i<nf; i++)
		p = mkpath(p, f[i]);
	return p;
}

static Vtime*
strtovtime(char *s)
{
	char *f[20], *g[3];
	int i, nf;
	Vtime *v, *x;

	if(strcmp(s, "Inf")==0)
		return infvtime();

	v = mkvtime();
	nf = getfields(s, f, nelem(f), 1, ",");
	for(i=0; i<nf; i++){
		if(getfields(f[i], g, nelem(g), 1, ":/") != 3){
			freevtime(v);
			return nil;
		}
		x = mkvtime1(g[0], atoi(g[1]), atoi(g[2]));
		maxvtime(v, x);
		freevtime(x);
	}
	return v;
}

static int
parsehex(char *s, uchar *a)
{
	int c, n;

	n = 0;
	while(s[0] && s[1]){
		c = s[2];
		s[2] = '\0';
		*a++ = strtoul(s, 0, 16);
		s[2] = c;
		s += 2;
	}
	return n;
}

static int
strstate(char *s)
{
	if(strcmp(s, "Dir")==0)
		return SDir;
	if(strcmp(s, "File")==0)
		return SFile;
	if(strcmp(s, "Nonexistent")==0)
		return SNonexistent;
	if(strcmp(s, "NonreplicatedDir")==0)
		return SNonreplicated|SDir;
	if(strcmp(s, "NonreplicatedFile")==0)
		return SNonreplicated|SFile;
	return -1;
}

static Stat*
strtostat(char *str)
{
	char *f[12+1], *g[2];
	int nf;
	Stat *s;

	nf = tokenize(str, f, nelem(f));
	if(nf != 12)
		return nil;

	s = mkstat();
	if((s->state = strstate(f[0])) == -1){
	Error:
		freestat(s);
		return nil;
	}
	if((s->synctime = strtovtime(f[1])) == nil)
		goto Error;
	if((s->mtime = strtovtime(f[2])) == nil)
		goto Error;
	if((s->ctime = strtovtime(f[3])) == nil)
		goto Error;

	if(getfields(f[4], g, nelem(g), 0, "/") != 2)
		goto Error;
	s->mode = strtoul(g[0], 0, 8);
	s->localmode = strtoul(g[1], 0, 8);

	if(getfields(f[5], g, nelem(g), 0, "/") != 2)
		goto Error;
	s->uid = estrdup(g[0]);
	s->localuid = estrdup(g[1]);

	if(getfields(f[6], g, nelem(g), 0, "/") != 2)
		goto Error;
	s->gid = estrdup(g[0]);
	s->localgid = estrdup(g[1]);

	if(getfields(f[7], g, nelem(g), 0, "/") != 2)
		goto Error;
	s->muid = estrdup(g[0]);
	s->localmuid = estrdup(g[1]);

	if(getfields(f[8], g, nelem(g), 0, "/") != 2)
		goto Error;
	s->sysmtime = strtoul(g[0], 0, 0);
	s->localsysmtime = strtoul(g[1], 0, 0);

	s->length = strtoll(f[9], 0, 0);

	memset(s->sha1, 0, sizeof s->sha1);
	if(parsehex(f[10], s->sha1) < 0)
		goto Error;

	if(strlen(f[11])%2)
		goto Error;

	s->localsig.n = strlen(f[11])/2;
	s->localsig.a = emalloc(s->localsig.n);
	if(parsehex(f[11], s->localsig.a) < 0)
		goto Error;

	return s;
}

static char*
cmdaddtime(int argc, char **argv)
{
	char *e;
	Path *p;
	Vtime *st, *mt;

	if(argc != 3)
		return "usage: addsynctime path synctime mtime";

	if((p = strtopath(argv[0])) == nil && argv[0][0])
		return "bad path";
	if((st = strtovtime(argv[1])) == nil){
		freepath(p);
		return "bad vtime";
	}
	if((mt = strtovtime(argv[1])) == nil){
		freepath(p);
		freevtime(st);
		return "bad vtime";
	}
	e = nil;
	if(rpcaddtime(repl, p, st, mt) < 0)
		e = rpcerror();
	freepath(p);
	freevtime(st);
	freevtime(mt);
	return e;
}

static char*
cmdclose(int argc, char **argv)
{
	int fd;

	if(argc != 1)
		return "usage: close fd";

	fd = atoi(argv[0]);
	if(rpcclose(repl, fd) < 0)
		return rpcerror();
	return nil;
}

static char*
cmdcommit(int argc, char **argv)
{
	char *e;
	Stat *s;
	int fd;

	if(argc != 2)
		return "usage: commit fd stat";

	fd = atoi(argv[0]);
	s = strtostat(argv[1]);
	if(s == nil)
		return "bad stat";

	e = nil;
	if(rpccommit(repl, fd, s) < 0)
		e = rpcerror();
	return e;
}

static struct
{
	char *s;
	int n;
} dtab[] =
{
	{ "coverage",	DbgCoverage, },
	{ "sync",	DbgSync, },
	{ "work",	DbgWork, },
	{ "rpc",	DbgRpc, },
	{ "ignore",	DbgIgnore, },
	{ "ghost",	DbgGhost, },
	{ "db",	DbgDb, },
	{ "all",	~0, }
};

static int
safedbglevel(char *s)
{
	int i;

	for(i=0; i<nelem(dtab); i++)
		if(strcmp(dtab[i].s, s) == 0)
			return dtab[i].n;
	fprint(2, "?unknown debug level %s\n", s);
	return 3;
}

static char*
cmddebug(int argc, char **argv)
{
	char *f[30];
	int i, nf, d, dd;

	if(argc != 1)
		return "usage: debug level,...";

	nf = getfields(argv[0], f, nelem(f), 1, ",");
	d = 0;
	for(i=0; i<nf; i++){
		if((dd = safedbglevel(f[i])) == 3)
			return "bad debug level";
		d |= dd;
	}
	if(rpcdebug(repl, debug) < 0)
		return rpcerror();
	return nil;
}


static char*
cmdhangup(int argc, char **argv)
{
	USED(argv);
	if(argc != 0)
		return "usage: hangup";

	if(rpchangup(repl) < 0)
		return rpcerror();
	return nil;
}
	
static char*
cmdkids(int argc, char **argv)
{
	int i, nk;
	Kid *k;
	Path *p;

	if(argc != 1)
		return "usage: kids path";

	p = strtopath(argv[0]);
	if(p == nil && argv[0][0])
		return "bad path";
	nk = rpckids(repl, p, &k);
	freepath(p);
	if(nk < 0)
		return rpcerror();
	for(i=0; i<nk; i++)
		print("\t%s %$\n", k[i].name, k[i].stat);
	freekids(k, nk);
	return nil;
}

static char*
cmdmeta(int argc, char **argv)
{
	char *s;

	if(argc != 1)
		return "usage: meta key";

	if((s = rpcmeta(repl, argv[0])) == nil)
		return rpcerror();
	print("%s\n", s);
	free(s);
	return nil;
}

static char*
cmdmkdir(int argc, char **argv)
{
	char *e;
	Path *p;
	Stat *s;

	if(argc != 2)
		return "usage: mkdir path time";

	if((p = strtopath(argv[0])) == nil && argv[0][0])
		return "bad path";
	if((s = strtostat(argv[1])) == nil){
		freepath(p);
		return "bad vtime";
	}
	e = nil;
	if(rpcmkdir(repl, p, s) < 0)
		e = rpcerror();
	freepath(p);
	freestat(s);
	return e;
}

static char*
cmdopen(int argc, char **argv)
{
	char *e;
	int fd;
	Path *p;

	if(argc != 2)
		return "usage: open path fd";

	if((p = strtopath(argv[0])) == nil && argv[0][0])
		return "bad path";
	e = nil;
	if((fd = rpcopen(repl, p, argv[1][0])) < 0)
		e = rpcerror();
	else
		print("%d\n", fd);
	freepath(p);
	return e;
}

static char*
cmdread(int argc, char **argv)
{
	int fd;
	int n;
	char *s;

	if(argc != 2)
		return "usage: read fd n";

	fd = atoi(argv[0]);
	n = atoi(argv[1]);
	s = emalloc(n+1);
	n = rpcread(repl, fd, s, n);
	if(n < 0){
		free(s);
		return rpcerror();
	}
	s[n] = '\0';
	print("\t'%s'\n", s);
	free(s);
	return nil;
}

static char*
cmdreadhash(int argc, char **argv)
{
	int fd;
	int i, n;
	uchar *s;

	if(argc != 2)
		return "usage: readhash fd n";

	fd = atoi(argv[0]);
	n = atoi(argv[1]);
	s = emalloc(n+1);
	n = rpcreadhash(repl, fd, s, n);
	if(n < 0){
		free(s);
		return rpcerror();
	}
	print("%d\n", n);
	for(i=0; i+2+SHA1dlen<=n; i+=2+SHA1dlen)
		print("\t%11d\t%.20H\n", SHORT(s+i), s+i+2);
	free(s);
	return nil;
}

static char*
cmdreadonly(int argc, char **argv)
{
	USED(argv);
	if(argc != 1)
		return "usage: readonly ignwr";

	if(rpcreadonly(repl, atoi(argv[0])) < 0)
		return rpcerror();
	return nil;
}

static char*
cmdremove(int argc, char **argv)
{
	char *e;
	Path *p;
	Stat *s;

	if(argc != 2)
		return "usage: addsynctime path time";

	if((p = strtopath(argv[0])) == nil && argv[0][0])
		return "bad path";
	if((s = strtostat(argv[1])) == nil){
		freepath(p);
		return "bad vtime";
	}
	e = nil;
	if(rpcremove(repl, p, s) < 0)
		e = rpcerror();
	freepath(p);
	freestat(s);
	return e;
}

static char*
cmdstat(int argc, char **argv)
{
	Path *p;
	Stat *s;

	if(argc != 1)
		return "usage: stat path";

	if((p = strtopath(argv[0])) == nil && argv[0][0])
		return "bad path";
	if((s = rpcstat(repl, p)) == nil){
		freepath(p);
		return rpcerror();
	}
	print("\t%$\n", s);
	freestat(s);
	freepath(p);
	return nil;
}

static char*
cmdwrite(int argc, char **argv)
{
	int fd, n;
	char *s;

	if(argc != 2)
		return "usage: write fd string";

	fd = atoi(argv[0]);
	s = argv[1];
	if((n = rpcwrite(repl, fd, s, strlen(s))) < 0)
		return rpcerror();
	print("%d\n", n);
	return nil;
}

static char*
cmdwritehash(int argc, char **argv)
{
	int i, fd, n, nb;
	uchar *s;

	if(argc < 2)
		return "usage: writehash fd hash...";

	fd = atoi(argv[0]);
	nb = (argc-1)*SHA1dlen;
	s = emalloc(nb);
	for(i=1; i<argc; i++){
		if(parsehex(argv[i], s+(i-1)*SHA1dlen) < 0){
			free(s);
			return "bad hash";
		}
	}
	if((n = rpcwritehash(repl, fd, s, nb)) < 0){
		free(s);
		return rpcerror();
	}
	print("%d\n", n);
	free(s);
	return nil;
}

static struct {
	char *cmd;
	char *(*fn)(int, char**);
} tab[] = {
 {	"addtime",		cmdaddtime,	},
 {	"close",			cmdclose,	},
 {	"commit",			cmdcommit,	},
 {	"debug",			cmddebug, },
 {	"hangup",			cmdhangup,	},
 {	"kids",			cmdkids,	},
 {	"meta",			cmdmeta,	},
 {	"mkdir",			cmdmkdir,	},
 {	"open",			cmdopen,	},
 {	"read",			cmdread,	},
 {	"readhash",		cmdreadhash, },
 {	"readonly",		cmdreadonly,	},
 {	"remove",			cmdremove,	},
 {	"stat",			cmdstat,	},
 {	"write",			cmdwrite,	},
 {	"writehash",		cmdwritehash, },
};

void
usage(void)
{
	fprint(2, "usage: testsrv replica-cmd\n");
	exits("usage");
}

void
threadmain(int argc, char **argv)
{
	int i, n;
	char *e, *tok[16], buf[512];

	fmtinstall('H', encodefmt);
	fmtinstall('P', pathfmt);
	fmtinstall('R', rpcfmt);
	fmtinstall('$', statfmt);
	fmtinstall('V', vtimefmt);

	ARGBEGIN{
	default:
		usage();
	}ARGEND

	if(argc != 1)
		usage();

	repl = dialreplica(argv[0]);
	if(clientbanner(repl, argv[0]) < 0)
		sysfatal("%s handshake: %r", argv[0]);

	threadcreate(replthread, repl);
	startclient();
	for(;;){
		print(">>> ");
		n = read(0, buf, sizeof buf);
		if(n <= 0)
			exits(nil);
		buf[n-1] = '\0';
		n = tokenize(buf, tok, nelem(tok));
		if(n <= 0)
			continue;
		for(i=0; i<nelem(tab); i++)
			if(strcmp(tok[0], tab[i].cmd) == 0){
				e = tab[i].fn(n-1, tok+1);
				if(e)
					print("?%s\n", e);
				break;
			}
		if(i==nelem(tab))
			print("?unknown command\n");
	}
	replclose(repl);
	exits(nil);
}
