
/* game.cc
   Collects information about the current game

   Copyright (C) 2000  Mathias Broxvall

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "general.h"
#include "game.h"
#include "map.h"
#include "player.h"
#include "animated.h"
#include <iterator>
#include "guile.h"
#include "gameHook.h"
#include "gamer.h"
#include "forcefield.h"
#include "settings.h"
#include "pipe.h"
#include "weather.h"

using namespace std;

Game *Game::current=NULL;

extern GLfloat colors[5][3];

Game::Game(char *name,Gamer *g) {
  Ball::reset();
  ForceField::reset();
  Pipe::reset();

  current = this;  
  objects = new set<class Animated *>();
  hooks = new set<class GameHook *>();
  map = NULL;
  player1 = NULL;
  gameTime = 0.0;
  nextLevel[0]=0;
  setDefaults();
  gamer = g;

  localPlayers=1;

  weather = new Weather();

  /* Load the bootup script */
  char scmname[256];
  sprintf(scmname,"%s/levels/boot.scm",SHARE_DIR);
  gh_eval_file(scmname);

  player1 = new Player(gamer);
  loadLevel(name);
  player1->restart(Game::current->map->startPosition);
  player1->timeLeft = startTime;
  player1->lives = 4 - Settings::settings->difficulty;

}

Game::~Game() {
  clearLevel();

  delete player1;
  delete map;
  delete objects;
  delete hooks;
  delete weather;
}

void Game::loadLevel(char *name) {
  char mapname[256];
  char scmname[256];

  if(player1) {
	// level scripts might have modified our appearance. Reset them
	for(int i=0;i<3;i++) 
	  player1->primaryColor[i] = colors[gamer->color][i];
	player1->texture = textures[gamer->textureNum];
  }

  setDefaults();  
  snprintf(mapname,sizeof(mapname)-1,"%s/.trackballs/levels/%s.map",getenv("HOME"),name);
  snprintf(scmname,sizeof(scmname)-1,"%s/.trackballs/levels/%s.scm",getenv("HOME"),name);
  if(!fileExists(mapname))
	sprintf(mapname,"%s/levels/%s.map",SHARE_DIR,name);
  if(!fileExists(scmname))
	sprintf(scmname,"%s/levels/%s.scm",SHARE_DIR,name);
  sprintf(levelName,"%s",name);

  if(map) delete map;
  map = new Map(mapname);
  gh_eval_file(scmname);

  if(player1)
    player1->timeLeft = startTime;

  fogThickness=wantedFogThickness;
}
void Game::setDefaults() {
  isNight = 0;
  fogThickness=0.0; wantedFogThickness=0.0;
  startTime = 0;
  fogColor[0] = fogColor[1] = fogColor[2] = fogColor[3] = 1.0;
  useGrid=1;
  jumpFactor=1.0;
  oxygenFactor=1.0;
  restartBonusTime = 45.0;
}

void Game::clearLevel() {
  Ball::reset();	  
  ForceField::reset();

  weather->clear();
  if(hooks) {
    set<GameHook*> *old_hooks = new set<GameHook*>(*hooks);
    set<GameHook*>::iterator ih = old_hooks->begin();
    set<GameHook*>::iterator endh = old_hooks->end();
    
    for(;ih!=endh;ih++) 
      if(*ih != player1)
	(*ih)->remove();
    delete old_hooks;
    GameHook::deleteDeadObjects();
    hooks->clear();
    objects->clear();

    hooks->insert(player1);
    objects->insert(player1);
  }
}

void Game::tick(Real t) {
  gameTime += t;
  if(fogThickness<wantedFogThickness) fogThickness += min(0.3*t,wantedFogThickness-fogThickness);
  if(fogThickness>wantedFogThickness) fogThickness -= min(0.3*t,fogThickness-wantedFogThickness);

  set<GameHook*> *old_hooks = new set<GameHook*>(*hooks);
  set<GameHook*>::iterator ih = old_hooks->begin();
  set<GameHook*>::iterator endh = old_hooks->end();

  for(;ih!=endh;ih++) (*ih)->tick(t);
  delete old_hooks;

  weather->tick(t);
}
void Game::draw() {
  /* Some GL defaults the objects can rely on being setup */
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glDepthFunc(GL_LEQUAL);

  set<Animated*>::iterator i = objects->begin();
  set<Animated*>::iterator end = objects->end();
  for(;i!=end;i++) (*i)->draw();
  glEnable(GL_ALPHA_TEST);
  glAlphaFunc(GL_GREATER,0.01); // fixes some problems with overlapping translucent graphics
  i = objects->begin();
  for(;i!=end;i++) (*i)->draw2();

  weather->draw2();
}
void Game::add(Animated *a) {
  objects->insert(a);
}
void Game::add(GameHook *a) {
  hooks->insert(a);
}
void Game::remove(Animated *a) {
  objects->erase(a);
  //  printf("game::remove after - num objects is %d\n",objects->size());
}
void Game::remove(GameHook *a) {
  hooks->erase(a);
}
