/* main.cc
   Contains the main of the game. See README for a description of the game.

   Copyright (C) 2000  Mathias Broxvall
   		       Yannick Perret

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

char *SHARE_DIR_DEFAULT=SHARE_DIR;
#include "general.h"
#include "gameMode.h"
#include "mainMode.h"
#include "editMode.h"
#include "guile.h"
#include "ball.h"
#include "font.h"
#include "glHelp.h"
#include "sound.h"
#include "menuMode.h"
#include "enterHighScoreMode.h"
#include "highScore.h"
#include <getopt.h>
#include "game.h"
#include "forcefield.h"
#include "hofMode.h"
#include <SDL/SDL_image.h>
#include <unistd.h>
#include <settingsMode.h>
#include <settings.h>
#include <setupMode.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/time.h>
#include "pipe.h"
#include "pipeConnector.h"
#include "helpMode.h"
#include "map.h"
#include "calibrateJoystickMode.h"

using namespace std;

/* Important globals */
SDL_Surface *screen;
const char* program_name;
int silent=0;
int debug_joystick,repair_joystick;

char effectiveShareDir[256];
int screenResolutions[4][2] = {{640,480}, {800,600}, {1024,768}, {1280,1024}}, nScreenResolutions=4;

void changeScreenResolution() {
  screenWidth = screenResolutions[Settings::settings->resolution][0]; 
  screenHeight = screenResolutions[Settings::settings->resolution][1];

  if(Settings::settings->colorDepth == 16 || 1) {
	SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 5);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 5);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 5);
  } else {
	SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
  }
  SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
  SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER,1);

  screen = SDL_SetVideoMode(screenResolutions[Settings::settings->resolution][0], 
							screenResolutions[Settings::settings->resolution][1], 
							Settings::settings->colorDepth, 
							SDL_SWSURFACE | SDL_OPENGL | (Settings::settings->is_windowed?0:SDL_FULLSCREEN));
  screenWidth = screen->w; screenHeight = screen->h;
  SDL_WarpMouse(screenWidth/2,screenHeight/2);	
}

void print_usage (FILE* stream, int exit_code) {
       fprintf (stream, "Usage: %s [-w, -m] [-e, -l -t <level>] [-r <width>] [-s <sensitivity>]\n", program_name);
       fprintf (stream, "   -h  --help         Display this usage information.\n"
				"   -e  --edit         Start as level editor.\n"
				"   -l  --level        Start from level.\n"
				"   -w  --windowed     Run in window (Default is fullscreen)\n"
				"   -m  --mute         Mute sound.\n"
				"   -r  --resolution   Set resolution to 640, 800 or 1024\n"
				"   -s  --sensitivity  Mouse sensitivity, default 1.0\n"
				"   -f  --fps          Displays framerate\n"
				"   -q  --quiet        Do not print anything to stdout\n"
				"   -v  --version      Prints current version number\n"
				"   -t  --touch        Updates a map to the latest format\n"
				"   -y  --low-memory   Attempt to conserve memory usage\n"
				"   -j  --repair-joystick Correct for bad joysticks\n"
			);
       exit (exit_code);
}

int testDir() {
  if(!strlen(effectiveShareDir)) return 0;
  DIR *dir = opendir(effectiveShareDir);
  //printf("Looking for %s\n", effectiveShareDir);
  if(!dir) return 0;
  else closedir(dir);
  char str[256];
  sprintf(str,"%s/levels",effectiveShareDir);
  dir=opendir(str);
  if(!dir) return 0;
  else closedir(dir);
  /* TODO. Test for all other essential subdirectories */
  return 1;
}

void innerMain(void *closure,int argc,char **argv) { 
  int is_running=1;
  int editMode=0,touchMode=0;
  int audio=SDL_INIT_AUDIO;  
  SDL_Event event;
  char str[256],*touchName;
  int i;

  const char* const short_options = "he:l:t:wmr:s:fqvyj";
  const struct option long_options[] = {
         { "help",     0, NULL, 'h' },
         { "edit",     1, NULL, 'e' },
         { "level",    1, NULL, 'l' },
         { "windowed", 0, NULL, 'w' },
         { "mute",     0, NULL, 'm' },
		 { "resolution", 1, NULL, 'r'},
		 { "sensitivity", 1, NULL, 's'},
		 { "fps", 0, NULL, 'f'},
		 { "quiet", 0, NULL, 'q'},
		 { "version", 0, NULL, 'v'},
		 { "touch",   1, NULL, 't'},
		 { "low-memory",   0, NULL, 'y'},
		 { "debug-joystick", 0, NULL, '9' },
		 { "repair-joystick", 0, NULL, 'j' },
         { NULL,       0, NULL, 0   }
  };
  int next_option;
  
  program_name = argv[0];
 
  Settings::init();
  Settings *settings = Settings::settings;
  settings->doSpecialLevel=0;
  low_memory=0; debug_joystick=0; repair_joystick=0;
  do {
#ifdef solaris
	next_option = getopt(argc,argv,short_options);
#else
	next_option = getopt_long (argc, argv, short_options, 
	long_options, NULL);
#endif
	switch (next_option) {
	case 'h':
	  print_usage (stdout, 0);
	case 'e':
	  editMode = 1;
	  sprintf(Settings::settings->specialLevel,"%s",optarg);
	  Settings::settings->doSpecialLevel=1;
	  break;
	case 't':
	  touchMode = 1;
	  touchName=optarg;
	  audio=0; // no audio
	  break;
	case 'l':
	  sprintf(Settings::settings->specialLevel,"%s",optarg);
	  Settings::settings->doSpecialLevel=1;
	  break;
	case 'w': settings->is_windowed=1;  break;
	case 'm':
	  audio = 0;
	  break;
	case 'r':
	  for(i=0;i<nScreenResolutions;i++) if(screenResolutions[i][0] == atoi(optarg)) break;
	  if(i<nScreenResolutions) settings->resolution=i;
	  else { printf("Unknown screen resolution of width %d\n",i); }
	  break;
	case 's':
	  Settings::settings->mouseSensitivity = atof(optarg);
	  break;
	case 'f':
	  Settings::settings->showFPS=1;
	  break;
	case '?':
	  print_usage (stderr, 1);
	case -1:
	  break;
	case 'q': silent=1;
	  break;
	case 'v': printf("%s v%s\n", PACKAGE, VERSION); exit(0); break;
	case 'y': low_memory=1; break;
	case '9': debug_joystick=1; break;
	case 'j': repair_joystick=1; break;
	default:
	  abort ();
	}
  }
  while (next_option != -1);
  
  if(!silent) {
	printf("Welcome to Trackballs. \n");
	printf("Using %s as gamedata dir\n",SHARE_DIR);
  }

  /* Initialize SDL */
  if((SDL_Init(SDL_INIT_VIDEO|audio|SDL_INIT_JOYSTICK)==-1)) {
	printf("failed!\n"); fflush(stdout);
	printf("Could not initialize SDL: %s.\n", SDL_GetError());
	exit(-1);
  }  
  atexit(SDL_Quit);

  char buffer[1024];
  sprintf(buffer, "/%s/ V%s", PACKAGE, VERSION);
  SDL_WM_SetCaption(buffer, buffer);

  sprintf(str,"%s/icons/trackballs-32x32.png",SHARE_DIR);
  SDL_Surface *wmIcon = IMG_Load(str);
  if(wmIcon) {  SDL_WM_SetIcon(wmIcon, NULL); }

  // MB: Until here we are using 7 megs of memory
  changeScreenResolution();
  // MB: Until here we are using 42 megs of memory

  SDL_ShowCursor(SDL_DISABLE);
  // set the name of the window

  double bootStart=((double)SDL_GetTicks()) /1000.0 ;
  sprintf(str,"%s/images/splashScreen.jpg",SHARE_DIR);
  SDL_Surface *splashScreen = IMG_Load(str);
  if(!splashScreen) { printf("Error: Failed to load %s\n",str); exit(0); }
  glViewport(0,0,screenWidth,screenHeight);
  glColor3f(1.0,1.0,1.0);
  drawSurface(splashScreen,0,0,screenWidth,screenHeight);
  SDL_GL_SwapBuffers();
  drawSurface(splashScreen,0,0,screenWidth,screenHeight);
  SDL_GL_SwapBuffers();

  // MB: Until here we are using 47 megs of memory. 
  // splashscreen is using 5 megs but it is ok since we are freeing it before the real game


  /* Initialize all modules */
  initGuileInterface();  // MB: 0 megs
  generalInit();         // MB: 0 megs
  glHelpInit();          // MB: 1.5 megs

  // MB: Until here we are using 49 megs of memory. 

  if(audio != 0) soundInit();
  Settings::settings->loadLevelSets();

  // MB: Until here we are using 51 megs of memory. 

  /* Initialize and activate the correct gameModes */
  if(touchMode) { 
	char mapname[512];
	
	snprintf(mapname,sizeof(mapname)-1,"%s/.trackballs/levels/%s.map",getenv("HOME"),touchName);
	if(!fileExists(mapname))
	  sprintf(mapname,"%s/levels/%s.map",SHARE_DIR,touchName);
	if(!fileExists(mapname))
	  snprintf(mapname,511,"%s",touchName);
	printf("Touching map %s\n",mapname);
	Map *map = new Map(mapname);
	map->save(mapname,(int) map->startPosition[0],(int) map->startPosition[1]);
	exit(0);
  } else if(editMode) {
	Font::init();
    GameMode::activate(new EditMode());
  } else {

	// MB: Reminder 51 megs it was
    	Font::init();	       // MB: Until here we are using 54 megs of memory. 
	MenuMode::init();      // MB: Until here we are using 55 megs of memory. 

	MainMode::init();
	HighScore::init();     // MB: 55

	EnterHighScoreMode::init(); // MB: 58
	HallOfFameMode::init();     // MB: 62
	SettingsMode::init();       // MB: 64	
    GameMode::activate(MenuMode::menuMode);	// MB: 65
	GameHook::init();
	CalibrateJoystickMode::init();

	SetupMode::init();          // MB: 71
	Ball::init();
	ForceField::init();         // MB: 71
	HelpMode::init();           // MB: 74
	Pipe::init();
	PipeConnector::init();
	volumeChanged();
	
	// Until here 74 megs

  }
  if(!silent) printf("Trackballs initialization successfull\n");


  /* Make sure splahsscreen has been shown for atleast 2.5 seconds */
  double timeNow=((double)SDL_GetTicks()) /1000.0 ;
  while(timeNow < bootStart + 2.5) {
	drawSurface(splashScreen,0,0,screenWidth,screenHeight);
	SDL_GL_SwapBuffers();
        timeNow=((double)SDL_GetTicks()) /1000.0 ;
  }
  SDL_FreeSurface(splashScreen);

  /*                 */
  /* Main event loop */
  /*                 */

  double oldTime,t,td;
  oldTime=((double)SDL_GetTicks()) /1000.0 ;
  SDL_WarpMouse(screenWidth/2,screenHeight/2);
  while(is_running) {
	t=((double)SDL_GetTicks()) /1000.0 ;
	td = t - oldTime;
	oldTime = t;
	// reduced to 5 fps
	if(td > 0.2) td = 0.2;
	// we may also add a bottom limit to 'td' to prevent
	//  precision troubles in physic computation
	/*
	// limited to 75 fps
	if (td < 0.013) td = 0.013;
	*/

	/* update font system */
	Font::tick(td);

	/* Update world */
	if(GameMode::current) GameMode::current->idle(td);	

	/* Make sure music is still playing */
	soundIdle();

	/* Draw world */
	glViewport(0,0,screenWidth,screenHeight);
	if(GameMode::current) GameMode::current->display();
	else {
	  glClear(GL_COLOR_BUFFER_BIT);
	}
	//Font::draw();
	SDL_GL_SwapBuffers();

	/* Process events */
	SDL_MouseButtonEvent *e=(SDL_MouseButtonEvent*)&event;
	while(SDL_PollEvent(&event)) 
	  switch(event.type) {
	  case SDL_QUIT: is_running=0; break;
	  case SDL_MOUSEBUTTONDOWN:
		if(GameMode::current) GameMode::current->mouseDown(e->button,e->x,e->y);
		break;
	  case SDL_KEYDOWN:
		if(event.key.keysym.sym == SDLK_ESCAPE) {
		  if(editMode) {
			((EditMode*)GameMode::current)->askQuit();
		  } else if((GameMode::current && GameMode::current == MenuMode::menuMode))
			is_running=0;
		  else GameMode::activate(MenuMode::menuMode);
		}
		else if(GameMode::current) GameMode::current->key(event.key.keysym.sym);
		break;
	  }
  }

  Settings::settings->closeJoystick();
  Settings::settings->save();
  if(GameMode::current)
	delete(GameMode::current); 

  SDL_Quit();
}

int main(int argc,char **argv) {
  int i;

  char guileLoadPath[256+16];/*longest effective share directory plus"GUILE_LOAD_PATH="*/

  program_name = argv[0];

  /*** Autmatic detection of SHARE_DIR ***/
  effectiveShareDir[0]=0;
  /* From environment variable */
  char *evar=getenv("TRACKBALLS"); 
  if(evar && strlen(evar) > 0) sprintf(effectiveShareDir,"%s",evar);
  //printf("Looking for %s\n", effectiveShareDir);
  if(!testDir()) {     
       char thisDir[256];
       /* From arg0/share/trackballs  */
       sprintf(thisDir,"%s",program_name);
       for(i=strlen(thisDir)-1;i>=0;i--)
         if(thisDir[i] == '/'
#ifdef WIN32
            || thisDir[i] == '\\'
#endif
         ) break;
       thisDir[i] = 0;

       /*If no directory breaks are found just use the current directory*/
       if( i == 0 ) sprintf(thisDir,".");

       sprintf(effectiveShareDir,"%s/share/trackballs",thisDir);

       if(!testDir()) {
         /* From arg0/../share/trackballs */
         sprintf(effectiveShareDir,"%s/../share/trackballs",thisDir);

         if(!testDir()) {
           /* From arg0/share */
           sprintf(effectiveShareDir,"%s/share",thisDir);

           if(!testDir()) {
             /* From arg0/../share */
             sprintf(effectiveShareDir,"%s/../share",thisDir);

             if(!testDir()) {
               /* From compilation default */
               sprintf(effectiveShareDir,"%s",SHARE_DIR_DEFAULT);

               if(!testDir()) {
                 printf("Error. Could not find resource directory(%s)\n",effectiveShareDir);
                 exit(0);
               }

             }

           }

         }

       }

  }

  if(NULL == getenv("GUILE_LOAD_PATH"))
  {
     sprintf(guileLoadPath,"GUILE_LOAD_PATH=%s",effectiveShareDir);
       
     putenv(guileLoadPath);
  }

  scm_boot_guile(argc,argv,innerMain,0);
  return 0;
}
