/*
 * @(#)MidiScheduler.h 3.00 14 May 1999
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifndef TSE3_SCHEDULER_H
#define TSE3_SCHEDULER_H

#include "tse3/listen/MidiScheduler.h"

#include "tse3/Notifier.h"
#include "tse3/Midi.h"

#include <cstddef>

namespace TSE3
{
    namespace Impl
    {
        /**
         * A utility function for doing calculations very quickly. It performs
         * a multiplication and a divison. It used because most systems have 32
         * bit integers and we want multiplication/division that can cope with
         * overflows.
         *
         * @param  val Value to perform muldiv on
         * @param  num Numerator
         * @param  div Divisor
         * @return val * num / div (which coped with integer overflows)
         */
        extern int muldiv(int val, int num, int div);
    };

    class MidiScheduler;

    /**
     * This class (or an inherited version of it) will generate an appropriate
     * @ref MidiScheduler for the current platform. It provides a measure of
     * platform indepenance in the generation of @ref MidiSchedulers.
     *
     * @short   MidiScheduler class factory
     * @author  Pete Goodliffe
     * @version 3.00
     */
    class MidiSchedulerFactory
    {
        public:

            MidiSchedulerFactory();
            virtual ~MidiSchedulerFactory();

            /**
             * Create the most appropriate (for some definition of appropriate
             * given by the version of the MidiSchedulerFactory class)
             * @ref MidiScheduler object.
             *
             * The @ref MidiScheduler is created with new; it is your
             * responsibility to ensure it is deleted.
             *
             * @return New appropriate @ref MidiScheduler object
             */
            virtual MidiScheduler *createScheduler() = 0;

        private:

            MidiSchedulerFactory &operator=(const MidiSchedulerFactory &);
            MidiSchedulerFactory(const MidiSchedulerFactory &);
    };

    /**
     * The MidiScheduler provides the interface to the MIDI system (be it
     * a software or hardware driver). On top of this it implements a timed
     * stream for reception and transmission of MidiEvents.
     *
     * It can notify back to it's clients if the timestream has moved,
     * or been started/stopped by a device on the MIDI connection.
     *
     * If the timestream is 'stopped' the MidiScheduler class will still
     * remember a 'current clock' value.
     *
     * Additionally, it provides a MIDI remote control facility, where
     * certain combinations of MIDI keys can cause transport start/stop.
     *
     * The MidiScheduler class does not incorporate a lot of fancy utilities,
     * like channel/port remapping since higher level TSE3 components do
     * this, for example the @ref MidiMapper class.
     *
     * @short   A MIDI interface providing scheduling facilities
     * @author  Pete Goodliffe
     * @version 3.00
     * @see     MidiEvent
     */
    class MidiScheduler : public Notifier<MidiSchedulerListener>
    {
        public:

            MidiScheduler();
            virtual ~MidiScheduler();

            /**
             * Returns a string describing the particular implementaton
             * of the MidiScheduler interface.
             *
             * @return Implementation name string
             */
            virtual const char *implementationName() = 0;

            /**
             * Returns the number of port addressable by this MidiScheduler.
             * If this changes at any point the
             * @ref MidiSchedulerListener::MidiScheduler_Ports event
             * is raised.
             *
             * @return Number of MIDI ports this MidiScheduler provides
             */
            virtual size_t ports() const = 0;

            /**
             * Returns a string describing the port with the given number.
             *
             * @return Port name string
             */
            virtual const char *portName(size_t port) const = 0;

            /**
             * Returns a string describing the type of the port with the given
             * number.
             *
             * @return Port type string
             */
            virtual const char *portType(size_t port) const = 0;

            /**
             * Returns whether or not the port is readable. If this function
             * returns false, then you will never recieve an event from
             * @ref rx from this port.
             *
             * @return Whether @ref MidiEvent objects are readable from this
             *         port
             * @see    portWritable
             */
            virtual bool portReadable(size_t port) const = 0;

            /**
             * Returns whether or not the port is writeable. If this function
             * returns false, then you can't send @ref MidiEvent objects out
             * via this port (with @ref tx). If you specify this port number
             * in a @ref MidiEvent given to @ref tx, the @ref MidiEvent will
             * be ignored.
             *
             * @return Whether @ref MidiEvent objects are readable from this
             *         port
             * @see    portReadable
             */
            virtual bool portWriteable(size_t port) const = 0;

            /**
             * Transmit a @ref MidiCommand immediately.
             *
             * @param c The MidiCommand to transmit
             */
            virtual void tx(MidiCommand mc) = 0;

            /**
             * Start the scheduler clock at the given time.
             *
             * @param startTime The time to start the scheduler at
             */
            virtual void start(Clock startTime) = 0;

            /**
             * Start the scheduler clock at the current 'resting' time.
             */
            void start() { start(restingClock); }

            /**
             * Stop the scheduler clock and flush the Tx buffer instantaneously.
             *
             * @param stopTime The time at which to stop (-1 means immediately)
             */
            virtual void stop(Clock stopTime = -1) = 0;

            /**
             * Enquire whether the scheduler clock is running.
             *
             * @return Whether the clock is running
             */
            bool running() const { return _running; }

            /**
             * Without stopping, move the scheduler clock to the given time
             * newTime at time moveTime. Any further calls to clock()
             * after this will return times in the new time line.
             *
             * @param moveTime Time at which to perform the move
             * @param newTime  Time to move to
             */
            virtual void moveTo(Clock moveTime, Clock newTime) = 0;

            /**
             * Without stopping, move the scheduler clock immediately.
             *
             * @param moveTime Time to move to
             */
            void moveTo(Clock moveTime)
            {
                moveTo(clock(), moveTime);
            }

            /**
             * Read the scheduler clock.
             *
             * This works whether the scheduler is running or not. If it has
             * been stopped then it returns the time the scheduler was stopped
             * at (or has since been moved to).
             *
             * @return MidiScheduler time value
             */
            virtual Clock clock() = 0;

            /**
             * Read the scheduler clock in milliseconds.
             *
             * @return MidiScheduler time value in milliseconds
             */
            virtual int msecs() = 0;

            /**
             * Read the tempo.
             *
             * @return Current tempo
             * @see    setTempo
             */
            int tempo() const { return _tempo; }

            /**
             * Set the tempo.
             *
             * @param newTempo   The new tempo value in beats per
             *                   minute (1-256)
             * @param changeTime Only used if the scheduler is running to
             *                   indicate when the tempo change occurs. Any
             *                   further calls to clock() before changeTime
             *                   will return bogus values, as they will be in
             *                   the new tempo's time scale.
             * @see   clock
             * @see   tempo
             */
            virtual void setTempo(int newTempo, Clock changeTime) = 0;

            /**
             * Enquire whether there is any data in input buffer.
             *
             * @return Whether there is any input data ready to be processed
             */
            virtual bool eventWaiting() = 0;

            /**
             * Return and remove a @ref MidiEvent from scheduler recieve buffer.
             *
             * @return A MidiEvent containing the data. If the status of
             *         the MidiCommand is zero, there wasn't a whole
             *         MidiCommand in the buffer.
             */
            virtual MidiEvent rx() = 0;

            /**
             * Adds an event to scheduler transmit buffer, to be transmitted at
             * the appropriate time.
             *
             * Notes must be given to this method <b>in time order</b>.
             *
             * <b>Note:</b> if this is a @ref MidiCommand_NoteOn, then the
             * matching MidiCommand_NoteOff part of the @ref MidiEvent is
             * ignored.
             *
             * @param event @ref MidiEvent to schedule to transmission
             */
            virtual void tx(MidiEvent event) = 0;

            /*
             * Transmit a MIDI "system exclusive" data section immeditately.
             * The data is prepended by a MidiSystem_SysExStart status byte,
             * and followed by a MidiSystem_SysExEnd status byhte.
             *
             * Although you use this method to send sysex data, you receive
             * it through the normal rx mechanism; you will receive a
             * @ref MidiEvent with a @ref MidiSystem_SysExStart status byte,
             * and the first data bytes (in data1 and data2). Subsequent data
             * bytes are received as MidiEvents with this same status byte. The
             * sysex data end is denoted by a @ref MidiEvent with the
             * @ref MidiSystem_SysExEnd status byte.
             *
             * @param data A buffer containing the sysex data to send
             * @param size The number of bytes in the @p data buffer
             */
            virtual void txSysEx(const unsigned char *data, size_t size) = 0;

            /*
             * Remote MIDI keyboard transport control methods
             */

            /**
             * Read the remote control status.
             *
             * @return true if remote control enabled, false if disabled
             * @see    setRemoteControl
             */
            bool remoteControl() const { return _remote; }

            /**
             * Sets the remote control status.
             *
             * @param New remote control status
             * @see   remoteControl
             */
            void setRemoteControl(bool s) { _remote = s; }

            /**
             * Returns the shift note. This note must be pushed first, in
             * combination with other remote control notes to alter the
             * Scheduler.
             *
             * @return Shift note
             * @see    setShiftNote
             */
            unsigned int shiftNote() const { return _shiftNote; }

            /**
             * Sets the shift note.
             *
             * @param n New shift note
             * @see   shiftNote
             */
            void setShiftNote(unsigned int n) { _shiftNote = n; }

            /**
             * Returns the play note. When pushed with the shift note held down,
             * this note will cause the scheduler to start.
             *
             * @return Play note
             * @see    setPlayNote
             */
            unsigned int playNote() const { return _playNote; }

            /**
             * Sets the play note.
             *
             * @param n New play note
             * @see   playNote
             */
            void setPlayNote(unsigned int n) { _playNote = n; }

            /**
             * Returns the stop note. When pushed with the shift note held down,
             * this note will cause the scheduler to stop.
             *
             * @return Stop note
             * @see    setStopNote
             */
            unsigned int stopNote() const { return _stopNote; }

            /**
             * Sets the stop note.
             *
             * @param n New stop note
             * @see   stopNote
             */
            void setStopNote(unsigned int n) { _stopNote = n; }

            /**
             * Returns the record note. When pushed with the shift note held
             * down, this note will cause the scheduler to start.
             *
             * NOTE: I HAVEN'T IMPLEMENTED THIS YET! I THINK THAT I'LL ONLY BE
             *       ABLE TO HANDLE PLAYBACK STARTING.
             *
             * @return Record note
             * @see    setRecordNote
             */
            unsigned int recordNote() const { return _recordNote; }

            /**
             * Sets the record note.
             *
             * @param New record note
             * @see   recordNote
             */
            void setRecordNote(unsigned int n) { _recordNote = n; }

        protected:

            int   _tempo;
            Clock startClock;
            Clock restingClock;
            bool  _running;

            /**
             * An internal method to convert @ref Clock values to
             * millisecond times.
             */
            int clockToMs(Clock time)
            {
                return Impl::muldiv(time-startClock, 60000/Clock::PPQN, _tempo);
            }

            /**
             * An internal method to convert millisecond time values
             * to @ref Clocks.
             */
            Clock msToClock(int ms)
            {
                return startClock + Impl::muldiv(ms, _tempo, 60000/Clock::PPQN);
            }

            /**
             * Process the given @ref MidiEvent for remove control actions.
             * This may result in the MidiScheduler stopping, starting,
             * or shifting time.
             *
             * This method also handles MIDI start and stop commands.
             *
             * If the @ref MidiEvent causes such activity, it may be set to
             * a MidiCommand_Invalid so that it is not also echoed/recorded.
             *
             * This function should be called from each MidiScheduler
             * implementation of rx.
             */
            void remoteControl(MidiEvent &event);

        private:

            MidiScheduler &operator=(const MidiScheduler &);
            MidiScheduler(const MidiScheduler &);

            /*
             * Variables used in the remote control shenanigans.
             */
            bool         _remote;
            bool         shifted;
            unsigned int _shiftNote;
            unsigned int _playNote;
            unsigned int _stopNote;
            unsigned int _recordNote;
    };
}

#endif
