#ifndef RECORDTYPE_CC
#define RECORDTYPE_CC

//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.
//---------------------------------------------------------------------------

#include "Types.hh"
#include "RecordType.hh"
#include "IntegerType.hh"
#include "EnumerationType.hh"
#include "VHDLKernel.hh"

RecordType::RecordType(ObjectBase::ObjectType objType, const RecordTypeInfo &typeInfo, ResolutionFnId_t resolveFnId) {

  recordInfo     = typeInfo;
  fields         = buildRecord(objType);
  
  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId = recordInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }
}

RecordType::RecordType( ObjectBase::ObjectType objType,
			const RecordTypeInfo &typeInfo,
			ResolutionFnId_t resolveFnId, 
			const RecordType & ){
  recordInfo     = typeInfo;
  fields         = buildRecord(objType);
  
  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId = recordInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  
}

RecordType::RecordType(ObjectBase::ObjectType objType, const RecordType &source) {
  recordInfo = source.getRecordTypeInfo();
  fields     = buildRecord(objType);
  
  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = recordInfo.getResolutionFunctionId();
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  operator=(source);
}

RecordType::RecordType( bool alias,
			ObjectBase::ObjectType,
			const RecordType &source ) : VHDLType(alias) {

  recordInfo = source.getRecordTypeInfo();
  fields     = new VHDLType*[get_number_of_fields()];
  int counter;

  for(counter = 0; (counter < get_number_of_fields()); counter++) {
    fields[counter] = (VHDLType *) &(source.get_field(counter + 1));
  }
  
  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = recordInfo.getResolutionFunctionId();
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  operator=(source);
}

RecordType::RecordType(const RecordType& source) :
  VHDLType() {
  recordInfo = source.getRecordTypeInfo();
  fields     = buildRecord(source.getKind());

  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = recordInfo.getResolutionFunctionId();
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }
  
  operator=(source);
}

RecordType::RecordType(ObjectBase::ObjectType objType, const RecordTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int noofElmntAssns, ...) {
  recordInfo = typeInfo;
  fields     = buildRecord(objType);

  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId = recordInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  if (noofElmntAssns <= 0) {
    return;
  }

  va_list ap; 
  ElementAssociation* elmtptr           = NULL;  
  ElementAssociation* othersAssociation = NULL; 
  ElementAssociation** elmtAssocArray   = NULL; 
  char* charptr                         = NULL;
  
  int i = 0; 
  int j = 0; 
  int numberOfFields = get_number_of_fields();
  
  charptr        = new char[numberOfFields+1]; 
  elmtAssocArray = (ElementAssociation **) new char[noofElmntAssns * 
						   sizeof(ElementAssociation*)]; 
  for (i = 1; (i <= numberOfFields); i++) { 
    charptr[i] = 'U'; 
  } 
 
  va_start(ap, noofElmntAssns); 
  for(i = 0; (i < noofElmntAssns); i++) { 
    elmtAssocArray[i] = va_arg(ap, ElementAssociation*); 
  } 
  va_end(ap); 
     
  for(i = 0; (i < noofElmntAssns); i++) { 
    elmtptr = elmtAssocArray[i]; 
    if (elmtptr->choice.operator==(Others)) { 
      othersAssociation = elmtptr; 
      break; 
    } 
    else { 
      if(elmtptr->choice.left() == elmtptr->choice.right()) { 
	get_field(elmtptr->choice.left()).assignVal(*elmtptr->value); 
	charptr[elmtptr->choice.left()] = 'I'; 
 
      } 
      else { 
	if(elmtptr->choice.dirn() == ArrayInfo::to) { 
	  for(j = elmtptr->choice.left(); (j <= elmtptr->choice.right()); j++) { 
	    get_field(j).assignVal(*elmtptr->value); 
	    charptr[j] = 'I'; 
	  } 
	} 
	else { 
	  for(j = elmtptr->choice.left(); (j >= elmtptr->choice.right()); j--) { 
	    this->get_field(j).assignVal(*elmtptr->value);	   
	    charptr[j] = 'I'; 
	  } 
	} 
      } 
    } 
  } 

  if (othersAssociation != NULL) {
    for(i = 1; (i <= numberOfFields); i++) { 
      if(charptr[i] == 'U') { 
	this->get_field(i).assignVal(*othersAssociation->value); 
      } 
    } 
  }
  
  for(i = 0; (i < noofElmntAssns); i++) { 
    delete  elmtAssocArray[i]->value; 
    delete  elmtAssocArray[i]; 
  } 
  
  if(elmtAssocArray != NULL) { 
    delete [] elmtAssocArray; 
  } 
  delete [] charptr;
}

RecordType::RecordType(bool alias, ObjectBase::ObjectType objType, const RecordTypeInfo& typeInfo, ResolutionFnId_t resolveFnId, int noofElmntAssns, ...) : VHDLType(alias) {
  recordInfo = typeInfo;
  fields     = buildRecord(objType);

  if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId = recordInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  if (noofElmntAssns <= 0) {
    return;
  }

  va_list ap; 
  ElementAssociation* elmtptr           = NULL;  
  ElementAssociation* othersAssociation = NULL; 
  ElementAssociation** elmtAssocArray   = NULL; 
  char* charptr                         = NULL;
  
  int i = 0; 
  int j = 0; 
  int numberOfFields = get_number_of_fields();
  
  charptr        = new char[numberOfFields+1]; 
  elmtAssocArray = (ElementAssociation **) new char[noofElmntAssns * 
						   sizeof(ElementAssociation*)]; 
  for (i = 1; (i <= numberOfFields); i++) { 
    charptr[i] = 'U'; 
  } 
 
  va_start(ap, noofElmntAssns); 
  for(i = 0; (i < noofElmntAssns); i++) { 
    elmtAssocArray[i] = va_arg(ap, ElementAssociation*); 
  } 
  va_end(ap); 
     
  for(i = 0; (i < noofElmntAssns); i++) { 
    elmtptr = elmtAssocArray[i]; 
    if( elmtptr->choice.operator==(Others) ){
      othersAssociation = elmtptr; 
      break; 
    } 
    else { 
      if(elmtptr->choice.left() == elmtptr->choice.right()) {
	if (alias == true) {
	  get_field(elmtptr->choice.left()).setObject(elmtptr->value->getObject());
	}
	else {
	  get_field(elmtptr->choice.left()).assignVal(*elmtptr->value);
	}
	charptr[elmtptr->choice.left()] = 'I'; 
 
      } 
      else { 
	if(elmtptr->choice.dirn() == ArrayInfo::to) { 
	  for(j = elmtptr->choice.left(); (j <= elmtptr->choice.right()); j++) {
	    if (alias == true) {
	      get_field(j).setObject(elmtptr->value->getObject());	      
	    }
	    else {
	      get_field(j).assignVal(*elmtptr->value);
	    }
	    charptr[j] = 'I'; 
	  } 
	} 
	else { 
	  for(j = elmtptr->choice.left(); (j >= elmtptr->choice.right()); j--) {
	    if (alias == true) {
	      get_field(j).setObject(elmtptr->value->getObject());
	    }
	    else {
	      this->get_field(j).assignVal(*elmtptr->value);
	    }
	    charptr[j] = 'I'; 
	  } 
	} 
      } 
    } 
  } 

  if (othersAssociation != NULL) {
    for(i = 1; (i <= numberOfFields); i++) { 
      if(charptr[i] == 'U') { 
	this->get_field(i).assignVal(*othersAssociation->value); 
      } 
    } 
  }
  
  for(i = 0; (i < noofElmntAssns); i++) { 
    delete  elmtAssocArray[i]->value; 
    delete  elmtAssocArray[i]; 
  } 
  
  if(elmtAssocArray != NULL) { 
    delete [] elmtAssocArray; 
  } 
  delete [] charptr;
}

RecordType::RecordType(ObjectBase::ObjectType) {
  fields = NULL;
  resolutionFunctionId = -1;
}

RecordType::~RecordType() {
  delete [] fields;
}

ObjectBase::ObjectType
RecordType::getKind() const {
  if(get_number_of_fields() > 0) {
    return get_field(1).getKind();
  }
  else {
    cerr << "Warning Don't know the Object Type" << endl;
    cerr << "I am assuming VARIABLE" << endl;
    return ObjectBase::VARIABLE;
  }
}

VHDLType*
RecordType::clone() const {
  return new RecordType(*this);
}

VHDLType&
RecordType::get_field(int fieldNumber) const {
  if ((fieldNumber > 0) && (fieldNumber <= get_number_of_fields())) {
    return *(fields[fieldNumber - 1]);
  }
  else {
    cerr << "Error: fieldNumber out of limits.\n";
    abort();
  }

  return IntegerType::INVALID_TYPE_OBJECT;
}

VHDLType&
RecordType::operator=(const VHDLType& source) {
  ASSERT ( source.get_kind() == RECORD_TYPE );

  return operator=((const RecordType &) source);
}

RecordType&
RecordType::operator=(const RecordType& source) {
  if (fields == NULL) {
    // Okay.. let's just duplicate the source record...
    // This is the case when a global record constant is being initialized...
    
    recordInfo = source.getRecordTypeInfo();
    fields = buildRecord(source.getKind());

    if ((isCompositeResolvedType = recordInfo.isCompositeResolvedType()) == true) {
      resolutionFunctionId = recordInfo.getResolutionFunctionId();
    }
    
    if (resolutionFunctionId != -1) {
      isCompositeResolvedType = true;
      setCompositeResolvedSignal(true);
      setParentCompositeType(this);
    }
  }
  
  ASSERT ( get_number_of_fields() == source.get_number_of_fields() );

  int fieldCount = get_number_of_fields();
  for(int counter = 0; (counter < fieldCount); counter++) {
    fields[counter]->operator=(source.get_field(counter + 1));
  }

  return *this;
}

bool
RecordType::operator==(const RValue &val) const {
  for(int field = 1; field <= get_number_of_fields(); field++) {
    if( !(get_field(field).operator==(dynamic_cast<const RecordType &>(val).get_field(field)))){
      return false;
    }
  }
  
  return true;
}  

bool
RecordType::operator!=( const RValue &val) const {
  return !operator==(val);
}

bool
RecordType::operator<( const RValue &) const {
  abort();
}
bool
RecordType::operator<=( const RValue &) const {
  abort();
}
bool
RecordType::operator>( const RValue &) const {
  abort();
}
bool
RecordType::operator>=( const RValue &) const {
  abort();
}

VHDLType&
RecordType::assignVal(const VHDLType& val) {
  int field_number;
  
  ASSERT( val.get_kind() == RECORD_TYPE );
  const RecordType& source = (const RecordType &) val;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).assignVal(source.get_field(field_number));
  }
  
  return *this;
}

void
RecordType::setResolutionFunctionId(ResolutionFnId_t resolutionFnId) {
  int field_number;

  resolutionFunctionId = resolutionFnId;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).setResolutionFunctionId(resolutionFnId);
  }
}

void
RecordType::setTypeConversionFunctionId(TypeConversionFnId_t typeConversionFnId) {
  int field_number;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).setTypeConversionFunctionId(typeConversionFnId);
  }
}

void
RecordType::setParentCompositeType(VHDLType* ptr) {
  int field_number;
  
  isCompositeResolvedType = true;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).setParentCompositeType(ptr);
  }
}

void
RecordType::setCompositeResolvedSignal(bool val) {
  int field_number;
  
  isCompositeResolvedType = val;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).setCompositeResolvedSignal(val);
  }
}

void
RecordType::setElaborationInfo(const VHDLType &obj_info) {
  int field_no;
  
  ASSERT (obj_info.get_kind() == RECORD_TYPE);
  const RecordType& source = (const RecordType &) obj_info;
  
  for(field_no = 1; (field_no <= get_number_of_fields()); field_no++) {
    get_field(field_no).setElaborationInfo(source.get_field(field_no));
  }  
}

void
RecordType::setAttrib(AttribType typ, VHDLType& attr) {
  int field_no;
  
  ASSERT (attr.get_kind() == RECORD_TYPE);
  RecordType& source = (RecordType&) attr;
  
  for(field_no = 1; (field_no <= get_number_of_fields()); field_no++) {
    get_field(field_no).setAttrib(typ, source.get_field(field_no));
  }  
}

//The TYPE's resolve is called only for composite resolved signals
//This resolve goes down to first sub-element of the VHDLType and
//calls the sub-elements resolve, but which actually does the resolution
//for the whole composite type
// The above given reason was the original reason for writing resolve
// in the RecordTypes. But when TypeConversion came into picture, we needed
// resolve to be written for RecordTypes  even if the RecordType was not a
// composite resolved signal. In case of not a composite resolved type,
// resolve method just goes down to the sub fields  and resolves the
// individual fields and then updates itself once the resolved value is
// obtained.

VHDLType*
RecordType::resolve(VHDLKernel* processPtr) {
  int index;
  VHDLType *retval = NULL, *field = NULL;
  
  if(_is_composite_resolved_type() == true) {
    if (get_number_of_fields() > 0) {
      retval = this->get_field(1).resolve(processPtr);
      return retval;
    }
    else {
      return NULL;
    }
  } else {
    for (index = 1; (index <= get_number_of_fields()); index++) {
      field  = &(this->get_field(index));
      retval = field->resolve(processPtr);
      
      if (field->_is_composite_resolved_type()) {
        *field = *retval;
      }
    }
    return NULL;
  }

  return NULL;
}

void
RecordType::updateEffVal(const VHDLType* ptr) {
  int field_number;
  
  ASSERT(ptr->get_kind() == RECORD_TYPE);
  RecordType *source = (RecordType *) ptr;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).updateEffVal(&(source->get_field(field_number)));
  }
}

bool
RecordType::_is_composite_resolved_type() const {
  return isCompositeResolvedType;
}

SignalBase*
RecordType::locateSig(int sigId) {
  int field_number = 1;
  SignalBase* ptr  = NULL;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    if ((ptr = get_field(field_number).locateSig(sigId)) != NULL) {
      return ptr;
    }
  }
  
  return NULL;
}

SignalBase*
RecordType::findSigInBlock(int sigId, VHDLKernel *srcId){
  int field_index;
  SignalBase *driver_ptr = NULL;

  for(field_index = 1; (field_index <= get_number_of_fields()); field_index++) {
    if ((driver_ptr = get_field(field_index).findSigInBlock(sigId, srcId)) != NULL) {
      return driver_ptr;
    }
  }
  
  return NULL;
}

int
RecordType::savantwrite(AccessVariable & line) const {
  int field_number;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).savantwrite(line);
  }
  
  return NORMAL_RETURN;
}

int
RecordType::savantwrite(AccessType &line) const {
  int field_number;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).savantwrite(line);
  }
  
  return NORMAL_RETURN;
}

int
RecordType::savantread(AccessVariable &line) {
  int field_number;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).savantread(line);
  }
  
  return NORMAL_RETURN;
}


int
RecordType::savantread(AccessType &line) {
  int field_number;
  
  for(field_number = 1; (field_number <= get_number_of_fields()); field_number++) {
    get_field(field_number).savantread(line);
  }

  return NORMAL_RETURN;
}

const RecordTypeInfo&
RecordType::getRecordTypeInfo() const {
  return recordInfo;
}

void
RecordType::print(ostream& os) const {
  int counter;
  for(counter = 1; counter <= get_number_of_fields(); counter++) {
    get_field(counter).print(os);
  }
}

void 
RecordType::Add( const VHDLType &rhs ){
  for(int i = 1; i <= get_number_of_fields(); i++) {
    get_field(i).Add( dynamic_cast<const RecordType &>(rhs).get_field(i) );
  }
}

void 
RecordType::Add( const SignalNetinfo *ptr ){
  for( int i = 1; i <= get_number_of_fields(); i++ ){
    get_field(i).Add( ptr);
  }
}

void 
RecordType::Add( VHDLKernel *processPtr ){
  ASSERT( processPtr != 0 );
  for( int i = 1; i <= get_number_of_fields() ; i++ ){
    get_field(i).Add( processPtr );
  }
}

void 
RecordType::Add( VHDLKernel *processPtr, int sigId ){
  ASSERT( processPtr != 0 );
  for( int i = 1; i <= get_number_of_fields() ; i++ ){
    get_field(i).Add( processPtr, sigId );
  }
}

void 
RecordType::copyId( const VHDLType &src ){
  const RecordType &as_record_type = dynamic_cast<const RecordType &>( src );
  for ( int i = 1 ; i <= get_number_of_elements(); i++){
    get_field(i).copyId( as_record_type.get_field(i) );
  }
}


VHDLType**
RecordType::buildRecord(ObjectBase::ObjectType objType) {
  int counter;

  VHDLType **newFields = new VHDLType*[get_number_of_fields()];

  for(counter = 0; (counter < get_number_of_fields()); counter++) {
    newFields[counter] = recordInfo.get_field_info(counter)->createObject(objType);
  }

  return newFields;
}

void
RecordType::resolveAndUpdate( VHDLKernel* processPointer ){
  if( _is_composite_resolved_type() == false ){
    for( int i = 1; i < get_number_of_elements(); i++ ){
      get_field(i).resolveAndUpdate( processPointer );
    }
  }
  else {
    processPointer->updateSignal(this);
  }
}


bool
RecordType::is_driver_already_set() const {
  return get_field(1).is_driver_already_set();
}

bool
RecordType::is_resolved_signal() const {
  return get_field(1).is_resolved_signal();
}

void
RecordType::set_sourcebase_delete_flag(bool flag) const {
  for ( int i = 1; i <= get_number_of_elements(); i++ ){
    get_field(i).set_sourcebase_delete_flag(flag);
  }
}

void 
RecordType::assignSignal( VHDLKernel *destProcess, 
			  VHDLKernel *srcId, 
			  const VHDLType& src,
			  const PhysicalType& delay, 
			  const PhysicalType& rejTime, 
			  const ArrayInfo& dinfo,
			  const ArrayInfo& sinfo ){
  ASSERT( destProcess != 0 );
  ASSERT( srcId != 0 );
  const RecordType &srcAsRecord = dynamic_cast<const RecordType &>(src);
  for( int i = 1; i <= get_number_of_fields(); i++) {
    destProcess->assignSignal( get_field(i), 
			       srcId,
			       srcAsRecord.get_field(i),
			       delay, 
			       rejTime, 
			       dinfo, 
			       sinfo );
  }
}



void 
assignVariable( RecordType& dest, const RecordType& src, 
		const ArrayInfo&, const ArrayInfo&) {
  int field_counter;
  
  for(field_counter = 1; (field_counter <= dest.get_number_of_fields());
      field_counter++) {
    dest.get_field(field_counter) = src.get_field(field_counter);
  }
}

ostream&
operator<<(ostream &os, const RecordType &rt) {
  int field;

  os << "(";
  for(field = 1; (field <= rt.get_number_of_fields()); field++) {
    if (field > 1) {
      os << ", ";
    }
    
    os << rt.get_field(field);
  }

  os << ")";
  
  return os;
}

#endif
