/* ==================================================== ======== ======= *
 *
 *  uevent.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uevent_hh
#define	_uevent_hh
//pragma ident	"@(#)uevent.hh	ubit:b1.11.6"

typedef union _XEvent* UX_Event;// X Event Pointer (hidden type)

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UItemData {
friend class UEvent;
public:
  // points to the item that is located under the mouse when calling 
  // UEvent::getItem()/getStr()
  UItem *item;
  ULink *itemLink;

  // the region that encloses this item. NOTE that these coordinates
  // are relative to the enclosing WINDOW (not the item parent!)
  // use methods getX(), getY() for obtaining local coordinates.
  URegion region;

  // graphical context. NOTE that itemCtx is automatically deleted 
  // when UItemData is deleted!
  class UContext *itemCtx;

  // true if Item is exactly located under the Mouse
  // false if Item is the last Item before the Mouse position
  u_bool exactMatch;

  // strpos = position of the mouse in the string if item is an UStr
  int strpos, strpos2;

  // auxiliary data
  void *aux;

public:
  UItemData(); 
  ~UItemData();

  // return the location of the item relatively to the source VIEW
  // (ie. the view returned by e.getView())
  u_pos getX(class UEvent*);
  u_pos getY(class UEvent*);

  // return the width and the height of the item
  u_dim getWidth();
  u_dim getHeight();

  //package_private: ====[internal implementation]===============
  void set(UContext&, UItem*, ULink*, const URegion&, 
	   int _strpos, u_bool exact_match);
  void merge(UContext&, UItem*, ULink*, const URegion&, 
	     u_bool exact_match);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UEvent {
  friend class UAppli;
  friend class UCtrl;
  friend class UGroup;
  friend class UBrick;
  friend class UBox;
  friend class UView;
  friend class UNatWin;
protected:
  UGroup*   source;
  UView*    sourceView;    // the view that contains the mouse
  UView*    winView;       // the view of the containing window
  class UWinContext *parentContext;
  class UContext    *sourceContext;
  UFlag**   flags;
  UBrick**  exts;
  unsigned short flagCount, extCount;
  u_time    time;	    // event time stamp
  UX_Event  xev;	    // pointer to X event (hidden type)
  u_eid     eid;	    // event id
  const UCond* cond;	    // UOn condition of the ucall()
public:
  int       detail;
  u_bool    propagate, redrawStatus;
  u_pos     xmouse, ymouse;  // current mouse position in the UWin
  u_pos     xdrag_ref, ydrag_ref;//screen coords of the View that got the Press
  UView*    firstTranspView; //1st transparent view in the view tree
  UBox*     through;         //events go through this transparent tool
  URegion   redrawClip;      // the region that must be redrawn

  // returns the UBox or the UGroup that got this event
  UGroup* getSource();

  // returns the specific Source VIEW that got this event
  // (remember that a Box can have several Views!)
  UView* getView() {return sourceView;}
 
  // xxx condition of the:  UOn::xxx / ucall(...)
  // NB: as the same UEvent can be catched by several ucall() conditions
  // this value should be checked immediately in the callback function
  // as it can change at a later time
  const UCond* getCond() {return cond;}

  // returns the time when this event was produced (when available)
  u_time getTime();

  // returns the event id (NOTE: subject to change!!!)
  u_eid getEid() {return eid;}
  u_eid getID() {return eid;}

  // returns the location of the mouse in the Source VIEW
  // (-1 if no source view was found)
  u_pos getX();
  u_pos getY();

  // returns the location of the mouse on the SCREEN
  // (available for mouse and key events)
  u_pos getXscreen();
  u_pos getYscreen();

  // returns the Window that caught this event
  // (note: which actually window gets the event is implementation dependent
  //  and it is not necessarily the UWin that contains the Source View)
  UWin*  getWin();
  UView* getWinView();

  // return the location of the mouse in the Window that caught this event
  // (ie. the window returned by getWin())
  u_pos getXwin() {return xmouse;}
  u_pos getYwin() {return ymouse;}

  // returns the pressed button number & the number of clicks when available
  // --- can only be used in 'UOn::mpress' and 'UOn::mrelease' callbacks
  int getClickCount();
  int getButtonNumber();

  // returns a mask that identifies which mouse buttons are being pressed
  // (or which have been pressed and are now released) and 0 if none
  int getButtons();

  // returns a combination of key and mouse button modifiers
  // --- can be used in Key and Mouse Button callbacks
  int getMods();

  // returns the ASCII typed character when available
  // --- can only be used in UOn::ktype/kpress/krelease callbacks
  int getKeyChar();

  // returns the X KeySym (but without taking modifiers into account)
  // --- can only be used in UOn::ktype/kpress/krelease callbacks
  long getKeySym();
 
  // returns the UItem that was under the mouse (null if no item found)
  UItem* getItem();
  // same function but discards UItem(s) that are not UStr(ings)
  UStr* getStr();

  // same functions but UItemData is filled with useful information
  // (the item location, its graphical properties, etc.) in ItemData
  UItem* getItem(UItemData&);
  // Note that the region returned in ItemData does not contain the entire string
  // but just the line that was beneath the mouse (in case of UFlowView rendering)
  UStr* getStr(UItemData&);

  // searches the item that is given as an argument
  // and returns useful information about this item in ItemData
  // Notes:
  // -- args. strpos1, strpos2 are used when searching for UStr(ings)
  // -- the region returned in ItemData does not contain the entire string
  //    but just its 'strpos1' to 'strpos2' subpart
  UItem* searchItem(UItemData&, const UItem *searched_item,
		    int strpos1 = -1, int strpos2 = -1);

  // implementation dependent : returns the X Event
  UX_Event getXEvent() {return xev;}

  // for retreiving Flags and Extensions in callback functions
  //!NOTE: the returned data is automatically destroyed when the UEvent
  //       is destroyed. NEVER delete it explicitely nor use it after
  //       the UEvent is deleted.
  UFlag** getFlags()    {return flags;}
  UBrick** getExts()    {return exts;}

  u_card getFlagCount() {return flagCount;}
  u_card getExtCount()  {return extCount;}

  // adds a Flag to the Event (for transparent tools)
  void addFlag(const UFlag*);

  // returns the graphical context of the parent (resp.) source view
  //class UContext *getContext();
  //!NOTE: the returned data is automatically destroyed when the UEvent
  //       is destroyed. NEVER delete it explicitely nor use it after
  //       the UEvent is deleted.
  class UContext* getParentContext();
  class UContext* getSourceContext(int pos = 0);

  // Standard Event IDs (see class 'UOn' in uctrl.h for details)
  enum EID {
    anyEvent  	= 0,		// for UOn::anyEvent
    mpress	= 1<<0, 
    mrelease	= 1<<1, 
    mmove 	= 1<<2, 
    mdrag 	= 1<<3, 
    kpress 	= 1<<4, 
    krelease 	= 1<<5, 
    enter 	= 1<<6, 
    leave 	= 1<<7, 
    change 	= 1<<8,
    add		= 1<<9,
    remove	= 1<<10,
    select 	= 1<<11,  
    viewPaint 	= 1<<12, 
    viewResize 	= 1<<13, 
    viewMove 	= 1<<14, 
    search	= 1<<15, 
    message 	= 1<<16  //!BEWARE: no comma after last item!
  };

  // Queries the mouse location and initializes the UEvent fields.
  // Returns true if the mouse is located on a UWin of the UAppli
  // Example:
  //     UEvent e;
  //     if (e.queryMouse()) {      // mouse over an appli window
  //        box = e.getSource(); view = e.getView(); 
  //        x_in_view = e.getX();
  //        x_in_win = e.getXwin();
  //     }
  //     e.whereInScreen(&x_in_screen, &y_in_sceen);
  //
  //NOT_AVAILABLE: u_bool queryMouse();

//package_private: ====[ubit intrinsics]===============================

  //!Note: 'win_view' must be a valid UWin view or null
  UEvent(u_eid event_id, UView *win_view, UX_Event);
  ~UEvent(); 

  // returns UView if found and null otherwise
  UView *locateSource(UView *searched_view);
  UView *locateSource(u_pos x_in_win, u_pos y_in_win);

  void setViewSource(UView*);   // sets View corresponding UBox
  void setGroupSource(UGroup*); // sets Group (when no UView is available)
  void setItemSource(UItem*);	// the source is an UItem 
  void setPropSource(UProp*);	// the source is an UProp

  // copies all UEvent *content* AND the content of all pointed fields
  void copy(const UEvent&);

  // copies the *content* of the UContext
  // att: usage special, l'event referant doit etre le meme 
  void dumpParentContext(const class UContext&);

  // actualize events with new Eid and new coordinates
  void actualize(u_eid event_id,u_pos x_in_win,u_pos y_in_win, UX_Event ev) {
    eid = event_id; xev = ev; xmouse = x_in_win; ymouse = y_in_win; 
  }
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */

