/* ==================================================== ======== ======= *
 *
 *  uugraph.cc
 *  Ubit Project [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ******************************X&*****************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */


//#pragma ident	"@(#)uugraph.cc	ubit:b1.11.5"
#include <ubrick.hh>
#include <ufont.hh>
#include <ucolor.hh>
#include <upix.hh>
#include <ustr.hh>
#include <uview.hh>
#include <ugraph.hh>
#include <uwin.hh>
#include <uappli.hh>
#include <ctype.h>
#include <math.h>
#include <unat.hh>
#ifdef GL
#include <GL/gl.h>
#include <GL/glx.h>
#endif

// polylines and polygons stored in the stack if card <= POLYCARD
// (heap memory used otherwise)
#define POLYCARD 100

/* !REMARQUE concernant X-Window et le parametre gcval.line_width :
 *
 * - width != 0 provoque des resultats quasi aleatoires
 *   a cause de pbms d arrondis pas franchement clairs (voir man pages)
 *   ainsi le dernier point apparait ou pas selon les cas!!!
 *
 * - DE PLUS: si line_width > 1 le trait deborde en epaisseur tantot
 *   a gauche, tantot a droite (s il est horizontal)
 *
 * ==> il est preferable de conserver line_width = 0 et d'imprimer 
 *     plusieurs lignes cote a cote quand on veut savoir PRECISEMENT
 *     ce que l'on va obtenir (en particulier pour les traits
 *     horizontaux et verticaux des widgets ou c'est au pixel pres)
 */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UGraph::UGraph() {
  xorMode = false;
  thickness = 0;
  boxview = null;
  disp = null;
  natwin = null;
  natgraph = null;
  doublebuf = null;
  transpbuf = null;
  subwin = false;
}

UGraph::UGraph(UView *view) {
  if (!view) {
    boxview  = null;
    disp     = null;
    natwin   = null;
    natgraph = null;
    uerror("UGraph::UGraph", "!Null BoxView");
    return;	// !faudrait une exception!
  }

  // !xorMode', 'thickness' must be initialized before lock()
  xorMode   = false;
  thickness = 0;
  boxview   = view;
  UWinGraph &wg = view->wg();
  disp      = wg.disp;
  natwin    = wg.natwin;
  // NB: double buffering: comme pour view
  doublebuf = wg.doublebuf;
  transpbuf = wg.transpbuf;
  subwin = false;

  // shared UNatGraphs(s) [one for each UDisp]
  // !att: ne convient pas si multi-threads du reaffichage standard
  // !att egalement aux couleurs et fonts qui peuvent varier suivant displays
  if (disp->natdisp->getClientGraph()->isLocked()) {
    // deja pris => creer un nouveau NatGraph
    natgraph = new UNatGraph(disp->natdisp);
    natgraph->lock(this);
  }
  else {
    // utiliser le NatGraph standard
    natgraph = disp->natdisp->getClientGraph();
    natgraph->lock(this);
    //...reset eventuel du GC
  }

  // ajuster la  zone de clipping a la PARTIE VISIBLE de la vue
  // et affecter la 'bonne' destination

  UEvent e(UEvent::search, view->getHardwinView(), null);
  e.locateSource(view);

  if (doublebuf) {
    natgraph->setDest(doublebuf,
		      wg.natgraph->x_win + view->x,
		      wg.natgraph->y_win + view->y);
    if (e.redrawStatus > 0) {
      natgraph->setClip(e.redrawClip.x + wg.natgraph->x_win,
			e.redrawClip.y + wg.natgraph->y_win,
			e.redrawClip.width,
			e.redrawClip.height);
    }
  }
  else {
    natgraph->setDest(natwin, view->x, view->y);
    if (e.redrawStatus > 0) {
      natgraph->setClip(e.redrawClip);
    }
  }
}

UGraph::~UGraph() {
  //!ATT: pas question de detruire les NatGraphs partages !
  if (natgraph && natgraph != disp->natdisp->getSharedGraph()) {
    if (natgraph == disp->natdisp->getClientGraph()) 
      natgraph->unlock();
    else {delete natgraph; natgraph = null;}
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// NOTE POUR UWinGraph UNIQUEMENT:
// - on ne doit JAMAIS changer thickness qui DOIT donc TOUJOURS valoir 0
//   ==> lignes d'epaisseur 1 avec affichage rapide optimise
// - raison: voir note sur pbm de precision en haut de fichier

UWinGraph::UWinGraph(class UWin *win) : UGraph() {
  hardwin  = win;
  natwin = new UNatWin();   //NEW: (deplace; etait avant dans init())
}

UWinGraph::~UWinGraph() {
  //!ATT: surtout ne PAS detruire hardwin!
  hardwin = null;

  //NB: le XDestroyWindow est necessaire car ~UNatWin ne detruit pas xwin
  // (xwin etant soit un XPixmap soit une XWindow). De plus PROBLEME !!!!! 
  // la destruction des XWindow est recursive ce qui peut            !!!!!
  // aboutir a des destructions multiples de la meme XWindow         !!!!!

  // NOTE:                                   !!! TEMPORAIRE : A REVOIR !!!
  if (natwin->xwidget) 
    XtDestroyWidget(natwin->xwidget);
  else if (natwin->xwin != None) 
    XDestroyWindow(natgraph->xdisplay, natwin->xwin);
  natwin->xwidget = null;
  natwin->xwin = null;

  delete natwin;
  natwin = null;
  // idiot puisque natgraph partage!
  // if (natgraph == natdisp->wingraph)  natgraph->unlock();
}

// initialize le WinGraph. dot obligatoirement etre appelee avant usage
// NB: cette fct ne cee pas la X Wndow. ce sera fait uterieurement par
// UWin::realize la 1ere fois que la fenetre apparaitra

u_bool UWinGraph::init(UDisp *d, UView* win_view) {
  if (!d || !win_view //EX: deplace: || !(natwin = new UNatWin())
      ) {
    uerror("!UWinGraph::realize", "!Fatal Error: could not realize UWinGraph");
    return false;       // faudrait une exception!
  }
  disp = d;
  doublebuf = null;  // no double buffering by default
  transpbuf = null;
  subwin = false;

  // !IMPORTANT:
  // Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
  // sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
  // seront incorrectes
  // (noter que par contre, UGraph permet de varier boxview, ce qui permet
  // (de travailler en ccords relatives)
  boxview = win_view;

  // shared UNatGraphs(s) [one for each UDisp]
  // !att: ne convient pas si multi-threads du reaffichage standard
  // !att egalement aux couleurs et fonts qui peuvent varier suivant displays
  natgraph = disp->natdisp->getSharedGraph();
  //natgraph->lock(this); idiot puisque natgraph partage!

  // pour initialiser x_win et y_Win a 0
  natgraph->setDest(natwin, 0, 0);

  // Note: on ne change pas le Clip: il faudra le faire explicitement
  // sur le natgraph partage !
  return true;
}

u_bool UWinGraph::isWinRealized() const {
  return (natwin && natwin->isRealized());
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UGraph::flush() {
  if (natgraph) {
#ifdef GL 
    glFlush();
#else
    XFlush(natgraph->xdisplay);
#endif
  }
}

// physical grab on this window (only for menus!)
int UGraph::grabPointer(class UCursor *c, u_bool confine_to_window) {
  //return true;
  return XGrabPointer(natgraph->xdisplay, 
		      natwin->getXWindow(),
		      //owner_events: si True, les windows sous le pointeur
		      //recoivents les Events comme d'habitude (et les coords
		      //sont relatives a celles-ci). Si False, c'est la grab-window
		      //qui recoit tous les events (et les coords sont relatives
		      //a la grab-window)
		      True,
		      ButtonPressMask|ButtonReleaseMask|PointerMotionMask,
		      GrabModeAsync,    // pointer_mode
		      GrabModeAsync,    // keyboard_mode
		      (confine_to_window ? natwin->getXWindow() : None),
		      None,             // cursor 
		      CurrentTime);     // time
}

void UGraph::ungrabPointer() {
  if (natgraph) XUngrabPointer(natgraph->xdisplay, CurrentTime);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

#ifdef GL
static void setGLcolor(color) {
  unsigned long 
    rMask = natgraph->getGLVisual()->red_mask,
    gMask = natgraph->getGLVisual()->green_mask,
    bMask = natgraph->getGLVisual()->blue_mask;

  long red   = (color & rMask) >> 16;
  long green = (color & gMask) >> 8;
  long blue  = (color & bMask);

  glColor3ub((GLubyte)red, (GLubyte)green, (GLubyte)blue);
}
#endif

/* ==================================================== ======== ======= */
// PaintMode is the default (overwrites pixels)

void UGraph::setPaintMode() {
  natgraph->xorMode = xorMode = false;
#ifdef GL
  //ca ne change pas le mode du GC! possible en GL ???
  setGLcolor(natgraph->color);
#else
  XGCValues gcval;
  gcval.foreground = natgraph->color;
  gcval.function = GXcopy;
  XChangeGC(natgraph->xdisplay, natgraph->gc, GCFunction|GCForeground, &gcval);
#endif
}

// setXORMode initiates drawing in XOR kode

void UGraph::setXORMode() {
  natgraph->xorMode = xorMode = true;
#ifdef GL
  //ca ne change pas le mode du GC! possible en GL ???
  setGLcolor(natgraph->color);
#else
  // seul le champ 'gcval.foreground' sert pour le XOR
  // cette valeur est elle meme un XOR de foreground ET alternate_color
  XGCValues gcval;
  gcval.foreground = natgraph->color ^ natgraph->bgcolor;
  gcval.function = GXxor;
  XChangeGC(natgraph->xdisplay, natgraph->gc, GCFunction|GCForeground, &gcval);
#endif
}

void UGraph::setColor(const UColor *c) {
  natgraph->color = disp->natdisp->getXPixel(c);
#ifdef GL
  //ca ne change pas le mode du GC! possible en GL ???
 if (xorMode) setGLcolor(natgraph->color ^ natgraph->bgcolor);
  else setGLcolor(natgraph->color);
#else
  XGCValues gcval;
  if (xorMode) gcval.foreground = natgraph->color ^ natgraph->bgcolor;
  else gcval.foreground = natgraph->color;
  XChangeGC(natgraph->xdisplay, natgraph->gc, GCForeground, &gcval);
#endif
}

void UGraph::setBgcolor(const UColor *c) {
  natgraph->bgcolor = disp->natdisp->getXPixel(c);
#ifdef GL
  //ca ne change pas le mode du GC! possible en GL ???
 if (xorMode) setGLcolor(natgraph->color ^ natgraph->bgcolor);
#else
  if (xorMode) {
    XGCValues gcval;
    gcval.foreground = natgraph->color ^ natgraph->bgcolor;
    XChangeGC(natgraph->xdisplay, natgraph->gc, GCForeground, &gcval);
  }
#endif
}

void UGraph::setCursor(const UCursor *c) {
  if (c) {
    Cursor curs = disp->natdisp->getXCursor(c);
    // NB: curs =eventuellement None
    XDefineCursor(natgraph->xdisplay, natwin->xwin, curs);
  }
  else XDefineCursor(natgraph->xdisplay, natwin->xwin, None);
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

// Changes line thickness when drawing lines, etc.
// - the default value is 0, which sets thickness to 1 pixel
//   and optimizes drawing routines
// !!a revoir avec UPen !!

void UGraph::setThickness(int th) {
#ifdef GL
  //??? a FAIRE         !!!!
#else
  XGCValues gcval;// !!a revoir avec UPen !!
  natgraph->thickness = thickness = th;// !!a revoir avec UPen !!
  gcval.line_width = th;
  XChangeGC(natgraph->xdisplay, natgraph->gc, GCLineWidth, &gcval);
#endif
}

void UGraph::setFont(const UFont *f) {
  UFontDesc fd;
  fd.set(f);
  setFont(&fd);
}

void UGraph::setFont(const UFontDesc *fd) {
#ifdef GL
  natgraph->glFontList = disp->natdisp->getGLXFont(fd);  //!!!!
#else
  natgraph->font = disp->natdisp->getXFont(fd);
  if (!natgraph->font) {
    uerror("UGraph::setFont", "Null Font!");
    return;
  }
  XGCValues gcval;
  gcval.font = natgraph->font->fid;
  XChangeGC(natgraph->xdisplay, natgraph->gc, GCFont, &gcval);
#endif
}

/* ==================================================== [Elc:01] ======= */
// ==== Double Buffering =================================================

UNatWin* UGraph::createAltBuffer(const URegion &r, const char *errmess) {
  if (natwin->xwin == None) {
    uerror("createAltBuffer", "Null X Window !");
    return null;
  }
 
  Window root_return;
  int x_return, y_return;
  unsigned int width_return, height_return, border_width_return, depth_return;
  Status stat = XGetGeometry(natgraph->xdisplay,
			     natwin->xwin,
			     &root_return, &x_return, &y_return,
			     &width_return, &height_return, 
			     &border_width_return,
			     &depth_return);
  if (stat != True || r.width <= 0 || r.height <= 0 || depth_return == 0) {
    // test important car X plante sur XCreatePixmap si une dimension
    // est nulle
    return null;
  }

  UNatWin *altbuf = new UNatWin();
  if (!altbuf 
      || (altbuf->xwin = XCreatePixmap(natgraph->xdisplay,
				       // vvv sert a trouver le Screen
				       natwin->xwin,
				       r.width, r.height, 
				       depth_return)
	  ) == None
      ) {
    uwarning("UGraph::createAltBuffer", errmess);
    if (altbuf) delete altbuf;
    return null;
  }

  return altbuf;
}


void UGraph::beginDoubleBuffering(const URegion &r) {
  if (doublebuf != null) {
    uwarning("UGraph::beginDoubleBuffering",
	     "Already in Double Buffering mode ==> request ignored");
    return;
  }

  doublebuf = createAltBuffer(r, "Not enough memory to enter Double Buffering mode ==> request ignored");

  if (doublebuf) {
    natgraph->setDest(doublebuf, -r.x, -r.y);
    natgraph->setClip(0, 0, r.width, r.height); 
  }
}

void UGraph::beginTranspBuffering(const URegion &r) {
  if (transpbuf != null) 
    deleteAltBuffer(transpbuf);

  transpbuf = createAltBuffer(r, "Not enough memory to enter Transparent mode ==> request ignored");

  if (transpbuf) {
    natgraph->setDest(transpbuf, -r.x, -r.y);
    natgraph->setClip(0, 0, r.width, r.height);  
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UGraph::deleteAltBuffer(UNatWin *&altbuf)  {
  //NB: le XFreePixmap est necessaire car ~UNatWin ne detruit pas xwin
  // (xwin etant soit un XPixmap soit une XWindow)
  if (altbuf->xwin != None) {

    // NOTE:                                   !!! TEMPORAIRE : A REVOIR !!!

    XFreePixmap(natgraph->xdisplay, altbuf->xwin);
    altbuf->xwidget = NULL;
    altbuf->xwin = None;
    delete altbuf;
  } 
  // on fait ce qui suit dans tous les cas par securite
  altbuf = null;
}

void UGraph::endDoubleBuffering(const URegion &r) {
  if (doublebuf == null) {
    //il y a des cas (region de taille nulle) ou il n'y avait rien 
    //a afficher et il est alors normal que doublebuf soit null
    //--> ne pas afficher de message d'erreur
    //uwarning("UGraph::endDoubleBuffering",
    //	     "Not in Double Buffering mode ==> request ignored");
    natgraph->setDest(natwin, 0, 0);  //!!!
    return;
  }

  if (doublebuf->xwin != None
      && doublebuf->xwin != natwin->xwin) { //double securite!

    natgraph->setClip(0, 0, boxview->getWidth(), boxview->getHeight());
    XCopyArea(natgraph->xdisplay, 
	      doublebuf->xwin,  //from
	      natwin->xwin,     //to (!att: PAS dest->xwin)
	      natgraph->gc,
	      0, 0,
	      r.width, r.height,
	      //r.x, r.y plus sur ?
	      -natgraph->x_win, -natgraph->y_win);
  }
  deleteAltBuffer(doublebuf);
  natgraph->setDest(natwin, 0, 0);  //!!!
}

void UGraph::endTranspBuffering(const URegion &r) {
  if (transpbuf == null) {
    uwarning("UGraph::endTranspBuffering",
	     "Not in Transparent mode ==> request ignored");
    if (doublebuf) natgraph->setDest(doublebuf, 0, 0);  //!!!
    else natgraph->setDest(natwin, 0, 0);  //!!!
    return;
  }

  XImage *xima1 = null, *xima2 = null;

  //ne pas faire setDest tout de suite
  UNatWin *dest;         // where graphics are drawn
  if (doublebuf) {
    dest = doublebuf;
    //fprintf(stderr, " - - - get bg from doublebuf\n");
  } 
  else {
    dest = natwin;
    //fprintf(stderr, " - - - get bg from win\n");
  }

  if (transpbuf->xwin && transpbuf->xwin != dest->xwin) { //double securite!
  
    // verif dest et dest->xwin NOT null
    natgraph->setClip(0, 0, boxview->getWidth(), boxview->getHeight());
    /*
    fprintf(stderr, "\nii=%d :: dest %d / dest.xwin %d (realized %d) :: r.x %d  r.y %d - r.w %d r.h %d\n",
	    ii, dest, dest->xwin, dest->isRealized(),
	    r.x, r.y, r.width, r.height);
    */

    //en fait ce getimage devrait etre fait avant/
    // -- 1 pour recuperer la bonne taille de fenetre
    // -- 2 pour faire une recopie dans le pixmap dans le cas des Shaped

    Window root_return;
    int x_return, y_return;
    unsigned int width_return, height_return, border_width_return, depth_return;
    //Status stat =
      XGetGeometry(natgraph->xdisplay,
		   dest->xwin,  //DEST!
		   &root_return, &x_return, &y_return,
		   &width_return, &height_return, &border_width_return,
		   &depth_return);
    /*
    fprintf(stderr, "EndTransp: stat %d / win %d /w %d h %d d %d \n",
	    stat,  dest->xwin, width_return, height_return, depth_return);
    */

    //test important car X plante sur XGetImage si dim trop grandes
    //if (stat != True || r.width <= 0 || r.height <= 0 || depth_return == 0)

    //LE CALCUL EST ERRONE dans le cas doublebuf : il faut les   !!!!!!!!
    //coords exactes avec le shift et non partir a O,O           !!!!!!!!

    URegion clip(r);
    int clipstat = clip.setInter(0, 0, width_return, height_return);

    if (clipstat == 0) {
      fprintf(stderr, " - - - !!PAS d INTERSECT\n");
      xima1 = null;
    }
    else {
      UXtry xtry;   // sert a catcher les X Errors safely
      //NB: le pbm ici c'est que les xwin ne sont pas forcement mappees
      //et completement visible (or X plante dans ces cas)

      xima1 = XGetImage(natgraph->xdisplay, dest->xwin,
			clip.x, clip.y,  clip.width, clip.height, 
			AllPlanes, ZPixmap);

      //fprintf(stderr, " - - - draw fg from pix\n");
      xima2 = XGetImage(natgraph->xdisplay, transpbuf->xwin,
			0, 0,  r.width, r.height, AllPlanes, ZPixmap);

      if (!xtry.status()) {
	printf("can't get X Image \n");
	xima1 = xima2 = NULL;
      }
    }
  }
    
  deleteAltBuffer(transpbuf);
  if (doublebuf) natgraph->setDest(doublebuf, 0, 0);  //!!!
  else natgraph->setDest(natwin, 0, 0);  //!!!

  // ce qui suit DOIT etre apres le setDest!
  if (!xima1 || !xima2) fprintf(stderr, "Null xima\n");
  else {
    XImage *blend = UNatIma::blend_XImages(disp->natdisp,
					   xima1, xima2,
					   0.75);
    if (!blend) {
      fprintf(stderr, "null blend \n");
      blend = xima2;
    } 

    //attention natima detruit blend!!!
    //UNatIma natima(disp->natdisp, blend, (XImage*)NULL);
    //drawIma(&natima, 0, 0);

    XPutImage(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	      blend,
	      0, 0,	     //from in source ima
	      r.x, r.y,    //to in target view
	      r.width, r.height);  // size
  }
  if (xima1) XDestroyImage(xima1);
  if (xima2) XDestroyImage(xima2); 
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// must be called when drawing is finished with the gmode
// that was returned by 'begin'

void UWinGraph::end() {}

void UWinGraph::endAll(int gmode, const URegion &r_obj) {
  switch(gmode) {
  case ON:
  case OFF:
    //nop;
    break;
  case SHIFTED:
    subwin = false;
      //!! va deconner dans cas TRANSP!
    natgraph->setDest(natwin, 0, 0);
    break;
  case BLENDED:
    endTranspBuffering(r_obj);
    break;
  default:
    uerror("UWinGraph::end", "invalid mode");
    break;
  }
}

//sets a shift (for sub hardwins)
int UWinGraph::beginShifted(const URegion& clip, u_pos xshift, u_pos yshift) {
  subwin = true;
  natgraph->setDest(natwin, xshift, yshift);
  begin(clip);
  return SHIFTED;
}

//starts blending (for transparency)
int UWinGraph::beginBlended(const URegion& clip, const URegion& r_obj) {
  //!att a l'ordre
  begin(clip);
  beginTranspBuffering(r_obj);
  return BLENDED;
}

int UWinGraph::begin(const URegion& clip) {
  if (transpbuf) {  //!!AREVOIR x_win,y_win partage
    natgraph->setDest(transpbuf);
    natgraph->setClip(clip.x + natgraph->x_win, 
		      clip.y + natgraph->y_win,
		      clip.width, clip.height);
  }
  else if (doublebuf) {  //!!AREVOIR x_win,y_win partage
    natgraph->setDest(doublebuf);
    natgraph->setClip(clip.x + natgraph->x_win, 
		      clip.y + natgraph->y_win,
		      clip.width, clip.height);
  }
  else if (subwin) {
    natgraph->setDest(natwin);
    natgraph->setClip(clip.x + natgraph->x_win, 
		      clip.y + natgraph->y_win,
		      clip.width, clip.height);
  }
  else {
    natgraph->setDest(natwin);
    natgraph->setClip(clip);
  }
  return ON;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UNatGraph::setClip(const URegion &r) {
#ifdef GL
  ?????????????
#else
  cliprect.x = r.x;
  cliprect.y = r.y;
  cliprect.width  = r.width;
  cliprect.height = r.height;
  XSetClipRectangles(xdisplay, gc, 0,0, &cliprect, 1, Unsorted);
#endif
}

void UNatGraph::setClip(u_pos x, u_pos y, u_dim width, u_dim height) {
#ifdef GL
  ?????????????
#else
  cliprect.x = x;
  cliprect.y = y;
  cliprect.width  = width;
  cliprect.height = height;
  XSetClipRectangles(xdisplay, gc, 0,0, &cliprect, 1, Unsorted);
#endif
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
//!!! BIEN EVIDEMMENT ce x_win,y_win partage va deconner!!
// si on utilse 2 transpbuf ou 1 transpbuf et un doublebuf
// ==> faut utiliser le end() ...
// de plus ca marche pas si element semi-transparents

//!!ATTENTION: pas le meme SetClip pour WinGraph et Graph !!
// note: UWinGraph: clip absolu dans la Window View

void UWinGraph::setClip(const URegion &r) {
  if (doublebuf||transpbuf||subwin)
    natgraph->setClip(r.x + natgraph->x_win,  //!!AREVOIR x_win,y_win partage
		      r.y + natgraph->y_win,
		      r.width, r.height);
  else
    natgraph->setClip(r);
}

void UWinGraph::setClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  if (doublebuf||transpbuf||subwin)
    natgraph->setClip(x + natgraph->x_win,   //!!AREVOIR x_win,y_win partage
		      y + natgraph->y_win,
		      width, height);
  else
    natgraph->setClip(x, y, width, height);
}

// setClip and setWinClip are similar for this class: 
// -- coordinates are relative to the origin of the WINDOW
// -- the view is no taken into account

void UWinGraph::setWinClip(const URegion &r) { 
  setClip(r);
}
void UWinGraph::setWinClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  setClip(x, y, width, height);
}

/* ==================================================== [Elc:01] ======= */
// UGraph: noter que le Clip est relatif a la View attachee
// - NB: verifier que le Clip est bien inclus dans la View et prendre le min
// -- !ATT: on ne tient pas compte dy cas ou r.x ou r.y  < 0  !!!

// !!ATT: DOUBLEBUFF et TRANSPBUFF pas pris en compte !!!

void UGraph::setClip(const URegion &r) {
  URegion xrect;
  if (r.x >= boxview->getWidth() || r.y >= boxview->getHeight()) {
    xrect.width = 0; xrect.height = 0;
  }
  else {
    // keep the min
    if (r.x + r.width > boxview->getWidth())
      xrect.width = boxview->getWidth() - r.x;
    else
      xrect.width = r.width;

    if (r.y + r.height > boxview->getHeight())
      xrect.height = boxview->getHeight() - r.y;
    else
      xrect.height = r.height;
  }

  // !ATT: on ne tient pas compte dy cas ou r.x ou r.y  < 0 
  //xrect.x = boxview->getXwin() + r.x;
  //xrect.y = boxview->getYwin() + r.y;

  natgraph->setClip(boxview->getXwin() + r.x,
		    boxview->getYwin() + r.y,
		    xrect.width,
		    xrect.height);
}

void UGraph::setClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  URegion r(x, y, width, height);
  setClip(r);
}

void UGraph::setWinClip(const URegion &r) {
  URegion r2(r.x - boxview->getXwin(), r.y - boxview->getYwin(),
	    r.width, r.height);
  setClip(r2);
}

void UGraph::setWinClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  URegion r2(x - boxview->getXwin(), y - boxview->getYwin(),
	    width, height);
  setClip(r2);
}

/* ==================================================== [Elc:01] ======= */
// ==== Coordinates ======================================================

// width and heigh of the current View
u_dim UGraph::getWidth()  {return boxview->getWidth();}
u_dim UGraph::getHeight() {return boxview->getHeight();}

// conversion from local View coords. to global Window coords.
u_pos UGraph::XToXwin(u_pos x_in_view) {return x_in_view + boxview->getXwin();}
u_pos UGraph::YToYwin(u_pos y_in_view) {return y_in_view + boxview->getYwin();}

// conversion from global Window coords. to local View coords.
u_pos UGraph::XwinToX(u_pos x_in_win) {return x_in_win - boxview->getXwin();}
u_pos UGraph::YwintoY(u_pos y_in_win) {return y_in_win - boxview->getYwin();}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// NOTE: UWinGraph en coords ABSOLUES dans la Window
//       et UGraph en coords LOCALES de la View

// !IMPORTANT:
// Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
// sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
// seront incorrectes (c'est le cas de clearRect, drawPolyline, drawPolygon..)

void UGraph::drawPoint(u_pos x, u_pos y) {
#ifdef GL
  glBegin(GL_POINTS);
  glVertex2i(x, -y);
  glEnd();
#else
  XDrawPoint(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	     natgraph->x_win + x, natgraph->y_win + y);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawLine(u_pos x1, u_pos y1, u_pos x2, u_pos y2) {
#ifdef GL
  glBegin(GL_LINE_STRIP);
  glVertex2i(natgraph->x_win + x1, -(natgraph->y_win + y1));
  glVertex2i(natgraph->x_win + x2, -(natgraph->y_win + y2));
  glEnd();
#else
  XDrawLine(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	    natgraph->x_win + x1, natgraph->y_win + y1, 
	    natgraph->x_win + x2, natgraph->y_win + y2);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawString(const UStr *s, u_pos x, u_pos y) {
  if (s) drawString(s->chars(), s->length(), x, y);
}

void UGraph::drawString(const char *str, int str_len, u_pos x, u_pos y) {
  if (str && str_len > 0) {
#ifdef GL
    glPushAttrib(GL_LIST_BIT | GL_CURRENT_BIT);
    glRasterPos2f((float)(natgraph->x_win+x), 
  		  (float)-(natgraph->y_win+y+natgraph->font->ascent));
    glListBase(natgraph->glFontList);
    glCallLists(str_len, GL_UNSIGNED_BYTE, str);
    glPopAttrib();
#else
    XDrawString(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc,
		natgraph->x_win + x, 
		natgraph->y_win + y + natgraph->font->ascent, 
		str, str_len); 
#endif
  }
}

/* ==================================================== ======== ======= */

void UGraph::drawRect(u_pos x, u_pos y, u_dim w, u_dim h) {
#ifdef GL
  glBegin(GL_LINE_LOOP);
  glVertex2i(natgraph->x_win + x    , -(natgraph->y_win + y));
  glVertex2i(natgraph->x_win + x + w, -(natgraph->y_win + y));
  glVertex2i(natgraph->x_win + x + w, -(natgraph->y_win + y + h));
  glVertex2i(natgraph->x_win + x    , -(natgraph->y_win + y + h));
  glEnd();
#else
  XDrawRectangle(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
		 natgraph->x_win + x, natgraph->y_win + y, w, h);
#endif
}

void UGraph::fillRect(u_pos x, u_pos y, u_dim w, u_dim h) {
#ifdef GL
  //glBegin ??
  glRecti(natgraph->x_win + x    , -(natgraph->y_win + y), 
  	  natgraph->x_win + x + w, -(natgraph->y_win + y + h));
  //glEnd ??
#else
  XFillRectangle(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
		 natgraph->x_win + x, natgraph->y_win + y, w, h);
#endif
}

// remplit le border ?
void UGraph::fillView() {
  fillRect(0, 0, boxview->getWidth(), boxview->getHeight());
}

/* ==================================================== ======== ======= */

void UGraph::drawArc(u_pos x, u_pos y, u_dim w, u_dim h,
		     float angle1, float angle2) {
#ifdef GL
  float rw    = w/2.0;
  float rh    = h/2.0;
  float delta = 1.0*M_PI/180 * (angle1-angle2);
  float alpha = 1.0+M_PI/180 * angle1;
  int step    = int(((w+h) * (w+h) * (angle1-angle2)) / (90*w*h));
  if (step < 0) step = -step;

  glBegin(GL_LINE_STRIP);
  for(int i=0; i<=step; i++)
    glVertex2f(rw*(cos(alpha+delta*i/step)+1) + natgraph->x_win+x, 
	       rh*(sin(alpha+delta*i/step)-1) - natgraph->y_win-y);
  glEnd();
#else
  XDrawArc(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	   natgraph->x_win + x, natgraph->y_win + y, w, h, 
	   (int)(angle1 * 64.), (int)(angle2 * 64.));     //!!att: au *64 !
#endif
};

void UGraph::fillArc(u_pos x, u_pos y, u_dim w, u_dim h,
		     float angle1, float angle2) {
#ifdef GL
  float rw    = w/2.0;
  float rh    = h/2.0;
  float delta = 1.0*M_PI/180 * (angle1-angle2);
  float alpha = 1.0+M_PI/180 * angle1;
  int step    = int(((w+h) * (w+h) * (angle1-angle2)) / (90*w*h));
  if (step<0) step = -step;

  glBegin(GL_POLYGON);
  for(int i=0; i<=step; i++)
    glVertex2f(rw*(cos(alpha+delta*i/step)+1)+x, 
	       -(rh*(sin(alpha+delta*i/step)-1)+y));
  glEnd();
#else
  XFillArc(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc,
	   natgraph->x_win + x, natgraph->y_win + y, w, h, 
	   (int)(angle1 * 64.), (int)(angle2 * 64.));  //!!att: au *64 !
#endif
};

/* ==================================================== [Elc:01] ======= */
// ==== Clearing routines ================================================
// Note: these functions use the background Color
//       and are NOT altered by the XOR mode

void UGraph::clearView(u_bool repaint_view) {
  if (repaint_view) {
    if (boxview) boxview->updatePaint();
  }
  else clearRect(0, 0, boxview->getWidth(), boxview->getHeight());
}

// -- if 'repaint_view' is true : repaints the normal content of the view
//    in the specified region
// -- otherwise: fills the area with the UGraph's background Color
// Note: this function is not altered by the XOR mode

void UGraph::clearRect(u_pos x, u_pos y, u_dim w, u_dim h, 
		       u_bool repaint_view) {
  if (repaint_view) {
    URegion rr(x, y, w, h);
    if (boxview) boxview->updatePaint(&rr);
  }
  else {
#ifdef GL
    setGLcolor(natgraph->bgcolor);   
    glRecti(natgraph->x_win + x    , -(natgraph->y_win + y), 
	    natgraph->x_win + x + w, -(natgraph->y_win + y + h));
    
    // retour a la normale
    if (xorMode) setGLcolor(natgraph->color ^ natgraph->bgcolor);
    else setGLcolor(natgraph->color);
  }
#else
    // sets the Paint mode in any case
    XGCValues gcval;
    gcval.foreground = natgraph->bgcolor;
    gcval.function   = GXcopy;
    XChangeGC(natgraph->xdisplay, natgraph->gc, 
	      GCFunction|GCForeground, &gcval);
    
    XFillRectangle(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
		   natgraph->x_win + x, natgraph->y_win + y, w, h);
    
    // retour a la normale
    if (xorMode) {
      gcval.foreground = natgraph->color ^ natgraph->bgcolor;
      gcval.function   = GXxor;
      XChangeGC(natgraph->xdisplay, natgraph->gc, 
		GCFunction|GCForeground, &gcval);
    }
    else {
      gcval.foreground = natgraph->color;
      XChangeGC(natgraph->xdisplay, natgraph->gc, GCForeground, &gcval);
    }
  }
#endif
}

void UGraph::copyArea(u_pos x, u_pos y, u_dim w, u_dim h,
		      u_dim delta_x, u_dim delta_y,
		      u_bool generate_refresh_events_when_obscured) {
  XGCValues gcval;
  if (generate_refresh_events_when_obscured) {
    gcval.graphics_exposures = true;
    XChangeGC(natgraph->xdisplay, natgraph->gc, GCGraphicsExposures, &gcval);
  }

  XCopyArea(natgraph->xdisplay, natgraph->dest->xwin, 
	    natgraph->dest->xwin, natgraph->gc,
	    natgraph->x_win + x, natgraph->y_win + y, 
	    w, h,
	    natgraph->x_win + x + delta_x, 
	    natgraph->y_win + y + delta_y);

  if (generate_refresh_events_when_obscured) {
    gcval.graphics_exposures = false;
    XChangeGC(natgraph->xdisplay, natgraph->gc, GCGraphicsExposures, &gcval);
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// !IMPORTANT:
// Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
// sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
// seront incorrectes (c'est le cas de drawPolyline, drawPolygon, fillPolygon)

// drawPolyline draws a sequence of connected lines:

void UGraph::drawPolyline(u_pos xpoints[], u_pos ypoints[], u_card card) {
#ifdef GL
  glBegin(GL_LINE_STRIP);
  for(int k=0; k<card; k++)
    glVertex2i(xpoints[k] + natgraph->x_win,
	       -(ypoints[k] + natgraph->y_win));
  glEnd();
#else
  //XDrawLines vraiment plus efficace que boucle avec XDrawLine ?
  XPoint points[POLYCARD];
  for (int shift = 0 ; shift < card; shift += POLYCARD) {
    int subcard = (card-shift <= POLYCARD) ? card-shift : POLYCARD;
    for (int k = 0; k < subcard; k++) {
      points[k].x = xpoints[k + shift] + natgraph->x_win;
      points[k].y = ypoints[k + shift] + natgraph->y_win;
    }
    XDrawLines(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	       points, card, CoordModeOrigin);
  }
#endif
}

/* ==================================================== ======== ======= */
// drawPolygon draws a polygon (which is automatically CLOSED):

void UGraph::drawPolygon(u_pos xpoints[], u_pos ypoints[], u_card card) {
#ifdef GL
  glBegin(GL_LINE_LOOP);
  for(int k=0; k<card; k++)
    glVertex2i(xpoints[k] + natgraph->x_win,
	       -(ypoints[k] + natgraph->y_win));
  glEnd();
#else
  //XDrawLines vraiment plus efficace que boucle avec XDrawLine ?
  XPoint points[POLYCARD+1];
  for (int shift = 0 ; shift < card; shift += POLYCARD) {
    int subcard = (card-shift <= POLYCARD) ? card-shift : POLYCARD;
    for (int k = 0; k < subcard; k++) {
      points[k].x = xpoints[k] + natgraph->x_win;
      points[k].y = ypoints[k] + natgraph->y_win;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XDrawLines(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, 
	       points, card+1, CoordModeOrigin);
  }
#endif
}

/* ==================================================== ======== ======= */
// fillPolygon fills a closed polygon (which is automatically CLOSED):
// NB: 'mode' can be set for optimisation purpose. 
// - default, mode = 0 = Complex : the polygon can have an arbitrary shape
// - mode = 1 = Nonconvex : assumes a non Convex but non Intersecting polygon.
// - mode =2  = Convex    : assumes a Convex polygon. 

// GL WARNING !!!
// with openGL, others modes than 2 are not supported for the moment  !!!
// be carefull when using this function !!!

void UGraph::fillPolygon(u_pos xpoints[], u_pos ypoints[],
			 u_card card, int mode) {
#ifdef GL
  if (mode != 2) {
    uwarning("UGraph::FillPolygon", 
	     "non convex polygons not supported with OpneGLn");
  }
  glBegin(GL_POLYGON);
  for(int k=0; k<card; k++)
    glVertex2i(xpoints[k] + natgraph->x_win,
	       -(ypoints[k] + natgraph->y_win));
  glEnd();
#else
  if (card <= POLYCARD) {
    XPoint points[POLYCARD+1];
    for (int k = 0; k < card; k++) {
      points[k].x = xpoints[k] + natgraph->x_win;
      points[k].y = ypoints[k] + natgraph->y_win;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
  }
  else {
    XPoint *points = (XPoint*)malloc((card+1) * sizeof(XPoint));
    for (int k = 0; k < card; k++) {
      points[k].x = xpoints[k] + natgraph->x_win;
      points[k].y = ypoints[k] + natgraph->y_win;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
    free(points);
  }
#endif
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

static int charDefaultWidth(UX_Font fs) {
  // va savoir pourquoi : parfois default_char vaut n'importe quoi !
  if (fs->default_char >= fs->min_char_or_byte2 
      && fs->default_char <= fs->max_char_or_byte2)
    return fs->per_char[fs->default_char].width;
  else
    return fs->per_char[0].width;
}

int UGraph::getCharPos(const UFontDesc *font, const char *s, 
		       int len, u_pos x) {
  int charpos = 0;
  UX_Font fs = disp->natdisp->getXFont(font);
  if (!fs) return 0;

  XCharStruct *cs;
  int width;
  int xpos = 0;

  for (charpos = 0; charpos < len; charpos++) {
    //!!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[charpos];
    
    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) width = cs->width;
      else width = charDefaultWidth(fs);
    }
    else width = charDefaultWidth(fs);

    //printf("--cwidth=%d\n\n",  width);
    if (x <= xpos + width / 2)  break;
    else xpos += width;
  }

  // lie a getSubTextSize: si on tombe sur un seprateur a la fin
  // alors il faut renvoyer position precedente car ce separateur n'est 
  // pas affiche car il y a un saut de ligne implicite
  if (len > 0 && isspace(s[len-1])) charpos--;
  return charpos;
}

/* ==================================================== ======== ======= */

int UGraph::getXPos(const UFontDesc *font, 
		    const char *s, int len, int charpos) {
  UX_Font fs = disp->natdisp->getXFont(font);
  if (!fs) return 0;
  /*
  printf("--getXPos: min_char_or_byte2 %d - max_char_or_byte2 %d  - min_byte1 %d - max_byte1 %d -  ll_chars_exist %d - n_properties %d\n ",
	 fs->min_char_or_byte2, fs->max_char_or_byte2,
	 fs->min_byte1, fs->max_byte1,  fs->all_chars_exist,
	 fs->n_properties);
  */

  if (charpos > len || charpos < 0) charpos = len;
  XCharStruct *cs;
  int xpos = 0;

  for (int k = 0; k < charpos; k++) {
    //!!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[k];

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) xpos += cs->width;
      else xpos += charDefaultWidth(fs);
    }
    else xpos += charDefaultWidth(fs);
  }
  return xpos;
}

/* ==================================================== ======== ======= */

int UGraph::getTextHeight(const UFontDesc *font) {
  UX_Font fs = disp->natdisp->getXFont(font);
  if (!fs) {
    uerror("UGraph::getTextHeight", "!Can't retreive Font");
    return 0;
  }
  else return fs->ascent + fs->descent;
}

/* ==================================================== ======== ======= */

int UGraph::getTextWidth(const UFontDesc *font, const char *s, int len) {
  UX_Font fs = disp->natdisp->getXFont(font);
  if (!fs) {
    uerror("UGraph::getTextWidth", "!Can't retreive Font");
    return 0;
  }
  else return XTextWidth(fs, s, len);
}

/* ==================================================== ======== ======= */

void UGraph::getTextSize(const UFontDesc *font, const char *s, int len, 
			 u_dim *w, u_dim *h) {
  UX_Font fs;
  if (!(fs = disp->natdisp->getXFont(font))) {
    *w = 0; *h = 0;
    uerror("UGraph::getTextSize", "!Can't retreive Font");
    return;
  }
  *w = XTextWidth(fs, s, len);
  *h = fs->ascent + fs->descent;
}

/* ==================================================== ======== ======= */
// return the substring length
int UGraph::getSubTextSize(const UFontDesc *font, const char *s, int len, 
			   u_dim maxwidth, u_dim chw,
			   u_dim *w, u_dim *h) {
  UX_Font fs;
  if (!(fs = disp->natdisp->getXFont(font))) {
    *w = 0; *h = 0;
    uerror("UGraph::getSubTextSize", "!Can't retreive Font");
    return 0;
  }

  *h = fs->ascent + fs->descent;
  *w = 0;
  int charcount = 0;
  int currentw = 0, lastw = 0;
  //int cw_moyen = (fs->min_bounds.width + fs->max_bounds.width) / 2;
  int cw;

  for (int charpos = 0; charpos < len; charpos++) {
    //!!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[charpos];
    //int cw = cw_moyen;
    lastw = currentw;

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      XCharStruct *cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) cw = cs->width;
      else cw = charDefaultWidth(fs);
    }
    else cw = charDefaultWidth(fs);

    currentw += cw;
    chw += cw;

    //if (c == ' ' || c == '\n' || c == '\t' || c == 13) {
    if (isspace(c)) {

      if (c == 13) {   ///????
	if (chw-cw != 0) {
	  // en theorie, 0, en pratique, il faut le retour a la ligne !
	  if (charcount == len-1) *w = -1;
	  else *w = -currentw-1;
	  charcount = charpos + 1;
	  return charcount;
	} 
	else {
	  lastw -= cw;
	  currentw -= cw;
	  chw -= cw;
	  charcount = charpos + 1;
	}
      }

      else {      
	// on n'a pas encore depasse' la limite => continuer, sauf si '\n'
	if (currentw < maxwidth) {
	  *w = currentw;
	  charcount = charpos+1;
	  if (c == '\n') {
	    *w = -currentw-1;	// negatif pour le retour a la ligne
	    return charcount;
	  }
	}
	else {
	  // enlever le separ : si ca depasse encore ET si on a deja
	  // trouve un separ precedent alors renvoyer le precedent
	  if (charcount > 0 && lastw >= maxwidth) 
	    return charcount;
	  else {			// sinon renvoyer separ courant
	    *w = currentw;
	    charcount = charpos+1;
	    return charcount;
	  }
	}
      }
    } // endif(c == ...)
  } // endfor

  // cas ou on n'a trouve aucun separateur : on renvoit tout
  *w = currentw;
  return charcount = len;
}

/* ==================================================== [Elc:01] ======= */
// ==== Images and Pixmaps =============================================

void UGraph::getImaSize(class UNatIma *nima, u_dim &w, u_dim &h) {
  w = nima->getWidth();
  h = nima->getHeight();
}

void UGraph::getPixSize(class UNatPix *nima, u_dim &w, u_dim &h) {
  w = nima->getWidth();
  h = nima->getHeight();
}

void UGraph::drawIma(const class UIma *ima, u_pos x, u_pos y) {
#ifdef GL
    drawGLTex(ima->getNatIma(), x, y);
#else
  const UPix *pix = dynamic_cast<const UPix*>(ima);
  if (pix && pix->getNatPix())
    drawIma(pix->getNatPix(), x, y);
  else
    drawIma(ima->getNatIma(), x, y);
#endif
}

/* ==================================================== ======== ======= */

#ifdef GL
static void drawGLIma(const class UNatPix *pix, u_pos x, u_pos y) {
  GLfloat x1 = natgraph->x_win + x;
  GLfloat y1 = -(natgraph->y_win + y);
  GLfloat x2 = x1 + pix->width;
  GLfloat y2 = y1 - pix->height;

  glPushAttrib(GL_CURRENT_BIT);
  glEnable(GL_TEXTURE_2D);
  //glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glEnable(GL_BLEND);

  glBindTexture(GL_TEXTURE_2D, pix->glIma);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glBegin(GL_POLYGON);
  glTexCoord2f(0.0, 0.0); glVertex2f(x1, y1);     
  glTexCoord2f(1.0, 0.0); glVertex2f(x2, y1);     
  glTexCoord2f(1.0, 1.0); glVertex2f(x2, y2);      
  glTexCoord2f(0.0, 1.0); glVertex2f(x1, y2);
  glEnd();

  glDisable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);  
  glPopAttrib();
}

static void drawGLPix(const class UNatIma *ima, u_pos x, u_pos y) {
  GLfloat x1 = natgraph->x_win + x;
  GLfloat y1 = -(natgraph->y_win + y);
  GLfloat x2 = x1 + ima->xima->width;
  GLfloat y2 = y1 - ima->xima->height;

  glPushAttrib(GL_CURRENT_BIT);
  glEnable(GL_TEXTURE_2D);
  //glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glEnable(GL_BLEND);

  glBindTexture(GL_TEXTURE_2D, ima->glIma);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glBegin(GL_POLYGON);
  glTexCoord2f(0.0, 0.0); glVertex2f(x1, y1);     
  glTexCoord2f(1.0, 0.0); glVertex2f(x2, y1);     
  glTexCoord2f(1.0, 1.0); glVertex2f(x2, y2);      
  glTexCoord2f(0.0, 1.0); glVertex2f(x1, y2);
  glEnd();

  glDisable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);
  glPopAttrib();
}
#endif

/* ==================================================== ======== ======= */

void UGraph::drawIma(const UNatPix *pix, u_pos x, u_pos y) {
  // calculer la zone de clip du pixmap sur la destination
  // (c'est en particulier necessaire s'il y a un mask car XSetClipMask 
  // prend le mask entier sans tenir compte du cliprect

  int clip_x = natgraph->cliprect.x - (x + natgraph->x_win);
  int clip_y = natgraph->cliprect.y - (y + natgraph->y_win);

  // mask => effet de transparence
  if (pix->xpixshape != None) {
    XSetClipOrigin(natgraph->xdisplay, natgraph->gc, 
		   natgraph->x_win + x, 
		   natgraph->y_win + y);
    XSetClipMask(natgraph->xdisplay, natgraph->gc, pix->xpixshape);
  }

  XCopyArea(natgraph->xdisplay, pix->xpix, natgraph->dest->xwin, natgraph->gc,
	    clip_x, clip_y,	     //from
	    natgraph->cliprect.width, 
	    natgraph->cliprect.height,  // size
	    natgraph->cliprect.x, 
	    natgraph->cliprect.y);  //to
  
  // !att: reset du Clip pour la suite
  if (pix->xpixshape != None) {
    XSetClipRectangles(natgraph->xdisplay, natgraph->gc, 
		       0,0, &natgraph->cliprect, 1, Unsorted);
  }
}

/* ==================================================== ======== ======= */

void UGraph::drawIma(const UNatIma *ima, u_pos x, u_pos y) {

  // optimisation du clipping: 
  // on ne peut pas se contenter d'utiliser le GC pour le clipping
  // car il y a un transfert client -> serveur de l'image
  // et il faut donc clipper l'image source AVANT le transfert 
  // pour minimiser la quaantite de donnees a transferer

  // ima rect: rectangle de la location de l'ima dans target view
  // si l'image n'etait pas clippee
  URegion imarect(x, y, ima->xima->width, ima->xima->height);

  // intersection d'imarect avec le clip courant
  u_bool clipstat = imarect.setInter(natgraph->cliprect.x - natgraph->x_win, 
				     natgraph->cliprect.y - natgraph->y_win,
				     natgraph->cliprect.width, 
				     natgraph->cliprect.height);
  // rien a afficher
  if (!clipstat) return;

  // calcul du decalage dans l'image d'origine:
  // eg.: si imarect.x == x apres le clip, il faut afficher l'image
  // depuis son origine, sinon il y a une partie a ne pas afficher
  // d'ou un decalage
  XPutImage(natgraph->xdisplay, natgraph->dest->xwin, natgraph->gc, ima->xima, 
	    imarect.x - x,	              //from in source ima
	    imarect.y - y,
	    imarect.x+ natgraph->x_win,      //to in target view 
	    imarect.y+ natgraph->y_win,
	    imarect.width, imarect.height);  // clipped size of source
}

/* ==================================================== [Elc:00] ======= */
// ==== OffScreen ========================================================

/*
UOffScreen::UOffScreen(UDisp *d, u_dim h, u_dim w) {
  natwin = null;
  //natdisp = d->getNatDisp();
  disp = d;
  height = h; 
  width = w;

  UAppli *a = null;
  UFrame *f = null;

  if (!(a = d->getAppli()) || !(f = a->getMainFrame())) {
    uerror("UOffScreen::UOffScreen",
	   "UAppli or Main UFrame not yet created");
    return;
  }

  UViewIndex ix = f->getViewIndex();
  UView *fview = null;
  UNatWin *fwin = null;

  if (!(fview = f->getView(ix)) || !(fwin = fview->wg().getNatWin())) {
    uerror("UOffScreen::UOffScreen",
	   "Main UFrame not yet realized");
    return;
  }

  Window root_return;
  int x_return, y_return;
  unsigned int width_return, height_return, border_width_return, depth_return;

  XGetGeometry(disp->getNatDisp()->getXDisplay(),
	       fwin->xwin,
	       &root_return, &x_return, &y_return,
	       &width_return, &height_return, &border_width_return,
	       &depth_return);

  if (! (natwin = new UNatWin(fview))
      || !(natwin->xwin = XCreatePixmap(disp->getNatDisp()->getXDisplay(),
					// vvv sert a trouver le Screen
					fwin->xwin,
					width, height, 
					depth_return))
      ) {
    uerror("UOffScreen::UOffScreen",
	   "Not enough memory => can't create OffScreen");
  }
}

UOffScreen::~UOffScreen() {
  if (natwin) {
    if (natwin->xwin) 
      XFreePixmap(disp->getNatDisp()->getXDisplay(), natwin->xwin);

    //NB: le XFreePixmap est necessaire car 'delete doublebuf' ne detruit
    //pas les Pixmaps (seulement les XWindos associees aux Widgets)
    delete natwin;
  }
  natwin = null;
}
*/

/*
UGraph::UGraph(UOffScreen *ofs) {
  if (!ofs) {
    boxview = null;
    //natdisp = null;
    disp = null;
    natwin = null;
    natgraph = null;
    uerror("UGraph::UGraph", "!Null UOffScreen");
    return;	// !faudrait une exception!
  }

  // !'xorMode', 'thickness' must be initialized before lock()
  xorMode = false;
  thickness = 0;
  boxview = null;
  //natdisp = ofs->natdisp;
  disp    = ofs->disp;
  natwin  = ofs->natwin;
  doublebuf = null;  // no double buffering by default
  transpbuf = null;

  // shared UNatGraphs(s) [one for each UDisp]
  // !att: ne convient pas si multi-threads du reaffichage standard
  // !att egalement aux couleurs et fonts qui peuvent varier suivant displays
  if (disp->natdisp->defgraph->isLocked()) {
    // deja pris => creer un nouveau NatGraph
    natgraph = new UNatGraph(disp->natdisp);
    natgraph->lock(this);
  }
  else {
    // utiliser le NatGraph standard
    natgraph = disp->natdisp->defgraph;
    natgraph->lock(this);
    //...reset eventuel du GC
  }

  natgraph->setDest(natwin, 0, 0);
  natgraph->setClip(0, 0, ofs->width, ofs->height);
}
*/

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
