/* ==================================================== ======== ======= *
 *
 *  uunat.cc			// native windows
 *  Ubit Project [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uunat.cc	ubit:b1.11.6"
#include <ubrick.hh>
#include <uconfig.hh>
#include <ufont.hh>
#include <ucolor.hh>
#include <upix.hh>
#include <ucursor.hh>
#include <ustr.hh>
#include <uview.hh>
#include <uviewImpl.hh>
#include <unat.hh>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <uwin.hh>
#include <uwinImpl.hh>
#include <uappli.hh>
#include <utimer.hh>
#include <ugraph.hh>
#include <ucontext.hh>
#include <ucall.hh>
#include <umenu.hh>

#define CURSOR_MAP_DEFAULT_SIZE   256
#define VIRTUAL_COLORMAP_DEFAULT_SIZE   256

/*
static const char* NN(const char *s) {
  return s ? s : "none";
}
UnknownNatPixWidth 18
UnknownNatPixHeight 18
static char UnknownNatPixData[] = {
 0xff,0xff,0x03,0xff,0xff,0x03,0xff,0xff,0x03,0x3f,0xf0,0x03,0x8f,0x8f,0x03,
 0xc7,0x1f,0x03,0xe7,0x3f,0x03,0x73,0x7e,0x02,0x3b,0x67,0x02,0x39,0x67,0x02,
 0x99,0x73,0x02,0x99,0x31,0x03,0x39,0x05,0x03,0x33,0x84,0x03,0x63,0xce,0x03,
 0x07,0xe3,0x03,0x0f,0xe0,0x03,0x3f,0xf0,0x03
};
*/
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UNatGraph::UNatGraph(UNatDisp *natdisp) {
  graph_lock = null;
  xorMode = false;
  thickness = 0;
  x_win = y_win = 0;
  dest = null;

  xdisplay = natdisp->getXDisplay();
  font = null;
  color = BlackPixelOfScreen(natdisp->getXScreen());
  bgcolor = WhitePixelOfScreen(natdisp->getXScreen());

  XGCValues gcval;
  gcval.function   = GXcopy;
  gcval.foreground = color;
  gcval.background = bgcolor;
  // graphics_exposures = true means that events are generated when
  // XCopyArea or XCopyPlanes is called and a part of the view is obscured
  // This is useless here except when scrolling data (the last argument 
  // of the UGraph::copyArea function can be used to change this
  // behavior (see fct. copyArea() in ugraph.hh for details)
  gcval.graphics_exposures = false;

  //Note: The GC can be used with any destination drawable having 
  //the same root and depth as the specified drawable. 
  gc = XCreateGC(xdisplay, natdisp->getXWindow(),
		 GCFunction | GCForeground | GCBackground | GCGraphicsExposures,
		 &gcval);
}


UNatGraph::~UNatGraph() {
  if (gc != None) XFreeGC(xdisplay, gc);
}

//inlined: void UNatGraph::setDest(UNatWin *d) {dest = d;}

void UNatGraph::setDest(UNatWin *d, u_pos x_origin, u_pos y_origin) {
  dest  = d;
  x_win = x_origin;
  y_win = y_origin;
}

void UNatGraph::lock(UGraph* graph) {
  ///if (graph_lock) graph_lock->natgraph = null; // !!!
  graph_lock = graph;

  XGCValues gcval;
  if (xorMode != graph->xorMode) {
    xorMode = graph->xorMode;
    gcval.function = (xorMode ? GXxor : GXcopy);
    XChangeGC(xdisplay, gc, GCFunction, &gcval);
  }
  if (thickness != graph->thickness) {
    gcval.line_width = thickness = graph->thickness;
    XChangeGC(xdisplay, gc, GCLineWidth, &gcval);
  }
}

void UNatGraph::unlock() {
  graph_lock = null;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UNatWin::UNatWin() {
  xwidget = null;
  xwin = None;
}

UNatWin::~UNatWin() {
  xwidget = null;
  xwin = None;
  // NOTE:                                   !!! TEMPORAIRE : A REVOIR !!!
  // on ne detruit pas xwin ici mais dans UWinGraph ou endDoubleBuffering
  // deux raisons:
  // a) xwin est soit une XWindow soit un XPimap
  // b) la destruction des XWindow est recursive ce qui pourrait
  //    aboutir a des destructions multiples de la meme XWindow
  // alors il faudra le detruire explicitement a l'endroit ad hoc
  // (c'est typiquement ce qui est fait dans endDoubleBuffering()
  // pour detruire le Pixmap intermediaire)
  //xwin = None;
}

/* ==================================================== ======== ======= */

void UNatWin::where(UNatDisp *natdisp, u_pos &_x, u_pos &_y) {
  /*ifdef WITH_XT
  if (xwidget) {
    Position xpos, ypos;
    XtVaGetValues(xwidget, XtNx, (XtArgVal)&xpos, XtNy, (XtArgVal)&ypos, NULL);
    _x = xpos; _y = ypos;
  }
  else 
  */
  if (xwin == None) {_x = -1; _y = -1;}
  else {
    //if (natdisp && natdisp->getXDisplay()) {
    //!NB: XGetGeometry ne marche pas dans le cas des Shells car ceux-ci
    //sont inclus dans une fenetre intermediaire cree par le WM
    Window childw;
    int rootx, rooty;
    XTranslateCoordinates(natdisp->getXDisplay(), xwin, 
			  RootWindowOfScreen(natdisp->getXScreen()), 
			  0, 0, &rootx, &rooty, &childw);
    _x = rootx; _y = rooty;
  }
}

void UNatWin::move(UNatDisp *natdisp, u_pos x, u_pos y) {
  /*ifdef WITH_XT
  if (xwidget) {
    XtVaSetValues(xwidget, XtNx, (XtArgVal)x, XtNy, (XtArgVal)y, NULL);
  }
  else 
  */
  if (xwin != None) XMoveWindow(natdisp->getXDisplay(), xwin, x, y);
}

void UNatWin::resize(UNatDisp *natdisp, u_dim w, u_dim h) {
  /*ifdef WITH_XT
  if (xwidget) {
    XtVaSetValues(xwidget, XtNwidth, (XtArgVal)w, XtNheight, (XtArgVal)h, NULL);
  }
  else
  */
  if (xwin != None) XResizeWindow(natdisp->getXDisplay(), xwin, w, h);
}

void UNatWin::show(UNatDisp *natdisp, u_bool state) {
  /*ifdef WITH_XT
  if (xwidget) {
    if (state) {
      XtPopup(xwidget, XtGrabNone);
      //XMapRaised(XtDisplay(xwidget), xwin);
    }
    else XtPopdown(xwidget);
  }
  else
  */
  if (xwin != None) {
    if (state) XMapRaised(natdisp->getXDisplay(), xwin);
    else XUnmapWindow(natdisp->getXDisplay(), xwin);
  }
}

void UNatWin::setTitle(UNatDisp*nd, const char*s) {
  /*ifdef WITH_XT
   if (xwidget)
     XtVaSetValues(xwidget, XtNtitle, (XtArgVal)s, NULL);
   else 
  */
  XStoreName(nd->getXDisplay(), xwin, s);
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

static void initWinAttr(UNatDisp *nd, 
			XSetWindowAttributes& wattr, 
			unsigned long &wattr_mask) {
  wattr_mask = 0;

  //mettre colormap appropriee par rapport au Visual sinon plantage!
  if (nd->getXColormap() != None) {
    wattr.colormap = nd->getXColormap();
    wattr_mask |= CWColormap;
  }

  //default Pixmap with same depth as UNatDisp
  //INDISPENSABLE, sinon XCreateWindow plante si depth != rootWindow's depth
  //MEME si border_width = 0 !
  wattr.border_pixmap = nd->getDefaultPix();
  wattr_mask |=CWBorderPixmap;

  //transparent background
  wattr.background_pixmap = None;
  wattr_mask |= CWBackPixmap;

  //wattr.background_pixel = WhitePixelOfScreen(nd->getXScreen());
  //wattr_mask |= CWBackPixel;

  //wattr.border_pixel = BlackPixelOfScreen(nd->getXScreen());
  //wattr_mask |= CWBorderPixel;

  // curseur standard
  wattr.cursor = nd->getXCursor(&UCursor::arrow);
  wattr_mask |= CWCursor;
}

/* ==================================================== ======== ======= */

u_bool UNatWin::realizeDialog(UNatDisp *nd, UNatWin *nw, UWin *win) {
  /*ifdef WITH_XT
    return UNatWin::realizeFrame(nd, nw, win);
  */
  if (UNatWin::realizeFrame(nd, nw, win)) {
    // l'iconification du MainFrame entraine l'iconification de ce Dialog
    XSetTransientForHint(nd->getXDisplay(),
	 		 nw->xwin,
			 nd->getXWindow());
    return true;
  }
  else return false;
}

u_bool UNatWin::realizeFrame(UNatDisp *nd, UNatWin *nw, UWin *win) {
  if (!nd || nd->getXWindow() == None) {
    uerror("UNatWin::realizeDialog", UError::CANT_REALIZE_WINDOW);
    return false;
  }
  XSetWindowAttributes wattr;
  unsigned long wattr_mask;

  initWinAttr(nd, wattr, wattr_mask);

  nw->xwidget = NULL;
  nw->xwin = XCreateWindow(nd->getXDisplay(),
			   RootWindowOfScreen(nd->getXScreen()),//fenetre mere
			   0,0,	         // x,y position
			   10, 10,       // width , height (0,0 crashes!)
			   0,	         // border_width
			   nd->getDepth(),
			   InputOutput,  // Class
			   nd->getXVisual(),
			   wattr_mask,
			   &wattr);
  if (nw->xwin != None) {
    manageEvents(nd, nw, win);
    return true;
  }
  else {
    uerror("UNatWin::realizeDialog", UError::CANT_CREATE_X_WINDOW);
    return false;
  }

  /*ifdef WITH_XT
  Widget xshell;
  if (!nd || !(xshell = nd->getXWidget())) {
    uerror("UNatWin::realizeDialog", UError::CANT_REALIZE_WINDOW);
    return false;
  }
  nw->xwidget = XtVaCreatePopupShell
    ("UDialog", 
     transientShellWidgetClass, 
     xshell,
     // le prog peut changer la taille de la window
     XtNallowShellResize, (XtArgVal)True,
     // n accepte pas le keyboard sans cette resource
     XtNinput,    (XtArgVal)True,

     // imposer valeurs specifiees par UNatDisp (ne pas laisser X
     // prendre des valeurs par defaut plus ou moins appropriees)
     XtNvisual,	  (XtArgVal)nd->getXVisual(),
     XtNcolormap, (XtArgVal)nd->getXColormap(),
     XtNdepth,	  (XtArgVal)nd->getDepth(),

     // pour eviter pbms ouverture de taille (0,0)
     XtNheight,	  (XtArgVal)10,
     XtNwidth,	  (XtArgVal)10,
     XtNborderWidth, (XtArgVal)0,

     // transparent background
     XtNbackgroundPixmap, (XtArgVal)None,
     NULL);

  if (nw->xwidget) {
    // ATT: Realize the widget BEFORE drawing and accessing the X Window
    XtRealizeWidget(nw->xwidget);
    nw->xwin = XtWindow(nw->xwidget);
    manageEvents(nd, nw, win);
  }
  if (nw->xwin != None) return true;
  else {
    uerror("UNatWin::realizeDialog", UError::CANT_CREATE_X_WINDOW);
    return false;
  }
  */
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// !!NB: un et un seul Main Frame (les autres Frames sont crees par la
// fct realizeFrame

u_bool UNatWin::realizeMainFrame(UNatDisp *nd, UNatWin *nw, UWin *win) {
  Widget xshell;
  if (!nd || !(xshell = nd->getXWidget())) {
    uerror("UNatWin::realizeMainFrame", UError::CANT_REALIZE_WINDOW);
    return false;
  }

  nw->xwidget = xshell;	    // the toplevel Xt widget itself
  if (nw->xwidget) {
    // rend la fenetre visible
    XtVaSetValues(nw->xwidget, XtNmappedWhenManaged, (XtArgVal)True, NULL);
    // ATT: Realize the widget BEFORE drawing and accessing the X Window
    // deja fait XtRealizeWidget(xwidget);
    nw->xwin = XtWindow(nw->xwidget);
    manageEvents(nd, nw, win);
  }
  if (nw->xwin != None) return true;
  else {
    uerror("UNatWin::realizeMainFrame", UError::CANT_CREATE_X_WINDOW);
    return false;
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

u_bool UNatWin::realizeMenu(UNatDisp *nd, UNatWin *nw, UWin *win) {
  if (!nd || nd->getXWindow() == None) {
    uerror("UNatWin::realizeMenu", UError::CANT_REALIZE_WINDOW);
    return false;
  }
  XSetWindowAttributes wattr;
  unsigned long wattr_mask;

  initWinAttr(nd, wattr, wattr_mask);

  //window pas geree par le WM (no borders...) car c'est un menu
  wattr.override_redirect = TRUE;
  wattr_mask |= CWOverrideRedirect;

  //sauve ce qui est sous le le menu pour economiser les reaffichages
  wattr.save_under = TRUE;
  wattr_mask |= CWSaveUnder;

  nw->xwidget = NULL;
  nw->xwin = XCreateWindow(nd->getXDisplay(),
			   RootWindowOfScreen(nd->getXScreen()),//fenetre mere
			   0, 0,         // x,y position
			   10, 10,       // width, height (0,0 crashes!)
			   0,	         // border_width
			   nd->getDepth(),
			   InputOutput,  // Class
			   nd->getXVisual(),
			   wattr_mask, &wattr);

  if (nw->xwin != None) {
    manageEvents(nd, nw, win);
    return true;
  }
  else {
    uerror("UNatWin::realizeMenu", UError::CANT_CREATE_X_WINDOW);
    return false;
  }

  /*ifdef WITH_XT
  Widget xshell;
  if (!nd || !(xshell = nd->getXWidget())) {
    uerror("UNatWin::realizeMenu", UError::CANT_REALIZE_WINDOW);
    return false;
  }
  nw->xwidget = XtVaCreatePopupShell
    ("UMenu",
     //overrideShellWidgetClass,
     shellWidgetClass,
     xshell,
     // window pas geree par le WM (no borders, etc)
     XtNoverrideRedirect, (XtArgVal)True,
     // sauver contenu du menu pour economiser les reaffichages
     XtNsaveUnder, (XtArgVal)True,

     // le prog peut changer la taille de la window
     XtNallowShellResize, (XtArgVal)True,
     // n'accepte pas le keyboard sans cette resource
     XtNinput, (XtArgVal)True,

     // imposer valeurs specifiees par UNatDisp (ne pas laisser X
     // prendre des valeurs par defaut plus ou moins appropriees)
     XtNvisual,	  (XtArgVal)nd->getXVisual(),
     XtNcolormap, (XtArgVal)nd->getXColormap(),
     XtNdepth,	  (XtArgVal)nd->getDepth(),

     // pour eviter pbms ouverture de taille (0,0)
     XtNheight, (XtArgVal)10,
     XtNwidth,  (XtArgVal)10,
     XtNborderWidth, (XtArgVal)0,

     // transparent background
     XtNbackgroundPixmap, (XtArgVal)None,
     NULL);

  if (nw->xwidget) {
    // ATT: Realize the widget BEFORE drawing and accessing the X Window
    XtRealizeWidget(nw->xwidget);
    nw->xwin = XtWindow(nw->xwidget);
    Cursor curs = nd->getXCursor(&UCursor::arrow);
    XDefineCursor(nd->getXDisplay(), nw->xwin, curs);
    manageEvents(nd, nw, win);
  }

  if (nw->xwin != None) return true;
  else {
    uerror("UNatWin::realizeMenu", UError::CANT_CREATE_X_WINDOW);
    return false;
  }
  */
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

u_bool UNatWin::realizeIncrust(UNatDisp *nd, UNatWin *nw, UWin *win){
  UIncrust *ic = dynamic_cast<UIncrust*>(win);

  if (!ic || !nd || !(nd->getXWidget())) {
    uerror("UNatWin::realizeIncrust", UError::CANT_REALIZE_WINDOW);
    return false;
  }
  nw->xwidget = null; //!att

  if (ic->is_external_win) {
    // incrusts a preexisting external X Window
    nw->xwin = ic->xwin;
    if (nw->xwin == None) {
      uerror("UNatWin::realizeIncrust", 
	     "Null external X Window passed as an argument\n ==> can't manage this window");
      return false;
    }
    else {
      //?? XSelectInput(nd->getXDisplay(),xwin, ExposureMask);
    }
  }
  
  else {  // !is_external_win ==> creates a true X Window 
    XSetWindowAttributes wattr;
    unsigned long wattr_mask;

    initWinAttr(nd, wattr, wattr_mask);

    // il faut rendre cette window sensible aux ExposeEvent car sa
    // parent window ne les recevra pas
    wattr.event_mask = ExposureMask; 
    wattr_mask |= CWEventMask;

    //!ATT BUG: c'est la Window du Disp qui est prise comme parent !BUG!
    //(cad le main frame). Normalement ca devrait etre la Window
    //qui contient reellement cette window

    ic->xwin = nw->xwin = XCreateWindow(nd->getXDisplay(),
					nd->getXWindow(), // Window du Disp!
				    // offsets of this win from toplevel origin
				    // (set by function GuiCreateWidgets)
					0, 0,
					10, 10, 0,  //avoid size = (0,0)!
					nd->getDepth(),
					(unsigned int)InputOutput, 
					nd->getXVisual(),
					wattr_mask, &wattr); 
    if (nw->xwin == None) {
      uerror("UNatWin::realizeIncrust", UError::CANT_CREATE_X_WINDOW);
      return false;
    }
    XMapWindow(nd->getXDisplay(), nw->xwin);
  }

  ic->add(UOn::viewPaint / ucall(nw, &UNatWin::reshapeCB, nd, ic->winview));
  return true;
}

void UNatWin::reshapeCB(UNatDisp *nd, UView *winview) {
  move(nd, winview->getXwin(), winview->getYwin());
  resize(nd, winview->getWidth(), winview->getHeight());
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// Event Management

// Error handler qui ne fait rien : ca sert a eviter des catastrophes
// dans certains cas (X etant particuliermenet bien concu en ce qui
// concerne la gestion des erreurs comme chacun sait ...)

//malheureusement cette var. est static car quietErrorHandler doit l'etre
int UXtry::xerror;

int UXtry::quietErrorHandler(Display*, XErrorEvent *xer) {
  xerror = xer->error_code;
  return 0;
}

static long MaxSelRequestSize(Display *disp) {
  long l = XMaxRequestSize(disp);
  if (l > 65536) l = 65536;  // min
  return (l << 2) - 100;
}

/* ==================================================== ======== ======= */
// Text selection (cut and paste)

void UNatWin::on_selection(UAppli *appli, UWin *win, UView*, XEvent *xev){
  //NB: no time!
  //fprintf(stderr, ">configureEH type=%d\n",ev->type);

  switch (xev->type) {
  case SelectionClear: {
      // effacer la selection
      UTextsel &ts = appli->getTextsel();
      ts.reset(true);
    } break;

  case SelectionRequest:
    // Cas ou une autre application demande a obtenir la valeur de la 
    // selection courante (detenue par 'this' appli)

    // on verifie qu'il s'agit bien de la section PRIMARY et le type
    // demande par le requestor (STRING) puis:
    // -- on change la property adequate avec la nouvelle valeur
    //    de la selection
    // -- puis on envoie un XSelectionEvent de confirmation ou d'echec
    //    DANS TOUS LES CAS y compris si on n'a rien change!

    // NOTES:
    // -- il peut y avoir plusieurs echanges: certaines applis peuvent
    //    d'abord demander les donnes sous un type specifique (par exple
    //    propre a Qt, etc.), puis, en cas d'echec (si on a renvoye
    //    un SelectionEvent d'echec), redemander la selection sous une
    //    autre forme
    // -- selection != property
    //    selection = PRIMARY = 1 tandis que property depend de l'appli
    //    qui demande les donnees
    {
      XSelectionRequestEvent *xsel = (XSelectionRequestEvent*)xev;
      u_bool property_changed = false;

      if (xsel->selection == XA_PRIMARY
	  && xsel->target == XA_STRING
	  ) {

	//fprintf(stderr, "SelRequest: xsel->selection %d, xsel->property %d, xsel->target %d, xsel->requestor %d \n", xsel->selection, xsel->property, xsel->target, xsel->requestor);

	UTextsel &ts = appli->getTextsel();
	UStr selval;
	ts.copyTo(selval);

	// troncature si la taille de selval est trop grande
	int req_max_length = MaxSelRequestSize(xsel->display);
	int nitems = selval.length();
	if (nitems > req_max_length) nitems = req_max_length;

	// NOTE: pour bien faire il faudrait faire une boucle pour
	// pouvoir envoyer les messages de longue taille qui excedent
	// les limites du serveur X (ie. MaxSelRequestSize)
	
	XChangeProperty(xsel->display, xsel->requestor, 
			xsel->property, xsel->target, 
			8/*format*/, PropModeReplace/*mode*/,
			(unsigned char*)selval.chars(), nitems);	
	property_changed = true;
      }

      // On envoie un XSelectionEvent de CONFIRMATION ou d'ECHEC
      // DANS TOUS LES CAS y compris si on n'a rien change!

      XSelectionEvent event_send;
      event_send.type       = SelectionNotify;
      event_send.serial     = xsel->serial;
      event_send.time       = xsel->time;
      event_send.send_event = xsel->send_event;
      event_send.display    = xsel->display;
      event_send.requestor  = xsel->requestor;
      event_send.selection  = xsel->selection;
      event_send.target     = xsel->target;

      // send and event with property == None if nothing changed
      if (property_changed)
	event_send.property = xsel->property;  // cas OK
      else
	event_send.property = None;            // cas d'echec
      
      XSendEvent(xsel->display, xsel->requestor, 
		 0,  //propagate
		 0,  //event_mask
		 (XEvent*)&event_send);
    }
    break;

  case SelectionNotify:
    {
      // cas ou 'this' application a demande le contenu de la selection
      // par XConvertSelection. celui-ci 'arrive' via un (ou des)
      // SelectionNotify event(s)
      XSelectionEvent *xsel = (XSelectionEvent*)xev;

      if (xsel->property != None
	  && xsel->selection == XA_PRIMARY
	  && xsel->target == XA_STRING
	  && appli->server_selection_str != null
	  ) {

	//fprintf(stderr, "SelNotify: xsel->selection %ld, xsel->property %ld, xsel->target %ld, xsel->requestor %ld \n", xsel->selection, xsel->property, xsel->target, xsel->requestor);
	
	long req_length = MaxSelRequestSize(xsel->display);
	long req_offset = 0;

	Atom type_return;
	int format_return;
	unsigned long nitems_return = 0, bytes_after_return = 1;
	unsigned char *prop_return = null;
   
	// NOTE: pour bien faire il faudrait faire une boucle pour
	// pouvoir recuperer les messages de longue taille qui excedent
	// les limites du serveur X (ie. MaxSelRequestSize)

	// ATT: dans ce cas il ne faudrait appeler XGetWindowProperty
	// avec delete_property == FALSE puis appeler XDeleteProperty
	// qunad tout a ete recupere

	Bool req_stat =
	  XGetWindowProperty(xsel->display, xsel->requestor, xsel->property,
			     req_offset, req_length,
			     TRUE,/*delete property*/
			     AnyPropertyType,/*req_type= Any type*/
			     &type_return, &format_return, 
			     &nitems_return, &bytes_after_return,
			     &prop_return);
	
	if (req_stat == Success
	    && nitems_return > 0
	    && prop_return != null
	    && type_return == XA_STRING
	    && format_return == 8
	    ) {
	  appli->server_selection_str
	    ->insert(appli->server_selection_pos, (char*)prop_return, true);
	}

	//desallouer la memoire allouee par XGetWindowProperty
	if (prop_return != null) XFree ((char*)prop_return);
      }

      // DEVRAIT debloquer appli !!!!
      appli->server_selection_str = null;
      appli->server_selection_pos = 0;
    }
    break;
  }
}

/* ==================================================== ======== ======= */

void UNatWin::on_message(UAppli* a, UWin* win, UView* winview, XEvent* xev) {
  UNatDisp *natdisp = a->getNatDisp();

  if (win->isDef(UMode::MESSAGE_CB, 0)) {
    UEvent e(UEvent::message, winview, xev);
    e.setViewSource(winview);
    //call callbacks
    win->fire(e, UOn::message);
  }

  //fermeture des windows par le WM:
  // on a clique' sur le bouton "Close' du WM et celui-ci le fait savoir
  if (xev->xclient.message_type == natdisp->atoms.WM_PROTOCOLS
      && ((Atom)xev->xclient.data.l[0]) == natdisp->atoms.WM_DELETE_WINDOW
      ){
    //this fct. hides or kills the UWin depending on subclass
    //(and possible redefinitions of the fct. by client)
    win->close();
  }

  //else fprintf(stderr, ">Other ClientMessage\n");
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// NB: une autre option serait de combiner les refresh events et ne reafficher 
// qu'une seule fois une zone complexe (if (ev->xexpose.count == 0) ...)

void UNatWin::on_expose(UAppli*, UWin* win, UView* winview, XEvent* xev){
  UEvent e(UEvent::viewPaint, winview, xev);
  e.setViewSource(winview);
  /*
  printf("expose win = %ld : %d,%d - %dx%d \n", xev->xexpose.window,
	 xev->xexpose.x, xev->xexpose.y, 
	 xev->xexpose.width, xev->xexpose.height);
  */
  switch (xev->type) {
  case Expose:
    // ne reafficher que la zone correspondant au xexpose
    e.redrawClip.set(xev->xexpose.x, xev->xexpose.y, 
		     xev->xexpose.width, xev->xexpose.height);
    break;
  case GraphicsExpose:
    // these events are generated when XCopyArea or XCopyPlanes is called 
    // and a part of the view is obscured. They are useful for repainting
    // these parts when scrolling data.
    // see fct. copyArea() in ugraph.hh for details
    e.redrawClip.set(xev->xgraphicsexpose.x, xev->xgraphicsexpose.y, 
		     xev->xgraphicsexpose.width, xev->xgraphicsexpose.height);
    break;
  case NoExpose:
  default:
    //nop
    return;
  }
  e.redrawStatus = true;
  win->updateView(e, winview, UUpdate::PAINT);
}

/* ==================================================== ======== ======= */
// moves and resizes

void UNatWin::on_configure(UAppli*, UWin* win, UView* winview, XEvent* xev){
  // PROBLEMES: 
  // -1- on ne peut savoir si c'est un move ou un resize 
  //     (ou autre chose comme chgt taille border)
  //  ==> on ne traite les cas ou la taille a change

  // -2- les XMoveWindow et les XResizeWindow generent des
  //     events X qui retournent a l'appli un peu plus tard ...
  //     ==> il n'y a donc aucun moyen de savoir (?) si c'est l'appli
  //     qui a fait le move/resize ou si c'est l'utilisateur
  //     ce qui peut entrainer des bouclages infinis ...
  //  ==> on teste WIN_MAPPED pour ne prendre les configure en compte
  //     que lorsque la window est mapped (voir MapNotify/UnmapNotify))

  if (win->isDef(0,UMode::WIN_MAPPED)
      && (winview->getWidth() != xev->xconfigure.width
	  || winview->getHeight() != xev->xconfigure.height)
      ) {
    win->resize(xev->xconfigure.width, xev->xconfigure.height);
  }
}
/* ==================================================== ======== ======= */

void UNatWin::on_misc(UAppli*, UWin* win, UView*, XEvent* xev) {
  switch (xev->type) {
  case MapNotify:		// la fenetre apparait
    win->setCmodes(UMode::CAN_SHOW, true);
    //ceci permet de ne pas prendre en compte les Configure Events
    // tant que la wndow n'a pas apparu (voir on_configure)
    win->setCmodes(UMode::WIN_MAPPED, true);
    break;

  case UnmapNotify:		// la fenetre disaparait
    win->setCmodes(UMode::CAN_SHOW, false);
    win->setCmodes(UMode::WIN_MAPPED, false);
    break;

  case DestroyNotify:		// la fenetre est detruite
    //fprintf(stderr, ">DestroyNotify!!\n");
    break;

  case PropertyNotify:
    //fprintf(stderr, ">propertyNotify\n");
    break;

  case ColormapNotify:
    //fprintf(stderr, ">ColormapNotify\n");
    break;
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
  /*
  printf("press x %d y %d - xwin %d ywin %d (%d %d) - source %p %s win %p %s\n",
	 e.getX(), e.getY(), e.getXwin(), e.getYwin(), e.xmouse, e.ymouse,
	 e.getSource(), 
	 e.getSource() ? e.getSource()->cname() : "none",
	 e.getWin(), 
	 e.getWin() ? e.getWin()->cname() : "none");
  */

void UNatWin::on_mpress(UAppli *a, UWin *win, UView *winview, XEvent *xev) {
  UEvent e(UEvent::mpress, winview, xev);
  e.time = xev->xbutton.time;
  e.locateSource(xev->xbutton.x, xev->xbutton.y);
  a->winMousePress(win, e);
}

void UNatWin::on_mrelease(UAppli *a, UWin *win, UView *winview, XEvent *xev) {
  UEvent e(UEvent::mrelease, winview, xev);
  e.time = xev->xbutton.time;
  e.locateSource(xev->xbutton.x, xev->xbutton.y);
  a->winMouseRelease(win, e);
}

void UNatWin::on_mmove(UAppli *a, UWin *win, UView *winview, XEvent *xev) {
  UEvent e(UEvent::mmove, winview, xev);
  e.time = xev->xmotion.time;
  e.locateSource(xev->xmotion.x, xev->xmotion.y);
  a->winMouseMove(win, e);
}

/* ==================================================== ======== ======= */

void UNatWin::on_kpress(UAppli *a, UWin *win, UView *winview, XEvent *xev){
  UEvent e(UEvent::kpress, winview, xev);
  e.time = xev->xkey.time;
  e.locateSource(xev->xkey.x, xev->xkey.y);
  a->winKeyPress(win, e);
}

void UNatWin::on_krelease(UAppli *a, UWin *win, UView *winview, XEvent *xev){
  UEvent e(UEvent::krelease, winview, xev);
  e.time = xev->xkey.time;
  e.locateSource(xev->xkey.x, xev->xkey.y);
  a->winKeyRelease(win, e);
}

/* ==================================================== ======== ======= */

void UNatWin::on_enter(UAppli *a, UWin *win, UView *winview, XEvent *xev){
  // la fenetre dans laquelle on rentre prend le focus
  // c'est en particulier obligatoire pour les menus sinon ils ne prennent
  // pas les entrees clavier
  if (xev->xcrossing.window != None) {
    UXtry xtry;  // sert a catcher les X Errors safely
    //aussi bizarre que ca paraisse, X est tellement mal fait que le seul
    //moyen d'eviter un plantage c'est de redefinir l'ErrorHandler via UXtry
    
    //fprintf(stderr, "TakeFocus xwin %ld \n", xev->xcrossing.window);
    XSetInputFocus(xev->xcrossing.display, xev->xcrossing.window,
		   RevertToParent, CurrentTime);
    // synchroniser pour que X detecte l'erreur maintenant, en mode
    // protege en non 10 mn plus tard!
    XSync(xev->xcrossing.display, False);

    //NB: c'est sans doute pas suffisant: il faut reattribuer le focus
    //ensuite au bon objet
  }

  // !!!! COMPLETER: EVENTS PERDUS !!!!
  // UEvent e(....)
  // e.time = xev->xcrossing.time;

  // CORRECT: ne PAS faire: win->boxMotion(boxlink,ev);
  // incompatible avec gestion des grabbedMenu !
}

void UNatWin::on_leave(UAppli *a, UWin *win, UView *winview, XEvent *xev){
  UEvent e(UEvent::leave, winview, xev);
  e.time = xev->xcrossing.time;
  e.locateSource(xev->xcrossing.x, xev->xcrossing.y);

  // enorme bug corrige ci-dessous qui generait des LeaveEvent (X type 8)
  // qui etaient traites comme des MotionEvent (X type 6) et foutaient
  // un bordel immonde dans la gestion des boxMotion
  // FAUX-EX: winview->getWin()->boxMotion(a, &e);  <- faux!
  a->winLeave(win, e);
}

void UNatWin::on_focus(UAppli *a, UWin *win, UView *winview, XEvent *xev){
  /*!!!                                         PBM avec certains WMs ???
  if (xev->type == FocusIn)
    fprintf(stderr, ">Focus In!!\n");
  else if (xev->type == FocusOut)
    fprintf(stderr, ">Focus Out!!\n");
  else 
    fprintf(stderr, ">Focus ???\n");
  */
}

void UNatWin::on_any_event(UAppli* a, UWin* win, UView* winview, XEvent* xev){
  //call callback sur un event X quelconque
  UEvent e(UEvent::anyEvent, winview, xev);
  e.setViewSource(winview);
  win->fire(e, UOn::anyEvent);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UNatDisp::dispatchEvents(XEvent *xev) {
  /*ifdef WITH_XT
     XtDispatchEvent(xev);
  */
  UX_Window xwin = xev->xany.window;
  UWin *win = null;

  for (ULink *l = disp->hardwins.first(); l!=null; l = l->next()) {
    win = static_cast<UWin*>(l->brick()); //par construction
    UGraph *g;
    if (win && win->impl.hard 
	&& (g = win->impl.hard->wingraph)
	&& (g->getNatWin()->getXWindow() == xwin)
	) {
      break;
    }
  }
  if (!win) return;

  //printf("-xev %d - win %p %s - xwin %d \n",
  //xev->type, win,win->cname(), xwin);

  UView  *v = win->winview;
  UAppli *a = v->getAppli();

  if (win->isDef(UMode::ANY_EVENT_CB, 0))
    UNatWin::on_any_event(a, win, v, xev);

  switch(xev->xany.type) {
  case MotionNotify:
    UNatWin::on_mmove(a, win, v, xev);
    break;
  case EnterNotify:
      UNatWin::on_enter(a, win, v, xev);
      break;
  case LeaveNotify:
    UNatWin::on_leave(a, win, v, xev);
    break;
  case Expose:
  case GraphicsExpose:
  case NoExpose:
    UNatWin::on_expose(a, win, v, xev);
    break;
  case ConfigureNotify:
    UNatWin::on_configure(a, win, v, xev);
    break;
  case ButtonPress:
    UNatWin::on_mpress(a, win, v, xev);
    break;
  case ButtonRelease:
    UNatWin::on_mrelease(a, win, v, xev);
    break;
  case KeyPress:
    UNatWin::on_kpress(a, win, v, xev);
    break;
  case KeyRelease:
    UNatWin::on_krelease(a, win, v, xev);
    break;
  case FocusIn:
  case FocusOut:
    UNatWin::on_focus(a, win, v, xev);
    break;
  case SelectionClear:
  case SelectionNotify:
  case SelectionRequest:
    UNatWin::on_selection(a, win, v, xev);
    break;
  case ClientMessage:
    UNatWin::on_message(a, win, v, xev);
    break;
  default:
    UNatWin::on_misc(a, win, v, xev);
    break;
  }
}

/* ==================================================== ======== ======= */

void UNatWin::manageEvents(UNatDisp* nd, UNatWin *nw, UWin* win){
  //recuperer Close fenetre
  XSetWMProtocols(nd->getXDisplay(), 
		  nw->xwin, 
		  &(nd->atoms.WM_DELETE_WINDOW), 1);

  XSelectInput(nd->getXDisplay(),
	       nw->xwin, 
	       PointerMotionMask
	       | ExposureMask
	       //NB: GrapicsExpose, NoExpose -> field graphics_expose of GC
	       | EnterWindowMask
	       | LeaveWindowMask
	       | ButtonPressMask
	       | ButtonReleaseMask
	       | KeyPressMask
	       | KeyReleaseMask
	       | FocusChangeMask
	       | KeymapNotify
	       | ColormapChangeMask
	       | PropertyChangeMask
	       //VisibilityChangeMask
	       //NB: ne pas utiliser ResizeRedirectMask: c'est pour le WM
	       //et ca fait deconner les Reise, Expose, etc.
	       // ResizeRedirectMask
	       | StructureNotifyMask
	       //SubstructureNotifyMask
	       //SubstructureRedirectMask
	       );

  /*ifdef WITH_XT
  // la souris sort ou on entre dans le Shell Window
  XtAddEventHandler(nw->xwidget, EnterWindowMask, False, enter, win);
  XtAddEventHandler(nw->xwidget, LeaveWindowMask, False, leave, win);
  // focus
  //XtAddEventHandler(nw->xwidget,FocusChangeMask,False,focusEH,(XtPointer)win);
  // fonction de gestion souris
  XtAddEventHandler(nw->xwidget, ButtonPressMask, False, mpress,(XtPointer)win);
  XtAddEventHandler(nw->xwidget, ButtonReleaseMask,False,mrelease,(XtPointer)win);
  // actif meme si bouton pas appuye
  XtAddEventHandler(nw->xwidget, PointerMotionMask,False,mmove,(XtPointer)win);
  // fonction de gestion clavier
  XtAddEventHandler(nw->xwidget, KeyPressMask, False, kpress,(XtPointer)win);
  XtAddEventHandler(nw->xwidget, KeyReleaseMask,False,krelease,(XtPointer)win);
  // fonction de reaffichage
  // (NB: les GraphicsExpose sont geres par configureEH)
  XtAddEventHandler(nw->xwidget, ExposureMask, False, refresh, (XtPointer)win);
  // la fenetre apparait / disparait, change de taille, etc
  // NB: ne PAS utiliser ResizeRedirectMask qui est destine au WM
  XtAddEventHandler(nw->xwidget, StructureNotifyMask, True,
  		    configure,(XtPointer)win);
  */
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* *** Fonts and FontFamilys *** */
/* ==================================================== ======== ======= */

UX_Font UNatDisp::getXFont(const UFontDesc *f) {
  if (f->family->ix >= fontFamilyCount || !fontFamilyMaps[f->family->ix])
    realizeFontFamily(f->family);

  if (!fontFamilyMaps[f->family->ix][f->font_ix]) realizeFont(f);

  // dans tous les cas (pas de else)
  return fontFamilyMaps[f->family->ix][f->font_ix];
}

u_bool UNatDisp::realizeFontFamily(const UFontFamily *family) {
  // increase the 'fontFamilyMaps' vector
  if (family->ix >= fontFamilyCount) {
    int newCount = family->ix + 1;
    if (!fontFamilyMaps)
      fontFamilyMaps =
	(UX_Font**)malloc(newCount * sizeof(UX_Font*));
    else
      fontFamilyMaps =
	(UX_Font**)realloc(fontFamilyMaps, newCount * sizeof(UX_Font*));

    if (!fontFamilyMaps) {
      uerror("UNatDisp::realizeFontFamily", 
	     "!Fatal Error: Not enough memory to augment Font Family List\n - (font family = %s)", family->name);
      // impossible car 'const UFontFamily':  *family = UFontFamily::standard;
      return false;
    }
    else {
      for (int k = fontFamilyCount; k < newCount; k++) fontFamilyMaps[k] = null;
      fontFamilyCount = newCount;
    }
  }

  // allocate the corresponding fontFamilyMap
  if (!fontFamilyMaps[family->ix]) {

    // Faudrait aussi verfier que cette Font Family existe !!!! A_REVOIR!!!

    UX_Font *ffmap =
      (UX_Font*)malloc(UFont::FONTMAP_SIZE * sizeof(UX_Font));

    if (!ffmap) {
      uerror("UNatDisp::realizeFontFamily", 
	     "!Fatal Error: Not enough memory to create Font Family map\n - (font family = %s)", family->name);
      // impossible car 'const UFontFamily':  *family = UFontFamily::standard;
      return false;
    }
    else {
      for (int kk = 0; kk < UFont::FONTMAP_SIZE; kk++) ffmap[kk] = null;
      fontFamilyMaps[family->ix] = ffmap;
    }
  }

  return true;			// everything OK!
}


/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */


u_bool UNatDisp::realizeFont(const UFont *font) {
  UFontDesc f;
  f.set(font);
  return realizeFont(&f);
}


u_bool UNatDisp::realizeFont(const UFontDesc *f) {
  u_bool stat = true;

  // initializes the font family (if needed)
  // Note: fontFamily ne peut etre NULL par construction

  if (f->family->ix >= fontFamilyCount || !fontFamilyMaps[f->family->ix])
    stat = realizeFontFamily(f->family);

  if (fontFamilyMaps[f->family->ix][f->font_ix]) // font already loaded
    return stat;

  UX_Font fs = loadNatFont(f->family, f->styles, f->lsize);
  
  if (fs) fontFamilyMaps[f->family->ix][f->font_ix] = fs;
  else {
    uwarning("UNatDisp::realizeFont", "Can't load Font variant; using default\n - (font family = %s)", f->family->name);
    stat = false;

    // if 'plain' exists (and this is not plain), this entry will point 
    // on the 'plain' entry of the same FontFamily               //!!cf SIZE
    //
    if (f->font_ix != 0 && fontFamilyMaps[f->family->ix][0])
      fontFamilyMaps[f->family->ix][f->font_ix] = fontFamilyMaps[f->family->ix][0];

    // try to load 'plain' of same FontFamily
    //
    else if (f->font_ix != 0 && (fs = loadNatFont(f->family, 0, f->lsize))) {
      // both map entry will point of fs
      fontFamilyMaps[f->family->ix][0] = fs;
      fontFamilyMaps[f->family->ix][f->font_ix] = fs;
    }

    // point on default 'standard' FontFamily (ix = 0 by construction)
    //
    else if (fontFamilyMaps[0][f->font_ix])
      fontFamilyMaps[f->family->ix][f->font_ix] = fontFamilyMaps[0][f->font_ix];
    
    else fontFamilyMaps[f->family->ix][f->font_ix] = fontFamilyMaps[0][0];
  }
  return stat;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

static char *newFontSpec(const char *orig, const char *newfield, int argno) {

  int n1 = 0;			// # chars in 1st substring
				// 2nd substring replace by 'newfield'
  const char *p3 = null;	// pointer of 3rd substring
  int cpt = 0;			// field counter (delimited by '-')

  for (const char *s = orig; *s; s++) {
    if (*s == '-') {
      cpt++;
      if (cpt == argno) n1 = s - orig + 1;
      else if (cpt == argno+1) p3 = s; 
    } 
  }

  // argno est trop grand
  if (n1 == 0) return u_strdup(orig);
  // argno est a la fin => p3 pas trouve car pas de - final
  if (p3 == null) p3 = orig + strlen(orig);

  int ll_newfield = strlen(newfield);
  char *newspec = 
    (char*)malloc((n1 + ll_newfield + strlen(p3) + 1) * sizeof(char));

  strncpy(newspec, orig, n1);
  strcpy(newspec + n1, newfield);
  strcpy(newspec + n1 + ll_newfield, p3);
  return newspec;
}


UX_Font UNatDisp::loadNatFont(const UFontFamily *family, int styles, int lsize) {
  UX_Font fs = null;

  // lsize==0 means "default"  ///!!!!PBM pas de borne min valable !!!
  if (lsize <1) lsize = UFont::MEDIUM_LSIZE; //bug11jan00 lsize = 3;  //!!!DEFAULT
  else if (lsize > UFont::MAX_LSIZE) {
    uerror("UNatDisp::loadNatFont", "Font size is too large");
    lsize = UFont::MAX_LSIZE;
  }
  int s = lsize-1; // !att: shift -1

  if (styles == 0) {		// PLAIN
    //printf("style = plain %s\n\n", family->natName);
    char *spec1 = newFontSpec(family->natName, family->fontSizes[s].natspec, 7);
    fs = XLoadQueryFont(getXDisplay(), spec1);
    free(spec1);
  }

  else if (styles & UFont::BOLD) { 
    if (styles & UFont::ITALIC) {	// BOLD + ITALIC
      char *spec1 = newFontSpec(family->natName, family->fontSizes[s].natspec, 7);
      char *spec2 = newFontSpec(spec1, family->boldStyle, 3);
      char *spec3 = newFontSpec(spec2, family->italicStyle, 4);
      //printf("style BI = %s\n\n", spec2);
      fs = XLoadQueryFont(getXDisplay(), spec3);
      free(spec1);
      free(spec2);
      free(spec3);
    } 
    else {			// BOLD ONLY
      char *spec1 = newFontSpec(family->natName, family->fontSizes[s].natspec, 7);
      char *spec2 = newFontSpec(spec1, family->boldStyle, 3);
      //printf("style B = %s\n\n", spec);
      fs = XLoadQueryFont(getXDisplay(), spec2);
      free(spec1);
      free(spec2);
    }
  }

  else if (styles & UFont::ITALIC) {	//  ITALIC ONLY
    char *spec1 = newFontSpec(family->natName, family->fontSizes[s].natspec, 7);
      char *spec2 = newFontSpec(spec1, family->italicStyle, 4);
    //printf("style I = %s\n\n", spec);
    fs = XLoadQueryFont(getXDisplay(), spec2);
    free(spec1);
    free(spec2);
  }

  return fs;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* *** Cursors *** */
/* ==================================================== ======== ======= */

UX_Cursor UNatDisp::getXCursor(const UCursor *c) {
  if (c->shape < cursorMapSize && cursorMap[c->shape] != None)
    return cursorMap[c->shape];

  else if (realizeCursor(c))
    return cursorMap[c->shape];
  
  else return None; 
}

u_bool UNatDisp::realizeCursor(const UCursor *c) {
  if (c->shape >= cursorMapSize) {
    if (!cursorMap) {
      cursorMapSize = CURSOR_MAP_DEFAULT_SIZE;
      cursorMap = (Cursor*)malloc(cursorMapSize * sizeof(Cursor)); 
      if (!cursorMap) {
	cursorMapSize = 0;
	uerror("NatDisp::realizeCursor", UError::NO_MEMORY);
	return false;
      }
      for (int k = 0; k < cursorMapSize; k++) cursorMap[k] = None;
    }

    else { // on pourrait aussi agrandir
      uerror("NatDisp::realizeCursor", "Cursor shape number is too large");
      return false;
    }
  }

  if (cursorMap[c->shape] == None) {
    cursorMap[c->shape] = XCreateFontCursor(getXDisplay(), c->shape);
  }
  return true;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
/* *** Colors *** */
/* ==================================================== ======== ======= */

UX_Pixel UNatDisp::getXPixel(const UColor *c) {
  // allocate the color if not already done
  if (c->ix == 0 || virtualColormap[c->ix] == (unsigned long)-1)
    realizeColor(c);
  return virtualColormap[c->ix];
}

// this fct is used for changing the virtual colormap dynamically
void UNatDisp::setXPixel(const UColor* c, UX_Pixel natcol) {
  virtualColormap[c->ix] = natcol;
}

// Loads the color designed by '' physically into the 
// VirtualColormap. returns false if color is not available.
// 'color ix' is automatically initialized if ix == 0
// (ix has internal specific values for standard colors)

// !ATT: COMPLETEMENT FAUX EN MULTI DISPLAY CAR LES INDICES NE SERONT
// PAS LES MEMES DANS TOUTES LES TABLES !!!!

u_bool UNatDisp::realizeColor(const UColor *c) {
  XColor xcolor;

  // couleur deja chargee ?
  if (c->ix != 0 && virtualColormap[c->ix] != (unsigned long)-1)
    return true;

  if (!c->natspec) {
    uwarning("realizeColor", "Can't allocate color : null color name",
	     c->natspec);
    return false;
  }


  if (!XParseColor(getXDisplay(), getXColormap(), c->natspec, &xcolor)) {
    uwarning("realizeColor", "Can't allocate color '%s' : unknown color name",
	     c->natspec);
    return false;
  }
  
  // !!!! ICI IL FAUDRAIT CHERCHER UNE COULEUR APPROCHEE DANS LA X-CMAP

  if (!XAllocColor(getXDisplay(), getXColormap(), &xcolor)) {
    uwarning("realizeColor", "Can't allocate color '%s': full colormap",
	     c->natspec);

    // if > to a certain value use white, otherwise, use black
    // [ 65535 is the max val for each color)
    if (xcolor.red + xcolor.blue + xcolor.green > 65535*2)
      c->setIx(UColor::white.ix);
    else
      c->setIx(UColor::black.ix);
 
    return false;
  }

  // trouver automatiquement la valeur de ix si == 0
  if (c->ix == 0) {
    int k;

    // find a free cell in the virtualColormap[] table
    // skip cell [0] which is devoted to the unknown color
    for (k = 1; k < virtualColormapSize; k++)
      // NB: virtualColormap[k] == 0 is a valid color
      if (virtualColormap[k] == (unsigned long)-1) {c->setIx(k); break;}

    if (c->ix == 0) {  // cell not found => increase table size
      virtualColormapSize += VIRTUAL_COLORMAP_DEFAULT_SIZE;
      virtualColormap = 
	(Pixel*)realloc(virtualColormap, sizeof(Pixel));
      if (!virtualColormap) {
	uerror("realizeColor", UError::NO_MEMORY);
	return false;
      }
      
      // rajouter le nouvel index et initialiser la suite de la table a -1
      // (sinon les futur recherches de cellules planteront)
      c->setIx(virtualColormapSize - VIRTUAL_COLORMAP_DEFAULT_SIZE);
      for (k = c->ix +1; k < virtualColormapSize; k++) 
	virtualColormap[k] = (unsigned long)-1;
    }
  }
  virtualColormap[c->ix] = xcolor.pixel;
  return true;
}


/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

UNatDisp::UNatDisp(UDisp *d) {
  disp = d;
  xdisplay = null;
  xscreen = null;
  xvisual = null;
  xcmap = None;
  xwin = None;
  xwidget = null;
  depth = 0;
  sharedGraph = null;
  clientGraph = null;
  virtualColormap = null; virtualColormapSize = 0;
  fontFamilyMaps = null; fontFamilyCount = 0;
  //obs: unknownPixmap = null;
  cursorMap = null; cursorMapSize = 0;
  defaultPix = None;
}

UNatDisp::~UNatDisp() {
  if (xdisplay && defaultPix != None)
    XFreePixmap(xdisplay, defaultPix);
}

/* ==================================================== ======== ======= */

UX_Window UNatDisp::getXRootWindow() const {
  return RootWindowOfScreen(xscreen);
}
int UNatDisp::getScreenWidth() const {
  return WidthOfScreen(xscreen);
}
int UNatDisp::getScreenHeight() const {
  return HeightOfScreen(xscreen);
}
int UNatDisp::getScreenDefaultDepth() const {
  return DefaultDepthOfScreen(xscreen);
}

// the Unknown Pixmap for *this* Display
/*
UNatPix* UNatDisp::getUnknownPixmap() {
  if (!unknownPixmap)  // NB: on pourrait aussi essayer de les partager
    unknownPixmap = UNatPix::createPix(this, null,
				       UnknownNatPixData,
				       UnknownNatPixWidth,
				       UnknownNatPixHeight,
				       &UColor::black, &UColor::white);
  return unknownPixmap;
}
*/
/* ==================================================== ======== ======= */
// return the closest available depth

int UNatDisp::matchVisualProps(int visual_class, int depth_hint, int scr_no) {
  XVisualInfo vi;
  return matchVisualProps(&vi, visual_class, depth_hint, scr_no);
}

int UNatDisp::matchVisualProps(UX_VisualInfo pvi, 
			       int visual_class, int depth_hint, int scr_no) {
  if (scr_no >= ScreenCount(xdisplay)) {
    uerror("UNatDisp::matchVisualProps", UError::INVALID_X_SCREEN_NUMBER);
    return 0;
  }
  else if (XMatchVisualInfo(xdisplay, scr_no, depth_hint, visual_class, pvi) != 0)
    return depth_hint;   // exact match OK
  else {
    // find closest available depth
    int count, possible_d = 0;
    int *tab = XListDepths(xdisplay, scr_no, &count);
    if (tab) {
      for (int k = 0; k < count; k++)
	if (XMatchVisualInfo(xdisplay, scr_no, tab[k], visual_class, pvi) != 0) {
	  if (possible_d == 0)
	    possible_d = tab[k];
	  else if (abs(tab[k] - depth_hint) < abs(possible_d - depth_hint))
	    possible_d = tab[k];
	}
      XFree(tab);
    }
    return possible_d;
  }
}

int UNatDisp::setVisualProps(int visual_class, int depth_hint, int scr_no,
			     UX_Colormap cmap) {
  XVisualInfo vi;
  depth_hint = matchVisualProps(&vi, visual_class, depth_hint, scr_no);

  if (depth_hint == 0) return 0;   //appropriate visual not found
  else return setVisualProps(&vi, cmap);
}


int UNatDisp::setVisualProps(UX_VisualInfo pvi, UX_Colormap c) {
  if (pvi->screen >= ScreenCount(xdisplay)) {
    uerror("UNatDisp::setVisualProps", UError::INVALID_X_SCREEN_NUMBER);
    return 0;
  }
  depth   = pvi->depth;
  xvisual = pvi->visual;
  xscreen = ScreenOfDisplay(xdisplay, pvi->screen);

  if (xcmap != None) XFreeColormap(xdisplay, xcmap);
  if (c != None) xcmap = c;
  //RootWindowOfScreen(natdisp->getXScreen()), 
  else xcmap = XCreateColormap(xdisplay, RootWindow(xdisplay, pvi->screen),
			       pvi->visual, AllocNone);
  if (xcmap == None)
    uwarning("UNatDisp::setVisualProps", "Could not create Colormap");

  return depth;
}

/* ==================================================== ======== ======= */

void UNatDisp::realize() {

  // si xwidget deja defini c'est : 
  // -- soit qu'on a deja appele cette fct
  // -- soit que xwidget a ete fourni comme tel a l'intialisation
  //    (ie. qu'il a ete cree par une autre appli)

  if (!xwidget) {

  Arg args[10] = {
    // evite d'ouvrir une fenetre merdeuse au prealable
    {(char*)XtNmappedWhenManaged, (XtArgVal)False},

    // le prog peut changer la taille de la window
    {(char*)XtNallowShellResize, (XtArgVal)True},

    // accepter les entrees clavier
    {(char*)XtNinput, (XtArgVal)True},	

    // pour eviter pbms ouverture de taille (0,0)
    {(char*)XtNwidth,  (XtArgVal)10},
    {(char*)XtNheight, (XtArgVal)10},

    // transparent background
    {(char*)XtNbackgroundPixmap, (XtArgVal)None}
  };

  Cardinal acount = 6; //!!!
  if (xvisual != null) {
    XtSetArg(args[acount], (char*)XtNvisual, xvisual); acount++;
  }
  if (xcmap != None) {
    XtSetArg(args[acount], (char*)XtNcolormap, xcmap); acount++;
  }
  if (depth != 0) {
    XtSetArg(args[acount], (char*)XtNdepth, depth); acount++;
  }

  xwidget = XtAppCreateShell(NULL,  //application_name;
			     NULL,  //application_class,
			     applicationShellWidgetClass,
			     xdisplay, 
			     args, acount);
  if (!xwidget) {
    uerror("!UNatDisp::realize", UError::CANT_CREATE_X_WINDOW);
    return;;
  }

  XtRealizeWidget(xwidget);
  xwin = XtWindow(xwidget);

  if (xwin == None) {
    uerror("!UNatDisp::realize", UError::CANT_CREATE_X_WINDOW);
    return;
  }

  //NB: probablement deja fait auparavant (selon fcts appelees)
  xdisplay = XtDisplay(xwidget);
  xscreen  = XtScreen(xwidget);

  if (!xdisplay || !xscreen) {
    uerror("!UNatDisp::realize", 
	   "Can't display GUI: null X display or null screen");
    return;
  }

  // Note: the visual can possibly be null (in which case it should not be given
  // as an arg. to certain functions such as, for instance, Xpm functions
  XtVaGetValues(xwidget, 
		XtNdepth,    &depth,
		XtNcolormap, &xcmap, 
		XtNvisual,   &xvisual, 
		NULL);
  if (xcmap == None) uwarning("UNatDisp::realize","null colormap");
  }

  // TOUJOURS FAIRE CE QUI SUIT!! (meme si create_widget == false)

  //default Pixmap with same depth as UNatDisp
  //(necessaire pour XCreateWindow !)
  defaultPix = XCreatePixmap(xdisplay, xwin, 10, 10, depth);

  atoms.PRIMARY_SELECTION   = XA_PRIMARY;
  atoms.SECONDARY_SELECTION = XA_SECONDARY;
  atoms.UBIT_SELECTION   = XInternAtom(xdisplay, "_UBIT_SELECTION", False);
  atoms.WM_PROTOCOLS     = XInternAtom(xdisplay, "WM_PROTOCOLS", False);
  atoms.WM_DELETE_WINDOW = XInternAtom(xdisplay, "WM_DELETE_WINDOW", False);
  atoms.WM_TAKE_FOCUS    = XInternAtom(xdisplay, "WM_TAKE_FOCUS", False);

  // used by UWinGraph objects for toolkit drawing
  // Ce natgraph sera partagee par tous les UWinGraph relatifs a ce UDisp
  sharedGraph = new UNatGraph(this);

  // used by UGraph objects for appli. drawing
  clientGraph = new UNatGraph(this);

  // DEFAULT FONTS
  // !NB: ATTENTION: xdisp DOIT AVOIR ETE INITALISES !
  // ces 4 fontes doivent obligatoirement etre realisees
  realizeFont(&UFont::standard);
  realizeFont(&UFont::normal);  //ex plain
  realizeFont(&UFont::bold);
  realizeFont(&UFont::italic);

  // DEFAULT COLORS
  // !NB: ATTENTION: xdisp et xcmap DOIVENT AVOIR ETE INITALISES !
  int k;
  virtualColormapSize = VIRTUAL_COLORMAP_DEFAULT_SIZE;
  virtualColormap = 
    (Pixel*)malloc(virtualColormapSize * sizeof(Pixel));

  // initialiser la table a -1 (sinon allocations de cellules planteront)
  for (k = 0; k < virtualColormapSize; k++)
    virtualColormap[k] = (unsigned long)-1;

  realizeColor(&UColor::white);
  UBgcolor::white.set(UColor::white);
  // ca sera la valeur par defaut !!! @Elc 16 june
  virtualColormap[0] = virtualColormap[UColor::white.ix];

  realizeColor(&UColor::black);
  UBgcolor::black.set(UColor::black);

  realizeColor(&UColor::disabled);

  realizeColor(&UColor::grey);
  UBgcolor::grey.set(UColor::grey);

  realizeColor(&UColor::lightgrey);
  UBgcolor::lightgrey.set(UColor::lightgrey);

  realizeColor(&UColor::darkgrey);
  UBgcolor::darkgrey.set(UColor::darkgrey);

  realizeColor(&UColor::navy);
  UBgcolor::navy.set(UColor::navy);

  realizeColor(&UColor::blue);
  UBgcolor::blue.set(UColor::blue);

  realizeColor(&UColor::red);
  UBgcolor::red.set(UColor::red);

  realizeColor(&UColor::green);
  UBgcolor::green.set(UColor::green);

  realizeColor(&UColor::yellow);
  UBgcolor::yellow.set(UColor::yellow);

  realizeColor(&UColor::orange);
  UBgcolor::orange.set(UColor::orange);

  realizeColor(&UColor::wheat);
  UBgcolor::wheat.set(UColor::wheat);

  realizeColor(&UColor::teal);
  UBgcolor::teal.set(UColor::teal);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
//!ATT ces fcts doivent etre appellee sur des Button events

u_bool UAppli::ownServerSelection(UEvent& e) {
  XButtonEvent *xev = &(e.getXEvent()->xbutton);

  if (xev->type != ButtonPress && xev->type != ButtonRelease) {
    uwarning("UNatDisp::ownXSelection", "Wrong event type");
    return false;
  }
  Atom selection = XA_PRIMARY;
  Window owner = xev->window;

  XSetSelectionOwner(xev->display, selection, owner, xev->time);
  return (XGetSelectionOwner(xev->display, selection) == owner);
}

void UAppli::insertServerSelection(UEvent& e, UStr *str, int pos) {
  XButtonEvent *xev = &(e.getXEvent()->xbutton);

  if (xev->type != ButtonPress && xev->type != ButtonRelease) {
    uwarning("UNatDisp::requestPrimarySelection", "Wrong event type");
    return;
  }
  Atom selection = XA_PRIMARY;
  Atom target    = XA_STRING;
  Atom property  = getNatDisp()->atoms.UBIT_SELECTION;

  XConvertSelection(xev->display, selection, target, property,
		    xev->window, xev->time);

  //ATT: pbm si str est detruite entre-temps : il faudrait bloquer l'appli !!
  server_selection_str = str;
  server_selection_pos = pos;

  // ceci entraine la generation d'un SelectionNotify event par le
  // serveur si tout marche bien
  // ce SelectionNotify event est ensuite recu par configureEH qui
  // a alors a charge de realiser l'operation d'insertion
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UNatAppli::UNatAppli(UAppli *app, int *argc, char *argv[]) : UNatDisp(app) {

  status = -1;  //application status
  XtToolkitInitialize();

  // chercher si option -display dans ligne de commande
  for (int k = 1; k < *argc; k++) 
    if (strcmp(argv[k], "-display") == 0 && k+1 < *argc) {
      app->display_name = strdup(argv[k+1]);
      break;
    }
  
  xdisplay = XOpenDisplay(app->display_name);
  if (!xdisplay) {
    uerror("!UNatAppli::UNatAppli", UError::CANT_OPEN_X_CONNECTION);
    exit(1);
  }
  // ecran par defaut du Display
  xscreen = DefaultScreenOfDisplay(xdisplay);
  xcmap = DefaultColormapOfScreen(xscreen);
  xcontext = XtCreateApplicationContext();

  XtDisplayInitialize(xcontext, xdisplay, 
		      app->appli_name,
		      app->style_name,
		      NULL, 0,       //options, num_options, 
		      argc, argv);
}

UNatAppli::~UNatAppli() {
  if (xdisplay) XCloseDisplay(xdisplay);
  xdisplay = null;
}

u_bool UNatAppli::isRealized() const {
  return (xcontext != null && xwidget != null);
}

void UNatAppli::realize() {
  realize(null, null);
}

//NB: argc, argv can be null
//
void UNatAppli::realize(int *argc, char *argv[]) {
  if (!xcontext) {
    int no_argc = 0;
    xcontext = XtCreateApplicationContext();
    XtDisplayInitialize(xcontext, xdisplay, 
			getAppli()->appli_name,
			getAppli()->style_name,
			NULL, 0, //options, num_options, 
			(argc ? argc : &no_argc), NULL); //args
  }
  //BUGGY!  if (!xwidget) UNatDisp::realize()
  UNatDisp::realize();
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

void UNatAppli::quit(int _status) {
  status = _status;
}

int UNatAppli::mainLoop() {
  //EX: XtAppMainLoop(xcontext);
  XEvent xev;

  while (status < 0) {
    //NB: cette fct traite les inputs et les timers
    XtAppNextEvent(xcontext, &xev);

    //NB: s'il y avait plusieurs Displays if faudrait appeler 
    //cette fonction pour chacun d'entre eux
    dispatchEvents(&xev);
  }
  return status;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

UNatTimer::UNatTimer(UNatAppli *natapp, UTimer *t) {
  timer = t;
  ////xappc = null; A REVOIR!!!
  ////xtid = null;A REVOIR!!!
  // dans cet ordre!
  ////if (!natapp || !(xappc = natapp->getXAppContext()))
  ////  uerror("UTimer", "UAppli argument is null or unrealized");
}

u_bool UNatTimer::run(int duration) {
  /*
  ///  if (!xappc) {
  ////  uerror("UTimer", "The Timer must be realized before using it!");
  ////  return false;
  //// }
  else if (xtid) {
    uerror("UTimer", "The Timer is already running");
    return true;
  }
  xtid = XtAppAddTimeOut(xappc, duration, XTimeOut, (XtPointer)this);
  return (xtid != null);
  */
  return 1;
}


void UNatTimer::stop() {
  /*
  if (!xappc || !xtid)
    uerror("UTimer::start", "UAppli or Timer not realized");
  else XtRemoveTimeOut(xtid);
  xtid = null;
  */
}

  
void UNatTimer::XTimeOut(XtPointer timer_ctx, XtIntervalId *id) {
  /*
  UNatTimer *tc = (UNatTimer*)timer_ctx;
  int next_duration = tc->timer->step();

  if (next_duration > 0)
    // re-enregistre le TimeOut (car automatiquement enleve)
    tc->xtid = XtAppAddTimeOut(tc->xappc, next_duration,
			       XTimeOut, (XtPointer)tc);
*/
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
