# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Handle meta-data about project: name, version."""


import logging
import os

import unperishlib


class ProjectInfo(unperishlib.Operation):

    """Add project info (name, version) options."""
    
    name = "project-info"
    
    def add_options(self, parser):
        parser.add_public_option("--project-name", metavar="NAME", 
                                 help="Project name.")
        parser.add_public_option("--project-version", metavar="VERSION",
                                 help="Project version.")


class ProjectInfoFromDistutils(unperishlib.Operation):

    """Determine project name and version from distutils setup.py."""

    name = "project-info-from-setup.py"
    
    provided_options = ["project_name", "project_version"]

    def add_options(self, parser):
        parser.add_public_option("--distutils",
                                 help="""\
Set the name of the Python distutils script to use. Default: %default.""")
        parser.set_defaults(distutils="setup.py")

    def do_it(self, options):
        distutils = options.distutils
        if distutils and os.path.basename(distutils) == distutils:
                distutils = os.path.join(options.directory, distutils)
        if distutils and os.path.exists(distutils):
            stdout = unperishlib.run(["python", distutils,
                                      "--name", "--version"])
            lines = stdout.splitlines()
            if len(lines) == 2:
                options.project_name = lines[0]
                options.project_version = lines[1]
        else:
            logging.debug("%s: cannot set project info" % self.name)
