# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Meta plugins for Unperish."""


import sys

import unperishlib


class ListOperations(unperishlib.Operation):

    """List all operations to the standard output.
    
    Write to the standard output all operations and their description,
    if --long-list has been used.
    
    """
    
    name = "list"
    
    def __init__(self):
        self.output = sys.stdout
    
    def add_options(self, parser):
        parser.add_public_option("--long-list", action="store_true",
                                 help="""\
Show descriptions of operations as well as their names in
the 'list' operation.""")

    def set_application(self, app):
        self.app = app

    def set_output(self, f):
        """Set file where we should output.
        
        This is used by unit testing.
        
        """
        self.output = f
        
    def remove_indent(self, lines):
        if lines:
            indent = min([len(line) - len(line.lstrip()) for line in lines])
            return [line[indent:] for line in lines]
        else:
            return []
    
    def add_indent(self, lines, amount):
        return ["%s%s" % (" "* amount, line) for line in lines]

    def do_it(self, options):
        opers = self.app.get_plugin_manager().get_operations()
        opers = [y for x, y in sorted((z.get_name(), z) for z in opers)]
        for oper in opers:
            self.output.write("%s\n" % oper.get_name())
            if oper.__doc__:
                lines = oper.__doc__.splitlines()
                short = lines[0]
                long = lines[1:]
                if options.long_list:
                    self.output.write("%s\n" % short)
                    if long:
                        long = self.remove_indent(long)
                        long = self.add_indent(long, 4)
                        self.output.write("%s\n" % "\n".join(long).rstrip())
            if options.long_list:
                self.output.write("\n")
