/*
    Copyright (C) 2001-2002  bjk <bjk@arbornet.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include "common.h"
#include "login.h"

static void lastinfo(INFO *info, char *str)
{
    int i = 0;
    char *tmp, *buf;

    tmp = str;

    while ((buf = strtok(tmp, ",")) != NULL) {
	switch (i) {
	    case 0:
		strncpy(info->lastinfo->tty, buf, 
			sizeof(info->lastinfo->tty));
		break;
	    case 1:
		strncpy(info->lastinfo->host, buf, 
			sizeof(info->lastinfo->host));
		break;
	    case 2:
		strncpy(info->lastinfo->time, buf, 
			sizeof(info->lastinfo->time));
		break;
	    default:
		break;
	}

	tmp = NULL;
	i++;
    }

    return;
}

static char *idle(INFO *info)
{
    char *str, filename[FILENAME_MAX];
    int count, n;
    time_t now, blah;
    struct stat mystat;
    UTMP utmp;

    if (!utmpfd) {
#ifdef HAVE_UTMPX_H
        if ((utmpfd = open(_PATH_UTMPX, O_RDONLY)) < 0) {
	    perror(_PATH_UTMPX);
#else
        if ((utmpfd = open(_PATH_UTMP, O_RDONLY)) < 0) {
	    perror(_PATH_UTMP);
#endif
	    return NULL;
        }
    }

    if (lseek(utmpfd, 0, SEEK_SET) == -1) {
#ifdef HAVE_UTMPX_H
	perror(_PATH_UTMPX);
#else
	perror(_PATH_UTMP);
#endif
	return NULL;
    }

    while ((count = read(utmpfd, &utmp, sizeof(UTMP))) == sizeof(UTMP)) {
#ifdef HAVE_UTMPX_H
	if (strncmp(info->passwd->pw_name, utmp.ut_user,
		    sizeof(utmp.ut_user)) == 0)
#else
	if (strncmp(info->passwd->pw_name, utmp.ut_name,
		    sizeof(utmp.ut_name)) == 0)
#endif
	    break;
    }

    snprintf(filename, sizeof(filename), "%s/%s", _PATH_DEV, utmp.ut_line);

    if (stat(filename, &mystat) == -1) 
	return UNKNOWN;

#ifdef HAVE_UTMPX_H
    if (utmp.ut_tv.tv_sec > mystat.st_atime)
#else
    if (utmp.ut_time > mystat.st_atime)
#endif
	str = "0";
    else {
	time(&now);
	blah = mystat.st_atime;

#ifdef HAVE_UTMPX_H
	if (blah < utmp.ut_tv.tv_sec)
	    blah = utmp.ut_tv.tv_sec;
#else
	if (blah < utmp.ut_time)
	    blah = utmp.ut_time;
#endif

	n = now - blah;

	if (n <= 0)
	    str = "0";
	else {
	    n = (now - blah) / 60;

	    if (n <= 0)
	        str = "0";
	    else
	        str = itoa(n);
	}
    }

    return str;
}

static char *lastlogin(INFO *info)
{
    int count;
    long offset;
    char *str = NULL, buf[UT_LINESIZE + UT_HOSTSIZE + TIMEBUF + 2];
    char tmp[TIMEBUF], htmp[UT_HOSTSIZE];
    static int firstrun;
    struct lastlog last;

    buf[0] = tmp[0] = htmp[0] = 0;

    if (!lastfp && firstrun)
	return UNKNOWN;

    if (!lastfp) {
	firstrun = 1;

	if ((lastfp = fopen(_PATH_LASTLOG, "r")) == NULL)
	    return UNKNOWN;
    }

    offset = (long)info->passwd->pw_uid * sizeof(struct lastlog);

    if (fseek(lastfp, offset, SEEK_SET) == -1) {
	perror(_PATH_LASTLOG);
	return UNKNOWN;
    }

    if ((count = fread(&last, sizeof(last), 1, lastfp)) < 1) {
	perror(_PATH_LASTLOG);
	return UNKNOWN;
    }

    if (last.ll_line[0] == 0)
	strncpy(buf, NONE, sizeof(buf));
    else
	strncpy(buf, last.ll_line, sizeof(buf));

    strncat(buf, ",", sizeof(buf));

    strncpy(htmp, last.ll_host, sizeof(htmp));
    htmp[UT_HOSTSIZE - 1] = 0;

    if (htmp[0] == 0)
	strncat(buf, NONE, sizeof(buf));
    else
	strncat(buf, htmp, sizeof(buf));

    strncat(buf, ",", sizeof(buf));

    if (last.ll_time)
	strncat(buf, stamp(last.ll_time, tf), sizeof(buf));
    else
	strncat(buf, NONE, sizeof(buf));

    str = buf;
    return str;
}

static char *utmpfud(INFO *info, int opt)
{
    int count;
    char *str = NULL;
    time_t epoch;
    static int firstrun;
    UTMP utmp;

    if (utmpfd < 0)
	return NULL;

    if (!utmpfd) {
	firstrun = 1;

#ifdef HAVE_UTMPX_H
        if ((utmpfd = open(_PATH_UTMPX, O_RDONLY)) < 0) {
	    perror(_PATH_UTMPX);
#else
        if ((utmpfd = open(_PATH_UTMP, O_RDONLY)) < 0) {
	    perror(_PATH_UTMP);
#endif
	    return NULL;
        }
    }

    if (lseek(utmpfd, 0, SEEK_SET) == -1) {
#ifdef HAVE_UTMPX_H
	perror(_PATH_UTMPX);
#else
	perror(_PATH_UTMP);
#endif
	return NULL;
    }

    while ((count = read(utmpfd, &utmp, sizeof(UTMP))) == sizeof(UTMP)) {
	char tmp[UT_HOSTSIZE];

#ifdef HAVE_UTMPX_H
	if (strncmp(info->passwd->pw_name, utmp.ut_user,
		    sizeof(utmp.ut_user)) == 0) {
#else
	if (strncmp(info->passwd->pw_name, utmp.ut_name,
		    sizeof(utmp.ut_name)) == 0) {
#endif
	    info->login->online = 1;

	    switch (opt) {
		case UTMP_LINE:
		    str = utmp.ut_line;
		    break;
		case UTMP_NAME:
#ifdef HAVE_UTMPX_H
		    str = utmp.ut_user;
#else
		    str = utmp.ut_name;
#endif
		    break;
		case UTMP_HOST:
		    if (utmp.ut_host[0] != 0) {
			tmp[0] = 0;
		        strncpy(tmp, utmp.ut_host, sizeof(tmp));
		        tmp[UT_HOSTSIZE - 1] = 0;
			str = tmp;
		    }
		    else
			str = UNKNOWN;
		    break;
		case UTMP_TIME:
#ifdef HAVE_UTMPX_H
		    str = stamp(utmp.ut_tv.tv_sec, tf);
#else
		    str = stamp(utmp.ut_time, tf);
#endif
		    break;
		case DURATION:
		    time(&epoch);

#ifdef HAVE_UTMPX_H
		    if ((epoch - utmp.ut_tv.tv_sec) > 60)
			str = itoa((epoch - utmp.ut_tv.tv_sec) / 60);
#else
		    if ((epoch - utmp.ut_time) > 60)
			str = itoa((epoch - utmp.ut_time) / 60);
#endif

		    if (!str)
			str = itoa(0);
		    break;
		default:
		    break;
	    }

	    if (str)
		break;
	}
    }
    
    return str;
}

static int msgstat(INFO *info)
{
    char filename[FILENAME_MAX], *tmp;
    struct stat mystat;

    tmp = utmpfud(info, UTMP_LINE);
    snprintf(filename, sizeof(filename), "%s/%s", _PATH_DEV, tmp);

    if (stat(filename, &mystat) == -1) 
	return 0;

    if (mystat.st_mode & S_IWGRP || mystat.st_mode & S_IWOTH)
	return 1;

    return 0;
}

void logininfo(INFO *info, const char *arg)
{
    int i;

    if ((info->login = (LOGIN *) malloc(sizeof(LOGIN))) == NULL) {
	perror("malloc");
	exit(errno);
    }

    bzero(info->login, sizeof(LOGIN));
    (void *) utmpfud(info, 0);

    for (i = 0; i < ARRAYCNT(optspec); i++) {
	char tmp[LINE_MAX];

	if (optspec[i] == 0)
	    break;

	switch (optspec[i]) {
	    case 'q':
		if (info->login->online)
		    strncpy(info->login->idle, idle(info),
			    sizeof(info->login->idle));
		else
		    strncpy(info->login->idle, UNKNOWN,
			    sizeof(info->login->idle));
		break;
	    case 'n':
		strncpy(tmp, lastlogin(info), sizeof(tmp));
		lastinfo(info, tmp);
		break;
	    case 'x':
		if (info->login->online)
		    strncpy(info->login->host, utmpfud(info, UTMP_HOST),
			    sizeof(info->login->host));
		else
		    strncpy(info->login->host, UNKNOWN,
			    sizeof(info->login->host));
		break;
	    case 'y':
		if (info->login->online)
		    strncpy(info->login->tty, utmpfud(info, UTMP_LINE), 
			    sizeof(info->login->tty));
		else
		    strncpy(info->login->tty, UNKNOWN,
			    sizeof(info->login->tty));
		break;
	    case 'e':
		if (info->login->online)
		    info->login->mesgstat = msgstat(info);
		break;
	    case 't':
		if (info->login->online)
		    strncpy(info->login->stamp, utmpfud(info, UTMP_TIME),
			    sizeof(info->login->stamp));
		else
		    strncpy(info->login->stamp, UNKNOWN, 
			    sizeof(info->login->stamp));
		break;
	    case 'b':
		if (info->login->online)
		    strncpy(info->login->duration, utmpfud(info, DURATION),
			    sizeof(info->login->duration));
		else
		    strncpy(info->login->duration, UNKNOWN,
			    sizeof(info->login->duration));
		break;
#ifdef __linux__
#ifdef HAVE_PROCFS
	    case 'C':
		strncpy(info->login->ppid, parentpid(info),
			sizeof(info->login->ppid));
		break;
#endif
#else
	    case 'C':
		strncpy(info->login->ppid, parentpid(info),
			sizeof(info->login->ppid));
		break;
#endif
	    default:
		break;
	}
    }

    return;
}
