/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <stdio.h>
#include <stdlib.h>

#include "renderingAtomic_d3.h"
#include <visu_rendering.h>
#include <visu_data.h>
#include <coreTools/toolFortran.h>

static gboolean read_nattyp_natnom(FILE *flux, GError **error, EndianId endian);
static void free_read_d3();
static int read_d3_file(VisuData *data, FILE *flux, GError **error);


static unsigned int ncheck, mcheck;
static char title[129];
static unsigned int nat, ntypeD3;
static unsigned int *nattyp;
static char **natnom;
static double *x, *y, *z;

static gboolean loadD3(VisuData *data, const gchar* filename,
		       FileFormat *format, int nSet, GError **error);

/******************************************************************************/

RenderingFormatLoad* atomicD3Init()
{
  char *typeD3[] = {"*.d3", (char*)0};
  char *descrD3 = _("Binary format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "Native format of V_Sim.";
  meth->fmt = fileFormatNew(descrD3, typeD3);
  if (!meth->fmt)
    {
      g_error("Can't initialize the rendering atomic method, aborting...\n");
    }
  meth->priority = 10;
  meth->load = loadD3;

  return meth;
}

/******************************************************************************/

static gboolean read_nattyp_natnom(FILE *flux, GError **error, EndianId endian)
{
  unsigned int k;
  gboolean valid;

  valid = toolFortranRead_flag(&ncheck, flux, error, endian);
  if (!valid) return valid;
  valid = toolFortranRead_integer(nattyp, ntypeD3, flux, error, endian, FALSE);
  if (!valid) return valid;
  for(k = 0; k < ntypeD3; k++)
    {
      valid = toolFortranRead_character(natnom[k], 8, flux, error, endian, FALSE);
      if (!valid) return valid;
      natnom[k][8] = '\0';
      g_strchomp(natnom[k]);
    }
  valid = toolFortranRead_flag(&mcheck, flux, error, endian);
  if (!valid) return valid;
  if (ncheck != mcheck)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("WARNING : wrong fortran syntax, flag size unmatched.\n"));
      return FALSE;
    }
  return TRUE;
}

/******************************************************************************/

static gboolean loadD3(VisuData *data, const gchar* filename,
		       FileFormat *format _U_, int nSet _U_, GError **error)
{
  int res;
  FILE *readFrom;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = fopen(filename, "rb");
  if (!readFrom)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("WARNING : impossible to open this file.\n"));
      return FALSE;
    }

  res = read_d3_file(data, readFrom, error);
  fclose(readFrom);

  if (res < 0)
    /* The file is not a D3 file. */
    return FALSE;
  else if (res > 0)
    /* The file is a D3 file but some errors occured. */
    return TRUE;
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}

static int read_d3_file(VisuData *data, FILE *flux, GError **error)
{
  int res;
  unsigned int snat;
  VisuElement *type;
  VisuElement **types;
  gchar *infoUTF8;
  gboolean valid;
  EndianId endian;
  unsigned int nn[2], i, j;
  double boxGeometry[6];
  float geometry[6], xyz[3];

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  natnom = (gchar**)0;
  nattyp = (unsigned int*)0;
  x = (double*)0;
  y = (double*)0;
  z = (double*)0;  

  /* Check the first element to find the endianness.
     The first element is an int equal to 128. We test Littl endian first. */
  DBG_fprintf(stderr, "Atomic D3 : test for endianness.\n");
  valid = toolFortranTest_endianness(128, flux, error, &endian);
  if (!valid)
    return -1;

  /* TITLE */
  DBG_fprintf(stderr, "Atomic D3 : read title.\n");
  valid = toolFortranRead_character(title, 128, flux, error, endian, TRUE);
  if (!valid)
    return -1;
  title[128] = '\0';
  g_strchomp(title);
  DBG_fprintf(stderr, " | '%s'\n", title);

  /* NAT-NTYPE */  
  DBG_fprintf(stderr, "Atomic D3 : read nat and ntype.\n");
  valid = toolFortranRead_integer(nn, 2, flux, error, endian, TRUE);
  if (!valid)
    return 1;
  nat = nn[0];
  ntypeD3 = nn[1];
  DBG_fprintf(stderr, " | %d %d\n", nat, ntypeD3);
               
  /* NATI-NOM */      
  nattyp = g_malloc(ntypeD3 * sizeof(int));
  natnom = g_malloc(ntypeD3 * sizeof(char *));
  for(i=0; i<ntypeD3; i++)
    natnom[i] = g_malloc(9 * sizeof(char));
  valid = read_nattyp_natnom(flux, error, endian);
  if(!valid)
    {
      DBG_fprintf(stderr, "! invalid nattyp or natnom.\n");
      free_read_d3();
      return 1;
    }
  snat = 0;
  for(i=0; i<ntypeD3; i++)
    snat = snat + nattyp[i];
  if(snat != nat)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("WARNING : wrong d3 syntax,"
			     " 'sum of nattyp's not equal to nat'\n"));
      free_read_d3();
      return 1;
    }
         
  /* DXX-DYX-DYY-DZX-DZY-DZZ */
  valid = toolFortranRead_double(boxGeometry, 6, flux, error, endian, TRUE);
  if (!valid)
    {
      DBG_fprintf(stderr, "! invalid box '%s'.\n", (*error)->message);
      free_read_d3();
      return 1;
    }
  for (i = 0; i < 6; i++)
    geometry[i] = (float)boxGeometry[i];
   
  /* X */      
  x = g_malloc(sizeof(double) * nat);
  valid = toolFortranRead_double(x, nat, flux, error, endian, TRUE);
  if (!valid)
    {
      DBG_fprintf(stderr, "! invalid x coordinates '%s'.\n", (*error)->message);
      free_read_d3();
      return 1;
    }
   
  /* Y */      
  y = g_malloc(sizeof(double) * nat);
  valid = toolFortranRead_double(y, nat, flux, error, endian, TRUE);
  if (!valid)
    {
      DBG_fprintf(stderr, "! invalid y coordinates '%s'.\n", (*error)->message);
      free_read_d3();
      return 1;
    }
   
  /* Z */      
  z = g_malloc(sizeof(double) * nat);
  valid = toolFortranRead_double(z, nat, flux, error, endian, TRUE);
  if (!valid)
    {
      DBG_fprintf(stderr, "! invalid z coordinates '%s'.\n", (*error)->message);
      free_read_d3();
      return 1;
    }

  DBG_fprintf(stderr, "Atomic d3 : nat = %d | ntypeD3 = %d\n", nat, ntypeD3);
  for(i = 0; i < ntypeD3; i++)
    DBG_fprintf(stderr, " | natnom %d = '%s', nb %d\n", i, natnom[i], nattyp[i]);
  /* Now we have all the element and the number of atoms per element.
     So we allocate the memory. */
  types = g_malloc(sizeof(VisuElement*) * ntypeD3);
  for(i = 0; i < ntypeD3; i++)
    {
      /* adding nomloc to the hashtable */
      type = visuElementGet_fromName(natnom[i]);
      if (!type)
	{
	  type = visuElementNew_withName(natnom[i]);
	  if (!type)
	    {
	      g_warning("Cannot create a new"
			" type for '%s'\n", natnom[i]);
	      return 1;
	    }
	  res = visuElementAdd(type);
	  if (res)
	    {
	      g_warning("Cannot add new element, maximum has been reached.");
	      return 1;
	    }
	}
      types[i] = type;
    }

  /* Set the commentary. */
  infoUTF8 = g_locale_to_utf8(title, -1, NULL, NULL, NULL);
  if (infoUTF8)
    {
      visuDataSet_fileCommentary(data, infoUTF8, 0);
      g_free(infoUTF8);
    }
  else
    g_warning("Can't convert '%s' to UTF8.\n", title);

  res = visuDataSet_population(data, ntypeD3, nattyp, types);
  g_free(types);
  if (!res)
    {
      g_error("FATAL ERROR! Can't store the nodes in the VisuData object.");
    }

  nat = 0;
  for (i = 0; i < ntypeD3; i++)
    {
      for (j = 0; j < nattyp[i]; j++)
	{
	  xyz[0] = (float)x[nat];
	  xyz[1] = (float)y[nat];
	  xyz[2] = (float)z[nat];
	  visuDataAdd_nodeFromIndex(data, i, xyz);
	  nat++;
	}
    }

  /* We finish with the geometry. */
  visuDataSet_boxGeometry(data, geometry, TRUE);
  
  free_read_d3();
   
  return 0;
}

void free_read_d3()
{
  unsigned int i;
  
  DBG_fprintf(stderr, "Atomic d3 : freeing tmp variables.\n");
  for(i = 0; i < ntypeD3; i++)
    if (natnom[i])
      g_free(natnom[i]);
  if (natnom)
    g_free(natnom);
  if(nattyp)
    g_free(nattyp);
  if (x)
    g_free(x);
  if (y)
    g_free(y);
  if (z)
    g_free(z);
}

/******************************************************************************/ 
/******************************************************************************/
